/****************************************************************************/ 
/*                                                                          */
/*                      ALLIANCE CAO & VLSI CAD                             */
/*                                                                          */
/*    Product : (S)ymbolic (2)to (R)eal translater                          */ 
/*    File    : merge.c                                                     */ 
/*                                                                          */ 
/*    (c) copyright 1992 MASI laboratory CAO & VLSI team                    */ 
/*    All Right Reserved                                                    */ 
/*    Support : e-mail cao-vlsi@masi.ibp.fr                                 */ 
/*                                                                          */ 
/****************************************************************************/
/*                                                                          */
/*                             Merge functions                              */
/*                                                                          */
/****************************************************************************/

#include "generic.h"
#include  MPH_H
#include  MUT_H
#include  RDS_H
#include  RPR_H
#include "rdsacces+.h"
#include "maxima.h"
#include "statistics.h"

/*****************************************************************************
 * name_transfert: given two rectangles, return name to keep after merge
 *                 if the names are different, a warning is done, to prevent
 *                 that the second one is erased  
 ****************************************************************************/

char * name_transfert(desc1, desc2)
rds_rec * desc1, * desc2;
{
  if (desc1->extract_type != RDS_NONE 
  ||  desc2->extract_type != RDS_NONE )
     return NULL;
  if (desc1->u_rec.name == NULL)
     return desc2->u_rec.name;
  if (desc2->u_rec.name == NULL)
     return desc1->u_rec.name;
/*\
 * some segments in different chanels made by scr(1) bloc has 
 * different name but are linked together throught cells, s2r
 * merged then, but can't say it's a error!
 *
 * if (desc1->u_rec.name != desc2->u_rec.name)
 *    fprintf(stderr,
 *            "**s2r error**different nodes merged, %s erased\n",
 *            desc2->u_rec.name);
\*/
  return( desc1->u_rec.name);
}

/***** macro definitions  *****/

#define lgx1 (x1+dx1)
#define lgy1 (yy1+dy1)
#define lgx2 (x2+dx2)
#define lgy2 (y2+dy2)

/******************************************************************************
 * ext_win_segs: given a model, a layer and a window; it returns a pointer to 
 *               a list of rectangles ( formed by the rectangles in the 
 *               specified model in the specified layer intersecting this window
 *               - or touching it).
 *               For consistency purpose these rectangles - which are in the 
 *               list - are extracted from the model and the new list merged is
 *               returned back to the model.
 *****************************************************************************/

rds_rec *ext_win_segs( model, layer_num, wx, wy, dx, dy)
register rds_fig *model;
register char layer_num;
register long wx, wy, dx, dy;
{
typedef rds_rec *rectangle_typ;
register rectangle_typ *rectanglep;
register rds_rec *temp_list;
register rds_rec *temp;
     
  temp_list = NULL;
  for( rectanglep = &(model->layertab[ layer_num]); *rectanglep;) {
    if( are_rect_intersec( (*rectanglep)->x, 
                           (*rectanglep)->y,
                           (*rectanglep)->dx, 
                           (*rectanglep)->dy,
                           wx, wy, dx, dy)) {
       temp = (*rectanglep)->next;
       (*rectanglep)->next = temp_list;
       temp_list = *rectanglep;
       *rectanglep = temp;
    } else 
       rectanglep = & ((*rectanglep)->next);
  }
  return( temp_list);
}

/**********************************************************************
 * inclus: determines if one of the given rectangles is included in the
 *         other and if it is the case:
 *       - it marks the one included "to be removed"
 *       - it marks the one that includes "used"
 *       - it sets modif to 1 to indicate that a modification happened
 *       - if rectangle1(desc1) was the one included it sets modif1 to1    
 *       - if rectangle2(desc2) was the one included it sets modif2 to1    
 **********************************************************************/
inclus(modif,modif1,modif2, desc1, desc2)
unsigned char *modif;  /*indique une modification de la base de donnees*/
unsigned char *modif1; /* indique que le desc 1 a ete elimine */
unsigned char *modif2; /* indique que le desc 2 a ete elimine */
rds_rec *desc1, *desc2;
{
register long x1, dx1, yy1, dy1, x2, dx2, y2, dy2;

  x1 = desc1->x;
  yy1 = desc1->y;
  dx1 = desc1->dx;
  dy1 = desc1->dy;
  x2 = desc2->x;
  y2 = desc2->y;
  dx2 = desc2->dx;
  dy2 = desc2->dy;

  if(x1==x2) {
    if(yy1==y2) {
      if(lgx1==lgx2) {
        if(lgy1==lgy2) {
          /* desc1 equal desc2 : only one must be erased, 
             but, because all flattened rectangles will be erased 
             it's better to delete the not flattened one */
          *modif=1;
          *modif2=1;
          *modif1=1;
          if( ! is_flattenres( desc2)) {
            mark_remove( desc2);
          } else {
            mark_remove( desc1);
          }
          return;
        } 
      }
    }
  }
  if(x1>=x2) {
    if(yy1>=y2) {
      if(lgx1<=lgx2) {
        if(lgy1<=lgy2) {
          if( ! is_flattenres( desc2)) {
            desc2->u_rec.name = name_transfert(desc1, desc2);
          }
          mark_remove( desc1);
          *modif=1;
          *modif1=1;
          return;
        }
      }
    }
  }
  if(x2>=x1) {
    if(y2>=yy1) {
      if(lgx2<=lgx1) {
        if(lgy2<=lgy1) {
          /* desc2 include in desc1 */
          if( ! is_flattenres( desc1)) {
            desc1->u_rec.name = name_transfert(desc1, desc2);
          }
          mark_remove( desc2);
          *modif=1;
          *modif2=1;
        }
      }
    }
  }
  return;
}

/************************************************************************
 * prolong: determines if the 2 given rectangles have a dimension in common
 *          and if it is possible to extend the second one so as to include
 *          the first one. If it is the case:
 *              - the first rectangle (desc1) is marked "to be removed".
 *              - the second rectangle (desc2) is extended and is marked "used".
 *              - modif and modif1 are set to 1.
 *************************************************************************/

prolong(modif,modif1, desc1, desc2)
unsigned char *modif; /* indique une modificatin de la base de donnees */
unsigned char *modif1; /* indique que desc 1 a ete elimine */
rds_rec *desc1, *desc2;
{
register long x1, dx1, yy1, dy1, x2, dx2, y2, dy2;

  x1 = desc1->x;
  yy1 = desc1->y;
  dx1 = desc1->dx;
  dy1 = desc1->dy;
  x2 = desc2->x;
  y2 = desc2->y;
  dx2 = desc2->dx;
  dy2 = desc2->dy;

  if(yy1==y2) {
    if(dy1==dy2) {
      if(x1<x2) {
        if(lgx1>=x2) {
/* cas 1                                                              */
/*            ________________                                        */
/*           | 1    | | 2     |                                       */
/*           |      | |       |                                       */
/*           |______|_|_______|                                       */
          *modif=1;
          *modif1=1;
          desc2->x = x1;
          desc2->dx = lgx2 - x1;
          mark_used( desc2);
          mark_remove( desc1);
          if( ! is_flattenres( desc2)) {
            desc2->u_rec.name = name_transfert(desc1, desc2);
          }
        }
      } else {
        if(x2<x1) {
          if(lgx2>=x1) {
/* cas 2                                                              */
/*            ________________                                        */
/*           | 2    | | 1     |                                       */
/*           |      | |       |                                       */
/*           |______|_|_______|                                       */
            *modif=1;
            *modif1=1;
            desc2->dx = lgx1 - x2;
            mark_used( desc2);
            mark_remove( desc1);
            if( ! is_flattenres( desc2)) {
              desc2->u_rec.name = name_transfert(desc1, desc2);
            }
          }
	}   
      }
    }
    return;
  }
  if(x1==x2) {
    if(dx1==dx2) {
      if(yy1<y2) {
        if(lgy1>=y2) {
/* cas 3                                                              */
/*               ________                                             */
/*              | 2      |                                            */
/*              |        |                                            */
/*              |________|                                            */
/*              |________|                                            */
/*              |        |                                            */
/*              | 1      |                                            */
/*              |________|                                            */
          *modif=1;
          *modif1=1;
          desc2->y = yy1;
          desc2->dy = lgy2 - yy1;
          mark_used( desc2);
          mark_remove( desc1);
          if( ! is_flattenres( desc2)) {
            desc2->u_rec.name = name_transfert(desc1, desc2);
	  }
        }
      } else {
        if(y2<yy1) {
          if(lgy2>=yy1) {
/* cas 4                                                              */
/*               ________                                             */
/*              | 1      |                                            */
/*              |        |                                            */
/*              |________|                                            */
/*              |________|                                            */
/*              |        |                                            */
/*              | 2      |                                            */
/*              |________|                                            */
            *modif=1;
            *modif1=1;
            desc2->dy = lgy1 - y2;
            mark_used( desc2);
            mark_remove( desc1);
            if( ! is_flattenres( desc2)) {
              desc2->u_rec.name = name_transfert(desc1, desc2);
	    }
          }
	}
      }
    }
  }
  return;
}

/*****************************************************************************
 * supegl_dimin: determines if the given dimension is greater than twice the 
 *               resize quantity plus the minimum width of a segment in this
 *               layer ( returns 1) or not (returns 0).
 ****************************************************************************/

supegl_dimin( val, layer_num) /* used in contact */
long val;
char layer_num;
{
LineOversize    *line_oversize;
LineLayerWidth  *line_layer_width;

  line_oversize     = GetOversizeParam (layer_num);
  line_layer_width  = GetLayerWidthParam (layer_num);
  if( val >= (long)(2 * GET_OVERSIZE(line_oversize) 
                    + GET_LAYER_WIDTH(line_layer_width))) {
    return(1);
  } else  {
    return(0);
  }
}

/*************************************************************************
 * contact: determines if the 2 given rectangles intersect each other.
 *          If it is the case:
 *              - it extends one of them and marks it "used".
 *          or  - it creates 1 or 2 new compensating rectangles and adds them 
 *                to the given list.( rectangles are not created unless they 
 *                have the dimensions greater than twice the resize quantity
 *                and only in the first pass (passage).).
 **************************************************************************/ 

contact( temp_list, layer_num, passage, desc1, desc2)
rds_rectanglep_typ *temp_list;
char  layer_num;
char  passage;
rds_rec *desc1, *desc2;
{
char  mbk_type;
register long  x1, dx1, yy1, dy1, x2, dx2, y2, dy2;

  mbk_type = desc1->mbk_type;
  x1 = desc1->x;
  yy1 = desc1->y;
  dx1 = desc1->dx;
  dy1 = desc1->dy;
  x2 = desc2->x;
  y2 = desc2->y;
  dx2 = desc2->dx;
  dy2 = desc2->dy;

  if (x1 >= x2) {
    if (yy1 < y2) {
      if (lgy1 < lgy2) {
        if (lgx1 <= lgx2) {
          if (lgy1 >= y2) {
/* cas 1                                                              */
/*             _________                                              */
/*            |    2    |                                             */
/*            |   ___   |                                             */
/*            |__|___|__|                                             */
/*               | 1 |                                                */
/*               |___|                                                */
            desc1->dy = lgy2 - yy1;
            mark_used( desc1);
            return;
          }
        } 
        else if (passage == 0) {
          if (x1 <= lgx2) {
            if (lgy1 > y2) {
/* cas 2                                                              */
/*           ______________                                           */
/*          | 2        ____|____                                      */
/*          |         |    | 1  |                                     */
/*          |_________|____|    |                                     */
/*                    |         |                                     */
/*                    |_________|                                     */
              if (x1 != lgx2) {
                if (supegl_dimin((lgx2 - x1), layer_num)) {
                  (*temp_list) = rds_rectangle_in(  (*temp_list), 
                                 x1,  yy1, lgx2 - x1,  lgy2 - yy1, 
                                 layer_num, mbk_type, 0, NULL,  RDS_NONE);
		}
	      }
              if (supegl_dimin((lgy1 - y2), layer_num)) {
                (*temp_list) = rds_rectangle_in(  (*temp_list), 
                               x2,  y2, lgx1 - x2,  lgy1 - y2, 
                               layer_num, mbk_type, 0, NULL,  RDS_NONE);
              }
              return;
            }
	  }
	}
      }
    }
    if (yy1 > y2) {
      if (lgx1 <= lgx2) {
        if (yy1 <= lgy2) {
          if (lgy1 > lgy2) {
/* cas 3                                                              */
/*                   _______                                          */
/*                  | 1     |                                         */
/*               ___|_______|____                                     */
/*              | 2 |_______|    |                                    */
/*              |                |                                    */
/*              |________________|                                    */
            desc1->y = y2;
            desc1->dy = lgy1 - y2;
            mark_used( desc1);
            return;
          }
	}
      }
    }
  } else {      /*  x1<x2 */
    if (lgx1 >= x2) {
      if (yy1 < y2) {
        if (passage == 0) {
          if (lgx1 <= lgx2) {
            if (lgy1 > y2) {
              if (lgx1 >= x2) {
                if (lgy1 < lgy2) {
/* cas 4                                                              */
/*                      _____________                                 */
/*                     | 2           |                                */
/*             ________|_______      |                                */
/*            | 1      |_______|_____|                                */
/*            |                |                                      */
/*            |________________|                                      */
                  if (x2 != lgx1) {
                    if (supegl_dimin((lgx1 - x2), layer_num)) {
                      (*temp_list) = rds_rectangle_in(  (*temp_list), 
                                     x2,  yy1, lgx1 - x2,  lgy2 - yy1, 
                                     layer_num, mbk_type, 0, NULL,  RDS_NONE);
                    }
                  }
                  if (supegl_dimin((lgy1 - y2), layer_num)) {
                    (*temp_list) = rds_rectangle_in(  (*temp_list), 
                                   x1,  y2, lgx2 - x1,  lgy1 - y2, 
                                   layer_num, mbk_type, 0, NULL,  RDS_NONE);
		  }
                  return;
                }
	      }
	    }
	  }
	}
      } else {/* yy1>=y2 */
        if (lgx1 >= x2) {
          if (lgx1 < lgx2) {
            if (lgy1 <= lgy2) {
/* cas 5                                                              */
/*                       _____________                                */
/*                      | 2           |                               */
/*               _______|_______      |                               */
/*              | 1     |       |     |                               */
/*              |_______|_______|     |                               */
/*                      |_____________|                               */
              desc1->dx = lgx2 - x1;
              mark_used( desc1);
              return;
            }
	  }
        }
      }
      if (lgx1 >= x2) {
        if (lgx1 < lgx2) {
          if (yy1 <= y2) {
            if (lgy1 >= lgy2) {
/* cas 6                                                              */
/*           _____________                                            */
/*          | 1           |                                           */
/*          |       ______|______                                     */
/*          |      |      |    2 |                                    */
/*          |      |______|______|                                    */
/*          |_____________|                                           */
              desc2->x = x1;
              desc2->dx = lgx2 - x1;
              mark_used( desc2);
              return;
            }
	  }
	}
      }
    }
  }
  if (x1 <= x2) {
    if (lgx1 < lgx2) {
      if (passage == 0) {
        if (lgx1 >= x2) {
          if (yy1 > y2) {
            if (lgy1 > lgy2) {
              if (yy1 <= lgy2) {
/* cas 7                                                              */
/*           ______________                                           */
/*          | 1        ____|____                                      */
/*          |         |    | 2  |                                     */
/*          |_________|____|    |                                     */
/*                    |_________|                                     */
                if (supegl_dimin((lgy2 - yy1), layer_num)) {
                  (*temp_list) = rds_rectangle_in(  (*temp_list), 
                                 x1,  yy1, lgx2 - x1,  lgy2 - yy1, 
                                 layer_num, mbk_type, 0, NULL,  RDS_NONE);
                }
                if (x2 != lgx1) {
                  if (supegl_dimin((lgx1 - x2), layer_num)) {
                    (*temp_list) = rds_rectangle_in(  (*temp_list), 
                                   x2,  y2, lgx1 - x2,  lgy1 - y2, 
                                   layer_num, mbk_type, 0, NULL,  RDS_NONE);
		  }
                  return;
                }
              }
	    }
	  }
	}
      }
    } else { /* x1+dx1>=x2+dx2 */
      if (yy1 < y2) {
        if (lgy1 >= y2) {
          if (lgy1 < lgy2) {
/* cas 8                                                              */
/*                   _________                                        */
/*                  | 2       |                                       */
/*            ______|_________|_______                                */
/*           | 1    |_________|       |                               */
/*           |                        |                               */
/*           |________________________|                               */
            desc2->y = yy1;
            desc2->dy = lgy2 - yy1;
            mark_used( desc2);
            return;
          }
	}
      }
      if (yy1 > y2) {
        if (lgy1 > lgy2) {
          if (yy1 <= lgy2) {
/* cas 9                                                              */
/*             _________                                              */
/*            |    1    |                                             */
/*            |   ___   |                                             */
/*            |__|___|__|                                             */
/*               | 2 |                                                */
/*               |___|                                                */
            desc2->dy = lgy1 - y2;
            mark_used( desc2);
            return;
          }
	}
      }
    }
  }
  if (x1 > x2) {
    if (x1 <= lgx2) {
      if (lgx1 > lgx2) {
        if (yy1 <= y2) {
          if (lgy1 >= lgy2) {
/* cas 10                                                             */
/*                       _____________                                */
/*                      | 1           |                               */
/*               _______|_______      |                               */
/*              | 2     |       |     |                               */
/*              |_______|_______|     |                               */
/*                      |_____________|                               */
            desc2->dx = lgx1 - x2;
            mark_used( desc2);
            return;
          }
	}
        if (yy1 >= y2) {
          if (lgy1 <= lgy2) {
/* cas 11                                                             */
/*           _____________                                            */
/*          | 2           |                                           */
/*          |       ______|______                                     */
/*          |      |      |    1 |                                    */
/*          |      |______|______|                                    */
/*          |_____________|                                           */
            desc1->x = x2;
            desc1->dx = lgx1 - x2;
            mark_used( desc1);
            return;
          }
	}
      }
      if (passage == 0) {
        if (lgx1 >= lgx2) {
          if (yy1 > y2) {
            if (lgy1 > lgy2) {
              if (yy1 <= lgy2) {
/* cas 12                                                             */
/*                      _____________                                 */
/*                     | 1           |                                */
/*             ________|_______      |                                */
/*            | 2      |_______|_____|                                */
/*            |                |                                      */
/*            |________________|                                      */
                if (x1 != lgx2) {
                  if (supegl_dimin((lgx2 - x1), layer_num)) {
                    (*temp_list) = rds_rectangle_in(  (*temp_list), 
                                   x1,  y2, lgx2 - x1,  lgy1 - y2, 
                                   layer_num, mbk_type, 0, NULL,  RDS_NONE);
		  }
		}
                if (supegl_dimin((lgy2 - yy1), layer_num)) {
                  (*temp_list) = rds_rectangle_in(  (*temp_list), 
                                 x2,  yy1, lgx1 - x2, lgy2 - yy1, 
                                 layer_num, mbk_type, 0, NULL,  RDS_NONE);
		}
                return;
              }
	    }
	  }
	}
      }
    }
  return;
  }
} /* END contac() */



/*****************************************************************************
 * rect_list_sup: compares the given rectangle ( given by its coordinates) to 
 *                every rectangle in the given list, using their quantities of 
 *                oversize, and determines if they are intersecting in the 
 *                normal case ( not oversized). If they are, it returns 1, else
 *                it returns 0.
 ******************************************************************************/
rect_list_sup( x, y, dx, dy, list, qty1, qty2)
register long x , y, dx, dy;
rds_rec *list;
register long qty1, qty2;
{
register rds_rec *rectp;   

  for( rectp = list; rectp; rectp = rectp->next) {
    /* 1 is added as intersec is true if the 2 rect. touch which is not our case */
    if( are_rect_intersec( x+qty1+1, y+qty1+1,
                          dx-2*qty1-2, dy-2*qty1-2,
                          rectp->x+qty2,rectp->y+qty2,
                          rectp->dx-2*qty2,rectp->dy-2*qty2)) {
      break;
    }
  }
  if( rectp) {
    return( 1);
  } else {
    return( 0);
  }   
}


/*************************************************************************
 * contact_2: determines if the 2 given rectangles intersect each other.
 *          If it is the case:
 *              - it extends one of them and marks it "used".
 *          or  - it creates 1 or 2 new compensating rectangles and adds them 
 *                to the given list.
 *              - If there is a need for compensating rectangles and they 
 *                cannot be created as they would have a dimension less than
 *                the minimum width permitted for a segment in the given layer:
 *                  + a pair of scotch will be considered and will be compared
 *                    to all rectangles in the complementary layers (given
 *                    by temp_list_s and temp_list_t). 
 *                      * If it does not intersect any of them one rectangle
 *                        of the pair is created and the other will be created 
 *                        implicitly in following passes.
 *                      * If it intersects one of them, another pair will be 
 *                        considered.
 *                  + The other pair will pass the same test.
 *                      * If it does not intersect any of them one rectangle
 *                        of the pair is created and the other will be created 
 *                        implicitly in following passes.
 *                      * If it intersects one of them an error will be 
 *                        signaled.
 *             Remarks:
 *                - Rectangles are created only in the first pass except if
 *                  one or both of them is a scotch.
 *                - If a scotch is created then the variable "modif" is set
 *                  to 1 to permit another pass; so as to consider the 
 *                  possibility of new merges due to the newly created scotch.  
 ******************************************************************************/

void contact_2(temp_list_p, temp_list_s, temp_list_t, layer_num, passage, desc1, desc2, modif)
rds_rectanglep_typ *temp_list_p;
rds_rec            *temp_list_s, *temp_list_t;
char                layer_num;
char                passage;
rds_rec            *desc1, *desc2;
unsigned char      *modif;
{
char  mbk_type;
register long  x1,  dx1,  yy1,  dy1, x2, dx2, y2, dy2;
register long  sx,  sdx,  sy,   sdy;
long           ssx, ssdx, ssy,  ssdy;
char           need_scotch = 0;
long           minwdth_os, qty1,  qty2;
rds_rec        * scotch1, *scotch2;
LineOversize   * line_oversize;
LineLayerWidth * line_layer_width;
LinePost       * line_postreat;

  mbk_type = desc1->mbk_type;
  x1 = desc1->x;
  yy1 = desc1->y;
  dx1 = desc1->dx;
  dy1 = desc1->dy;
  x2 = desc2->x;
  y2 = desc2->y;
  dx2 = desc2->dx;
  dy2 = desc2->dy;

  if (x1 >= x2) {
    if (yy1 < y2) {
      if (lgy1 < lgy2) {
        if (lgx1 <= lgx2) {
          if (lgy1 >= y2) {
/* cas 1                                                              */
/*             _________                                              */
/*            |    2    |                                             */
/*            |   ___   |                                             */
/*            |__|___|__|                                             */
/*               | 1 |                                                */
/*               |___|                                                */
            desc1->dy = lgy2 - yy1;
            mark_used( desc1);
            return;
          }
        } 
        else if ((passage == 0) || (is_scotch(desc1)) || (is_scotch(desc2))) {
          if (x1 <= lgx2) {
            if (lgy1 > y2) {
/* cas 2                                                              */
/*           ______________                                           */
/*          | 2        ____|____                                      */
/*          |         |    | 1  |                                     */
/*          |_________|____|    |                                     */
/*                    |         |                                     */
/*                    |_________|                                     */
              if (x1 != lgx2) {
                if (supegl_dimin((lgx2 - x1), layer_num)) {
                  (*temp_list_p) = rds_rectangle_in( (*temp_list_p), 
                                   x1,  yy1, lgx2 - x1,  lgy2 - yy1, 
                                   layer_num, mbk_type, 0, NULL, RDS_NONE);

                } else {
                  need_scotch = 1;
                }
              }
              if (supegl_dimin((lgy1 - y2), layer_num)) {
                (*temp_list_p) = rds_rectangle_in(  (*temp_list_p), 
                                 x2, y2, lgx1 - x2,  lgy1 - y2, 
                                 layer_num, mbk_type, 0, NULL, RDS_NONE);
                need_scotch = 0;
              }
              if ( need_scotch) {
                line_oversize   = GetOversizeParam (layer_num);
                line_layer_width= GetLayerWidthParam (layer_num);
                line_postreat   = GetPostParam (layer_num);
                qty1            = GET_OVERSIZE( line_oversize);
                if( (qty2 = GET_POST_COMPLEMENTARY (line_postreat)) != -1) {
                  qty2 = GET_OVERSIZE( GetOversizeParam (qty2)); 
                } else {
                  qty2 = 0;
                }
                minwdth_os      = GET_LAYER_WIDTH( line_layer_width) + 2 *qty1;
                STAT_SCTCH_RQRD++;
                sx = lgx2 - minwdth_os;
                sy = lgy1 - minwdth_os;
                sdx = minwdth_os;
                sdy = lgy2 - lgy1 + minwdth_os;
                if ( !rect_list_sup( sx, sy, sdx, sdy,  temp_list_s, qty1, qty2))  {
                   (*temp_list_p) = rds_rectangle_in((*temp_list_p), 
                                    sx,  sy,  sdx,  sdy, 
                                    layer_num, mbk_type, 0, NULL, RDS_NONE);

                    mark_scotch(*temp_list_p);
                    STAT_SCTCH_CRTD++;
                    *modif = 1;
                } else {
                  sx = x2;
                  sy = y2;
                  sdx = x1 - x2 + minwdth_os;
                  sdy = minwdth_os;
                  if ( !rect_list_sup( sx, sy, sdx, sdy,  temp_list_s, qty1, qty2)) {
                    (*temp_list_p) = rds_rectangle_in(  (*temp_list_p), 
                                     sx,  sy,  sdx,  sdy, 
                                     layer_num, mbk_type, 0, NULL, RDS_NONE);
                    mark_scotch(*temp_list_p);
                    STAT_SCTCH_CRTD++;
                    *modif = 1;
                  } else {
                    fprintf(stderr, "***s2r error***, scotch error\n");
                  }
                }
              }
              return;
            }
          }
        }
      }
    }
    if (yy1 > y2) {
      if (lgx1 <= lgx2) {
        if (yy1 <= lgy2) {
          if (lgy1 > lgy2) {
/* cas 3                                                              */
/*                   _______                                          */
/*                  | 1     |                                         */
/*               ___|_______|____                                     */
/*              | 2 |_______|    |                                    */
/*              |                |                                    */
/*              |________________|                                    */
            desc1->y = y2;
            desc1->dy = lgy1 - y2;
            mark_used( desc1);
            return;
          }
        }
      }
    }
  } else {      /*  x1<x2 */
    if (lgx1 >= x2) {
      if (yy1 < y2) {
        if ((passage == 0) || (is_scotch(desc1)) || (is_scotch(desc2))) {
          if (lgx1 <= lgx2) {
            if (lgy1 > y2) {
              if (lgx1 >= x2) {
                if (lgy1 < lgy2) {
/* cas 4                                                              */
/*                      _____________                                 */
/*                     | 2           |                                */
/*             ________|_______      |                                */
/*            | 1      |_______|_____|                                */
/*            |                |                                      */
/*            |________________|                                      */
                  if (x2 != lgx1) {
                    if (supegl_dimin((lgx1 - x2), layer_num)) {
                      (*temp_list_p) = rds_rectangle_in(  (*temp_list_p), 
                                       x2,  yy1, lgx1 - x2,  lgy2 - yy1, 
                                       layer_num, mbk_type, 0, NULL, RDS_NONE);
                    } else {
                      need_scotch = 1;
                    }
                  }
                  if (supegl_dimin((lgy1 - y2), layer_num)) {
                    (*temp_list_p) = rds_rectangle_in(  (*temp_list_p), 
                                     x1,  y2, lgx2 - x1,  lgy1 - y2, 
                                     layer_num, mbk_type, 0, NULL, RDS_NONE);
                    need_scotch = 0;
                  }
                  if ( need_scotch) {
                    line_oversize   = GetOversizeParam (layer_num);
                    line_layer_width= GetLayerWidthParam (layer_num);
                    line_postreat   = GetPostParam (layer_num);
                    qty1  = GET_OVERSIZE( line_oversize);
                    if( (qty2 = GET_POST_COMPLEMENTARY (line_postreat)) != -1) {
                      qty2 = GET_OVERSIZE( GetOversizeParam (qty2)); 
                    } else {
                      qty2 = 0;
                    }
                    minwdth_os      = GET_LAYER_WIDTH( line_layer_width) + 2 * qty1;
                    STAT_SCTCH_RQRD++;
                    sx = lgx1 - minwdth_os;
                    sy = y2;
                    sdx = lgx2 - lgx1 + minwdth_os;
                    sdy = minwdth_os;
                    if (!rect_list_sup( sx, sy, sdx, sdy,  temp_list_s, qty1, qty2)) {
                      (*temp_list_p) = rds_rectangle_in(  (*temp_list_p), 
                                       sx,  sy,  sdx,  sdy, 
                                       layer_num, mbk_type, 0, NULL, RDS_NONE);
                      mark_scotch(*temp_list_p);
                      STAT_SCTCH_CRTD++;
                      *modif = 1;
                    } else {
                      sx = x2;
                      sy = lgy1 - minwdth_os;
                      sdx = minwdth_os;
                      sdy = lgy2 - lgy1 + minwdth_os;
                      if ( !rect_list_sup( sx, sy, sdx, sdy,  temp_list_s, qty1, qty2)) {
                        (*temp_list_p) = rds_rectangle_in(  (*temp_list_p), 
                                         sx,  sy,  sdx,  sdy, 
                                         layer_num, mbk_type, 0, NULL, RDS_NONE);
                        mark_scotch(*temp_list_p);
                        STAT_SCTCH_CRTD++;
                        *modif = 1;
                      } else {
                        fprintf(stderr, "***s2r error***, scotch error\n");
                      }
                    }
                  }
                  return;
                }
              }
            }
          }
        }
      } else { /* yy1>=y2 */
        if (lgx1 >= x2) {
          if (lgx1 < lgx2) {
            if (lgy1 <= lgy2) {
/* cas 5                                                              */
/*                       _____________                                */
/*                      | 2           |                               */
/*               _______|_______      |                               */
/*              | 1     |       |     |                               */
/*              |_______|_______|     |                               */
/*                      |_____________|                               */
              desc1->dx = lgx2 - x1;
              mark_used( desc1);
              return;
            }
          }
        }
      }
      if (lgx1 >= x2) {
        if (lgx1 < lgx2) {
          if (yy1 <= y2) {
            if (lgy1 >= lgy2) {
/* cas 6                                                              */
/*           _____________                                            */
/*          | 1           |                                           */
/*          |       ______|______                                     */
/*          |      |      |    2 |                                    */
/*          |      |______|______|                                    */
/*          |_____________|                                           */
              desc2->x = x1;
              desc2->dx = lgx2 - x1;
              mark_used( desc2);
              return;
            }
          }
        }
      }
    }
  }
  if (x1 <= x2) {
    if (lgx1 < lgx2) {
      if ((passage == 0) || (is_scotch(desc1)) || (is_scotch(desc2))) {
        if (lgx1 >= x2) {
          if (yy1 > y2) {
            if (lgy1 > lgy2) {
              if (yy1 <= lgy2) {
/* cas 7                                                              */
/*           ______________                                           */
/*          | 1        ____|____                                      */
/*          |         |    | 2  |                                     */
/*          |_________|____|    |                                     */
/*                    |_________|                                     */
                if (supegl_dimin((lgy2 - yy1), layer_num)) {
                  (*temp_list_p) = rds_rectangle_in(  (*temp_list_p), 
                                   x1,  yy1, lgx2 - x1,  lgy2 - yy1, 
                                   layer_num, mbk_type, 0, NULL, RDS_NONE);
                } else {
                  need_scotch = 1;
                }
                if (x2 != lgx1) {
                  if (supegl_dimin((lgx1 - x2), layer_num)) {
                    (*temp_list_p) = rds_rectangle_in(  (*temp_list_p), 
                                     x2,  y2, lgx1 - x2,  lgy1 - y2, 
                                     layer_num, mbk_type, 0, NULL, RDS_NONE);
                    need_scotch = 0;
                  }
                }
                if ( need_scotch) {
                  line_oversize   = GetOversizeParam (layer_num);
                  line_layer_width= GetLayerWidthParam (layer_num);
                  line_postreat   = GetPostParam (layer_num);
 
                  qty1            = GET_OVERSIZE( line_oversize);
                  if( (qty2 = GET_POST_COMPLEMENTARY (line_postreat)) != -1) {
                    qty2 = GET_OVERSIZE( GetOversizeParam (qty2)); 
                  } else {
                    qty2 = 0;
                  }
                  minwdth_os      = GET_LAYER_WIDTH( line_layer_width) + 2 * qty1;
                  STAT_SCTCH_RQRD++;
                  sx = lgx1 - minwdth_os;
                  sy = lgy2 - minwdth_os;
                  sdx = minwdth_os;
                  sdy = lgy1 - lgy2 + minwdth_os;
                  if ( !rect_list_sup( sx, sy, sdx, sdy,  temp_list_s, qty1, qty2)) {
                    (*temp_list_p) = rds_rectangle_in(  (*temp_list_p), 
                                     sx, sy, sdx, sdy, 
                                     layer_num, mbk_type, 0, NULL, RDS_NONE);
                    mark_scotch(*temp_list_p);
                    STAT_SCTCH_CRTD++;
                    *modif = 1;
                  } else {
                    sx = x2;
                    sy = y2;
                    sdx = minwdth_os;
                    sdy = yy1 - y2 + minwdth_os;
                    if ( !rect_list_sup( sx, sy, sdx, sdy,  temp_list_s, qty1, qty2)) {
                      (*temp_list_p) = rds_rectangle_in(  (*temp_list_p), 
                                       sx, sy, sdx, sdy, 
                                       layer_num, mbk_type, 0, NULL, RDS_NONE);
                      mark_scotch(*temp_list_p);
                      STAT_SCTCH_CRTD++;
                      *modif = 1;
                    } else {
                      fprintf(stderr, "***s2r error***, scotch error\n");
                    }
                  }
                }
                return;
              }
            }
          }
        }
      }
    } else { /* x1+dx1>=x2+dx2 */
      if (yy1 < y2) {
        if (lgy1 >= y2) {
          if (lgy1 < lgy2) {
/* cas 8                                                              */
/*                   _________                                        */
/*                  | 2       |                                       */
/*            ______|_________|_______                                */
/*           | 1    |_________|       |                               */
/*           |                        |                               */
/*           |________________________|                               */
            desc2->y = yy1;
            desc2->dy = lgy2 - yy1;
            mark_used( desc2);
            return;
          }
        }
      }
      if (yy1 > y2) {
        if (lgy1 > lgy2) {
          if (yy1 <= lgy2) {
/* cas 9                                                              */
/*             _________                                              */
/*            |    1    |                                             */
/*            |   ___   |                                             */
/*            |__|___|__|                                             */
/*               | 2 |                                                */
/*               |___|                                                */
            desc2->dy = lgy1 - y2;
            mark_used( desc2);
            return;
          }
        }
      }
    }
  }
  if (x1 > x2) {
    if (x1 <= lgx2) {
      if (lgx1 > lgx2) {
        if (yy1 <= y2) {
          if (lgy1 >= lgy2) {
/* cas 10                                                             */
/*                       _____________                                */
/*                      | 1           |                               */
/*               _______|_______      |                               */
/*              | 2     |       |     |                               */
/*              |_______|_______|     |                               */
/*                      |_____________|                               */
            desc2->dx = lgx1 - x2;
            mark_used( desc2);
            return;
          }
        }
        if (yy1 >= y2) {
          if (lgy1 <= lgy2) {
/* cas 11                                                             */
/*           _____________                                            */
/*          | 2           |                                           */
/*          |       ______|______                                     */
/*          |      |      |    1 |                                    */
/*          |      |______|______|                                    */
/*          |_____________|                                           */
            desc1->x = x2;
            desc1->dx = lgx1 - x2;
            mark_used( desc1);
            return;
          }
        }
      }
      if ((passage == 0) || (is_scotch(desc1)) || (is_scotch(desc2))) {
        if (lgx1 >= lgx2) {
          if (yy1 > y2) {
            if (lgy1 > lgy2) {
              if (yy1 <= lgy2) {
/* cas 12                                                             */
/*                      _____________                                 */
/*                     | 1           |                                */
/*             ________|_______      |                                */
/*            | 2      |_______|_____|                                */
/*            |                |                                      */
/*            |________________|                                      */
                if (x1 != lgx2) {
                  if (supegl_dimin((lgx2 - x1), layer_num)) {
                    (*temp_list_p) = rds_rectangle_in(  (*temp_list_p), 
                                     x1,  y2, lgx2 - x1,  lgy1 - y2, 
                                     layer_num, mbk_type, 0, NULL, RDS_NONE);
                  } else {
                    need_scotch = 1;
                  }
                }
                if (supegl_dimin((lgy2 - yy1), layer_num)) {
                  (*temp_list_p) = rds_rectangle_in(  (*temp_list_p), 
                                   x2,  yy1, lgx1 - x2,  lgy2 - yy1, 
                                   layer_num, mbk_type, 0, NULL, RDS_NONE);
                  need_scotch = 0;
                }
                if ( need_scotch) {
                  line_oversize   = GetOversizeParam (layer_num);
                  line_layer_width= GetLayerWidthParam (layer_num);
                  line_postreat   = GetPostParam (layer_num);
                
                  qty1            = GET_OVERSIZE( line_oversize);
                  if( (qty2 = GET_POST_COMPLEMENTARY (line_postreat)) != -1) {
                    qty2 = GET_OVERSIZE( GetOversizeParam (qty2)); 
                  } else {
                    qty2 = 0;
                  }
                  minwdth_os      = GET_LAYER_WIDTH( line_layer_width) + 2 * qty1;
                  STAT_SCTCH_RQRD++;
                  sx = lgx2 - minwdth_os;
                  sy = yy1;
                  sdx = lgx1 - lgx2 + minwdth_os;
                  sdy = minwdth_os;
                  if (!rect_list_sup( sx, sy, sdx, sdy,  temp_list_s, qty1, qty2)) {
                    (*temp_list_p) = rds_rectangle_in(  (*temp_list_p), 
                                     sx,  sy,  sdx,  sdy, 
                                     layer_num, mbk_type, 0, NULL, RDS_NONE);
                    mark_scotch(*temp_list_p);
                    STAT_SCTCH_CRTD++;
                    *modif = 1;
                  } else {
                    sx = x2;
                    sy = lgy2 - minwdth_os;
                    sdx = x1 - x2 + minwdth_os;
                    sdy = minwdth_os;
                    if (!rect_list_sup( sx, sy, sdx, sdy,  temp_list_s, qty1, qty2)) {
                      (*temp_list_p) = rds_rectangle_in(  (*temp_list_p), 
                                       sx,  sy,  sdx,  sdy, 
                                       layer_num, mbk_type, 0, NULL, RDS_NONE);
                
                      mark_scotch(*temp_list_p);
                      STAT_SCTCH_CRTD++;
                      *modif = 1;
                    } else {
                      fprintf(stderr, "***s2r error***, scotch error\n");
                    }
                  }
                }
                return;
              }
            }
          }
        }
      }
    }
  }
  return;
} /* END contact_2() */

/*****************************************************************************
 * merge_layer_win: merges the rectangles of the given list - given also their
 *                  layer.
 *                  Merge consists of:
 *                      - eliminating redundancy (inclusion).
 *                      - extending rectangles.
 *                      - creating compensation rectangles.
 ****************************************************************************/
 
void merge_layer_win( temp_list, layer_num)
register rds_rectanglep_typ *temp_list;
register char layer_num;
{
unsigned char modif,modif1,modif2;
register char passage=0;
register rds_rec *desc1 ;
register rds_rec *desc2 ;

  do
  {
    modif=0;
    for( desc1 = (*temp_list); desc1; desc1 = desc1->next) {
      if(is_remove( desc1) 
      || desc1->extract_type == RDS_CON_EXTER
      || desc1->extract_type == RDS_REF_REF
      || desc1->extract_type == RDS_REF_CON)
        continue;

    for( desc2 = desc1->next; desc2; desc2 = desc2->next) {
      if(is_remove( desc2)
      || desc2->extract_type == RDS_CON_EXTER
      || desc2->extract_type == RDS_REF_REF
      || desc2->extract_type == RDS_REF_CON)
        continue;

      modif1=0;
      modif2=0;
      /* appel des tests permettant l'unification  */
      inclus (&modif,&modif1,&modif2, desc1, desc2);

      if( modif1 == 0 ) {
        if( modif2 == 0 ) {
          prolong(&modif,&modif1, desc1, desc2);
          if( modif1 == 0 ) {
            contact(temp_list, layer_num, passage, desc1, desc2);
          } else 
            break;
        } 
      } else 
        break;
    }/* fin for desc2 */
  }/* fin for desc1 */
   /* les creations de masques sont faites dans la premiere passe du while(), passage=0.
      Ceci signifie que un masque genere n'entraine pas de creation de masques. Le fait de
      travailler avec un desc2 toujours > a desc1, elimine toute redondance d'unification
    */
    rds_remove_rectangle( temp_list);  
    passage++;
  } while(modif==1);
}/* END merge_layer_win() */


/*****************************************************************************
 * merge_layer_win_2: merges the rectangles of the given list - given also their
 *                  layer.
 *                  Merge consists of:
 *                      - eliminating redundancy (inclusion).
 *                      - extending rectangles.
 *                      - creating compensation rectangles.
 *                      - creating scotchs.
 ****************************************************************************/
 
void merge_layer_win_2( layer_num, temp_list_p, temp_list_s, temp_list_t)
register char layer_num;
register rds_rectanglep_typ *temp_list_p;
register rds_rec *temp_list_s, *temp_list_t;
{
unsigned char modif,modif1,modif2;
register char passage=0;
register rds_rec *desc1 ;
register rds_rec *desc2 ;

  do{
    modif=0;
    for( desc1 = (*temp_list_p); desc1; desc1 = desc1->next) {
      if(is_remove( desc1)
      || desc1->extract_type == RDS_CON_EXTER
      || desc1->extract_type == RDS_REF_REF
      || desc1->extract_type == RDS_REF_CON)
        continue;

      for( desc2 = desc1->next; desc2; desc2 = desc2->next) {
        if(is_remove( desc2)
        || desc2->extract_type == RDS_CON_EXTER
        || desc2->extract_type == RDS_REF_REF
        || desc2->extract_type == RDS_REF_CON)
          continue;
        modif1=0;
        modif2=0;
    
        /* appel des tests permettant l'unification  */
        inclus(&modif,&modif1,&modif2, desc1, desc2);
        if(modif1==0){
          if(modif2==0){
            prolong(&modif,&modif1, desc1, desc2);
            if(modif1==0)
              contact_2(temp_list_p,temp_list_s,temp_list_t,
                        layer_num,passage,desc1,desc2,&modif);
            else 
              break;
          }
        } else 
          break;
      }/* fin for desc2 */
        if(is_scotch(desc1))
          demark_scotch(desc1);
    }/* fin for desc1 */
    /* les creations de masques sont faites dans la premiere passe du while(), passage=0.
       Ceci signifie que un masque genere n'entraine pas de creation de masques. Le fait de
       travailler avec un desc2 toujours > a desc1, elimine toute redondance d'unification
    */
    rds_remove_rectangle( temp_list_p);  
    passage++;
  } while(modif==1);
}/* END merge_layer_win2 () */


/****************************************************************************
 * calculate_window: calculates the window used in merging the rectangles of
 *                   the specified layer of the specified model.
 *                   The dimensions of this window are put in dx and dy; and
 *                   the number of times it will partition the x dimension
 *                   and the y dimension of the envelop of the model are put
 *                   in nx and ny respectively.
 ***************************************************************************/
 
void calculate_window( model, layer_num, dx, dy, nx, ny)
rds_fig *model;
char layer_num;
long *dx, *dy, *nx, *ny;
{
double tot_num_seg, nb_win, sq_win;
rds_rec *rect;
double temp1, temp2;

   if( is_empty(model,layer_num))
       printf("THere is an error\n\n");
   for( rect=model->layertab[ layer_num], tot_num_seg =0;
        rect;
        rect = rect->next,tot_num_seg++); 
/************** Nasser
   temp1 = tot_num_seg / MRG_EXT_RATIO;
   temp2 = sqrt(temp1);
   if( temp2 > MAX_WIN_SEG)
      temp2 = MAX_WIN_SEG;
   if( temp2 < MIN_WIN_SEG)
      temp2 = MIN_WIN_SEG;
   temp1 = tot_num_seg / temp2;

   temp1++;
   temp2 = sqrt( temp1);
   if( ( temp2 * temp2) < temp1)
      *nx = (long) temp2 + 1;
   else
      *nx = (long) temp2;
   *ny = (long) temp2;
   *dx = model->bboxtab[layer_num]->dx / (*nx) + 1;
   *dy = model->bboxtab[layer_num]->dy / (*ny) + 1;
**************/
/************** Franck */
   if (tot_num_seg == 0 ) {
      *dx = *dy = *nx = *ny = 0;
   } else {
    	nb_win = 2 * sqrt( tot_num_seg );
    	sq_win = sqrt( nb_win);
   	*nx = (long)sq_win + 1;
   	*ny = *nx;
   	*dx = model->bboxtab[layer_num]->dx / (*nx) + 1;
   	*dy = model->bboxtab[layer_num]->dy / (*ny) + 1;
   }
/************** Ludovic 
   if (tot_num_seg == 0 ) {
      *dx = *dy = *nx = *ny = 0;
   } else {
      *dx = *dy = 1+(long) ( sqrt (  model->bboxtab[layer_num]->dx *
                                   model->bboxtab[layer_num]->dy )
                           / sqrt ( sqrt ( tot_num_seg ) ) );
      *nx =  1 + model->bboxtab[layer_num]->dx / *dx ;
      *ny =  1 + model->bboxtab[layer_num]->dy / *dy ;
   }
**************/
}


/******************************************************************************
 * concat_list: concatenates the rectangle list 2 to the rectangle list 1.
 *****************************************************************************/

void concat_list( list1, list2)
rds_rectanglep_typ *list1;
rds_rec *list2;
{
   typedef rds_rec *rds_rectangle_typ;
   rds_rectangle_typ *layer_tail;
   register rds_rec *rectp;

   /* get the tail of list1 */
   for( rectp = (*list1); rectp && rectp->next; rectp = rectp->next);
   if( rectp)
      layer_tail = &(rectp->next);
   else
      layer_tail = list1;
   *layer_tail = list2;
}

/******************************************************************************
 * merge_layer: merges the rectangles of the specified layer of the specified
 *              model.
 *****************************************************************************/

merge_layer(model,layer_num)
rds_fig *model;
char layer_num;
{
long             corn_width,    dx,  dy,  nx,  ny,  wx,  wy,  countx,  county;
rds_rec         *temp_list;
LineRingWidth   *line_ring_width;

  if( model->layertab[ layer_num]) {
    line_ring_width  = GetRingWidthParam (layer_num);
    corn_width       = GET_RING_WIDTH( line_ring_width);
    calculate_window ( model, layer_num, &dx, &dy, &nx, &ny);

    for( countx      = 0; countx < nx; countx++) {
      for( county    = 0; county < ny; county++) {
        wx           = model->bboxtab[layer_num]->x + countx * dx - corn_width;
        wy           = model->bboxtab[layer_num]->y + county * dy - corn_width;

        temp_list    = ext_win_segs(model, layer_num, wx, wy,
                                    dx + 2 * corn_width, dy + 2 * corn_width);

        merge_layer_win(&(temp_list), layer_num);
        concat_list( &(model->layertab[layer_num]), temp_list);
      }
    }
  }
}

/******************************************************************************
 * merge_layer_2: merges the rectangles of the specified layer of the specified
 *              model.
 *              remarks: 
 *              - It differs with merge_layer() in that it calls
 *              merge_layer_win_2() instead of merge_layer_win().
 *              - IT makes another merge in the reverse windowing direction
 *              to compensate changes that might be done by scotchs.
 *****************************************************************************/

merge_layer_2  ( model, layer_num )
rds_fig *model;
char layer_num;
{
long             corn_width,   countx,       county, wx, wy, dx, dy, nx, ny; 
rds_rec         *temp_list_p, *temp_list_s, *temp_list_t;
long             dif_ws_wp;
char             comp_layer;
LineRingWidth   *line_ring_width;
LinePost        *line_postreat;

  if( model->layertab[ layer_num]) {
    line_ring_width  = GetRingWidthParam (layer_num);
    line_postreat    = GetPostParam (layer_num);
    corn_width       = GET_RING_WIDTH( line_ring_width);
    calculate_window( model, layer_num, &dx, &dy, &nx, &ny);
    for(countx = 0; countx < nx; countx++) {
      for( county = 0; county < ny; county++) {
        wx=model->bboxtab[layer_num]->x+countx * dx - corn_width;
        wy=model->bboxtab[layer_num]->y+county * dy - corn_width;
        temp_list_p = ext_win_segs( model, layer_num, wx, wy, 
                                    dx+2*corn_width, dy+2*corn_width);

        if( (comp_layer = GET_POST_COMPLEMENTARY( line_postreat)) != -1) {
           temp_list_s    = model->layertab[comp_layer];
        } else {
           temp_list_s    = NULL;
        }

        merge_layer_win_2( layer_num,& ( temp_list_p ),
                           temp_list_s, temp_list_t);

        concat_list( &(model->layertab[layer_num]), temp_list_p);
      }
    }

    /* another pass beginning with the windows in opposite direction
     * to resolve the problems that might appear if creating of scotchs
     * result in creation of rectangles in a window previously treated
     */

    for ( countx = nx - 1; countx >= 0 ; countx--) {
      for ( county = ny - 1; county >= 0 ; county--) {
        wx = model->bboxtab[layer_num]->x+countx * dx - corn_width;
        wy = model->bboxtab[layer_num]->y+county * dy - corn_width;

        temp_list_p = ext_win_segs ( model, layer_num, wx, wy,
                                     dx+2*corn_width, dy+2*corn_width);

        merge_layer_win_2 ( layer_num,&(temp_list_p),
                            temp_list_s, temp_list_t);

        concat_list ( &(model->layertab[layer_num]), 
                      temp_list_p);

      }
    } 
  } 
} 
