/*******************************************************************************
*                                                                              *
*                      Chaine de CAO & VLSI   Alliance                         *
*                                                                              *
*    Rectangle data structure to cif driver                                    *
*    (c) copyright 1991 Laboratoire MASI equipe CAO & VLSI                     *
*    All rigths reserved                                                       *
*    Support : e-mail cao-vlsi@masi.ibp.fr                                     *
*                                                                              *
*    based upon Franck Wajsburt `mkcif'                                        *
*    Modified by Frederic Petrot & Franck Wajsburt   on : 04/05/92             *
*    Modified by Frederic Petrot                     on : 03/08/92             *
*    Modified by Frederic Petrot                     on : 04/09/92             *
*    Modified by Frederic Petrot & Franck Wajsburt   on : 04/05/92             *
*******************************************************************************/

/* cif driver :
   this piece of code is strongly dedicated to vti cif dialect. All
   documentation concerning this dialect is available with the Alliance cad
   tools documentation. */

#ident "@(#)rds2cif driver 1.02 04/09/92"

#include <values.h>
#include MUT_H
#include MPH_H /* shall be erased sooner or later, here for phcon prototype */
#include RDS_H
#include RPR_H
#include RUT_H
#define DRIVECIF_C
#include "drive_cif.h"
#include "cif.h"

#define RDS2CIF_UNIT(x) \
	((((x) * CIF_UNIT / RDS_UNIT_PARAM) * ds_b) / ds_a)

/*******************************************************************************
* macros                                                                       *
*******************************************************************************/
#define ABS(x) (((x) > 0) ? (x) : -(x))
enum {NO_MODEL, BAD_TRSF, NO_TOP, FILE_OPEN, FILE_CLOSE, NO_CON_NAME};

/*******************************************************************************
* error messages                                                               *
*******************************************************************************/
static void
	error(type, ptr, y)
int type;
void *ptr;
long y;
{
	(void)fflush(stdout);
	(void)fprintf(stderr, "*** rds error ***\ncif driver failed :\n");
	switch (type) {
		case NO_TOP :
			(void)fprintf(stderr, "figure `%s' does not exist in rds!\n", ptr);
			break;
		case NO_MODEL :
			(void)fprintf(stderr, "instance model `%s' does not exist in rds!\n",
											ptr);
			break;
		case BAD_TRSF :
			(void)fprintf(stderr, "unknown instance transformation on `%s'\n",
								ptr);
			break;
		case FILE_OPEN :
			(void)fprintf(stderr, "cif driver could not open file `%s'\n", ptr);
			break;
		case FILE_CLOSE :
			(void)fprintf(stderr, "cif driver could not close file `%s'\n", ptr);
			break;
		case NO_CON_NAME :
			(void)fprintf(stderr, "rds connector (%d, %d) has no mane!\n", 
								(long)ptr, y);
	}
	EXIT(1);
}

/*******************************************************************************
* change rds name to legal cif name                                            *
*******************************************************************************/
static char
	*busname(name)
char *name;
{
char buffer[256], *s, *t;
char one = 1;

	if (!name)
		return NULL;

	s = name;
	t = buffer;
	while (*s) {
		if (*s == ' ')
			if (one) {
				*t++ = '[';
				s++;
				one = 0;
			} else {
				*t++ = ']';
				*t++ = '[';
				s++;
			}
		if (*s == SEPAR && !one) {
			*t++ = ']';
			one = 1;
		}
		*t++ = *s++;
	}
	if (!one)
		*t++ = ']';
	*t = '\0';
	return buffer;
}

/*******************************************************************************
* writing mother figure                                                        *
*******************************************************************************/
static void
	wr_figure(iob, model_num)
FILE *iob;
long model_num;
{
	(void)fprintf(iob, "C%ld;\n", model_num);
}

/*******************************************************************************
* writing instance                                                             *
*******************************************************************************/
static void
	wr_ins(iob, ins_name, model_num, x, y, transf, ds_a, ds_b)
FILE *iob;
char *ins_name;
long model_num;
long x, y, transf, ds_a, ds_b;
{
char cif_transf[100];
long cif_x;
long cif_y;

	cif_x = RDS2CIF_UNIT(x);
	cif_y = RDS2CIF_UNIT(y);

	switch (transf) {
		case RDS_NOSYM :
			(void)sprintf(cif_transf, "T%ld, %ld", cif_x, cif_y);
			break;
		case RDS_SYM_X :
			(void)sprintf(cif_transf, "MX T%ld, %ld", cif_x, cif_y);
			break;
		case RDS_SYM_Y :
			(void)sprintf(cif_transf, "MY T%ld, %ld", cif_x, cif_y);
			break;
		case RDS_SYMXY :
			(void)sprintf(cif_transf, "R-1, 0 T%ld, %ld", cif_x, cif_y);
			break;
		case RDS_ROT_P :
			(void)sprintf(cif_transf, "R0, 1 T%ld, %ld", cif_x, cif_y);
			break;
		case RDS_ROT_M :
			(void)sprintf(cif_transf, "R0, -1 T%ld, %ld", cif_x, cif_y);
			break;
		case RDS_SY_RP :
			(void)sprintf(cif_transf, "MX R0, -1 T%ld, %ld", cif_x, cif_y);
			break;
		case RDS_SY_RM :
			(void)sprintf(cif_transf, "MX R0, 1 T%ld, %ld", cif_x, cif_y);
			break;
		default :
			error(BAD_TRSF, (void *)ins_name, 0);
	}
	(void)fprintf(iob, "4I %s;", ins_name);
	(void)fprintf(iob, "C%ld %s;\n", model_num, cif_transf);
}

/*******************************************************************************
* writing connector                                                            *
*******************************************************************************/
static void
	wr_con(iob, name, index, width, x, y, dx, dy, cif_layer, ds_a, ds_b)
FILE *iob;
char *name;
long index;
long width, x, y, dx, dy, ds_a, ds_b;
char *cif_layer;
{
long  cif_width;
 
	if (name == NULL)
		error(NO_CON_NAME, (void *)x, y);
	if (cif_layer)
		(void)fprintf(iob, "L%s;", cif_layer);
	(void)fprintf(iob, "4X %s %ld ", busname(name), index);
	(void)fprintf(iob, "%ld %ld %ld",
					RDS2CIF_UNIT(x) + RDS2CIF_UNIT(dx) / 2,
					RDS2CIF_UNIT(y) + RDS2CIF_UNIT(dy) / 2,
					RDS2CIF_UNIT(width));
	(void)fprintf(iob, " %s;", name);
	(void)fprintf(iob, "B%ld %ld %ld %ld;\n",
					RDS2CIF_UNIT(ABS(dx)),
					RDS2CIF_UNIT(ABS(dy)),
					RDS2CIF_UNIT(x) + RDS2CIF_UNIT(dx) / 2,
					RDS2CIF_UNIT(y) + RDS2CIF_UNIT(dy) / 2);
}

/*******************************************************************************
* writing segments                                                             *
*******************************************************************************/
static void
	wr_rect(iob, name, x, y, dx, dy, cif_layer, ds_a, ds_b)
FILE *iob;
char *name;
long x, y, dx, dy, ds_a, ds_b;
char *cif_layer;
{
	if (cif_layer)
		(void)fprintf(iob, "L%s;", cif_layer);
	if (name != NULL && *name != '*' && *name != '\0') 
		(void)fprintf(iob, "4N %s %ld %ld;", busname(name), 
					RDS2CIF_UNIT(x) + RDS2CIF_UNIT(dx) / 2,
					RDS2CIF_UNIT(y) + RDS2CIF_UNIT(dy) / 2);
	(void)fprintf(iob, "B%ld %ld %ld %ld;\n",
					RDS2CIF_UNIT(ABS(dx)),
					RDS2CIF_UNIT(ABS(dy)),
					RDS2CIF_UNIT(x) + RDS2CIF_UNIT(dx) / 2,
					RDS2CIF_UNIT(y) + RDS2CIF_UNIT(dy) / 2);
}

/*******************************************************************************
* writing figure header                                                        *
*******************************************************************************/
static void
	wr_header(iob, figname)
FILE *iob;
char *figname;
{
long counter;

	(void)time(&counter);
	(void)fprintf(iob, "(rds to CIF driver version 1.02\n");
	(void)fprintf(iob, "technology %s\n", RDS_TECHNO_NAME);
	(void)fprintf(iob, "%s", ctime(&counter));
	(void)fprintf(iob, "%s\n", figname);
	(void)fprintf(iob, "%s);\n\n", getenv("USER"));
}

/*******************************************************************************
* writing figure tail                                                          *
*******************************************************************************/
static void
	wr_tail(iob, x, y, dx, dy)
FILE *iob;
long x, y, dx, dy;
{
	(void)fprintf(iob, "(AB : %.2f, %.2f %.2f, %.2f in micron); \n", 
							(float)x / RDS_UNIT_PARAM, (float)y / RDS_UNIT_PARAM,
							(float)(x + dx) / RDS_UNIT_PARAM,
							(float)(y + dy) / RDS_UNIT_PARAM);
}

/*******************************************************************************
* writing model header                                                         *
*******************************************************************************/
static void
	wr_model_header(iob, model_name, model_num, ds_a, ds_b)
FILE *iob;
char *model_name;
long model_num, ds_a, ds_b;
{
	(void)fprintf(iob, "DS%ld %ld %ld;\n", model_num, ds_a, ds_b);
	(void)fprintf(iob, "9 %s;\n", model_name);
}

/*******************************************************************************
* writing model abutment box                                                   *
*******************************************************************************/
static void
	wr_model_abox(iob, x, y, dx, dy, ds_a, ds_b)
FILE *iob;
long x, y, dx, dy, ds_a, ds_b;
{
	(void)fprintf(iob, "(AB : %.2f, %.2f %.2f, %.2f in micron);\n", 
							(float)x / RDS_UNIT_PARAM,
							(float)y / RDS_UNIT_PARAM,
							(float)(x + dx) / RDS_UNIT_PARAM,
							(float)(y + dy) / RDS_UNIT_PARAM);
	(void)fprintf(iob, "4A %ld %ld %ld %ld; \n", 
							RDS2CIF_UNIT(x),
							RDS2CIF_UNIT(y),
							RDS2CIF_UNIT(x + dx),
							RDS2CIF_UNIT(y + dy));
}

/*******************************************************************************
* writing model tail                                                           *
*******************************************************************************/
static void
	wr_model_tail(iob)
FILE *iob;
{
	(void)fprintf(iob, "DF;\n\n");
}

/* rds usage :
   a hierarchical list of rds models is build for proper driving, since in cif
   no forward references should be done. */

extern rds_fig *HeadFigureRds;

/*******************************************************************************
* returns the cif number of a rds model, using the sorted hierarchical list    *
*******************************************************************************/
static long
	getcifmodel(head, name)
ptype_list *head;
char *name;
{
ptype_list *ptype;

	for (ptype = head; ptype != NULL; ptype = ptype->NEXT)
		if (((rds_fig *)ptype->DATA)->name == name)
			return ptype->TYPE;
	return 0;
}

/*******************************************************************************
* compute the smallest usable symbol scales for file size minimization         *
*******************************************************************************/
static void
	compute_cif_unit(rds_unit, a, b)
long rds_unit;
long *a, *b;
{
	/* why ds_a, ds_b ?
	   the simple of driving CIF would be to take both as 1, and have on
	   each coordinate a (x * CIF_UNIT)/rds_unit).
	   The only interest here is to shorten the lenght of the numbers in the
	   CIF output.
	   ds_a and ds_b are defined as follow :
	   (ds_a / ds_b) * rds_unit = CIF_UNIT
	   see CIF documentation for details.
	   the coodinates for CIF will be computed like
	   (x * ds_b) / (ds_a * rds_unit).
	   The computation algorithm is equivalent to the one given in the
	   function ComputeRdsUnit, so take a look there for details. */
	if (rds_unit < CIF_UNIT) {
		for (*a = 1, *b = 1; (*a) * rds_unit != CIF_UNIT * (*b); (*b)++) {
			for ((*a)--; (*a) * rds_unit < CIF_UNIT * (*b); (*a)++);
			if ((*a) * rds_unit == CIF_UNIT * (*b))
				break;
		}
	} else {
		for (*a = 1, *b = 1; (*a) * rds_unit != CIF_UNIT * (*b); (*a)++) {
			for ((*b)--; (*a) * rds_unit > CIF_UNIT * (*b); (*b)++);
			if ((*a) * rds_unit == CIF_UNIT * (*b))
				break;
		}
	}
}

/*******************************************************************************
* drives the model on disk properly                                            *
*******************************************************************************/
void
	savecif(top)
char *top;
{
FILE *iob;
ptype_list *model, *list;
rds_fig *f;
long ds_a, ds_b;

	if ((f = GetFigureRds(top)) == NULL)
		error(NO_TOP, (void *)top, 0L);

	compute_cif_unit(RDS_UNIT_PARAM, &ds_a, &ds_b);

	/* ds_b is multiplied by 2 :
	   I need to be able to describe boxes in cif, it means that the
	   center of *each* rectangle must be on a virtual grid step, without
	   loosing precision.
	   This implies that the biggest representable coordinates on a long
	   is divided by two. 
   ds_b *= 2;
      but to make correct computations in rds, coordinates have to be multi
      ple of 2, since physical grid and rds unit have already been multiplied 
      by 2 it's not necessary to do it again for ds_b
   */
	list = model = (ptype_list *)reverse((chain_list *)rds_model_list(f));

	if ((iob = mbkfopen(top, "cif", "w")) == NULL)
		error(FILE_OPEN, (void *)top, 0L);

	wr_header(iob, top);
	while (model) {
		wr_model_header(iob, ((rds_fig *)model->DATA)->name, model->TYPE, ds_a, ds_b);
		if (((rds_fig *)model->DATA)->layertab[RDS_ABOX])
			wr_model_abox(iob,
							((rds_fig *)model->DATA)->layertab[RDS_ABOX]->x, 
							((rds_fig *)model->DATA)->layertab[RDS_ABOX]->y, 
							((rds_fig *)model->DATA)->layertab[RDS_ABOX]->dx, 
							((rds_fig *)model->DATA)->layertab[RDS_ABOX]->dy,
							ds_a, ds_b);
		drive_model(iob, list, ((rds_fig *)model->DATA), ds_a, ds_b);
		wr_model_tail(iob);
		model = model->NEXT;
	}
	wr_figure(iob, getcifmodel(list, f->name));
	if (f->layertab[RDS_ABOX])
		wr_tail(iob, f->layertab[RDS_ABOX]->x,
						f->layertab[RDS_ABOX]->y,
						f->layertab[RDS_ABOX]->dx,
						f->layertab[RDS_ABOX]->dy);
  	(void)fprintf(iob, "E \n");

	if (fclose(iob))
		error(FILE_CLOSE, (void *)top, 0L);
}


static void
	drive_model(iob, model, f, ds_a, ds_b)
FILE *iob;
ptype_list *model;
rds_fig *f;
long ds_a, ds_b;
{
rds_rec *r;
rds_ins *i;
long j, index = 0;
int k;
char *layer;

	for (j = 0; j < RDS_MAX_LAYER; j++) {
		layer = GET_CIF_LAYER(GetCifLayerParam(j));
		if (*layer == '?') /* unknown cif layers for vti */
			continue;
		for (r = f->layertab[j]; r; r = r->next) {
			/* drive connector :
			   put the connector name in vti cif dialect.
			   if the rds doesn't come from mbk, the type may be set but the
			   mbk pointer may be null. */
			if (r->extract_type == RDS_CON_EXTER) {
				/* mbk pointer is null :
				   this situation is ambigous, since the width of the
				   connector may be known only given its orientation.
				   In order not to fail in this case, we chose the widest of
				   dx, dy. */
				if (!(phcon_list *)r->mbk)
      				wr_con(iob, r->u_rec.name, ++index,
							r->dx > r->dy ? r->dx : r->dy,
							r->x, r->y, r->dx, r->dy, layer, ds_a, ds_b);
				else if (((phcon_list *)r->mbk)->ORIENT == NORTH
						|| ((phcon_list *)r->mbk)->ORIENT == SOUTH)
      				wr_con(iob, r->u_rec.name, ++index, r->dx, r->x, r->y,
							r->dx, r->dy, layer, ds_a, ds_b);
				else
      				wr_con(iob, r->u_rec.name, ++index, r->dy, r->x, r->y,
							r->dx, r->dy, layer, ds_a, ds_b);
			} else if (!IsLinkedRectangleRds(r))
				/* node names :
				   rectangles that may have names on them must be driven with
				   it. */
				wr_rect(iob, r->u_rec.name, r->x, r->y, r->dx , r->dy, layer,
							ds_a, ds_b);
			else
				wr_rect(iob, NULL, r->x , r->y , r->dx , r->dy, layer, ds_a, ds_b);
			/* layer choice :
			   the layer is driven only once, since in cif when a layer is defined,
			   all further objects are build with this layer. This is close of the
			   rds representation, so the writting functions print a layer when the
			   parameter is not NULL, nothing if it is. */
			if (layer)
				layer = NULL;
		}
	}
 
	for (i = f->instance; i; i = i->next)
		wr_ins(iob, i->ins_name, getcifmodel(model, i->mod_name),
					i->x, i->y, (long)i->transf, ds_a, ds_b);
}
