/*******************************************************************************
*                                                                              *
* This file is part of the sources of the masi cao-vlsi cad tools              *
*                version 2.0                                                   *
*                                                                              *
*   version 1.0 was develloped by Edmond Janssen & Eric Playe, in june 1991,   *
*               under the supervision of Frederic Petrot                       *
*                                                                              *
*   version 2.0 is basically a version 1.0 extension written in august 1991    *
*               by Frederic Petrot                                             *
*                                                                              *
*   version 2.x supports colors, multipage formats, and so on                  *
*               by Frederic Petrot                                             *
*   Please report the bugs at e-mail : cao-vlsi@masi.ibp.fr                    *
*                                                                              *
*   File : mbk2ps.c                                                            *
*                                                                              *
*******************************************************************************/
#include <ctype.h>
#include <string.h>
#include <time.h>
#include MUT_H
#include MPH_H
#include "ps.h"
#include "mrgltypes.h"

extern phfig_list *WORK_PHFIG;
extern path_link *merge_layer();

/******************************************************************************/
/* Constants and global variables*/
/******************************************************************************/

enum {NO_ERROR, NEED_CELLNAME, BAD_OPTION, UPDATE_MAXLAYER, NO_PS_FILE,
		OPEN_FAILED};

#define MAXLAYER (LAST_LAYER + 1)

static int RTrans = 3;
static int WTrans = 8;
static int SizeVia = 6;

static int Extension[ MAXLAYER ] = {
   /*  NWELL  */   0,
   /*  PWELL  */   0,
   /*  NTIE   */   1,
   /*  PTIE   */   1,
   /*  NDIF   */   1,
   /*  PDIF   */   1,
   /*  NTRANS */   0,
   /*  PTRANS */   0,
   /*  POLY   */   1,
   /*  ALU1   */   1,
   /*  ALU2   */   2,
   /*  ALU3   */   2,
   /*  TPOLY  */   1,
   /*  TALU1  */   1,
   /*  TALU2  */   2,
   /*  TALU3  */   2 };

static char *SegColors[ MAXLAYER ] = {
	"0.823529 0.705882 0.54902",
	"1.0 1.0 0.878431",
	"0.486275 0.988235 0.0",
	"0.703922 0.803922 0.196078",
	"0.0 1.0 0.498039",
	"0.955555 0.988235 0.196078",
	"0.0 0.0 0.0", /* NTRANS used for contact hole, sucks, aint'it */
	"1 0.498039 0.313725", /* PTRANS used for reference, not bad too */
	"1.0 0.0 0.0",
	"0.254902 0.411765 0.902353",
	"0.0 1.0 1.0",
	"1.0 0.854902 0.72549",
	"1.0 0.713725 0.756863",
	"0.690196 0.768627 0.870588",
	"0.878431 1.0 1.0",
	"1.0 0.894118 0.768627"
};

/******************************************************************************/
/* Types                                                                      */
/******************************************************************************/

struct for_request {
   int   Segments,
         Connectors,
         Vias,
         References,
         Instances,
         ConnectorName,
         SegmentName,
         ReferenceName,
         InstanceName,
         ModelName,
         AbutmentBox,
         MergedLayers,
         Silent,
         Rotate,
         Flatten,
         Patterns,
			TeX,
			Color,
         Split;
   int   Layer[MAXLAYER];
   };

typedef struct of_drw_layer {   /* drawable layer */
   rect_link   *LIST;
   int         NUMBER;
   } drw_layer;

typedef struct of_rect {   /* rectangle */
   lambda X0, Y0, X1, Y1;
   } rect;

typedef struct of_drw_via_link {   /* drawable via */
   struct of_drw_via_link *NEXT;
   lambda                  X, Y;
   char                    LAYER1, LAYER2;
   } drw_via_link;

typedef struct of_string_link {
   struct of_string_link   *NEXT;
   char                    *STRING;
   lambda                  X, Y;
   int                     ANGLE;
   } string_link;

/******************************************************************************/
/* Macros                                                                     */
/******************************************************************************/

#define HALF(X)   ((X) >> 1)
#define DBLE(X)   ((X) << 1)
#define MIN(X, Y)   ((X) < (Y) ?  (X) : (Y))
#define MAX(X, Y)   ((X) > (Y) ?  (X) : (Y))

static rect BoundingBox;
static float Scale;

/******************************************************************************/
/* Functions declaration                                                      */
/******************************************************************************/

void getoptions();
void getsegments();
void getconnectors();
void getvias();
void getreferences();
void getinstances();
void getboundingbox();
void delpaths();
void boundingbox();
void frame();
void describ_paths();
void describ_rectangles();
void describ_vias();
void describ_references();
void strokeAB();
void write_strings();
void translate();
float scale();
void usage();
void comments();
void banner();
char *pstime();
char *busname();
void pserror();
void patternsheet();

/******************************************************************************/
/*  Program                                                                   */
/******************************************************************************/

main(ArgCount, ArgValue)
int ArgCount;
char *ArgValue[];
{
char *CellName;
phfig_list *Cell;
struct for_request Request;
char Pattern[MAXLAYER];
drw_layer RectsOfLayer[MAXLAYER];
drw_via_link *DrawableViasList = NULL;
string_link *ListOfStrings = NULL;
rect AbutmentBox, Frame;
path_link *PathsList;
FILE *PS_file;
char SetUp[ BUFSIZ ];
int FontSize;
int Paper;
int Rows;
int Columns;
int Layer;

	WTrans = HALF(WTrans * SCALE_X);
	RTrans = HALF(RTrans * SCALE_X);
	SizeVia = HALF(SizeVia * SCALE_X);
	for (Layer = 0; Layer < MAXLAYER; Layer++) {
		RectsOfLayer[Layer].LIST = NULL;
		Pattern[Layer] = RectsOfLayer[Layer].NUMBER = 0;
		Extension[Layer] = HALF(Extension[Layer] * SCALE_X);
	}

	if (ArgCount == 1)
		usage(ArgValue[0], NEED_CELLNAME);

	getoptions(&Request, Pattern, &Frame, &FontSize, &Paper, &Rows, &Columns,
					ArgCount, ArgValue);

	if (Request.Silent)
		banner();

	mbkenv();
	CellName = namealloc(ArgValue[1]);
	if (!strcmp(CellName, "patternsheet"))
		patternsheet();
	Cell = getphfig(CellName, 'A');

	if (!Request.Flatten)
		rflattenphfig(Cell, YES, YES);

	if (!Request.Rotate) {
	void genDEF_PHFIG();
	void genDEF_AB();
	chain_list *genCOPY_UP_CON();
	void genPLACE();
	void genFLATTEN_PHFIG();
	phcon_list *Connector, *NewConnector;
	char *ConnectorName;

		genDEF_PHFIG("tmp");
		genPLACE(CellName, "instance", ROT_P, 0, 0);
		genDEF_AB(0, 0, 0, 0);
		for (Connector = Cell->PHCON; Connector; Connector = Connector->NEXT) {
			ConnectorName = busname(Connector->NAME);
			genCOPY_UP_CON(Connector->INDEX, ConnectorName, "instance",
							ConnectorName);
		}
		/* index ordering varies with rotation ... so we cheat */
		WORK_PHFIG->PHCON = (phcon_list *)reverse((chain_list *)WORK_PHFIG->PHCON);
		for (NewConnector = WORK_PHFIG->PHCON, Connector = Cell->PHCON;
				NewConnector && Connector;
				NewConnector = NewConnector->NEXT, Connector = Connector->NEXT)
			NewConnector->INDEX = Connector->INDEX;
		/* back to normal */
		genFLATTEN_PHFIG("instance", NO);
		delphfig(Cell);
		Cell = WORK_PHFIG;
		Cell->NAME = CellName;
	}

	BoundingBox.X0 = AbutmentBox.X0 = Cell->XAB1;
	BoundingBox.Y0 = AbutmentBox.Y0 = Cell->YAB1;
	BoundingBox.X1 = AbutmentBox.X1 = Cell->XAB2;
	BoundingBox.Y1 = AbutmentBox.Y1 = Cell->YAB2;

	if (Request.Instances) 
		getinstances(Cell->PHINS, RectsOfLayer,
						Request.InstanceName ? &ListOfStrings : NULL,
						Request.ModelName);
	if (Cell->PHINS)
		Request.MergedLayers = 0;

	if (Request.Segments)
		getsegments(Cell->PHSEG, Request.Layer, RectsOfLayer,
						Request.SegmentName ? &ListOfStrings : NULL);

	if (Request.Vias)
		getvias(Cell->PHVIA, &DrawableViasList, RectsOfLayer, Pattern);

	if (Request.References)
		getreferences(Cell->PHREF, RectsOfLayer,
							Request.ReferenceName ? &ListOfStrings : NULL);

	if (Request.Connectors)
		getconnectors(Cell->PHCON, Request.Layer, RectsOfLayer,
							Request.ConnectorName ? &ListOfStrings : NULL);

	delphfig(Cell);

	if ((PS_file = mbkfopen(CellName, "ps", "w")) == NULL) {
		pserror(OPEN_FAILED);
		PS_file = stdout;
	}
	/* Paper choice :
	   Computes paper size and frame size according to the Row and Columns
	   expected. */
	if (!(Paper % 2)) {
		Frame.X0 += 25;
		Frame.Y0 += 25;
		Frame.X1 *= (1 << ( 2 - (Paper / 2)));
		Frame.Y1 *= (1 << ( 2 - (Paper / 2)));
		Frame.X1 -= 25;
		Frame.Y1 -= 25;
	} else {
		Frame.X0 += 25;
		Frame.Y0 += 25;
		Layer = Frame.Y1 * (1 << ( 2 - ((Paper + 1) / 2)));
		Frame.Y1 = Frame.X1 * (1 << ( 2 - (Paper / 2)));
		Frame.X1 = Layer;
		Frame.X1 -= 25;
		Frame.Y1 -= 25;
	}
	if (Request.Split) {
		Frame.X1 = (Frame.X1 * Rows) - (Rows > 1 ? Frame.X0 : 0);
		Frame.Y1 = (Frame.Y1 * Columns) - (Columns > 1 ? Frame.Y0 : 0);
	}
	header(PS_file, CellName, "mbk2ps V2.0", "MBKtoPostScript", "PatternFont");
	Scale = scale(Request.TeX, Frame);
	if (Request.TeX)
		boundingbox(PS_file, Frame);
	endheader(PS_file);
	if (Request.Patterns) {
		if(!includefont(PS_file, "patternfont.ps"))
			pserror(NO_PS_FILE);
		if(!prologue(PS_file, "mbk2ps.dict"))
			pserror(NO_PS_FILE); ;
	} else if(!prologue(PS_file, "mbk2ps_outline.dict"))
			pserror(NO_PS_FILE);
	sprintf(SetUp, "%.2f setlinewidth\n%u setlinecap\n%u setlinejoin", .1, 2, 0);
	setup(PS_file, SetUp);
	if (!Request.TeX)
		frame(PS_file, CellName, AbutmentBox, Frame);
	translate(PS_file, CellName, Request.TeX, Frame);
	if (Request.Color)
		fputs("\t0.0 0.0 0.0 setrgbcolor\n", PS_file);
	if (Request.AbutmentBox)
		strokeAB(PS_file, AbutmentBox);
	write_strings(PS_file, ListOfStrings, FontSize);
	if ((FontSize = Request.Color ? (int)(3 / Scale) : (int)(15 / Scale)) < 2)
		FontSize = 2;
	fprintf(PS_file, "/PatternFontScale %d def\n", FontSize);
#if 0
	fprintf(PS_file,"/%s {\n\tgsave\n", CellName);
#else
	fprintf(PS_file,"\tgsave\n", CellName);
#endif
	if (Request.Segments || Request.Connectors) {
		for (Layer = 0; Layer < MAXLAYER; Layer++) {
			if (Request.Layer[Layer]) {
				if (Request.MergedLayers) {
			 		PathsList = merge_layer(RectsOfLayer[Layer].LIST,
													RectsOfLayer[Layer].NUMBER);
			 		describ_paths(PS_file, PathsList, Pattern[Layer], 
										Request.Color ? SegColors[Layer] : NULL);
			 		delpaths(&PathsList);
		 		} else /* don't merge layers */
			 		describ_rectangles(PS_file, RectsOfLayer[Layer].LIST,
										Pattern[Layer],
										Request.Color ? SegColors[Layer] : NULL);
			}
		}
	}
	fputs("\tgrestore\n", PS_file);
#if 0
	if (!Request.Split) {
		fputs("\tgrestore\n} def\n", PS_file);
		fprintf(PS_file, "/PatternFontScale %d def\n", FontSize);
		if (!Request.TeX)
			fputs("frame\n", PS_file);
		fprintf(PS_file,"%s\n", CellName);
	} else {
		fputs("\tgrestore\n} def\n", PS_file);
		if (!Request.TeX)
			fputs("frame\n", PS_file);
		fprintf(PS_file, "/PatternFontScale %d def\n", FontSize);
		fputs("/LeftMargin 20 def\n", PS_file);
		fputs("/BottomMargin 20 def\n", PS_file);
		if (!(Paper % 2)) {
			fprintf(PS_file, "/PageWidth %d def\n",
									(1 << ( 2 - (Paper / 2))) * 595 - 40);
			fprintf(PS_file, "/PageHeight %d def\n",
									(1 << ( 2 - (Paper / 2))) * 842 - 40);
		} else {
			fprintf(PS_file, "/PageWidth %d def\n",
									(1 << ( 2 - ((Paper + 1) / 2))) * 842 - 40);
			fprintf(PS_file, "/PageHeight %d def\n",
									(1 << ( 2 - (Paper / 2))) * 595 - 40);
		}
		fprintf(PS_file,"{%s} %d %d splitted_pages\n", CellName, Rows, Columns);
	}
	if (Paper == 3)
		fputs("a3\na3tray\n", PS_file);
#endif
	fputs("showpage\n", PS_file);
	trailer(PS_file);
	fclose(PS_file);

	if (Request.Silent)
		comments(ArgValue[0], ArgValue[1], &Frame);

	return NO_ERROR;
}

/******************************************************************************** function busname                                                             ********************************************************************************/
static char *busname(name)
char *name;
{
static char buffer[255];
char *s, *t;
char one = 1;
 
	if (!name)
		return NULL;
 
	s = name;
	t = buffer;
	while (*s) {
		if (*s == ' ')
			if (one) {
				*t++ = '[';
				s++;
				one = 0;
			} else {
				*t++ = ']';
				*t++ = '[';
				s++;
			}
		if (*s == '_' && !one) { /* was SEPAR and not / */
			*t++ = ']';
			one = 1;
		}
		*t++ = *s++;
	}
	if (!one)
		*t++ = ']';
	*t = '\0';
	return buffer;
}

/******************************************************************************/
/*	Functions definition                                                       */
/******************************************************************************/

void getoptions(Request, Pattern, Frame, FontSize, Paper, Rows, Columns,
						ArgCount, ArgValue)
struct for_request *Request;
char Pattern[];
rect *Frame;
int *FontSize;
int *Paper;
int *Rows;
int *Columns;
int ArgCount;
char *ArgValue[];
{
int WantedLayer;
int ArgNumber;
unsigned int WantedPattern;
unsigned WantedX0, WantedY0, WantedX1, WantedY1;

	/*	Default values */
	(*Request).Layer[NWELL] = 1;
	(*Request).Layer[PWELL] = 0;
	(*Request).Layer[NDIF] = 1;
	(*Request).Layer[PDIF] = 1;
	(*Request).Layer[NTIE] = 1;
	(*Request).Layer[PTIE] = 1;
	(*Request).Layer[NTRANS] = 1;
	(*Request).Layer[PTRANS] = 1;
	(*Request).Layer[POLY] = 1;
	(*Request).Layer[ALU1] = 1;
	(*Request).Layer[ALU2] = 1;
	(*Request).Layer[TPOLY] = 1;
	(*Request).Layer[TALU1] = 1;
	(*Request).Layer[TALU2] = 1;
	(*Request).Instances	= 1;
	(*Request).InstanceName	= 1;
	(*Request).ModelName	= 0;
	(*Request).Segments = 1;
	(*Request).SegmentName = 1;
	(*Request).Connectors = 1;
	(*Request).ConnectorName = 1;
	(*Request).Vias = 1;
	(*Request).References = 1;
	(*Request).ReferenceName = 1;
	(*Request).AbutmentBox = 1;
	(*Request).MergedLayers = 0;
	(*Request).Silent = 1;
	(*Request).Rotate = 1;
	(*Request).Flatten = 1;
	(*Request).Patterns = 1;
	(*Request).TeX = 0;
	(*Request).Color = 0;
	(*Request).Split = 0;
	Pattern[NWELL] = 1;
	Pattern[PWELL] = 1;
	Pattern[NTIE] = 2;
	Pattern[PTIE] = 3;
	Pattern[NDIF] = 2;
	Pattern[PDIF] = 3;
	Pattern[NTRANS] = 12;
	Pattern[PTRANS] = 11;
	Pattern[POLY] = 4;
	Pattern[ALU1] = 5;
	Pattern[ALU2] = 6;
	Pattern[TPOLY] = 12;
	Pattern[TALU1] = 1;
	Pattern[TALU2] = 10;
	(*Frame).X0 = 00;
	(*Frame).Y0 = 00;
	(*Frame).X1 = 595;
	(*Frame).Y1 = 842;
	Scale = 0.0;
	*FontSize = 12;
	*Paper = 4;
	*Rows = 0;
	*Columns = 0;

	for (ArgNumber = 2; ArgNumber < ArgCount; ArgNumber++) {
	char *Arg = ArgValue[ArgNumber];
	char *s;
		if (*Arg == '-') {
			switch (*++Arg) {
				case '1' :
				case '2' :
				case '3' :
				case '4' :
				case '5' :
				case '6' :
				case '7' :
				case '8' :
				case '9' :
				/*
					if (s = strchr(Arg, (int)'x')) {
						(*Request).Split = 1;
						*Rows = atoi(s + 1);
						*s = '\0';
						*Columns = atoi(Arg);
						if (!*Rows || !*Columns)
							usage(ArgValue[0], BAD_OPTION);
					} else
				*/
						usage(ArgValue[0], BAD_OPTION);
					continue;
				case 'a' :
					if (isdigit(*++Arg)) {
						if ((*Paper = (int)(*Arg - '0')) > 4)
							usage(ArgValue[0], BAD_OPTION);
						if (*++Arg == '\0')
							continue;
					} else
						usage(ArgValue[0], BAD_OPTION);
				case 'b' :
					if (*++Arg == '\0')
						(*Request).AbutmentBox = 0;
					else
						usage(ArgValue[0], BAD_OPTION);
					continue;
				case 'c' :
					if (*++Arg == '\0')
						(*Request).Connectors = 0;
					else if (*Arg == 'n')
						(*Request).ConnectorName = 0;
					else if (*Arg == 'o')
						(*Request).Color = 1;
					else
						usage(ArgValue[0], BAD_OPTION);
					continue;
				case 'i' :
					if (*++Arg == '\0')
						(*Request).Instances = 0;
					else if (*Arg == 'n')
						(*Request).InstanceName = 0;
					else
						usage(ArgValue[0], BAD_OPTION);
					continue;
				case 'm' :
					if (*++Arg == '\0')
						(*Request).MergedLayers = 0;
					else if (*Arg == 'n')
						(*Request).ModelName = 0;
					else
						usage(ArgValue[0], BAD_OPTION);
					continue;
				case 'p' :
					if (*++Arg == '\0')
						(*Request).Patterns = 0;
					else
						usage(ArgValue[0], BAD_OPTION);
					continue;
				case 'r' :
					if (*++Arg == '\0')
						(*Request).References = 0;
					else if (*Arg == 'n')
						(*Request).ReferenceName = 0;
					else if (*Arg == 'o' && *++Arg == 't')
						(*Request).Rotate = 0;
					else
						usage(ArgValue[0], BAD_OPTION);
					continue;
				case 's' :
					if (*++Arg == '\0')
						(*Request).Segments = 0;
					else if (*Arg == 'n')
						(*Request).SegmentName = 0;
					else if (*Arg == 'i')
						(*Request).Silent = 0;
					else if (*Arg == 'c') {
						if (++ArgNumber > ArgCount)
							usage(ArgValue[0], BAD_OPTION);
						else if (sscanf(ArgValue[ArgNumber], "%f", &Scale) != 1)
							usage(ArgValue[0], BAD_OPTION);
					} else
						usage(ArgValue[0], BAD_OPTION);
					continue;
				case 'v' :
					if (*++Arg == '\0')
						(*Request).Vias = 0;
					else
						usage(ArgValue[0], BAD_OPTION);
					continue;
				case 't' :
					if (*++Arg == '\0')
						(*Request).TeX = 1;
					else
						usage(ArgValue[0], BAD_OPTION);
					continue;
				case 'l' :
					if (++ArgNumber > ArgCount)
						usage(ArgValue[0], BAD_OPTION);
					if (sscanf(ArgValue[ArgNumber], "%u=%u",
							&WantedLayer, &WantedPattern) == 2) {
						if (WantedLayer < MAXLAYER && WantedPattern < 256) {
							if (WantedPattern)
								Pattern[ WantedLayer ] = WantedPattern;
							else
								(*Request).Layer[ WantedLayer ] = 0;
							continue;
						} else {
							fprintf(stderr, "LayerCode have to be in [0..%u]\n",
										MAXLAYER);
							fputs("PatternCode have to be in [0..255]\n", stderr);
						}
					}
				case 'f' :
					if (*++Arg == '\0') {
						if (++ArgNumber > ArgCount)
							usage(ArgValue[0]);
						if (sscanf(ArgValue[ArgNumber], "%u,%u,%u,%u",
								&WantedX0, &WantedY0, &WantedX1, &WantedY1) == 4){
							(*Frame).X0 = WantedX0;
							(*Frame).Y0 = WantedY0;
							(*Frame).X1 = WantedX1;
							(*Frame).Y1 = WantedY1;
							continue;
						}
					} else if (*Arg == 's') {
						if (++ArgNumber > ArgCount)
							usage(ArgValue[0], BAD_OPTION);
						else if (sscanf(ArgValue[ArgNumber], "%d", FontSize) != 1)
							usage(ArgValue[0], BAD_OPTION);
					} else if (*Arg == 'l')
						(*Request).Flatten = 0;
					else
						usage(ArgValue[0], BAD_OPTION);
					continue;
				default :
					usage(ArgValue[0], BAD_OPTION);
			}
			usage(ArgValue[0], BAD_OPTION);
		}
		usage(ArgValue[0], BAD_OPTION);
	}
}

void usage(Program, ERROR)
char *Program;
int ERROR;
{
	fprintf(stderr,
				"Usage : %s cellname | patternsheet [-b] [-c] [-cn] [-s] [-sn]",
				Program);
	fprintf(stderr, " [-r] [-rn] [-v] [-i] [-in] [-mn] [-m] [-p] [-sc scale]");
	fprintf(stderr, " [-f x0,y0,x1,y1] [-a n] [-si] [-rot] [-fl] [-fs size]");
	fprintf(stderr, " [-co] [-t]\n" /* [<r>x<c>] */);
	exit(ERROR);
}

void banner ()
{
	puts(" ");
	puts("                     @     @  @@@@@ ");
	puts("             @ @@@   @     @ @     @");
	puts("            @ @   @  @@@@  @@      @");
	puts("            @ @   @  @   @ @ @    @");
	puts("            @ @   @   @@@  @  @  @   @@@   @@@ ");
	puts("                                @   @   @ @   ");
	puts("                               @    @@@@   @@ ");
	puts("                              @     @        @");
	puts("                             @@@@@@@@     @@@");
	puts(" ");
	puts("        Alliance CAD system,             mbk2ps 2.3");
	puts("        Tool: mbk2ps, symbolic layout to PostScript");
	puts("        (c) copyright 1990-1993 MASI, CAO-VLSI team");
	puts("        E-mail support:        cao-vlsi@masi.ibp.fr");
	puts(" ");
	fflush(stdout);
}

void comments(Program, File, Frame)
char *Program;
char *File;
rect *Frame;
{
	fprintf(stdout, "\n%s generated the PostScript file %s.ps", Program, File);
	fprintf(stdout, " in the current directory\n");
	fprintf(stdout, "The scale value is %f\n", Scale);
	fprintf(stdout, "The frame coordinates, in pt, are (%d, %d, %d, %d)\n",
				(*Frame).X0, (*Frame).Y0, (*Frame).X1, (*Frame).Y1);
	fprintf(stdout, "The bounding box width and height, in pt, are (%d, %d)\n",
				(int)((BoundingBox.X1 - BoundingBox.X0) * Scale),
				(int)((BoundingBox.Y1 - BoundingBox.Y0) * Scale));
}

void add_string(HeadOfStringsList, String, Angle, X, Y)
string_link **HeadOfStringsList;
char *String;
int Angle;
lambda X;
lambda Y;
{
register string_link *Auxiliary;
register char *s = String;

	if (!s)
		return;

	while (*s)
		if (!isprint(*s++))
			return;

	Auxiliary = (string_link *)malloc(sizeof(string_link));
	Auxiliary->STRING = String;
	Auxiliary->X = X;
	Auxiliary->Y = Y;
	Auxiliary->ANGLE = Angle;
	Auxiliary->NEXT = (*HeadOfStringsList);
	(*HeadOfStringsList) = Auxiliary;
}

void add_rectangle(RectsOfOneLayer, X0, Y0, X1, Y1)
drw_layer *RectsOfOneLayer;
lambda X0;
lambda Y0;
lambda X1;
lambda Y1;
{
register rect_link *Auxiliary;

	if (X0 == X1 || Y0 == Y1)
		return;

	Auxiliary = (rect_link *)malloc(sizeof(rect_link));
	Auxiliary->X0 = X0;
	Auxiliary->Y0 = Y0;
	Auxiliary->X1 = X1;
	Auxiliary->Y1 = Y1;
	Auxiliary->NEXT = (*RectsOfOneLayer).LIST;
	(*RectsOfOneLayer).LIST = Auxiliary;
	(*RectsOfOneLayer).NUMBER++;
	/* updating BoundingBox coordinates */
	BoundingBox.X0 = MIN (BoundingBox.X0, X0);
	BoundingBox.Y0 = MIN (BoundingBox.Y0, Y0);
	BoundingBox.X1 = MAX (BoundingBox.X1, X1);
	BoundingBox.Y1 = MAX (BoundingBox.Y1, Y1);
}

void getinstances(HeadOfInstancesList, RectsOfLayer, ListOfStrings, model)
phins_list *HeadOfInstancesList;
drw_layer RectsOfLayer[];
string_link **ListOfStrings;
int model;
{
register phins_list *InstancePtr;
register phfig_list *ModelPtr;
lambda Xbl, Ybl, Xtr, Ytr;	/* bl : bottom left; tr : top right	*/
int Angle;

	for (InstancePtr = HeadOfInstancesList; InstancePtr;
			InstancePtr = InstancePtr->NEXT) {
		ModelPtr = getphfig(InstancePtr->FIGNAME, 'P');
		Xbl = InstancePtr->XINS;
		Ybl = InstancePtr->YINS;
		switch (InstancePtr->TRANSF) {
			case NOSYM :
			case SYM_X :
			case SYM_Y :
			case SYMXY :
				Xtr = Xbl + ModelPtr->XAB2 - ModelPtr->XAB1;
				Ytr = Ybl + ModelPtr->YAB2 - ModelPtr->YAB1;
				break;
			case ROT_P :
			case ROT_M :
			case SY_RP :
			case SY_RM :
				Xtr = Xbl + ModelPtr->YAB2 - ModelPtr->YAB1;
				Ytr = Ybl + ModelPtr->XAB2 - ModelPtr->XAB1;
				break;
		}
		if (ListOfStrings) {
			Angle = (Xtr - Xbl) > (Ytr - Ybl) ? 0 : 90;
			if (model)
				add_string(ListOfStrings, InstancePtr->FIGNAME, Angle,
								(Xbl + Xtr) / 2, (Ybl + Ytr) / 2);
			else
				add_string(ListOfStrings, InstancePtr->INSNAME, Angle,
								(Xbl + Xtr) / 2, (Ybl + Ytr) / 2);
		}
		add_rectangle(&RectsOfLayer[TALU1], Xbl, Ybl, Xtr, Ytr);
	}
}

void getsegments(HeadOfSegmentsList, RequiredLayer, RectsOfLayer, ListOfStrings)
phseg_list *HeadOfSegmentsList;
int RequiredLayer[];
drw_layer RectsOfLayer[];
string_link **ListOfStrings;
{
register phseg_list *SegmentPtr;
lambda Xbl, Ybl, Xtr, Ytr;	/* bl : bottom left */
						/* tr : top right	*/

	for (SegmentPtr = HeadOfSegmentsList; SegmentPtr;
			SegmentPtr = SegmentPtr->NEXT) {
		if (SegmentPtr->LAYER >= MAXLAYER) {
			printf("mbk2ps actual MAXLAYER : %u\n", MAXLAYER);
			exit(UPDATE_MAXLAYER);
		}
		if (RequiredLayer[ SegmentPtr->LAYER ]) {
			if (ListOfStrings)
				add_string(ListOfStrings, SegmentPtr->NAME, 0,
								HALF(SegmentPtr->X1 + SegmentPtr->X2),
								HALF(SegmentPtr->Y1 + SegmentPtr->Y2));
			if (SegmentPtr->TYPE == HOR) {
				Xbl = SegmentPtr->X1 - Extension[ SegmentPtr->LAYER ];
				Ybl = SegmentPtr->Y1 - HALF(SegmentPtr->WIDTH);
				Xtr = SegmentPtr->X2 + Extension[ SegmentPtr->LAYER ];
				Ytr = Ybl + SegmentPtr->WIDTH;
				switch (SegmentPtr->LAYER) {
					case NTRANS :
						add_rectangle(&RectsOfLayer[ POLY ], Xbl, Ybl, Xtr, Ytr);
						Xbl += RTrans;
						Ybl -= HALF(WTrans);
						Xtr -= RTrans;
						Ytr += HALF(WTrans);
						add_rectangle(&RectsOfLayer[ NDIF ], Xbl, Ybl, Xtr, Ytr);
						break;
					case PTRANS :
						add_rectangle(&RectsOfLayer[ POLY ], Xbl, Ybl, Xtr, Ytr);
						Xbl += RTrans;
						Ybl -= HALF(WTrans);
						Xtr -= RTrans;
						Ytr += HALF(WTrans);
						add_rectangle(&RectsOfLayer[ PDIF ], Xbl, Ybl, Xtr, Ytr);
						break;
					default :
						add_rectangle(&RectsOfLayer[ SegmentPtr->LAYER ],
											Xbl, Ybl, Xtr, Ytr);
				}	/* switch about SegmentPtr->LAYER */
			} else {		/* SegmentPtr->TYPE == VER */
				Xbl = SegmentPtr->X1 - HALF(SegmentPtr->WIDTH);
				Ybl = SegmentPtr->Y1 - Extension[ SegmentPtr->LAYER ];
				Xtr = Xbl + SegmentPtr->WIDTH;
				Ytr = SegmentPtr->Y2 + Extension[ SegmentPtr->LAYER ];
				switch (SegmentPtr->LAYER) {
					case NTRANS :
						add_rectangle(&RectsOfLayer[ POLY ], Xbl, Ybl, Xtr, Ytr);
						Xbl -= HALF(WTrans);
						Ybl += RTrans;
						Xtr += HALF(WTrans);
						Ytr -= RTrans;
						add_rectangle(&RectsOfLayer[ NDIF ], Xbl, Ybl, Xtr, Ytr);
						break;
					case PTRANS :
						add_rectangle(&RectsOfLayer[ POLY ], Xbl, Ybl, Xtr, Ytr);
						Xbl -= HALF(WTrans);
						Ybl += RTrans;
						Xtr += HALF(WTrans);
						Ytr -= RTrans;
						add_rectangle(&RectsOfLayer[ PDIF ], Xbl, Ybl, Xtr, Ytr);
						break;
					default :
						add_rectangle(&RectsOfLayer[SegmentPtr->LAYER],
											Xbl, Ybl, Xtr, Ytr);
				}
			}	
		}
	}
}

void getconnectors(HeadOfConnectorsList, RequiredLayer, RectsOfLayer, ListOfStrings)
phcon_list *HeadOfConnectorsList;
int RequiredLayer[];
drw_layer RectsOfLayer[];
string_link **ListOfStrings;
{
register struct phcon *ConnectorPtr;
lambda Xbl, Ybl, Xtr, Ytr; /* bl : bottom left; tr : top right */

	for (ConnectorPtr = HeadOfConnectorsList; ConnectorPtr;
			ConnectorPtr = ConnectorPtr->NEXT) {
		if (ConnectorPtr->LAYER >= MAXLAYER) {
			fprintf(stderr, "mbk2ps actual MAXLAYER : %u\n", MAXLAYER);
			exit(UPDATE_MAXLAYER);
		}
		if (RequiredLayer[ ConnectorPtr->LAYER ]) {
			if (ListOfStrings)
				add_string(ListOfStrings,
								nameindex(ConnectorPtr->NAME, ConnectorPtr->INDEX), 0,
								ConnectorPtr->XCON, ConnectorPtr->YCON);
			switch (ConnectorPtr->ORIENT) {
				case NORTH :
				case SOUTH :
					Xbl = ConnectorPtr->XCON - HALF(ConnectorPtr->WIDTH);
					Ybl = ConnectorPtr->YCON - Extension[ ConnectorPtr->LAYER ];
					Xtr = Xbl + ConnectorPtr->WIDTH;
					Ytr = Ybl + DBLE(Extension[ ConnectorPtr->LAYER ]);
					break;
				case EAST  :
				case WEST  :
					Xbl = ConnectorPtr->XCON - Extension[ ConnectorPtr->LAYER ];
					Ybl = ConnectorPtr->YCON - HALF(ConnectorPtr->WIDTH);
					Xtr = Xbl + DBLE(Extension[ ConnectorPtr->LAYER ]);
					Ytr = Ybl + ConnectorPtr->WIDTH;
					break;
			}
			add_rectangle(&RectsOfLayer[ConnectorPtr->LAYER], Xbl, Ybl, Xtr, Ytr);
		}
	}
}

void getvias(HeadOfViasList, DrawableViasList, RectsOfLayer, Pattern)
phvia_list *HeadOfViasList;
drw_via_link **DrawableViasList;
drw_layer RectsOfLayer[];
char Pattern[];
{
#define SIZE_HOLE    SCALE_X
#define SIZE_L_1     (2 * SCALE_X)
#define SIZE_L_2     (3 * SCALE_X)
#define SIZE_C_X     (5 * SCALE_X)
register phvia_list *ViaPtr;
char layer1, layer2;

	for (ViaPtr = HeadOfViasList; ViaPtr; ViaPtr = ViaPtr->NEXT) {
		switch (ViaPtr->TYPE) {
			case C_X_N :
				add_rectangle(&RectsOfLayer[ POLY ],
					ViaPtr->XVIA - HALF(SCALE_X), ViaPtr->YVIA - HALF(SCALE_X),
					ViaPtr->XVIA + HALF(SCALE_X), ViaPtr->YVIA + HALF(SCALE_X));
				add_rectangle(&RectsOfLayer[ NDIF ],
					ViaPtr->XVIA - HALF(SIZE_C_X), ViaPtr->YVIA - HALF(SIZE_C_X),
					ViaPtr->XVIA + HALF(SIZE_C_X), ViaPtr->YVIA + HALF(SIZE_C_X));
				continue;
			case C_X_P  :
				add_rectangle(&RectsOfLayer[ POLY ],
					ViaPtr->XVIA - HALF(SCALE_X), ViaPtr->YVIA - HALF(SCALE_X),
					ViaPtr->XVIA + HALF(SCALE_X), ViaPtr->YVIA + HALF(SCALE_X));
				add_rectangle(&RectsOfLayer[ PDIF ],
					ViaPtr->XVIA - HALF(SIZE_C_X), ViaPtr->YVIA - HALF(SIZE_C_X),
					ViaPtr->XVIA + HALF(SIZE_C_X), ViaPtr->YVIA + HALF(SIZE_C_X));
				continue;
			case CONT_POLY :
				add_rectangle(&RectsOfLayer[ NTRANS ],
					ViaPtr->XVIA - HALF(SIZE_HOLE), ViaPtr->YVIA - HALF(SIZE_HOLE),
					ViaPtr->XVIA + HALF(SIZE_HOLE), ViaPtr->YVIA + HALF(SIZE_HOLE));
				add_rectangle(&RectsOfLayer[ ALU1 ],
					ViaPtr->XVIA - HALF(SIZE_L_1), ViaPtr->YVIA - HALF(SIZE_L_1),
					ViaPtr->XVIA + HALF(SIZE_L_1), ViaPtr->YVIA + HALF(SIZE_L_1));
				add_rectangle(&RectsOfLayer[ POLY ],
					ViaPtr->XVIA - HALF(SIZE_L_2), ViaPtr->YVIA - HALF(SIZE_L_2),
					ViaPtr->XVIA + HALF(SIZE_L_2), ViaPtr->YVIA + HALF(SIZE_L_2));
				break;
			case CONT_VIA :
				add_rectangle(&RectsOfLayer[ NTRANS ],
					ViaPtr->XVIA - HALF(SIZE_HOLE), ViaPtr->YVIA - HALF(SIZE_HOLE),
					ViaPtr->XVIA + HALF(SIZE_HOLE), ViaPtr->YVIA + HALF(SIZE_HOLE));
				add_rectangle(&RectsOfLayer[ ALU1 ],
					ViaPtr->XVIA - HALF(SIZE_L_1), ViaPtr->YVIA - HALF(SIZE_L_1),
					ViaPtr->XVIA + HALF(SIZE_L_1), ViaPtr->YVIA + HALF(SIZE_L_1));
				add_rectangle(&RectsOfLayer[ ALU2 ],
					ViaPtr->XVIA - HALF(SIZE_L_2), ViaPtr->YVIA - HALF(SIZE_L_2),
					ViaPtr->XVIA + HALF(SIZE_L_2), ViaPtr->YVIA + HALF(SIZE_L_2));
				break;
			case CONT_DIF_N :
				add_rectangle(&RectsOfLayer[ NTRANS ],
					ViaPtr->XVIA - HALF(SIZE_HOLE), ViaPtr->YVIA - HALF(SIZE_HOLE),
					ViaPtr->XVIA + HALF(SIZE_HOLE), ViaPtr->YVIA + HALF(SIZE_HOLE));
				add_rectangle(&RectsOfLayer[ ALU1 ],
					ViaPtr->XVIA - HALF(SIZE_L_1), ViaPtr->YVIA - HALF(SIZE_L_1),
					ViaPtr->XVIA + HALF(SIZE_L_1), ViaPtr->YVIA + HALF(SIZE_L_1));
				add_rectangle(&RectsOfLayer[ NDIF ],
					ViaPtr->XVIA - HALF(SIZE_L_2), ViaPtr->YVIA - HALF(SIZE_L_2),
					ViaPtr->XVIA + HALF(SIZE_L_2), ViaPtr->YVIA + HALF(SIZE_L_2));
				break;
			case CONT_DIF_P :
				add_rectangle(&RectsOfLayer[ NTRANS ],
					ViaPtr->XVIA - HALF(SIZE_HOLE), ViaPtr->YVIA - HALF(SIZE_HOLE),
					ViaPtr->XVIA + HALF(SIZE_HOLE), ViaPtr->YVIA + HALF(SIZE_HOLE));
				add_rectangle(&RectsOfLayer[ ALU1 ],
					ViaPtr->XVIA - HALF(SIZE_L_1), ViaPtr->YVIA - HALF(SIZE_L_1),
					ViaPtr->XVIA + HALF(SIZE_L_1), ViaPtr->YVIA + HALF(SIZE_L_1));
				add_rectangle(&RectsOfLayer[ PDIF ],
					ViaPtr->XVIA - HALF(SIZE_L_2), ViaPtr->YVIA - HALF(SIZE_L_2),
					ViaPtr->XVIA + HALF(SIZE_L_2), ViaPtr->YVIA + HALF(SIZE_L_2));
				break;
			case CONT_BODY_N :
				add_rectangle(&RectsOfLayer[ NTRANS ],
					ViaPtr->XVIA - HALF(SIZE_HOLE), ViaPtr->YVIA - HALF(SIZE_HOLE),
					ViaPtr->XVIA + HALF(SIZE_HOLE), ViaPtr->YVIA + HALF(SIZE_HOLE));
				add_rectangle(&RectsOfLayer[ ALU1 ],
					ViaPtr->XVIA - HALF(SIZE_L_1), ViaPtr->YVIA - HALF(SIZE_L_1),
					ViaPtr->XVIA + HALF(SIZE_L_1), ViaPtr->YVIA + HALF(SIZE_L_1));
				add_rectangle(&RectsOfLayer[ NTIE ],
					ViaPtr->XVIA - HALF(SIZE_L_2), ViaPtr->YVIA - HALF(SIZE_L_2),
					ViaPtr->XVIA + HALF(SIZE_L_2), ViaPtr->YVIA + HALF(SIZE_L_2));
				break;
			case CONT_BODY_P :
				add_rectangle(&RectsOfLayer[ NTRANS ],
					ViaPtr->XVIA - HALF(SIZE_HOLE), ViaPtr->YVIA - HALF(SIZE_HOLE),
					ViaPtr->XVIA + HALF(SIZE_HOLE), ViaPtr->YVIA + HALF(SIZE_HOLE));
				add_rectangle(&RectsOfLayer[ ALU1 ],
					ViaPtr->XVIA - HALF(SIZE_L_1), ViaPtr->YVIA - HALF(SIZE_L_1),
					ViaPtr->XVIA + HALF(SIZE_L_1), ViaPtr->YVIA + HALF(SIZE_L_1));
				add_rectangle(&RectsOfLayer[ PTIE ],
					ViaPtr->XVIA - HALF(SIZE_L_2), ViaPtr->YVIA - HALF(SIZE_L_2),
					ViaPtr->XVIA + HALF(SIZE_L_2), ViaPtr->YVIA + HALF(SIZE_L_2));
				break;
		}
	}
}

void getreferences(HeadOfRefsList, RectsOfLayer, ListOfReferences)
phref_list	*HeadOfRefsList;
drw_layer RectsOfLayer[];
string_link **ListOfReferences;
{
#define SIZE_REF SCALE_X
register struct phref *RefPtr;

	for (RefPtr = HeadOfRefsList; RefPtr; RefPtr = RefPtr->NEXT)
		if (ListOfReferences) {
			add_string(ListOfReferences, RefPtr->NAME, 0,
							RefPtr->XREF, RefPtr->YREF);
			add_rectangle(&RectsOfLayer[ PTRANS ],
					RefPtr->XREF - HALF(SIZE_REF), RefPtr->YREF - HALF(SIZE_REF),
					RefPtr->XREF + HALF(SIZE_REF), RefPtr->YREF + HALF(SIZE_REF));
		}
}

#define STACK_SIZE 10 /* needed to avoid stack overflow on big circuits */

void describ_paths(PS_file, PathLinkPtr, Pattern, Color)
FILE *PS_file;
register path_link *PathLinkPtr;
char Pattern;
char *Color;
{
register point_link *PointLinkPtr;
register int i = STACK_SIZE;

	if (PathLinkPtr) {
		if (Color)
			fprintf(PS_file, "%s setrgbcolor\n", Color);
#ifndef GS
		fprintf(PS_file, "%u [\n", Pattern);
		while (PathLinkPtr) {
			fputs("\t{ [ ", PS_file);
			for (PointLinkPtr = PathLinkPtr->PATH; PointLinkPtr;
					PointLinkPtr = PointLinkPtr->NEXT) {
				fprintf(PS_file, "{ %li %li } ", PointLinkPtr->X, PointLinkPtr->Y);
#if 0
				if (i-- < 0) {
					fputc('\n', PS_file);
					i = STACK_SIZE;
				}
#endif
			}
			fputc(']', PS_file);
			fprintf(PS_file, " %li %li }\n", (PathLinkPtr->PATH)->X,
						(PathLinkPtr->PATH)->Y);
			PathLinkPtr = PathLinkPtr->NEXT;
		}
		fputs("] draw_paths\n", PS_file);
#else
		while (PathLinkPtr) {
			i = 0;
			for (PointLinkPtr = PathLinkPtr->PATH; PointLinkPtr;
					PointLinkPtr = PointLinkPtr->NEXT) {
				fprintf(PS_file, "%li %li %s ", 
							PointLinkPtr->X, PointLinkPtr->Y, i++ == 0 ? "moveto" : "lineto");
			}
			fprintf(PS_file, "%d patternfill stroke\n", Pattern);
			PathLinkPtr = PathLinkPtr->NEXT;
		}
#endif
	}
}

void delpaths(HeadOfPathsList)
path_link **HeadOfPathsList;
{
path_link  *PathLinkPtr,  *FreePathLink;
point_link *PointLinkPtr, *FreePointLink;

	PathLinkPtr = *HeadOfPathsList;
	while (PathLinkPtr) {
		PointLinkPtr = PathLinkPtr->PATH;
		while (PointLinkPtr) {
			FreePointLink = PointLinkPtr;
			PointLinkPtr = PointLinkPtr->NEXT;
			free(FreePointLink);
		}
		FreePathLink = PathLinkPtr;
		PathLinkPtr = PathLinkPtr->NEXT;
		free(FreePathLink);
	}
	*HeadOfPathsList = NULL;
}

void describ_rectangles(PS_file, RectLinkPtr, Pattern, Color)
FILE		*PS_file;
register rect_link *RectLinkPtr;
char		 Pattern;
char      *Color;
{
register int i = STACK_SIZE;/* needed to avoid stack overflow on big circuits */

	if (!RectLinkPtr)
		return;

	if (Color)
		fprintf(PS_file, "%s setrgbcolor\n", Color);
	while (1) {
		fprintf(PS_file, "%u [\n", Pattern);
		while (RectLinkPtr && i--) {
			fprintf(PS_file, "{ %li %li %li %li } ", RectLinkPtr->X0,
						RectLinkPtr->Y0, RectLinkPtr->X1, RectLinkPtr->Y1);
			RectLinkPtr = RectLinkPtr->NEXT;
		}
		fputs("\n] draw_rectangles\n", PS_file);
		if (!RectLinkPtr)
			break;
		i = STACK_SIZE;
	}
}

void strokeAB(PS_file, AbutmentBox)
FILE *PS_file;
rect  AbutmentBox;
{
	fprintf(PS_file, "%li %li %li %li strokeAB\n",
				AbutmentBox.X0, AbutmentBox.Y0, AbutmentBox.X1, AbutmentBox.Y1);
}

float scale(Tex, Frame)
int Tex;
rect Frame;
{
float Xbl, Ybl, Xtr, Ytr;
float WidthOfCell, HeightOfCell;

	Xbl = Frame.X0;
	Ybl = Frame.Y0;
	Xtr = Frame.X1;
	Ytr = Frame.Y1;
	WidthOfCell = BoundingBox.X1 - BoundingBox.X0;
	HeightOfCell = BoundingBox.Y1 - BoundingBox.Y0;

	if (!Tex)
		Xbl += 14, Ybl += 26, Xtr -=14, Ytr -= 6;
	return MIN((Xtr - Xbl) / WidthOfCell, (Ytr - Ybl) / HeightOfCell);
}

void translate(PS_file, CellName, Tex, Frame)
FILE *PS_file;
char *CellName;
int Tex;
rect Frame;
{
float Xbl, Ybl, Xtr, Ytr;
float WidthOfCell, HeightOfCell;

	Xbl = Frame.X0;
	Ybl = Frame.Y0;
	Xtr = Frame.X1;
	Ytr = Frame.Y1;

	fprintf(PS_file, "%f dup scale\n", Scale);
	if (!Tex)
		Xbl += 14, Ybl += 26, Xtr -=14, Ytr -= 6;
	fprintf(PS_file, "%f %f translate\n\n", 
				((Xbl + Xtr) / Scale - BoundingBox.X0 - BoundingBox.X1) / 2,
				((Ybl + Ytr) / Scale - BoundingBox.Y0 - BoundingBox.Y1) / 2);
}

void boundingbox(PS_file, Frame)
FILE *PS_file;
rect Frame;
{
float X, Y;

	X = ((Frame.X0 + Frame.X1) / Scale - BoundingBox.X0 - BoundingBox.X1) / 2;
	Y = ((Frame.Y0 + Frame.Y1) / Scale - BoundingBox.Y0 - BoundingBox.Y1) / 2;
	/* BoundingBox coordinates :
	   this comment should be as close as possible of the beginning of the
	   file if I want to avoid a TeX error, looking like `file to big'
	   during the \psfig{figure=big.ps}.
	   The values given below are *not* the real Bounding box ones, since I
	   don't want to bother calculating the texts size.
	   It nevertheless works ok since we can assume that the circuit more
	   than the text is to be centered in the page. */
	fprintf(PS_file, "%%%%BoundingBox: %2.f %2.f %2.f %2.f\n",
				(X + BoundingBox.X0) * Scale, (Y + BoundingBox.Y0) * Scale,
				(X + BoundingBox.X1) * Scale, (Y + BoundingBox.Y1) * Scale);
}

void frame(PS_file, CellName, AbutmentBox, Frame)
FILE *PS_file;
char *CellName;
rect AbutmentBox, Frame;
{
#define POINT_PICA 0.2834 /* 72 pt = 1 in = 2.54 cm => 1 cm = 28.34 pt */
#define SIXTEEN 32
#define TWELVE 42
float Xbl, Ybl, Xtr, Ytr, X, Y;
long  Time;
int FontSize;

	Xbl = Frame.X0;
	Ybl = Frame.Y0;
	Xtr = Frame.X1;
	Ytr = Frame.Y1;

	fputs("gsave\n", PS_file);
	fprintf(PS_file, "\t%f %f %f %f draw_rectangle stroke\n",
				Xbl, Ybl, Xtr, Ytr);
	fprintf(PS_file, "\t%f %f %f %f draw_rectangle stroke\n",
				Xbl + 2, Ybl + 2, Xtr - 2, Ybl + 20);
	fprintf(PS_file, "\t%f %f moveto\n", Xbl + 7, Ybl + 6);
	/* %d used to be 16, now adapts moreless to sheet size when small */
	FontSize = (Xtr - Xbl) / SIXTEEN;
	if (FontSize > 16)
		FontSize = 16;
	fprintf(PS_file, "\t/Times-Roman findfont %d scalefont setfont\n",
						FontSize);
	fputs("\tgsave\n", PS_file);
	fputs("\t\t(cao-vlsi) dup false charpath stroke\n", PS_file);
	fputs("\tgrestore\n", PS_file);
	fputs("\tstringwidth pop 0 rmoveto\n", PS_file);
	/* %d used to be 12, now adapts moreless to sheet size when small */
	FontSize = (Xtr - Xbl) / TWELVE;
	if (FontSize > 12)
		FontSize = 12;
	fprintf(PS_file, "\t/Times-Roman findfont %d scalefont setfont\n",
				FontSize);
	fprintf(PS_file, "\t( %s for %s) show\n", pstime(time(&Time)),
				getenv("USER"));
	fprintf(PS_file, "\t%f %f moveto\n", Xtr - 7, Ybl + 6);

	/* introduction of Symbols for lambda output */
	fprintf(PS_file, "\t( = %.2f mm)", Scale / POINT_PICA);
	fputs("dup stringwidth pop neg 0 rmoveto gsave show grestore\n", PS_file);
	fprintf(PS_file, "\t/Symbol findfont %d scalefont setfont\n", FontSize);
	fputs("\t\t(l) dup stringwidth pop neg 0 rmoveto gsave show grestore\n",
			PS_file);
	fprintf(PS_file, "\t/Times-Roman findfont %d scalefont setfont\n",
				FontSize);
	fputs("\t(  Scale : )", PS_file);
	fputs("dup stringwidth pop neg 0 rmoveto gsave show grestore\n", PS_file);
	fprintf(PS_file, "\t/Symbol findfont %d scalefont setfont\n", FontSize);
	fputs("\t(l) dup stringwidth pop neg 0 rmoveto gsave show grestore\n",
			PS_file);
	fprintf(PS_file, "\t/Times-Roman findfont %d scalefont setfont\n",
				FontSize);
	fprintf(PS_file, "\t( * %d)", (AbutmentBox.Y1 - AbutmentBox.Y0) / SCALE_X);
	fputs("dup stringwidth pop neg 0 rmoveto gsave show grestore\n", PS_file);
	fprintf(PS_file, "\t/Symbol findfont %d scalefont setfont\n", FontSize);
	fputs("\t(l) dup stringwidth pop neg 0 rmoveto gsave show grestore\n",
			PS_file);
	fprintf(PS_file, "\t/Times-Roman findfont %d scalefont setfont\n",
				FontSize);
	fprintf(PS_file, "\t(Cell : %s  Size : %d)", CellName,
				(AbutmentBox.X1 - AbutmentBox.X0) / SCALE_X);
	fputs(" dup stringwidth pop neg 0 rmoveto show\n", PS_file);
	fputs("grestore\n", PS_file);
}

void write_strings(PS_file, StringLinkPtr, FontSize)
FILE *PS_file;
register string_link *StringLinkPtr;
int FontSize;
{
	if (!StringLinkPtr)
		return;
	fprintf(PS_file, "/Courier findfont %d scalefont setfont\n",
				FontSize < 4 ? (int)(4/Scale) : (int)(FontSize/Scale));
	while (StringLinkPtr) {
		fprintf(PS_file, "%li %li moveto (%s) %d showstring\n",
					StringLinkPtr->X, StringLinkPtr->Y,
					StringLinkPtr->STRING, StringLinkPtr->ANGLE);
		StringLinkPtr = StringLinkPtr->NEXT;
	}
}

char *pstime(Time)
time_t Time;
{
#define TIMESIZE 19
char FormatedTime[TIMESIZE]; /* mmm dd yyyy HH:MM */

#ifdef sparc
	strftime(FormatedTime, TIMESIZE, "%h %d %Y %R", localtime(&Time));
	return FormatedTime;
#else
	return "No time today";
#endif
}

void pserror(why)
int why;
{
	switch (why) {
		case NO_PS_FILE:
			(void)fputs("mbk2ps: Installation problem ", stderr);
			(void)fputs(": could not find my own PostScript dictionary\n", stderr);
			break;
		case OPEN_FAILED:
			(void)fputs("mbk2ps: Could not open the PostScript file here", stderr);
			(void)fputs(" : output to be done on stdout\n", stderr);
			return;
	}
	exit(why);
}

#include "genlib.h"
void patternsheet()
{
	DEF_PHFIG("patternsheet");
	DEF_AB(1, 1, 11, 30);
	PHSEG(NWELL, 4, "n_well", 2, 28, 10, 28);
	PHSEG(NTIE, 2, "n_polarisation", 2, 23, 10, 23);
	PHSEG(PTIE, 2, "p_polarisation", 2, 19, 10, 19);
	PHSEG(NDIF, 2, "n_diffusion", 2, 23, 10, 23);
	PHSEG(PDIF, 2, "p_diffusion", 2, 19, 10, 19);
	PHSEG(POLY, 1, "polysilicium", 2, 15, 10, 15);
	PHSEG(ALU1, 1, "metal_1", 2, 12, 10, 12);
	PHSEG(TALU1, 1, "metal_1_through_route", 2, 9, 10, 9);
	PHSEG(ALU2, 2, "metal_2", 2, 6, 10, 6);
	PHSEG(TALU2, 2, "metal_2_through_route", 2, 2, 10, 2);
	PHVIA(C_X_P, 16, 4);
	PHSEG(TALU1, 1, "c_x_p", 21, 4, 22, 4);
	PHVIA(C_X_N, 16, 11);
	PHSEG(TALU1, 1, "c_x_n", 21, 11, 22, 11);
	PHVIA(CONT_VIA, 16, 17);
	PHSEG(TALU1, 1, "via", 21, 17, 22, 17);
	PHVIA(CONT_DIF_P, 16, 25);
	PHSEG(TALU1, 1, "cont_dif_p", 21, 25, 22, 25);
	PHVIA(CONT_DIF_N, 16, 29);
	PHSEG(TALU1, 1, "cont_dif_n", 21, 29, 22, 29);
	PHVIA(CONT_POLY, 16, 21);
	PHSEG(TALU1, 1, "cont_poly", 21, 21, 22, 21);
}
