/*

Formal specification of the YASP trajectory file
================================================
 
FMP 23 Aug 90
 
1) Header
=========
 
The Header  contains information  which is stored  only  once  on  the
trajectory  file. It precedes  all individual configurations (frames).
 
- character*80 title
  write (io_unit) title
  write (io_unit) number_of_records_per_frame (i*4)
 
2) Frames
=========
 
A "frame" is the information characterising the simulated  system at a
certain  time.    Each frame contains   as many  (Fortran)  records as
specified in the header.
 
- "Frame"
  write (io_unit) configuration_number, number_of_atoms, time
	(i*4, i*4, r*8)
 
  The configuration_number is the index number of the frame as it  ap-
  pears in the simulation,  which is  in general not  the same as  the
  index number of  the frame on  the trajectory file.  The time is the
  simulated time (e.g. ps), not the cpu time.
 
- "Box"
  write (io_unit) (a(i), i = 1, 3), (b(i), i = 1, 3), (c(i), i = 1, 3)
	(all r*8)
 
  The current size and shape of the periodic box is  described by  the
  its cell unit vectors a, b and c.  Note that a, b and c refer to the
  dimensions of the entire box.  The coordinate  origin for the atomic
  positions is,  however,  in the centre of the box.  At present, YASP
  only uses rectangular boxes (all angles between cell vectors are  90
  degrees). Hence the cell vectors look like
	a = (xbox, 0,    0   )
	b = (0,    ybox, 0   )
	c = (0,    0,    zbox)
  with xbox, ybox, zbox beind the box  lengths in the  three cartesian
  directions.
 
- "Pressure"
  write (io_unit) total_isotropic_pressure,
 $		  Pxx,
 $                Pyx, Pyy,
 $                Pzx, Pyx, Pzz
	(all r*8)
  The total pressure is the isotropic pressure of the system which can
  in principle be calculated from the components of the pressure ten-
  sor that follow. This is just to avoid recalculation. All pressures
  are in kPa.
 
- "Energies"
  write (io_unit) number_of_energies, (e(i), i = 1, number_of_energies)
	(i*4, rest r*8)
  As many energy-related quantities as there are. Presently there are
	e(1) = total_energy
	e(2) = potential_energy
	e(3) = kinetic_energy
	e(4) = temperature
  YASP energies are in kJ/mole, temperatures in K.
 
- "X-coordinates"
  write (io_unit) (x(i), i = 1, natom) (all r*8)
  Coordinates (atomic positions) are in nm.
 
- "Y-coordinates"
  write (io_unit) (y(i), i = 1, natom) (all r*8)
  Coordinates (atomic positions) are in nm.
 
- "Z-coordinates"
  write (io_unit) (z(i), i = 1, natom) (all r*8)
  Coordinates (atomic positions) are in nm.
 
- "X-velocities"
  write (io_unit) (vx(i), i = 1,natom) (all r*8)
  Atomic velocities are in nm/ps.
 
- "Y-velocities"
  write (io_unit) (vy(i), i = 1,natom) (all r*8)
  Atomic velocities are in nm/ps.
 
- "Z-velocities"
  write (io_unit) (vz(i), i = 1,natom) (all r*8)
  Atomic velocities are in nm/ps.
 
- "Others"
  write (io_unit) (q(i), i = 1, natom) (all r*8)
  As many of these records as necessary, all describing order(natom)
  quantities (energy per atom and the like). These are presently
  not used.
 
>From this description it can be seen that the information in a frame
naturally decomposes into three parts.
a) system description
  "Frame", "Box", and "Pressure"
b) "Scalar" quantities. Their number does not depend on the number of
   atoms. Let's call them O(natom**0)
  "Energies"
c) "Vector" quantities. Their number depends linearly on the number of
   atoms. O(natom**1).
There is probably no need to include atom-pair-related quantities
of O(natom**2) or higher into the file.

*/

#include <stdio.h>
#include <math.h>
#include <malloc.h>
#include <string.h>


#define MOD(a,b)  (((a+1)/b)*b - (a+1))

/***************************************************************************/
rtraj_frame_yasp(alt)  /* read one frame from a yasp trajectory   */
/***************************************************************************/


    int alt;     /* mode of operation (=0) first time in read
                                      (>0) trajectory number */
{
/* externals    */
    extern FILE *charmm_oc;
    extern float *x,*y,*z;
    extern int numat;
    extern char traj_file[BUFF_LEN];

   static int memory;
   static int every_step;              /* reading only every 'every_step' from
                                          the trajectory file */
   static int read_loop;
   static long record_len=0;         /* the "record" length in bytes of one
                                       record containing the x,y and z
                                       coordinates plus the information
                                       in between the coordinates */
   static int prev_frame=0;         /* frame number of previous frame */
   static long ret_fseek;

   static char title[BUFF_LEN];  /* yasp title */
   static int conf_num;    /* yasp configuration number */
   static int natom;       /* number of yasp atoms */
   static double time_ps;  /* yasp simulation time in ps */
   static double yasp_box[6]; /* yasp box */
   static double yasp_energ[20];  /* yasp energies */
   static double yasp_press[7];   /* yasp pressure */
   static double *tx,*ty,*tz;     /* pointers to yasp coordinates */
   static double *vx,*vy,*vz;     /* pointers to yasp velocities */
   static double *util1,*util2,*util3; /* yasp utilities */

   int numset,  /* number of data sets */

   every_step = 1;

/*                                      */
    if((alt - prev_frame) < 0 ) {
    /* rewind one "record" length of bytes */
    ret_fseek = fseek(yasp_oc,(-2*record_len),1);}

    if(alt == 0) {rewind(yasp_oc);
                  prev_frame = 0;} /* reset the counter = no previous frame */


   if(alt == 0) {  /* start of 1 */

/*  start reading  */

#ifdef CRAY
  
/* read controll record */
    icount = fread(&record,sizeof(int), 1 ,charmm); 
    icount = fread(&hdr,sizeof(float), 1 ,charmm); 
    icount = fread(icntrl,sizeof(int), 20 ,charmm); 
    icount = fread(&record,sizeof(int), 1 ,charmm); 
    icount = fread(&ntitl,sizeof(int), 1 ,charmm);

    for(i = 0 ; i < ntitl ; i++) {
    icount = fread(title,10*sizeof(double),1,charmm);
    title[79] = '\0';
    }

    icount = fread(&record,sizeof(int), 1 ,charmm);
    icount = fread(&natom,sizeof(int), 1 , charmm);
    icount = fread(&record,sizeof(int), 1 ,charmm);

#else
  
/* read controll record */
    icount = fread(&record,sizeof(int), 1 , charmm_oc);
    if(icount != 80) {
      printf("?ERROR - can't read yasp dynamics header \n");
        return;}
/* read 80 characters   */
    icount = fread(title,sizeof(char), 80 ,charmm_oc);

#endif

/* determine the "record length"    */
    record_len = 6 * sizeof(int) + /* the record contribution */
                 3 * numat * sizeof(float); /* x-,y- and z-contribution */


/*                                  */

    istep = alt;

    if(istep == 0) read_loop = 0;
    else
    if(MOD(istep,every_step) == 0) read_loop++;

#ifdef DEBUG
    printf(" Retrieving frame number : %d \n",read_loop+1);
#endif

#ifdef CRAY

    icount = fread(x,sizeof(float),natom,charmm);
    icount = fread(&record,sizeof(int), 1 ,charmm);
    icount = fread(y,sizeof(float),natom,charmm);
    icount = fread(&record,sizeof(int), 1 ,charmm);
    icount = fread(z,sizeof(float),natom,charmm);
    icount = fread(&record,sizeof(int), 1 ,charmm);

#else

/* next records are specific for one frame */
/* read controll record */
    icount = fread(&record,sizeof(int), 1 , charmm_oc);
    if(icount != (sizeof(float) * 2 + sizeof(double)) {
      printf("?ERROR - in 'FRAME' record \n");
        return;}

    icount = fread(&conf_num,sizeof(int), 1 ,charmm_oc);
    icount = fread(&natom,sizeof(int), 1 , charmm_oc);
    icount = fread(&time_ps,sizeof(double), 1 , charmm_oc);
    icount = fread(&record,sizeof(int), 1 , charmm_oc);

/* read controll record */
    icount = fread(&record,sizeof(int), 1 ,charmm_oc);
    if(icount != (sizeof(double) * 8)) {
      printf("?ERROR - in 'BOX' record \n");
       return;}

    icout = fread(&yasp_box,sizeof(double), 6 ,charmm_oc);
    icount = fread(&record,sizeof(int), 1 , charmm_oc);

/* read controll record */
    icount = fread(&record,sizeof(int), 1 ,charmm_oc);
    if(icount != (sizeof(double) * 20)) {
      printf("?ERROR - in 'ENERGIES' record \n");
        return;}

    icount = fread(&yasp_energ,sizeof(double), 20 , charmm_oc);
    icount = fread(&record,sizeof(int), 1 , charmm_oc);

/* read controll record */
    icount = fread(&record,sizeof(int), 1 ,charmm_oc);
    if(icount != (sizeof(double) * 7)) {
      printf("?ERRO- int 'PRESSURE' record \n");
        return;}

    icount = fread(&yasp_press,sizeof(double), 7 , charmm_oc);
    icount = fread(&record,sizeof(int), 1 , charmm_oc);

/* read controll record */
    icount = fread(&record,sizeof(int), 1 ,charmm_oc);
    if(icount != (sizeof(double) * 3 * natom)) {
      printf("?ERROR - in 'COORDINATES' record \n");
        return;}

    icount = fread(tx,sizeof(double),natom,charmm_oc);
    icount = fread(ty,sizeof(double),natom,charmm_oc);
    icount = fread(tz,sizeof(double),natom,charmm_oc);
    icount = fread(&record,sizeof(int), 1 , charmm_oc);

/* put coordinates into program coordinates */

    for(i = 0 ; i < natom ; i++) {
       x[i] = (float) tx[i];
       x[i] = (float) tx[i];
       x[i] = (float) tx[i];}

/* read controll record */
    icount = fread(&record,sizeof(int), 1 ,charmm_oc);
    icount3 = icount / 3;
    icount = fread(vx,sizeof(double),icount3,charmm_oc);
    icount = fread(vy,sizeof(double),icount3,charmm_oc);
    icount = fread(vz,sizeof(double),icount3,charmm_oc);
    icount = fread(&record,sizeof(int), 1 , charmm_oc);

/* read controll record */
    icount = fread(&record,sizeof(int), 1 ,charmm_oc);
    icount3 = icount / 3;
    icount = fread(util1,sizeof(double),icount3,charmm_oc);
    icount = fread(util2,sizeof(double),icount3,charmm_oc);
    icount = fread(util3,sizeof(double),icount3,charmm_oc);
    icount = fread(&record,sizeof(int), 1 , charmm_oc);


#endif

/*                                  */

    prev_frame = alt; /* just to be able to see which was the previous frame
                         when I jump into here next time */
}
