/*
 * timer.c
 *
 * Forms Object class: TIMER
 *
 * Written by: Mark Overmars
 *
 * Version 2.1 a
 * Date: Sep 29,  1992
 */

#include <malloc.h>
#include <stdio.h>
#include <strings.h>
#include <sys/types.h>
#include <sys/time.h>
#include "forms.h"

/* Extra information need for input boxes. */
typedef struct {
   int on;			/* whether running */
   long lastsec,lastusec;	/* last time the delay was adapted */
   float delay;			/* the time left to wait */
} SPEC;

static void get_time(long *sec, long *usec)
/* returns current time */
{
  struct timeval tp;
  struct timezone tzp;
  gettimeofday(&tp,&tzp);
  *sec = tp.tv_sec;
  *usec = tp.tv_usec;
}

static void draw_timer(FL_OBJECT *ob)
/* draws the timer */
{
  int tt;
  int col;
  char str[32];
  SPEC *sp = ((SPEC *)(ob->spec));
  if (!sp->on || sp->delay>0.0)
    col = ob->col1;
  else if ((int) (sp->delay / FL_TIMER_BLINKRATE) % 2)
    col = ob->col1;
  else
    col = ob->col2;
  fl_drw_box(ob->boxtype,ob->x,ob->y,ob->w,ob->h,col,FL_TIMER_BW);
  if (ob->type == FL_VALUE_TIMER && ob->align == FL_ALIGN_CENTER)
    fl_drw_text_beside(FL_ALIGN_LEFT,ob->x,ob->y,ob->w,ob->h,
			ob->lcol,ob->lsize,ob->lstyle,ob->label);
  else
    fl_drw_text_beside(ob->align,ob->x,ob->y,ob->w,ob->h,
			ob->lcol,ob->lsize,ob->lstyle,ob->label);
  if (ob->type == FL_VALUE_TIMER && sp->delay>0.0)
  {
    if (sp->delay <60.0)
      sprintf(str,"%.1f",sp->delay);
    else
    {
      tt = (int) ((sp->delay+0.05) / 60.0);
      sprintf(str,"%d:%2.1f", tt, sp->delay - 60.0 * tt);
    }
    fl_drw_text(FL_ALIGN_CENTER,ob->x,ob->y,ob->w,ob->h,
			ob->lcol,ob->lsize,ob->lstyle,str);
  }
}

static int handle_timer(FL_OBJECT *ob,int event,float mx,float my,char key)
/* Handles an event */
{
  SPEC *sp = ((SPEC *)(ob->spec));
  long sec,usec;
  float lastdelay;
  switch (event)
  {
    case FL_DRAW:
	if (ob->type != FL_HIDDEN_TIMER) draw_timer(ob);
        return 0;
    case FL_RELEASE:
	if (ob->type != FL_HIDDEN_TIMER && sp->delay <0.0)
	  fl_set_timer(ob,0.0);
	return 0;
    case FL_STEP:
	if (! sp->on) return 0;
        lastdelay = sp->delay;
	get_time(&sec,&usec);
	sp->delay -= (float) (sec - sp->lastsec) +
			(float) (usec - sp->lastusec) / 1000000.0;
	sp->lastsec = sec; sp->lastusec = usec;
        if (sp->delay > 0.0)
	{
	  if (ob->type==FL_VALUE_TIMER &&
		 (int)(10.0*sp->delay) != (int)(10.0*lastdelay))
	    fl_redraw_object(ob);
	}
	else if (lastdelay > 0.0)
	{
	  if (ob->type == FL_HIDDEN_TIMER)
	    fl_set_timer(ob,0.0);
	   else
	    fl_redraw_object(ob);
	  return 1;
        }
        else if ( (int) (lastdelay / FL_TIMER_BLINKRATE) !=
			(int) (sp->delay / FL_TIMER_BLINKRATE) )
	  fl_redraw_object(ob);
	return 0;
    case FL_FREEMEM:
	free(ob->spec);
	return 0;
  }
  return 0;
}

/*------------------------------*/

FL_OBJECT *fl_create_timer(int type, float x, float y,
				float w, float h, char label[])
/* creates an object */
{
  FL_OBJECT *ob;
  ob = fl_make_object(FL_TIMER,type,x,y,w,h,label,handle_timer);
  ob->boxtype = FL_TIMER_BOXTYPE;
  ob->col1 = FL_TIMER_COL1;
  ob->col2 = FL_TIMER_COL2;
  ob->align = FL_TIMER_ALIGN;
  ob->lcol = FL_TIMER_LCOL;
  ob->automatic = 1;

  ob->spec = (int *) fl_malloc(sizeof(SPEC));
  ((SPEC *)(ob->spec))->delay = 0.0;
  ((SPEC *)(ob->spec))->on = 0;

  return ob;
}

FL_OBJECT *fl_add_timer(int type, float x, float y,
				float w, float h, char label[])
/* Adds an object */
{
  FL_OBJECT *ob;
  ob = fl_create_timer(type,x,y,w,h,label);
  fl_add_object(fl_current_form,ob);
  return ob;
}

void fl_set_timer(FL_OBJECT *ob, float delay)
/* Sets the timer clock to the particular delay. (0.0 to reset) */
{
  SPEC *sp = ((SPEC *)(ob->spec));
  sp->delay = delay;
  sp->on = (delay>0.0);
  get_time(&(sp->lastsec),&(sp->lastusec));
  if (ob->type != FL_HIDDEN_TIMER) fl_redraw_object(ob);
}

float fl_get_timer(FL_OBJECT *ob)
/* returns the amount of time left */
{
   if (((SPEC *)(ob->spec))->delay >0.0)
     return ((SPEC *)(ob->spec))->delay;
   else
     return 0.0;
}
