/*
 * slider.c
 *
 * Forms Object class: SLIDER, VALSLIDER
 *
 * Written by: Mark Overmars
 *
 * Version 2.1 b
 * Date: Sep 29, 1992
 */

#include <malloc.h>
#include <sys/types.h>
#include <stdio.h>
#include "forms.h"

/* The special information for sliders. */
typedef struct {
   float min;		/* minimal value of slider */
   float max;		/* maximal value of slider */
   float val;		/* current value of slider */
   float step;		/* step size */
   float slsize;	/* size of the slider */
   int prec;            /* precision when printing value */
   int always;		/* whether always returning value */
} SPEC;

static void draw_slider(FL_OBJECT *ob)
/* Draws a slider */
{
  SPEC *sp = (SPEC *) ob->spec;
  char valstr[32];
  float val;
  float sxx = ob->x, syy = ob->y, sww = ob->w, shh = ob->h;
  float bxx = ob->x, byy = ob->y, bww = ob->w, bhh = ob->h;

  /* Draw the value box */
  if (ob->objclass == FL_VALSLIDER)
  {
    if (ob->type == FL_VERT_SLIDER || ob->type == FL_VERT_FILL_SLIDER ||
		ob->type == FL_VERT_NICE_SLIDER)
      { byy = syy + shh - 25.0; bhh = 25.0; shh -= 25.0; }
    else if (ob->type == FL_HOR_SLIDER || ob->type == FL_HOR_FILL_SLIDER ||
		ob->type == FL_HOR_NICE_SLIDER)
      { bww = 50.0; sxx += 50.0; sww -= 50.0; }
    sprintf(valstr,"%.*f",sp->prec,sp->val);
    fl_drw_box(ob->boxtype,bxx,byy,bww,bhh,ob->col1,FL_SLIDER_BW1);
    fl_drw_text_beside(FL_ALIGN_CENTER,bxx,byy,bww,bhh,
                          ob->lcol,ob->lsize,ob->lstyle,valstr);
  }
  /* Draw the slider */
  if (sp->min == sp->max)
    val = 0.5;
  else
    val = (sp->val - sp->min)/(sp->max - sp->min);
  if (ob->align == FL_ALIGN_CENTER)
    fl_drw_slider( ob->boxtype,sxx,syy,sww,shh,ob->col1,ob->col2,
			ob->type,sp->slsize,val,ob->label);
  else
  {
    fl_drw_slider( ob->boxtype,sxx,syy,sww,shh,ob->col1,ob->col2,
			ob->type,sp->slsize,val,"");
    fl_drw_text_beside(ob->align,ob->x,ob->y,ob->w,ob->h,
			ob->lcol,ob->lsize,ob->lstyle,ob->label);
  }
}

static int handle_mouse(FL_OBJECT *ob,float mx, float my)
/* Handle a mouse position change */
{
  SPEC *sp = (SPEC *) ob->spec;
  float oldval,newval;
  float sxx = ob->x, syy = ob->y, sww = ob->w, shh = ob->h;
  /* Change the slider size */
  if (ob->objclass == FL_VALSLIDER)
  {
    if (ob->type == FL_VERT_SLIDER || ob->type == FL_VERT_FILL_SLIDER ||
		ob->type == FL_VERT_NICE_SLIDER)
      { shh -= 25.0; }
    else if (ob->type == FL_HOR_SLIDER || ob->type == FL_HOR_FILL_SLIDER ||
		ob->type == FL_HOR_NICE_SLIDER)
      { sxx += 50.0; sww -= 50.0; }
  }
  /* Calculate the value */
  if (sp->min == sp->max)
    oldval = 0.5;
  else
    oldval = (sp->val - sp->min)/(sp->max - sp->min);
  fl_get_pos_in_slider(sxx,syy,sww,shh,ob->type,sp->slsize,
			mx,my,oldval,&newval);
  newval = sp->min + newval*(sp->max - sp->min);
  if (sp->step != 0.0) newval = (int) (newval/sp->step +0.5) * sp->step;
  if (newval < sp->min) newval = sp->min;
  if (newval > sp->max) newval = sp->max;
  if (sp->val != newval) 
    { sp->val = newval; fl_redraw_object(ob); return 1; }
  else
     return 0;
}

static int handle_slider(FL_OBJECT *ob,int event,float mx,float my,char key)
/* Handles an event */
{
  SPEC *sp = (SPEC *) ob->spec;
  switch (event)
  {
    case FL_DRAW:
	draw_slider(ob);
	return 0;
    case FL_ENTER:
    case FL_LEAVE:
	return 0;
    case FL_PUSH:
    case FL_MOUSE:
        return ( handle_mouse(ob,mx,my)  && sp->always);
    case FL_RELEASE:
        return (! sp->always);
    case FL_FREEMEM:
	free(ob->spec);
	return 0;
  }
  return 0;
}

/*------------------------------*/

static FL_OBJECT *create_it(int objclass, int type, float x, float y,
				float w, float h, char label[])
/* creates an object */
{
  FL_OBJECT *ob;
  ob = fl_make_object(objclass,type,x,y,w,h,label,handle_slider);
  ob->boxtype = FL_SLIDER_BOXTYPE;
  ob->col1 = FL_SLIDER_COL1;
  ob->col2 = FL_SLIDER_COL2;
  ob->align = FL_SLIDER_ALIGN;
  ob->lcol = FL_SLIDER_LCOL;

  ob->spec = (int *) fl_malloc(sizeof(SPEC));
  ((SPEC *)(ob->spec))->min = 0.0;
  ((SPEC *)(ob->spec))->max = 1.0;
  ((SPEC *)(ob->spec))->val = 0.5;
  ((SPEC *)(ob->spec))->step = 0.0;
  ((SPEC *)(ob->spec))->slsize = FL_SLIDER_WIDTH;
  ((SPEC *)(ob->spec))->prec = 2;
  ((SPEC *)(ob->spec))->always = TRUE;

  return ob;
}

static FL_OBJECT *add_it(int objclass, int type, float x, float y,
				 float w, float h, char label[])
/* Adds an object */
{
  FL_OBJECT *ob;
  ob = create_it(objclass,type,x,y,w,h,label);
  fl_add_object(fl_current_form,ob);
  return ob;
}

FL_OBJECT *fl_create_slider(int type, float x, float y, float w, float h,
				char label[])
  { return create_it(FL_SLIDER,type,x,y,w,h,label); }

FL_OBJECT *fl_add_slider(int type, float x, float y, float w, float h,
				 char label[])
  { return add_it(FL_SLIDER,type,x,y,w,h,label); }

FL_OBJECT *fl_create_valslider(int type, float x, float y, float w, float h,
				char label[])
  { return create_it(FL_VALSLIDER,type,x,y,w,h,label); }

FL_OBJECT *fl_add_valslider(int type, float x, float y, float w, float h,
				 char label[])
  { return add_it(FL_VALSLIDER,type,x,y,w,h,label); }

/*-------------------------------------------*/

void fl_set_slider_value(FL_OBJECT *ob,float val)
{
  ((SPEC *)(ob->spec))->val = val;
  fl_redraw_object(ob);
}
void fl_set_slider_bounds(FL_OBJECT *ob,float min,float max)
{
  ((SPEC *)(ob->spec))->min = min;
  ((SPEC *)(ob->spec))->max = max;
  fl_redraw_object(ob);
}

float fl_get_slider_value(FL_OBJECT *ob)
/* Returns value of the slider */
  { return ((SPEC *)(ob->spec))->val; }

void fl_get_slider_bounds(FL_OBJECT *ob,float *min,float *max)
/* Returns the slider bounds. */
{
  *min = ((SPEC *)(ob->spec))->min;
  *max = ((SPEC *)(ob->spec))->max;
}

void fl_set_slider_return(FL_OBJECT *ob, int value)
/* Sets whether to return value all the time */
  { ((SPEC *)(ob->spec))->always = value; }

void fl_set_slider_step(FL_OBJECT *ob, float value)
/* Sets the step size to which values are rounded. */
  { ((SPEC *)(ob->spec))->step = value; }

void fl_set_slider_size(FL_OBJECT *ob,float size)
/* Sets the portion of the slider box covered by the slider */
{
  if (size <= 0.0) ((SPEC *)(ob->spec))->slsize = 0.001;
  else if (size >= 1.0) ((SPEC *)(ob->spec))->slsize = 1.0;
  else ((SPEC *)(ob->spec))->slsize = size;
  fl_redraw_object(ob);
}

void fl_set_slider_precision(FL_OBJECT *ob, int prec)
/* Only for value sliders. */
  { ((SPEC *)(ob->spec))->prec = prec; fl_redraw_object(ob);}
