/*
 * sldraw.c
 *
 * Drawing routines for sliders
 *
 * Written by: Mark Overmars
 *
 * Version 2.1 b
 * Date: Nov  4, 1992
 */

#include <gl/gl.h>
#include <gl/device.h>
#include "forms.h"

static float flinear(float val, float smin, float smax, float gmin, float gmax)
/* performs linear interpolation */
{
  if (smin == smax) return gmax;
  else return gmin + (gmax - gmin) * (val - smin) / (smax - smin);
}

/***
  Routine to draw a slider
***/

void fl_drw_slider(
 	int boxtype,		/* Boxtype for slider */
	float x, float y, float w, float h, /* Bounding box */
	int col1, int col2,	/* Colors used */
	int sltype,		/* Slider type */
	float size,		/* Size of slider in box [0..1] */
	float val,		/* Value (0=bottom, 1=top) */
        char str[])		/* Label to be drawn in the slider */

/* Draws a slider */
{
  float xsl,ysl,hsl,wsl;
  int slbox;

  /* Calculate the slider size */
  if (sltype == FL_VERT_SLIDER || sltype == FL_VERT_NICE_SLIDER)
  {
    hsl = size * (h - 2.0*FL_SLIDER_BW1);
    ysl = flinear(val,0.0,1.0,y+FL_SLIDER_BW1,y+h-FL_SLIDER_BW1-hsl);
    wsl = w - 2.0 * FL_SLIDER_BW1;
    xsl = x + FL_SLIDER_BW1;
  }
  else if (sltype == FL_HOR_SLIDER || sltype == FL_HOR_NICE_SLIDER)
  {
    wsl = size * (w - 2.0*FL_SLIDER_BW1);
    xsl = flinear(val,0.0,1.0,x+FL_SLIDER_BW1,x+w-FL_SLIDER_BW1-wsl);
    hsl = h - 2.0 * FL_SLIDER_BW1;
    ysl = y + FL_SLIDER_BW1;
  }
  else if (sltype == FL_VERT_FILL_SLIDER)
  {
    hsl = val*(h-2.0*FL_SLIDER_BW1);
    ysl = y + FL_SLIDER_BW1;
    wsl = w - 2.0 * FL_SLIDER_BW1;
    xsl = x + FL_SLIDER_BW1;
  }
  else if (sltype == FL_HOR_FILL_SLIDER)
  {
    wsl = val*(w-2.0*FL_SLIDER_BW1);
    xsl = x + FL_SLIDER_BW1;
    hsl = h - 2.0 * FL_SLIDER_BW1;
    ysl = y + FL_SLIDER_BW1;
  }

  /* Draw the slider */
  fl_drw_box(boxtype,x,y,w,h,col1,FL_SLIDER_BW1);
  if (sltype == FL_VERT_NICE_SLIDER)
  {
    fl_drw_box(FL_FLAT_BOX,x+w/2.0-2.0,y+FL_SLIDER_BW1,
		4.0,h-2.0*FL_SLIDER_BW1,0,0.0);
    fl_drw_box(FL_UP_BOX,xsl,ysl,wsl,hsl,col1,FL_SLIDER_BW1);
    fl_drw_box(FL_DOWN_BOX,xsl+2.0,ysl+hsl/2.0-2.5,
		wsl-5.0,6.0,col2,FL_SLIDER_BW1-2.0);
  }
  else if (sltype == FL_HOR_NICE_SLIDER)
  {
    fl_drw_box(FL_FLAT_BOX,x+FL_SLIDER_BW1,y+h/2.0-2.0,
		w-2.0*FL_SLIDER_BW1,4.0,0,0.0);
    fl_drw_box(FL_UP_BOX,xsl,ysl,wsl,hsl,col1,FL_SLIDER_BW1);
    fl_drw_box(FL_DOWN_BOX,xsl+wsl/2.0-2.5,ysl+2.0,
		5.0,hsl-4.0,col2,FL_SLIDER_BW1-2.0);
  }
  else
  {
    switch (boxtype)
    {
      case FL_UP_BOX:		slbox = FL_UP_BOX; break;
      case FL_DOWN_BOX:		slbox = FL_UP_BOX; break;
      case FL_FRAME_BOX:	slbox = FL_FRAME_BOX; break;
      case FL_ROUNDED_BOX:	slbox = FL_ROUNDED_BOX; break;
      case FL_RFLAT_BOX:	slbox = FL_ROUNDED_BOX; break;
      case FL_RSHADOW_BOX:	slbox = FL_ROUNDED_BOX; break;
      default:			slbox = FL_BORDER_BOX; break;
    }
    fl_drw_box(slbox,xsl,ysl,wsl,hsl,col2,FL_SLIDER_BW2);
  }
  /* Draw the label */
  fl_drw_text(FL_ALIGN_CENTER,xsl,ysl,wsl,hsl,0,FL_SMALL_FONT,
			FL_NORMAL_STYLE,str);
}

/***
  Routine to determine the value indicated by the mouse position.
  The routine return the following possible value
	-2  do a page down (mouse click was below sliding part)
	-1  do a line down (mouse click in down button if available (not yet))
	 0  simple adapt value (mouse click in sliding part)
	 1  line up
	 2  page up
  In all cases, in newval the new value is return as if 0 was returned
  (i.e., you can ignore the returned value and simple set the slider
  to the new value if you don't want page down's etc.).
***/

int fl_get_pos_in_slider(
	float x, float y, float w, float h,  /* Bounding box */
	int sltype,			/* Slider type */
	float size,			/* Slider size */
	float xpos, float ypos,		/* Mouse position */
        float oldval,			/* The old value */
	float *newval)			/* The new value */
{
  float v;
  int ret = 0;
  xpos -= x+FL_SLIDER_BW1; ypos -= y+FL_SLIDER_BW1;
  h -= 2.0*FL_SLIDER_BW1; w -= 2.0*FL_SLIDER_BW1;
  if (sltype == FL_VERT_SLIDER || sltype == FL_VERT_NICE_SLIDER)
  {
    v = flinear(ypos,0.5*size*h,h-0.5*size*h,0.,1.);
    if (ypos < oldval * (1.0-size) * h) ret = -2;
    if (ypos > (oldval * (1.0-size) + size) * h) ret = 2;
  }
  else if (sltype == FL_HOR_SLIDER || sltype == FL_HOR_NICE_SLIDER)
  {
    v = flinear(xpos,0.5*size*w,w-0.5*size*w,0.,1.);
    if (xpos < oldval * (1.0-size) * w) ret = -2;
    if (xpos > (oldval * (1.0-size) + size) * w) ret = 2;
  }
  else if (sltype == FL_VERT_FILL_SLIDER)
    { v = flinear(ypos,0.0,h,0.,1.); }
  else if (sltype == FL_HOR_FILL_SLIDER)
    { v = flinear(xpos,0.0,w,0.,1.); }
  if (v < 0.0) v = 0.0;
  if (v > 1.0) v = 1.0;
  if (getbutton(LEFTSHIFTKEY) || getbutton(RIGHTSHIFTKEY))
    *newval = oldval + (v-oldval)*FL_SLIDER_FINE;
  else
    *newval = v;
  return ret;
}
