/******************************************************************************
* $Header: /usr/local/steve/sdm/src/udape/udres/RCS/udrdb.c,v 1.6 1992/01/24 16:49:05 steve Exp $
*
* file:		udrdb.c
*
* written by:	m baltuch
* date:		3/18/91
*
* Copyright (c)1991 University Corporation for Atmospheric Research
*	Not for Direct Resale. All copies to include this notice.
*
* usage:	udrdb shell_name prog_name class_name var:res [var:res ...]
*
*			shell_name	- name of shell executable (ie sh,csh,
*						tcsh, bash, ksh)
*			prog_name	- program name for resource
*			class_name	- class name for resource
*						if NULL program name will be
*						used with first letter
*						capitalized
*			var:res		- shell variable name:resource name
*					  pairs. the resource name should not
*					  include either the class or program
*					  name.
*
* description:	this program loads a ud resource database and translates the
*		resource name information passed in via argv[] into the
*		approriate resource values.  these values are written to
*	        stdout in an appropriate command string for the shell
*		environment to set shell variables. this allows the program
*		to be used within a shell script with the eval command to set
*		variables for use within the script.
*
*		the program supports the following shell environments:
*
*			bourne shell 	- sh
*			c shell		- csh
*			tc shell	- tcsh
*			korn shell	- ksh
*			bourne again	- bash
*
*		the resources are passed in via the command line in the form:
*
*			variable:resource
*
*		where variable is the name of the shell variable to be set
*		and resource is the name of the ud resource for which the
*		value is desired.
*
*		the environment variable UDRESPATH must be set for a resource
*		file to be loaded.  see the udres package documentation for
*		details.
*
*	        the following example shows how to use the program from within
*		a c shell script:
*
*		eval `udrdb foo Foo var1:res1 var2:res2`
*
*		this will set 2 variables, var1 and var2 to the values of
*		foo.res1 and foo.res2 resources respectively.
*
* input files:	specified resource file(s)
*
* output files:	none
*
* modification history
*  date		name	modification
*
*  3/28/91	msb	added command line argument for shell name
*
*  3/28/91	msb	corrected output syntax for sh, bash, ksh
******************************************************************************/

#include	"udposix.h"
#include	<stdio.h>
#include	<string.h>
#include	<stdlib.h>
#include	"udres.h"

#ifndef TRUE
#define	TRUE		1
#endif

#ifndef FALSE
#define	FALSE		0
#endif

/* define shell types */

#define	BOURNE_SH	0
#define C_SHELL		1
#define KORN_SH		2
#define	BASH_SH		3
#define	TC_SHELL	4

extern char	*udstrdup();

main(argc,argv)

int	argc;
char	*argv[];

{
  short		sh_flg, 				  /* shell type flag */
                i; 				    /* loop control variable */

  char		*var_name, 			      /* shell variable name */
                *res_name, 				    /* resource name */
                *res_val; 			    /* resource value string */

  static UdKeyTab	key_tab = {
    {NULL}
  }; 						/* null key table for udinit */

  static UdPosTab	pos_tab = {
    {NULL}
  }; 					 /* null positional table for udinit */

/* make sure we have something to translate */

  if (argc < 5) {			      /* invalid number of arguments */
    (void) fprintf(stderr,"udrdb: invalid number of arguments\n");
    (void) printf("exit\n"); 				 /* force shell exit */
    exit(TRUE);

  } /*if*/

/* which shell are we using */

  if (!strcmp(argv[1],"sh")) 				     /* bourne shell */
    sh_flg = BOURNE_SH;

  else if (!strcmp(argv[1],"csh")) 				  /* c shell */
    sh_flg = C_SHELL;

  else if (!strcmp(argv[1],"ksh")) 			       /* korn shell */
    sh_flg = KORN_SH;

  else if (!strcmp(argv[1],"bash")) 		       /* bourne again shell */
    sh_flg = BASH_SH;

  else if (!strcmp(argv[1],"tcsh")) 				 /* tc shell */
    sh_flg = TC_SHELL;

  else {						    /* unknown shell */
    (void) fprintf(stderr,"udrdb: Unknown shell type %s\n",argv[1]);
    (void) printf("exit\n"); 			  /* force shell script exit */
    exit(TRUE);

  } /*else*/

/* set up resource database */

  if (!udinit(key_tab,pos_tab,&argc,argv,argv[2],argv[3])) {   /* bad return */
    (void) fprintf(stderr,"udrdb:udinit: Resource database init failure\n");
    (void) printf("exit\n"); 			  /* force shell script exit */
    exit(TRUE);

  } /*if*/

/* main translation loop */

  for (i = 4;i < argc;i++) { 			/* step though val:res pairs */

/* parse pair for variable and resource names */

    var_name = udstrdup(strtok(argv[i],":"));
    res_name = udstrdup(strtok((char *)NULL,"\0"));

/* get resource value string */

    res_val = udgetres(res_name,(char*)NULL);

/* write command string */

    switch (sh_flg) {
    case BOURNE_SH: 					     /* bourne shell */
      (void) printf("%s=%s; export %s;",var_name,res_val,var_name);
      break; /*case BOURNE_SH*/

    case C_SHELL: 						  /* c shell */
      (void) printf("set %s=%s\n",var_name,res_val);
      break; /*case C_SHELL*/

    case KORN_SH: 					       /* korn shell */
      (void) printf("%s=%s; export %s;",var_name,res_val,var_name);
      break; /*case KORN_SH*/

    case BASH_SH: 					       /* bash shell */
      (void) printf("%s=%s; export %s;",var_name,res_val,var_name);
      break; /*case BASH_SH*/

    case TC_SHELL: 					       /* tcsh shell */
      (void) printf("set %s=%s\n",var_name,res_val);
      break; /*case TC_SHELL*/

    default: 					 /* we should never get here */
      break; /*default*/

    } /* switch */

  } /*for*/

/* beat feet */

  return FALSE;

} /* main */
