/*
 * This file contains implementations of supporting functions for the 
 * command-line abstract data type.  The function ClCreate() is in a 
 * separate file to allow for a VMS implementation.
 */

/*LINTLIBRARY*/

#include "udposix.h"
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <stddef.h>		/* for size_t */
#include <ctype.h>		/* for is...() macros/functions */

#include "uderrmsg.h"
#include "udalloc.h"
#include "valuestring.h"
#include "cmdline.h"

#ifndef lint
    static char	rcsid[]	= "$Id: cmdline.c,v 2.12 1992/01/24 16:59:29 steve Exp $";
    static char	afsid[]	= "$__Header$";
#endif

extern char	*udstrdup();

typedef struct {
    int		num;
    char	**strings;
}   StringList;

typedef struct {
    char	*name;		/* Name of key */
    int		ielt;		/* Index of first argument */
    int		nelt;		/* Number of arguments */
    int		iarg;		/* Argument-index of key on command-line */
}   KeyNode;

typedef struct ClTag {
    char	**eltv;		/* Completely-exploded value buffer */
    char	**values;	/* Returned values */
    KeyNode	*key;		/* Keyed-parameter list */
    StringList	*StringLists;	/* String-values for each non-key argument */
    enum {
	unknown,
	keyed,
	positional
    }		CurrParam;	/* Type of current parameter */
    char	**argv;		/* Original argument list */
    int		*iargv;		/* Command-line argument indices corresponding
				 * to value-buffer elements */
    int		argc;		/* Number of original arguments */
    int		nelt;		/* Total number of values */
    int		nkey;		/* Number of keyed-parameters */
    int		ipos;		/* Positional-parameter value-index */
    int		ikey;		/* Keyed-parameter index */
    int		nleft;		/* Number of values left, i.e. number of un-
				 * deleted values to the left and including
				 * the current positional value */
    bool	FixedNumber;	/* Fixed number of positional values? */
    bool	WasError;	/* fatal error occured? */
}   ClObject;			/* the command-line data-object*/

static char	GroupEnd[]	= "";	/* In completely-exploded argument
				 * list, indicates end of list of values */


/*
 * WHAT:   Free a command-line structure.
 *
 * INPUT:  A pointer to a command-line data-object.
 *
 * OUTPUT  The number of remaining arguments in the "argv" vector (i.e. the
 *	   new (argc").
 *
 * HOW:    Release allocated memory.
 */

    int
ClFree(cl)
    ClPtr	cl;		/* command-line structure */
{
    int		argc	= 0;

    if (cl != NULL) {
	if (cl->eltv != NULL)
	    (void)FREE(cl->eltv);
	if (cl->argv != NULL) {
	    int		i;
	    char	**out	= cl->argv;

	    for (i = 0; i < cl->argc; ++i)
		if (cl->argv[i] != NULL)
		    *out++	= cl->argv[i];
	    argc	= out - cl->argv;
	}
	if (cl->iargv != NULL)
	    (void)FREE(cl->iargv);
	if (cl->values != NULL)
	    (void)FREE(cl->values);
	if (cl->key != NULL)
	    (void)FREE(cl->key);
	if (cl->StringLists != NULL) {
	    int		iarg;

	    for (iarg = 0; iarg < cl->argc; ++iarg) {
		StringList	*sl	= cl->StringLists + iarg;

		if (sl != NULL)
		    if (sl->strings != NULL)
			(void)VsFree(sl->num, sl->strings);
	    }

	    (void)FREE(cl->StringLists);
	}

	(void)FREE(cl);
    }

    return argc;
}


/*
 * WHAT:   Create a command-line from the given word-list.  This is the
 *	   common-function called by either the Unix or VMS version of
 *	   ClCreate().
 *
 * INPUT:  The number of words and the word list.  The (argc, argv) combination
 *	   should exclude any non-appropriate words (e.g. the first word of an
 *	   invocation-line, argv[0]).
 *
 * OUTPUT: A pointer to an initialized command-line data-object.
 *
 * HOW:    Allocate memory for the command-line data-object and initialize
 *	   it from the list of command-line words.  The options are handled
 *	   specially so that they may be retrieved in an ordered sequence.
 */

    static ClPtr	
createcl(argc, argv)
    int		argc;		/* number of words */
    char	**argv;		/* pointers to words */
{
    int		iarg;
    int		ikey		= -1;
    int		ielt		= 0;
    int		nval;
    bool	success		= NO;
    bool	KeySeen		= NO;
    bool	HaveValues	= NO;
    ClPtr	cl		= UD_ALLOC(1, ClObject);
    static char	me[]		= "createcl";

    assert(argc >= 0);
    assert(!(argc > 0 && argv == NULL));

    if (cl == NULL) {
	udadvise("%s: Couldn't allocate command-line structure.", me);
    } else {
	cl->eltv	= NULL;
	cl->iargv	= NULL;
	cl->values	= NULL;
	cl->key		= NULL;
	cl->StringLists	= NULL;
	cl->CurrParam	= unknown;
	cl->nelt	= 0;
	cl->nkey	= 0;
	cl->ikey	= -1;
	cl->ipos	= 0;
	/* cl->nleft	= 0; */
	cl->WasError	= NO;
	cl->argv	= argv;
	cl->argc	= argc;

	if (argc <= 0) {
	    success	= YES;
	} else {
	    if ((cl->StringLists = UD_ALLOC(argc, StringList)) == NULL) {
		udadvise("%s: Couldn't allocate list of strings.", me);

	    } else {
		/*
		 * Size the completely-exploded argument list and create a 
		 * string-list for each non-key argument.  Special-case nil 
		 * arguments, the end-of-group argument "--", negative
		 * numbers, and a lone '-'.
		 */
		for (iarg = 0; iarg < argc; ++iarg) {
		    if (argv[iarg] != NULL) {
			if (strcmp(argv[iarg], "--") == 0) {
			    cl->StringLists[iarg].strings	= NULL;
			    cl->StringLists[iarg].num	= 0;
			    ++cl->nelt;

			} else if (argv[iarg][0] == '-' &&
				   argv[iarg][1] != 0   &&
				   argv[iarg][1] != '.' &&
				   !isdigit(argv[iarg][1])) {
			    cl->StringLists[iarg].strings	= NULL;
			    ++cl->nkey;

			} else {
			    cl->StringLists[iarg].strings	= 
				VsSplit(argv[iarg], &cl->StringLists[iarg].num);
			    cl->nelt		+= cl->StringLists[iarg].num;
			}
		    }
		}

		/* cl->nleft	= cl->nelt; */

		/* Create the keyed-parameter information array */
		if (cl->nkey > 0 &&
		    (cl->key = UD_ALLOC(cl->nkey, KeyNode)) == NULL) {

		    udadvise("%s: Couldn't allocate keyed-parameter array.", 
			me);

		} else {
		    /*
		     * Create the completely-exploded argument list.  For the
		     * end-of-group argument, "--", use the address of the 
		     * end-of-group array, GroupEnd[]. 
		     */
		    if (cl->nelt > 0 &&
			((cl->eltv = UD_ALLOC(cl->nelt, char*)) == NULL ||
			 (cl->iargv = UD_ALLOC(cl->nelt, int))) == NULL) {

			udadvise(
			    "%s: Couldn't allocate completely-exploded list.",
			    me);

		    } else {
			for (iarg = 0; iarg < argc; ++iarg) {

			    /* Use only non-nil command-line arguments */
			    if (argv[iarg] != NULL) {

				if (strcmp(argv[iarg], "--") == 0) {
				    cl->eltv[ielt++]	= GroupEnd;
				    KeySeen		= NO;
				    HaveValues	= NO;

				} else if (argv[iarg][0] == '-' &&
					   argv[iarg][1] != 0   &&
					   !isdigit(argv[iarg][1])) {
				    ++ikey;
				    cl->key[ikey].name	=
					udstrdup(argv[iarg]+1);
				    cl->key[ikey].ielt	= ielt;
				    cl->key[ikey].iarg	= iarg;
				    cl->key[ikey].nelt	= 0;
				    KeySeen		= YES;
				    HaveValues	= NO;

				} else {
				    int	i;
				    nval	= cl->StringLists[iarg].num;

				    for (i = ielt; i < ielt + nval; ++i) {
					cl->eltv[i]	= 
					    cl->StringLists[iarg].
					    strings[i-ielt];
					cl->iargv[i]	= iarg;
				    }
				    if (KeySeen) {
					cl->key[ikey].nelt	= nval;
					KeySeen	= NO;
					HaveValues	= NO;
				    } else {
					HaveValues	= YES;
				    }
				    ielt	+= nval;
				}
			    }			/* argument is non-nil */
			}			/* argument loop */

			success	= YES;
		    }				/* completely-exploded
						 * argument list allocated */
		}				/* keyed-parameter structure
						 * allocated */
		if (KeySeen && HaveValues)
		    cl->key[ikey].nelt	= nval;

		if (success) {
		    if (cl->nelt > 0 &&
			(cl->values = UD_ALLOC(cl->nelt, char*)) == NULL) {

			udadvise("%s: Couldn't allocate value-vector.", me);
			success	= NO;
		    }
		}
	    }					/* list-of-strings allocated */
	}					/* argc > 0 */

	if (!success)
	    (void) ClFree(cl);
    }						/* command-line struct
						 * allocated */

    return success ? cl : NULL;
}


/*
 */

    int
ClNextKey(cl)
    ClPtr	cl;		/* command-line structure */
{
    bool	success	= NO;		/* routine status = no more options */

    assert(cl != NULL);

    if (cl->ikey < cl->nkey - 1) {
	++cl->ikey;
	cl->CurrParam	= keyed;
	success	= YES;
    }

    return success;
}


/*
 */

    int
ClNextPos(cl)
    ClPtr	cl;		/* command-line structure */
{
    bool	success	= NO;		/* routine status = no more */

    assert(cl != NULL);

    for (; cl->ipos < cl->nelt; ++cl->ipos) {

	if (cl->eltv[cl->ipos] != NULL && cl->eltv[cl->ipos] != GroupEnd) {
	    int		ielt;
	    int		nval	= 1;

	    cl->FixedNumber	= NO;

	    for (ielt = cl->ipos + 1; ielt < cl->nelt; ++ielt) {
		if (cl->eltv[ielt] == GroupEnd) {
		    cl->FixedNumber	= YES;
		    break;
		} else if (cl->eltv[ielt] != NULL) {
		    ++nval;
		}
	    }

	    cl->CurrParam	= positional;
	    cl->nleft		= nval;
	    success		= YES;
	    break;
	}
    }

    return success;
}


/*
 * WHAT	   Find a keyed-parameter.
 */

    int
ClKeyFind(cl, name)
    ClPtr	cl;		/* Command-line object */
    char	*name;		/* Keyed-parameter name */
{
    bool	success	= NO;		/* Routine status = not found */
    KeyNode	*kn;			/* Traverses key-node list */

    assert(cl != NULL);
    assert(name != NULL);

    for (kn = cl->key; kn < cl->key + cl->nkey; ++kn) {
	if (strncmp(name, kn->name, strlen(kn->name)) == 0) {
	    cl->ikey		= kn - cl->key;
	    cl->CurrParam	= keyed;
	    success		= YES;
	}
    }

    return success;
}


/*
 * WHAT:   Return the name of the current named item.
 *
 * HOW:    Return the name in the current-node structure.
 *
 * INPUT:  A pointer to a valid command-line object which has a named
 *	   item as its current item.
 *
 * OUTPUT: The name of the current named item.
 */

    char*
ClName(cl)
    ClPtr	cl;		/* command-line structure */
{
    assert(cl != NULL);
    assert(cl->CurrParam == keyed);

    return cl->key[cl->ikey].name;
}


/*
 * Return the apparent number of values of the current item.
 */

    int
ClNval(cl)
    ClPtr	cl;		/* command-line structure */
{
    int		nval	= 0;

    assert(cl != NULL);
    assert(cl->CurrParam == keyed || cl->CurrParam == positional);

    if (cl->CurrParam == keyed) {
	nval	= cl->key[cl->ikey].nelt;
    } else {
	nval	= cl->nleft;
    }

    return nval;
}


/*
 * Return the maximum possible number of values left on the command-line.
 */

    int
ClNleft(cl)
    ClPtr	cl;		/* command-line structure */
{
    int		ielt;
    int		nval	= 0;

    assert(cl != NULL);
    assert(cl->CurrParam == keyed || cl->CurrParam == positional);

    if (cl->CurrParam == keyed) {
	ielt	= cl->key[cl->ikey].ielt;
    } else {
	ielt	= cl->ipos;
    }

    for (; ielt < cl->nelt; ++ielt)
	if (cl->eltv[ielt] != GroupEnd && cl->eltv[ielt] != NULL)
	    ++nval;

    return nval;
}


/*
 * Return the values of the current item.
 */

    char**
ClValues(cl)
    ClPtr	cl;		/* command-line structure */
{
    int		nelt;			/* Number of values */
    int		ielt;			/* Starting index */
    char	**eltv;
    char	**ValuePtr	= NULL;

    assert(cl != NULL);
    assert(cl->CurrParam == keyed || cl->CurrParam == positional);

    if (cl->CurrParam == keyed) {
	ielt	= cl->key[cl->ikey].ielt;
	nelt	= cl->key[cl->ikey].nelt;
    } else {
	ielt	= cl->ipos;
	nelt	= cl->nleft;
    }

    for (eltv = cl->values; nelt > 0; ++ielt) {
	assert(eltv != NULL);
	assert(ielt < cl->nelt);
	assert(cl->eltv[ielt] != GroupEnd);

	if (cl->eltv[ielt] != NULL) {
	    *eltv++	= cl->eltv[ielt];
	    --nelt;
	}
    }

    ValuePtr	= cl->values;

    return ValuePtr;
}


/*
 * WHAT	   Skip over the given number of values of the current item.
 */

    int
ClSkip(cl, nval)
    ClPtr	cl;		/* Command-line object */
    int		nval;		/* Number of values to skip */
{
    int		success	= YES;		/* Return status */
    static char	me[]	= "ClSkip";

    assert(cl != NULL);
    assert(nval >= 0);
    assert(cl->CurrParam == keyed || cl->CurrParam == positional);

    if (nval > 0) {
	int	ielt;

	if (cl->CurrParam == keyed) {

	    if (nval != cl->key[cl->ikey].nelt) {
		udadvise(
		    "%s: Keyed-parameter \"%s\" has wrong number of values.",
		    me, cl->key[cl->ikey].name);
		cl->WasError	= YES;
		success	= NO;
		nval	= cl->key[cl->ikey].nelt;
	    }

	    ielt	= cl->key[cl->ikey].ielt;

	    /* Delete command-line argument corresponding to key-name */
	    cl->argv[cl->key[cl->ikey].iarg]	= NULL;

	} else {
	    /*
	    if (cl->FixedNumber && nval > cl->nleft) {
		udadvise(
		    "%s: Positional-parameter has too few values.", 
		    me);
		cl->WasError	= YES;
		success	= NO;
		nval	= cl->nleft;
	    }
	    */

	    ielt	= cl->ipos;
	    assert(nval <= cl->nleft);
	}

	/* cl->nleft	-= nval; */

	for (; nval > 0; ++ielt) {
	    assert(ielt < cl->nelt);

	    if (cl->eltv[ielt]	!= NULL) {
		cl->eltv[ielt]	= NULL;

		/* Delete command-line arguments corresponding to values */
		cl->argv[cl->iargv[ielt]]	= NULL;

		--nval;
	    }
	}
    }

    return success;
}


/*
 * Indicate whether or not a fatal error occurred.
 */

    int
ClWasError(cl)
    ClPtr	cl;
{
    assert(cl != NULL);

    return cl->WasError;
}


#if !defined(vms) && !defined(vax11c)

/*
 *	The Following is the UNIX version of ClCreate().
 */


/*
 * WHAT:   Create a command-line from the given word-list.
 *
 * INPUT:  The number of words and the word list.  The (argc, argv) combination
 *	   should exclude any non-appropriate words (e.g. the first word of an
 *	   invocation-line, argv[0]).
 *
 * OUTPUT: A pointer to an initialized command-line data-object.
 *
 * HOW:    Call the common function createcl(argc, argv).
 */

    ClPtr	
ClCreate(argc, argv)
    int		argc;		/* number of words */
    char	**argv;		/* pointers to words */
{
    return createcl(argc, argv);
}

#else	/* defined(vms) || defined(vax11x) */

/*
 *	The Following is the VMS version of ClCreate().
 */

#ifndef lint
#   include <descrip>
#endif

#include <ctype.h>

#define SCREWUP(s)	(((s) & 0x1) == 0)	/* System status check	*/
#define SUCCESS(s)	(((s) & 0x1) != 0)	/* System status check	*/

static char	MetaChars[]	= "\"/ (,)=";

/*
 *	Keep EOS_INDEX equal to the index of the EOS in MetaChars[] because
 *	strchr(MetaChars, 0) might return a pointer to the EOS.
 */
#define EOS_INDEX	(sizeof(MetaChars)-1)	/* Must match table */
#define ALNUM_INDEX	(sizeof(MetaChars)+0)	/* Must match table */
#define OTHER_INDEX	(sizeof(MetaChars)+1)	/* Must match table */
#define NUM_TOKENS	(sizeof(MetaChars)+2)	/* Must match table */

/*
 * Finite-automaton actions:
 */
typedef enum {
    Nop,			/* No operation */
    Add,			/* Add character */
    Addlow,			/* Add lower-case version of character */
    Begkey,			/* Begin a keyed-parameter */
    End,			/* End word.  Advance to next character. */
    Emit,			/* End word.  Don't advance. */
    EndG,			/* End word.  End group next time. */
    Err,			/* Current character is in error */
}   Action;

/*
 * Finite-automaton modes.  The order must match the FA table.
 */
typedef enum {
    NEW,			/* Start mode for each new word */
    IQP,			/* In-quoted-parameter */
    EQP,			/* End-quoted-parameter */
    IP,				/* In-parameter */
    SQ,				/* Start-qualifier */
    IQQ,			/* In-quoted-qualifier */
    EQQ,			/* End-quoted-qualifier */
    IQ,				/* In-qualifier */
    SV,				/* Start-value (of qualifier) */
    IQV,			/* In-quoted-value */
    EQV,			/* End-quoted-value */
    IV,				/* In-value */
    SVLV,			/* Start value-list value */
    IQVLV,			/* In quoted value-list value */
    EQVLV,			/* End quoted value-list value */
    IVLV,			/* In value-list value */
}   Mode;

/*
 * A row in the finite-automaton table:
 */
typedef struct {
    Action	action[NUM_TOKENS];
    Mode	mode[NUM_TOKENS];
}   Row;

/*
 *	Table for driving finite-automaton.  Indexed down by current mode and 
 *	across by current character-type.  Each mode has two rows.  Top row 
 *	indicates action to take; bottom row indicates new mode to enter.  The
 *	NEW mode for the "error" action is just a place-holder.
 */
static Row	fa_tab[] = {
/*
"	/	SPACE	(	,	)	=	EOS	alnum	other
*/
/* NEW (new-word) */
Nop,	Begkey,	Nop,	Err,	Err,	Err,	Err,	Emit,	Addlow,	Addlow,
IQP,	SQ,	NEW,	NEW,	NEW,	NEW,	NEW,	NEW,	IP,	IP,
/* IQP (in-quoted-parameter) */
Nop,	Add,	Add,	Add,	Add,	Add,	Add,	Emit,	Add,	Add,
EQP,	IQP,	IQP,	IQP,	IQP,	IQP,	IQP,	NEW,	IQP,	IQP,
/* EQP (End-quoted-parameter) */
Add,	Emit,	EndG,	Err,	End,	Err,	Err,	Emit,	Addlow,	Addlow,
IQP,	NEW,	NEW,	NEW,	NEW,	NEW,	NEW,	NEW,	IP,	IP,
/* IP (in-parameter) */
Nop,	Emit,	EndG,	Err,	End,	Err,	Err,	Emit,	Addlow,	Addlow,
IQP,	NEW,	NEW,	NEW,	NEW,	NEW,	NEW,	NEW,	IP,	IP,
/* SQ (start-qualifier) */
Nop,	Nop,	Err,	Err,	Err,	Err,	Err,	Err,	Addlow,	Err,
IQQ,	SQ,	NEW,	NEW,	NEW,	NEW,	NEW,	NEW,	IQ,	NEW,
/* IQQ (in-quoted-qualifier) */
Nop,	Add,	Err,	Add,	Add,	Add,	Add,	Emit,	Add,	Add,
EQQ,	IQQ,	NEW,	IQQ,	IQQ,	IQQ,	IQQ,	NEW,	IQQ,	IQQ,
/* EQQ (End-quoted-qualifier) */
Add,	Emit,	EndG,	Err,	End,	Err,	End,	Emit,	Addlow,	Err,
IQQ,	NEW,	NEW,	NEW,	NEW,	NEW,	SV,	NEW,	IQ,	NEW,
/* IQ (in-qualifier) */
Nop,	Emit,	EndG,	Err,	End,	Err,	End,	Emit,	Addlow,	Err,
IQQ,	NEW,	NEW,	NEW,	NEW,	NEW,	SV,	NEW,	IQ,	NEW,
/* SV (start-value) */
Nop,	Err,	Err,	Nop,	Err,	Err,	Err,	Err,	Addlow,	Addlow,
IQV,	NEW,	NEW,	SVLV,	NEW,	NEW,	NEW,	NEW,	IV,	IV,
/* IQV (in-quoted-value) */
Nop,	Add,	Add,	Add,	Add,	Add,	Add,	Emit,	Add,	Add,
EQV,	IQV,	IQV,	IQV,	IQV,	IQV,	IQV,	NEW,	IQV,	IQV,
/* EQV (End-quoted-value) */
Add,	Emit,	EndG,	Err,	End,	Err,	Err,	Emit,	Addlow,	Addlow,
IQV,	NEW,	NEW,	NEW,	NEW,	NEW,	NEW,	NEW,	IV,	IV,
/* IV (in-value) */
Nop,	Emit,	EndG,	Err,	End,	Err,	Err,	Emit,	Addlow,	Addlow,
IQV,	NEW,	NEW,	NEW,	NEW,	NEW,	NEW,	NEW,	IV,	IV,
/* SVLV (start value-list value) */
Nop,	Err,	Nop,	Err,	Err,	Err,	Err,	Err,	Addlow,	Addlow,
IQVLV,	NEW,	SVLV,	NEW,	NEW,	NEW,	NEW,	NEW,	IVLV,	IVLV,
/* IQVLV (in quoted value-list value) */
Nop,	Add,	Add,	Add,	Add,	Add,	Add,	Err,	Add,	Add,
EQVLV,	IQVLV,	IQVLV,	IQVLV,	IQVLV,	IQVLV,	IQVLV,	NEW,	IQVLV,	IQVLV,
/* EQVLV (End quoted value-list value) */
Add,	Err,	Err,	Err,	Add,	EndG,	Err,	Err,	Add,	Add,
IQVLV,	NEW,	NEW,	NEW,	SVLV,	NEW,	NEW,	NEW,	IVLV,	IVLV,
/* IVLV (in value-list value) */
Nop,	Err,	Err,	Err,	Add,	EndG,	Err,	Err,	Add,	Add,
IQVLV,	NEW,	NEW,	NEW,	SVLV,	NEW,	NEW,	NEW,	IVLV,	IVLV,
/*
"	/	SPACE	(	,	)	=	EOS	alnum	other
*/
};


/*
 * WHAT	   End the current command-line word.
 *
 * INPUT   Pointer to current-word buffer; pointer to next character.
 *
 * OUTPUT  EOS-terminated current-word buffer.  Incremented next character
 *	   pointer.
 */

    static void
endword(word, wp)
    char	**word;
    char	**wp;
{
#if DEBUG
    (void)printf("Ending word\n");
#endif
    *(*wp)++	= 0;
    *word	= realloc(*word, (size_t)(*wp-*word));	/* Trim storage */
}


/*
 * WHAT	   Return the next UNIX-word extracted from a DCL command-line.
 *
 * INPUT   On first call: the command-line; on subsequent calls: NULL 
 *	   (indicating a continuation -- similar to strtok(3)).  Place to put 
 *	   address of malloc()ed space containing next word.
 *
 * OUTPUT  Status flag: 1 => success; 0 => error.  If the pointer to the next
 *	   word is NULL, then there are no more words.
 *
 * SIDE    Prints error messages using the Unidata error-messenger.
 *
 * HOW	   Run the finite-automaton by visiting the cell determined by the
 *	   current mode and the current input character, execute the indicated
 *	   function and change to the indicated mode.
 */

    static	bool
NextWord(cmdline, word)
    char	*cmdline;	/* DCL command-line */
    char	**word;		/* Will point to malloc'ed word */
{
    int		errmode	= uderrmode(UD_VERBOSE);
    bool	success	= YES;		/* Routine status = success */
    char	*wp;			/* Points into word buffer */
    static bool	EndGroup;		/* Emit end-of-group? */
    static char	*cp;			/* Traverses command-line */
    static char	*line;			/* Saves command-line */
    static char	me[]	= "NextWord";	/* This routine's name */
    static Mode	mode;			/* Mode of finite automaton */

    /* Initialize if first invocation */
    if (cmdline != NULL) {
	cp		= line	= cmdline;
	mode		= NEW;
	EndGroup	= NO;
    }

    /* Continue only if there's a command-line to parse */
    if (*cp == 0) {
	*word	= NULL;

    /* Emit end-of-group if previous parsing so indicates */
    } else if (EndGroup) {
	*word		= udstrdup("--");
	EndGroup	= NO;

    /* Identify and emit the next word in the command-line */
    } else {
	bool	done	= NO;		/* done with word? */

	/* Allocate storage for word based on remaining command-line length */
	if ((wp = *word = UD_ALLOC(strlen(cp)+1, char)) == NULL) {
	    udadvise("%s: Couldn't allocate word-storage.", me);
	    success	= NO;

	} else {
	    /* Run the FA until done or an error occurs */
	    while (!done && success) {
		int		token;		/* Index of token */
		char	*dp;
		bool	advance	= YES;	/* Default = advance to next letter 
					     * after action */

		/* Map whitespace to space */
		if (isspace(*cp))
		    *cp	= ' ';

#if DEBUG
		(void)printf("Character = 0%o (%c)\n", (unsigned)*cp, *cp);
#endif

		dp	= strchr(MetaChars, *cp);

		/* Determine index of current character (i.e. token) */
		if (dp == NULL) {
		    if (*cp == 0) {
			token	= EOS_INDEX;		/* End-of-string */
		    } else if (isalnum(*cp) || *cp == '$' || *cp == '_') 
							{/* VMS: '$' valid */
			token	= ALNUM_INDEX;		/* Alphanumeric */
		    } else {
			token	= OTHER_INDEX;		/* Other */
		    }
		} else {
		    token	= dp - MetaChars;
		}

#if DEBUG
		(void)printf("mode = %d\n", (int)mode);
		(void)printf("token = %d\n", token);
#endif

		/* Perform the action determined by mode and token */
		switch (fa_tab[(int)mode].action[token]) {

		case Add:
#if DEBUG
		    (void)printf("Adding '%c'\n", *cp);
#endif
		    *wp++	= *cp;
		    break;

		case Addlow:
#if DEBUG
		    (void)printf("Adding lowercase '%c'\n", *cp);
#endif
		    *wp++	= isupper(*cp) ? tolower(*cp) : *cp;
		    break;

		case Begkey:
#if DEBUG
		    (void)printf("Beginning key\n");
#endif
		    *wp++	= '-';
		    break;

		case End:			/* End word */
		    endword(word, &wp);
		    done	= YES;
		    break;

		case Emit:			/* End word. Don't advance */
		    endword(word, &wp);
		    advance	= NO;
		    done	= YES;
		    break;

		case EndG:			/* End word. End group */
		    endword(word, &wp);
		    EndGroup	= YES;
		    done		= YES;
		    break;

		case Err:
		    endword(word, &wp);

		    udadvise("%s: Syntax-error at indicated position:", me);
		    udadvise(line);

		    /* Print the command-line and indicate the point-of-error */
		    {
			int	nspace	= cp - line;
			char	*buf	= UD_ALLOC(nspace+2, char);
			char	*bp	= buf;

			if (buf == NULL) {
			    udadvise("%s: Couldn't allocate print-buffer.", me);
			} else {
			    while (nspace-- > 0)
				*bp++	= ' ';
			    *bp++	= '^';
			    *bp	= 0;

			    udadvise(buf);
			    FREE(buf);
			}
		    }
		    success	= NO;

		default:				/* Do nothing */
		    ;
		}

		/* Change the mode of the FA */
		mode	= fa_tab[(int)mode].mode[token];

		/* Advance to next character if appropriate */
		if (advance)
		    ++cp;
	    }
	}
    }

    (void)uderrmode(errmode);

    return success;
}


/*
 * WHAT	   Add a word (string) to a list of words (strings).
 *
 * INPUT   Pointer to base of malloc()ed string list (may be modified) and 
 *	   pointer to word to be added.
 *
 * SIDE	   Array of string-pointers may be moved as necessary to accomodate
 *	   the additional word.
 *
 * HOW	   If there's no more room in the string-pointer array, reallocate it.
 *	   Add the given pointer to the end of the array.
 */

    static int
AddWord(argvp, wp)
    char	***argvp;	/* List of words */
    char	*wp;		/* Word to be added */
{
    int		success	= YES;		/* Routine status */
    static char	**next;			/* Spot for next pointer */
    static char	**out;			/* Limit of "next" */
    static int	size;			/* Current buffer size */
    static int	NewSize;		/* Next size for buffer */
    static char	me[]	= "AddWord";	/* This routine's name */

    /* Initialize if first invocation */
    if (*argvp == NULL) {
	next	= out	= NULL;
	size	= 0;
	NewSize	= 32;
    }

    /* Insure space for new word-pointer */
    if (next == out) {
	char	**base	= UD_REALLOC(*argvp, NewSize, char*);

	if (base == NULL) {
	    udadvise("%s: Couldn't reallocate array.", me);
	    success	= NO;
	} else {
	    *argvp	= base;
	    next	= base + size;
	    size	= NewSize;
	    out	= base + size;
	    NewSize	*= 1.618034;		/* Golden ratio */
	}
    }

    *next++	= wp;

    return success;
}


/*
 * WHAT    Create a command-line object on a VMS system.
 *
 * INPUT   The contents of the argc/argv arguments are ignored.
 *
 * OUTPUT  A pointer to an initialized command-line data-object or NULL if an
 *	   error occurred.
 *
 * HOW     Obtain the CLI command-line and convert it into shell words, then
 *	   call the common worker-function, createcl(argc, argv).
 */

    ClPtr	
ClCreate(argc, argv)
    int		argc;		/* Number of words.  Not used. */
    char	**argv;		/* Pointers to words.  Not used. */
{
    bool		success	= NO;	/* Routine status */
    int			VmsStatus;
    char		cmdline[10240];
    ClPtr		cl	= NULL;
    unsigned short	len;
    static char		me[]	= "ClCreate";
#ifdef lint
    char		*line_d;
    char		*cmdline_d;
#else
    $DESCRIPTOR(line_d, "$LINE");
    $DESCRIPTOR(cmdline_d, cmdline);
#endif

    VmsStatus	= cli$get_value(&line_d, &cmdline_d, &len);
    if (SCREWUP(VmsStatus)) {
	uderror(me);
	udadvise("%s: Couldn't get command-line.", me);

    } else {
	char	*wp;				/* Points to next word */

	cmdline[len]	= 0;
#if DEBUG
	(void)printf("cmdline = \"%s\"\n", cmdline);
#endif

	argc	= 0;
	argv	= NULL;

	/* Parse the words.  Stop if an error occurs. */
	for (success = NextWord(cmdline, &wp); 
	    success;
	    success = NextWord((char*)NULL, &wp)) {

	    /*
	     *	If there's no more words, then Terminate the argument-list,
	     *	call the command-line creation routine, and stop processing.
	     */
	    if (wp == NULL) {
#if DEBUG
		int		i;

		(void)printf("argc = %d\n", argc);
		for (i = 0; i < argc; ++i)
		    (void)printf("argv[%d] = \"%s\"\n", i, argv[i]);
#endif
		/*
		 *	Terminate the Unix argument-list and call the command-
		 *	line creation routine.
		 */
		if (!AddWord(&argv, (char*)NULL)) {
		    udadvise("%s: Couldn't add word.", me);
		    success	= NO;
		} else {
		    success	= (cl = createcl(argc-1, argv+1)) != NULL;
		}

		break;
	    }

	    /* Add the next word */
	    (void) AddWord(&argv, wp);

	    ++argc;
	}					/* Command-line word-loop */

	/*	Cleanup if something went wrong */
	if (!success && argv != NULL) {
	    int		i;

	    for (i = 0; i < argc; ++i)
		if (argv[i] != NULL)
		    (void)free(argv[i]);

	    (void)free((char*)argv);
	}
    }						/* Got DCL line */

    return cl;
}

#endif	/* defined(vms) || defined(vax11x) */
