#include "snd.h"

static void W_info_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,"Minibuffer",get_info_help());
}

static void W_play_Help_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_help((snd_state *)clientData,"Play",get_play_help());
}

static char info_sep_help[] =
"When reading a very large file, Snd tries to\n\
keep an overview at hand of the channels so\n\
that you can move around quickly in very large\n\
data sets; when first read in, these overviews\n\
are set underway, and when they are finally\n\
ready for use, the line after the file name\n\
appears.  If you try to zoom out to a large\n\
view before the separator line appears, the\n\
graphics update process may be slow.\n\
";

static void W_info_sep_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,"Name Separator",info_sep_help);
}

static char amp_help[] = 
"This scrollbar controls the amplitude\n\
at which the sound is played.\n\
";

static void W_amp_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,"Amp",amp_help);
}

static char srate_help[] =
"This scrollbar controls the sampling rate\n\
at which the sound is played.  The arrow\n\
controls the direction (forwards or backwards)\n\
of playback.\n\
";

static void W_srate_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  ssnd_help((snd_state *)clientData,"Srate",srate_help,"\n",get_speed_menu_help(),NULL);
}

static char srate_arrow_help[] = 
"This button determines which direction\n\
the sound file is played.  When pointing\n\
to the right, the sound is played forwards;\n\
to the left, backwards.\n\
";

static void W_srate_arrow_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,"Srate Arrow",srate_arrow_help);
}

static char expand_help[] =
"This scrollbar controls the tempo at which\n\
the sound is played back, using granular\n\
synthesis. The expand button must be down\n\
to get any expansion.\n\
";

static void W_expand_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  ssnd_help((snd_state *)clientData,"Expand",expand_help,"\n",get_expand_menu_help(),NULL);
}

static void W_expand_button_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,"Expand Button","This button turns on expansion\n");
}

static char contrast_help[] =
"This scrollbar controls the amount of\n\
'contrast enhancement' applied during\n\
playback.  The contrast button must be\n\
down to get any effect.\n\
";

static void W_contrast_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  ssnd_help((snd_state *)clientData,"Contrast",contrast_help,"\n",get_contrast_menu_help(),NULL);
}

static void W_contrast_button_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,"Contrast Button","This button turns on contrast enhancement\n");
}

static char revscl_help[] =
"This scrollbar controls the amount of the\n\
sound that is fed into the reverberator.\n\
The reverb button must be down to get any\n\
reverb during playback.\n\
";

static void W_revscl_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  ssnd_help((snd_state *)clientData,"Revscl",revscl_help,"\n",get_reverb_menu_help(),NULL);
}

static char revlen_help[] =
"This scrollbar controls the lengths of\n\
the various delay lines in the reverb.\n\
It only takes effect when the reverb is\n\
created, that is, only when the play\n\
operation starts from silence.\n\
";

static void W_revlen_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  ssnd_help((snd_state *)clientData,"Revlen",revlen_help,"\n",get_reverb_menu_help(),NULL);
}

static void W_reverb_button_Help_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_help((snd_state *)clientData,"Reverb Button","This button turns on reverberation\n");
}

static char filter_help[] =
"The Snd filter is an FIR filter of arbitrary\n\
order.  You specify the filter you want by\n\
defining the frequency response as an envelope\n\
in the 'env' window; set the desired order in\n\
the 'order' window; then turn it on by pushing\n\
the filter button at the right.  The filter\n\
design algorithm uses frequency sampling.\n\
The higher the order, the closer the filter\n\
can approximate the envelope you draw.\n\
You can also specify the filter coefficients\n\
in a file of floats, then load them into the\n\
Snd filter by typing the file name in the\n\
filter envelope text window.\n\
";

static void W_filter_Help_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_help((snd_state *)clientData,"Filter",filter_help);
} 

static char filter_order_help[] =
"The filter order determines how closely\n\
the filter approximates the frequency response\n\
curve you drew in the 'env' window.\n\
";

static void W_filter_order_Help_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_help((snd_state *)clientData,"Filter Order",filter_order_help);
}

static char filter_envelope_help[] =
"The filter envelope is a line-segment\n\
description of the frequency response\n\
you want.  It consists of a sequence of\n\
x,y pairs; normally the x axis goes\n\
from 0 to .5 or 0 to 1.0.  For example,\n\
a low-pass filter envelope could be:\n\
0.0 1.0 .25 1.0 .5 0.0 1.0 0.0\n\
";

static void W_filter_envelope_Help_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_help((snd_state *)clientData,"Filter Envelope",filter_envelope_help);
}

static void W_filter_button_Help_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_help((snd_state *)clientData,"Filter Button","This button turns on the filter\n");
}

static void W_sync_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Sync Button",
"This button causes edit operations on one\n\
channel to be applied to all channels at the\n\
same time.\n\
");
}

static void W_combine_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Combine Button",
"This button causes all channels to be\n\
displayed in one window, sharing the various\n\
channel controls.  Two extra scrollbars on\n\
the right provide scroll and zoom for the\n\
overall set of channel graphs. The default\n\
multichannel display style can be set in\n\
the Snd initialization file by setting\n\
the variable combine-channels.\n\
");
}

static void W_record_Help_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_help((snd_state *)clientData,
	   "Record",
"The Record button causes Snd to rememeber\n\
any settings or changes to the settings within\n\
the control panel while the sound is playing.\n\
You can subsequently replay that 'take' by pressing\n\
the Replay button, and save it as an edit to\n\
the file with the Apply button.\n\
");
}

static void W_replay_Help_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_help((snd_state *)clientData,
	   "Replay",
"The Replay button replays the last\n\
recorded run (see Record) of the current\n\
sound.\n\
");
}

static void W_apply_Help_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_help((snd_state *)clientData,
	   "Apply",
"The Apply button saves the last recorded\n\
run over the current file (see Record) as\n\
an edit of the current file.\n\
");
}

static void W_set_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Save",
"The 'Save' button saves the current control\n\
panel state for a subsequent 'Restore'.\n\
");
}

static void W_reset_Help_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_help((snd_state *)clientData,
	   "Restore",
"The 'Restore' button returns the control\n\
panel to the state at the time of the\n\
last 'Save', or the initial state if there\n\
has been no 'Save'.\n\
");
}

static void W_name_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  ssnd_help((snd_state *)clientData,
	   "Minibuffer",
"This portion of the snd display has several\n\
parts: the sound file name, with an asterisk if\n\
the file has unsaved edits; a minibuffer for\n\
various expression evaluations; a sync button\n\
that causes operations on one channel to be\n\
applied to all channels; and a play button\n\
that causes the sound to be played.  The\n\
lower portion of the pane, normally hidden,\n\
contains a variety of sound manipulation\n\
controls that can be applied while it is\n\
playing:\n\
\n\
    file name, followed by a separator\n\
\n",
info_sep_help,
"\n\n\
    minibuffer, sync, and play buttons\n\
    amplitude scrollbar\n\
\n",
amp_help,
"\n\n\
    srate scrollbar\n\
\n",
srate_help,
"\n\
    srate arrow button\n\
\n",
srate_arrow_help,
"\n\n\
    expand scrollbar and button\n\
\n",
expand_help,
"\n\n\
    contrast scrollbar and button\n\
\n",
contrast_help,
"\n\n\
    reverb amount scrollbar\n\
\n",
revscl_help,
"\n\
    reverb length scrollbar\n\
\n",
revlen_help,
"\n\n\
    filter section\n\
\n",
filter_help,
"\n\
    filter order text window\n\
\n",
filter_order_help,
"\n\
    filter envelope window\n\
\n",
filter_envelope_help,
NULL);
}




Widget snd_widget(snd_info *sp,int w)
{ 
  if ((sp) && (sp->sgx))
    return((sp->sgx)->snd_widgets[w]);
  else return(NULL);
}


#define SND_TXT_BUF_SIZE 256
static char snd_txt_buf[SND_TXT_BUF_SIZE];

int sound_unlock_ctrls(snd_info *sp, void *ptr)
{
  XtManageChild(snd_widget(sp,W_snd_ctrls));
  XtVaSetValues(snd_widget(sp,W_snd_ctrls),XmNpaneMaximum,1000,XmNpaneMinimum,1,NULL);
  return(0);
}

int sound_lock_ctrls(snd_info *sp, void *ptr)
{
  snd_state *ss;
  ss = (snd_state *)(sp->state);
  XtUnmanageChild(snd_widget(sp,W_snd_ctrls));
  XtVaSetValues(snd_widget(sp,W_snd_ctrls),XmNpaneMaximum,ss->ctrls_height,XmNpaneMinimum,ss->ctrls_height,NULL);
  return(0);
}

static void fill_number(char *fs, char *ps)
{
  int i,j;
  j=strlen(fs);
  if (j>4) j=4;
  if (j<4) {ps[4] = '\0'; ps[3]='0'; ps[2]='0'; ps[1]=snd_string_decimal;}
  if ((*fs) == snd_string_decimal) {*ps++ = '0'; if (j==4) j=3;}
  for (i=0;i<j;i++) (*ps++) = (*fs++);
}

static char number_one[5]={'1',snd_string_decimal,'0','0','\0'};
static char number_zero[5]={'0',snd_string_decimal,'0','0','\0'};

static char amp_number_buffer[5]={'1',snd_string_decimal,'0','0','\0'};

static void reset_amp_number(Widget w) {strcpy(amp_number_buffer,number_one); make_name_label(w,amp_number_buffer);}

void snd_amp_changed(snd_info *sp, int val)
{
  char *sfs;
  if (val == 0) sp->amp = 0.0;
  else sp->amp = exp((float)(val-50)/20.0);
  sfs=prettyf(sp->amp,2);
  fill_number(sfs,amp_number_buffer);
  make_name_label(snd_widget(sp,W_snd_amp_number),amp_number_buffer);
  if (sp->recording) record_amp_change(sp,val,sp->amp);
  free(sfs);
}

static void W_amp_Drag_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_amp_changed((snd_info *)clientData,((XmScrollBarCallbackStruct *)callData)->value);
}

static void W_amp_ValueChanged_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_amp_changed((snd_info *)clientData,((XmScrollBarCallbackStruct *)callData)->value);
}



static char srate_number_buffer[5]={'1',snd_string_decimal,'0','0','\0'};

static void reset_srate_number(Widget w) {strcpy(srate_number_buffer,number_one); make_name_label(w,srate_number_buffer);}

void snd_srate_changed(snd_info *sp, int val)
{
  char *sfs;
  sp->srate = exp((float)(val-450)/150.0);
  sfs=prettyf(sp->srate,2);
  fill_number(sfs,srate_number_buffer);
  make_name_label(snd_widget(sp,W_snd_srate_number),srate_number_buffer);
  if (sp->recording) record_speed_change(sp,val,sp->srate);
  free(sfs);
}

static void W_srate_Drag_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_srate_changed((snd_info *)clientData,((XmScrollBarCallbackStruct *)callData)->value);
}

static void W_srate_ValueChanged_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_srate_changed((snd_info *)clientData,((XmScrollBarCallbackStruct *)callData)->value);
}

void toggle_direction_arrow(snd_info *sp, int state)
{
  XmToggleButtonSetState(snd_widget(sp,W_snd_srate_arrow),state,TRUE);
}



static char expand_number_buffer[5]={'1',snd_string_decimal,'0','0','\0'};

static void reset_expand_number(Widget w) {strcpy(expand_number_buffer,number_one); make_name_label(w,expand_number_buffer);}

void snd_expand_changed(snd_info *sp, int val)
{
  char *sfs;
  sp->expand = exp((float)(val-450)/150.0);
  sfs=prettyf(sp->expand,2);
  fill_number(sfs,expand_number_buffer);
  make_name_label(snd_widget(sp,W_snd_expand_number),expand_number_buffer);
  if ((sp->recording) && (sp->expanding)) record_expand_change(sp,val,sp->expand);
  free(sfs);
}

static void W_expand_Drag_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_expand_changed((snd_info *)clientData,((XmScrollBarCallbackStruct *)callData)->value);
}

static void W_expand_ValueChanged_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_expand_changed((snd_info *)clientData,((XmScrollBarCallbackStruct *)callData)->value);
}


static void Expand_button_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss;
  snd_info *sp = (snd_info *)clientData;
  ss = sp->state;
  sp->expanding = XmToggleButtonGetState(w);
  if (!(ss->using_schemes)) XmChangeColor(snd_widget(sp,W_snd_expand),(Pixel)((sp->expanding) ? ((ss->sgx)->scale) : ((ss->sgx)->main)));
}

void toggle_expand_button(snd_info *sp, int state)
{
  XmToggleButtonSetState(snd_widget(sp,W_snd_expand_button),state,TRUE);
}



static char contrast_number_buffer[5]={'0',snd_string_decimal,'0','0','\0'};

static void reset_contrast_number(Widget w) {strcpy(contrast_number_buffer,number_zero); make_name_label(w,contrast_number_buffer);}

void snd_contrast_changed(snd_info *sp, int val)
{
  char *sfs;
  sp->contrast = (float)val/10.0;
  sfs=prettyf(sp->contrast,2);
  fill_number(sfs,contrast_number_buffer);
  make_name_label(snd_widget(sp,W_snd_contrast_number),contrast_number_buffer);
  if ((sp->recording) && (sp->contrasting)) record_contrast_change(sp,val,sp->contrast);
  free(sfs);
}

static void W_contrast_Drag_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_contrast_changed((snd_info *)clientData,((XmScrollBarCallbackStruct *)callData)->value);
}

static void W_contrast_ValueChanged_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_contrast_changed((snd_info *)clientData,((XmScrollBarCallbackStruct *)callData)->value);
}


static void Contrast_button_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss;
  snd_info *sp = (snd_info *)clientData;
  ss = sp->state;
  sp->contrasting = XmToggleButtonGetState(w);
  if (!(ss->using_schemes)) XmChangeColor(snd_widget(sp,W_snd_contrast),(Pixel)((sp->contrasting) ? ((ss->sgx)->scale) : ((ss->sgx)->main)));
}

void toggle_contrast_button(snd_info *sp, int state)
{
  XmToggleButtonSetState(snd_widget(sp,W_snd_contrast_button),state,TRUE);
}



static char revscl_number_buffer[7]={'0',snd_string_decimal,'0','0','0','0','\0'};
static char number_long_zero[7]={'0',snd_string_decimal,'0','0','0','0','\0'};

static void reset_revscl_number(Widget w) {strcpy(revscl_number_buffer,number_long_zero); make_name_label(w,revscl_number_buffer);}

void snd_revscl_changed(snd_info *sp, int val)
{
  char *fs,*ps,*sfs;
  int i,j;
  sp->revscl = cube((float)val/60.0);
  sfs=prettyf(sp->revscl,3);
  fs=sfs;
  ps=(char *)(revscl_number_buffer);
  j=strlen(fs);
  if (j>6) j=6;
  if (j<6) 
    {
      revscl_number_buffer[5]='0';
      revscl_number_buffer[4]='0'; 
      revscl_number_buffer[3]='0';
      revscl_number_buffer[2]='0'; 
      revscl_number_buffer[1]=snd_string_decimal;
    }
  for (i=0;i<j;i++) (*ps++) = (*fs++);
  make_name_label(snd_widget(sp,W_snd_revscl_number),revscl_number_buffer);
  if ((sp->recording) && (sp->reverbing)) record_reverb_change(sp,val,sp->revscl);
  free(sfs);
}

static void W_revscl_Drag_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_revscl_changed((snd_info *)clientData,((XmScrollBarCallbackStruct *)callData)->value);
}

static void W_revscl_ValueChanged_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_revscl_changed((snd_info *)clientData,((XmScrollBarCallbackStruct *)callData)->value);
}



static char revlen_number_buffer[5]={'1',snd_string_decimal,'0','0','\0'};

static void reset_revlen_number(Widget w) {strcpy(revlen_number_buffer,number_one); make_name_label(w,revlen_number_buffer);}

void snd_revlen_changed(snd_info *sp, int val)
{
  char *sfs;
  sp->revlen = (float)val/20.0;
  sfs=prettyf(sp->revlen,2);
  fill_number(sfs,revlen_number_buffer);
  make_name_label(snd_widget(sp,W_snd_revlen_number),revlen_number_buffer);
  free(sfs);
}

static void W_revlen_Drag_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_revlen_changed((snd_info *)clientData,((XmScrollBarCallbackStruct *)callData)->value);
}

static void W_revlen_ValueChanged_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_revlen_changed((snd_info *)clientData,((XmScrollBarCallbackStruct *)callData)->value);
}



static void Reverb_button_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss;
  snd_info *sp = (snd_info *)clientData;
  ss = sp->state;
  sp->reverbing = XmToggleButtonGetState(w);
  if (!(ss->using_schemes))
    {
      XmChangeColor(snd_widget(sp,W_snd_revlen),(Pixel)((sp->reverbing) ? ((ss->sgx)->scale) : ((ss->sgx)->main)));
      XmChangeColor(snd_widget(sp,W_snd_revscl),(Pixel)((sp->reverbing) ? ((ss->sgx)->scale) : ((ss->sgx)->main)));
    }
}

void toggle_reverb_button(snd_info *sp, int state)
{
  XmToggleButtonSetState(snd_widget(sp,W_snd_reverb_button),state,TRUE);
}

static void Filter_button_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_info *sp = (snd_info *)clientData;
  sp->filtering = XmToggleButtonGetState(w);
}

void toggle_filter_button(snd_info *sp, int state)
{
  XmToggleButtonSetState(snd_widget(sp,W_snd_filter_button),state,TRUE);
}


static void filter_textfield_deactivate(snd_info *sp)
{
  chan_info *active_chan;
  Widget graph;
  active_chan = current_channel(sp);
  graph = chan_widget(active_chan,W_chn_graph);
  if ((XmIsTraversable(graph)) && (XmGetVisibility(graph) != XmVISIBILITY_FULLY_OBSCURED))
    XmProcessTraversal(graph,XmTRAVERSE_CURRENT);
}

static void Filter_Order_activate_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  char *str;
  int order;
  snd_info *sp = (snd_info *)clientData;
  str = XmTextGetString(w);
  if ((str) && (*str))
    {
      sscanf(str,"%d",&order);
      sp->filter_order = order;
    }
  filter_textfield_deactivate(sp);
}

/* TODO: fft-filter (i.e. envelope window superimposed on spectrum) */

static void Filter_activate_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  /* make an envelope out of the data */
  snd_info *sp = (snd_info *)clientData;
  char *str;
  int end_of_text;
  str = XmTextGetString(w);
  while ((str) && ((*str) == ' ')) str++;
  if (isalpha(*str))
    {
      sp->filter_is_envelope = 0;
      sp->filter_env = load_envelope(str);
      if (!(sp->filter_env))
	{
	  end_of_text = XmTextGetLastPosition(w);
	  XmTextInsert(w,end_of_text,snd_string_not_found);
	}
    }
  else
    {
      sp->filter_is_envelope = 1;
      sp->filter_env = scan_envelope(XmTextGetString(w));
    }
  filter_textfield_deactivate(sp);
}

void set_play_button(snd_info *sp, int val)
{
  XmToggleButtonSetState(snd_widget(sp,W_snd_play),val,FALSE);
  set_file_browser_play_button(sp->shortname,val);
}

static void Play_button_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_info *sp = (snd_info *)clientData;
  int play;
  if (sp->playing) stop_playing(sp->playing);
  play = XmToggleButtonGetState(w);
  set_file_browser_play_button(sp->shortname,play);
  goto_graph(any_selected_channel(sp));
  if (play) start_playing(sp,0);
}


static void Play_arrow_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_info *sp = (snd_info *)clientData;
  int dir;
  dir = XmToggleButtonGetState(w);
  if (dir) sp->play_direction = -1; else sp->play_direction = 1;
  if (sp->recording) record_direction_change(sp,dir,sp->play_direction);
}

void syncb(snd_info *sp)
{
  sp->syncing = 1;
  XmToggleButtonSetState(snd_widget(sp,W_snd_sync),TRUE,FALSE);
}

void unsyncb(snd_info *sp)
{
  sp->syncing = 0;
  XmToggleButtonSetState(snd_widget(sp,W_snd_sync),FALSE,FALSE);
}

static void Sync_button_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_info *sp = (snd_info *)clientData;
  chan_info *cp;
  sp->syncing = XmToggleButtonGetState(w);
  cp = any_selected_channel(sp);
  goto_graph(cp);
  if (sp->syncing) apply_x_axis_change(cp->axis,cp,sp);
}

void combineb(snd_info *sp, int val)
{
  XmToggleButtonSetState(snd_widget(sp,W_snd_combine),(val) ? 1 : 0,FALSE);
  if (val) combine_sound(sp); else separate_sound(sp);
  map_over_sound_chans(sp,update_graph,NULL);
}

static void Combine_button_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_info *sp = (snd_info *)clientData;
  if (XmToggleButtonGetState(w)) combine_sound(sp); else separate_sound(sp);
  map_over_sound_chans(sp,update_graph,NULL);
}


static Boolean xrun_amp_env (XtPointer sp)
{
  /* this extra step is needed to get around various X-isms */
  return(run_amp_env((snd_info *)sp));
}

void start_amp_env(snd_info *sp, int anew)
{
  /* fire up work proc for amp env */
  snd_context *sgx;
  sgx = sp->sgx;
  if (!(sgx->env_in_progress))
    {
      XtVaSetValues(snd_widget(sp,W_snd_info_sep),XmNseparatorType,XmNO_LINE,NULL);
      sp->env_anew = anew;
      sgx->env_in_progress = XtAppAddWorkProc(XtWidgetToApplicationContext(main_PANE(sp)),xrun_amp_env,(XtPointer)sp);
    }
}

void signal_amp_env_done(snd_info *sp)
{
  XtVaSetValues(snd_widget(sp,W_snd_info_sep),XmNseparatorType,XmSHADOW_ETCHED_IN,NULL);
}

void remove_amp_env(snd_info *sp)
{
  /* assume someone else is dealing with the arrays */
  snd_context *sgx;
  sgx = sp->sgx;
  if (sgx->env_in_progress)
    {
      signal_amp_env_done(sp);
      XtRemoveWorkProc(sgx->env_in_progress);
      sgx->env_in_progress = 0;
    }
}

void remove_apply(snd_info *sp)
{
  snd_context *sgx;
  if ((sp) && (sgx = sp->sgx) && (sgx->apply_in_progress))
    {
      XtRemoveWorkProc(sgx->apply_in_progress);
      sgx->apply_in_progress = 0;
    }
}

static void Info_activate_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  /* can be response to various things */
  snd_info *sp = (snd_info *)clientData;
  XmAnyCallbackStruct *cb = (XmAnyCallbackStruct *)callData;
  XKeyEvent *ev;
  KeySym keysym;
  ev = (XKeyEvent *)(cb->event);
  keysym = XKeycodeToKeysym(XtDisplay(w),(int)(ev->keycode),0);
  snd_info_activate(sp,keysym);
}

static void Record_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  /* merge (record) changes in ctrl state into current state */
  snd_info *sp = (snd_info *)clientData;
  sp->recording = XmToggleButtonGetState(w);
  if (sp->recording) 
    {
      initialize_record(sp);
      XmToggleButtonSetState(snd_widget(sp,W_snd_apply),FALSE,TRUE);
    }
}

static void Replay_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  /* run dac using current ctrls state (and envs, if any)
   * if any changes made (and record on), merge them into current state 
   */
  snd_info *sp = (snd_info *)clientData;
  sp->replaying = XmToggleButtonGetState(w);
  if (sp->replaying) 
    {
      initialize_replay(sp);
      XmToggleButtonSetState(snd_widget(sp,W_snd_apply),FALSE,TRUE);
    }
  /* assume play explicit (may want to set several of these before starting) */
}

static Boolean xrun_apply(XtPointer sp)
{
  return(apply_controls((void *)sp));
}

static void Apply_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  /* create temp file of run over current file using the current (saved) ctrls state */
  snd_info *sp = (snd_info *)clientData;
  snd_context *sgx;
  sgx = sp->sgx;
  sp->applying = XmToggleButtonGetState(w);
  if (sp->applying) 
    {
      XmToggleButtonSetState(snd_widget(sp,W_snd_replay),FALSE,TRUE);
      XmToggleButtonSetState(snd_widget(sp,W_snd_record),FALSE,TRUE);
      sgx->apply_in_progress = XtAppAddWorkProc(XtWidgetToApplicationContext(main_PANE(sp)),xrun_apply,(XtPointer)make_apply_state(sp));
    }
  else 
    {
      stop_applying(sp);
    }
}

/* apply is only safe if the DAC is currently inactive and remains safe only
 * if all other apply buttons are locked out (and play).
 */

static int lockapply(snd_info *sp, void *up) 
{
  if (sp != up) XtSetSensitive(snd_widget(sp,W_snd_apply),FALSE);
  return(0);
}

void lock_apply(snd_state *ss, snd_info *sp)
{
  /* if playing or applying, set other applys to insensitive */
  map_over_sounds(ss,lockapply,(void *)sp);
}

static int lockplay(snd_info *sp, void *up) 
{
  XtSetSensitive(snd_widget(sp,W_snd_replay),FALSE);
  XtSetSensitive(snd_widget(sp,W_snd_record),FALSE);
  XtSetSensitive(snd_widget(sp,W_snd_play),FALSE);
  return(0);
}

void lock_play_and_record(snd_state *ss, snd_info *sp)
{
  /* if applying, set plays, records, and replays to insensitive */
  map_over_sounds(ss,lockplay,(void *)sp);
}

static int unlockapply(snd_info *sp, void *up) 
{
  if (sp != up) XtSetSensitive(snd_widget(sp,W_snd_apply),TRUE);
  return(0);
}

void unlock_apply(snd_state *ss,snd_info *sp)
{
  map_over_sounds(ss,unlockapply,(void *)sp);
}

static int unlockplay(snd_info *sp, void *up) 
{
  XtSetSensitive(snd_widget(sp,W_snd_replay),TRUE);
  XtSetSensitive(snd_widget(sp,W_snd_record),TRUE);
  XtSetSensitive(snd_widget(sp,W_snd_play),TRUE);
  return(0);
}

void unlock_play_and_record(snd_state *ss,snd_info *sp)
{
  map_over_sounds(ss,unlockplay,(void *)sp);
}


static void Set_Callback(Widget w,XtPointer clientData,XtPointer callData) {save_control_state((snd_info *)clientData);}
static void Reset_Callback(Widget w,XtPointer clientData,XtPointer callData) {restore_control_state((snd_info *)clientData);}

static void add_sound_callbacks(Widget *sw, XtPointer sps)
{
  int i;
  snd_info *sp;
  chan_info *cp;
  snd_state *ss;
  sp=(snd_info *)sps;
  ss=(snd_state *)(sp->state);
  XtAddCallback(sw[W_snd_play],XmNvalueChangedCallback,Play_button_Callback,sps);
  XtAddCallback(sw[W_snd_srate_arrow],XmNvalueChangedCallback,Play_arrow_Callback,sps);
  XtAddCallback(sw[W_snd_expand_button],XmNvalueChangedCallback,Expand_button_Callback,sps);
  XtAddCallback(sw[W_snd_contrast_button],XmNvalueChangedCallback,Contrast_button_Callback,sps);
  XtAddCallback(sw[W_snd_reverb_button],XmNvalueChangedCallback,Reverb_button_Callback,sps);
  XtAddCallback(sw[W_snd_filter_button],XmNvalueChangedCallback,Filter_button_Callback,sps);
  XtAddCallback(sw[W_snd_info],XmNactivateCallback,Info_activate_Callback,sps);
  XtAddCallback(sw[W_snd_info],XmNfocusCallback,textfield_focus_Callback,ss);
  XtAddCallback(sw[W_snd_info],XmNlosingFocusCallback,textfield_unfocus_Callback,ss);
  XtAddCallback(sw[W_snd_filter_order],XmNactivateCallback,Filter_Order_activate_Callback,sps);
  XtAddCallback(sw[W_snd_filter_order],XmNfocusCallback,textfield_focus_Callback,ss);
  XtAddCallback(sw[W_snd_filter_order],XmNlosingFocusCallback,textfield_unfocus_Callback,ss);
  XtAddCallback(sw[W_snd_filter],XmNactivateCallback,Filter_activate_Callback,sps);
  XtAddCallback(sw[W_snd_filter],XmNfocusCallback,textfield_focus_Callback,ss);
  XtAddCallback(sw[W_snd_filter],XmNlosingFocusCallback,textfield_unfocus_Callback,ss);
  XtAddCallback(sw[W_snd_sync],XmNvalueChangedCallback,Sync_button_Callback,sps);
  if (sp->nchans > 1) XtAddCallback(sw[W_snd_combine],XmNvalueChangedCallback,Combine_button_Callback,sps);
  XtAddCallback(sw[W_snd_record],XmNvalueChangedCallback,Record_Callback,sps);
  XtAddCallback(sw[W_snd_replay],XmNvalueChangedCallback,Replay_Callback,sps);
  XtAddCallback(sw[W_snd_apply],XmNvalueChangedCallback,Apply_Callback,sps);
  XtAddCallback(sw[W_snd_set],XmNactivateCallback,Set_Callback,sps);
  XtAddCallback(sw[W_snd_reset],XmNactivateCallback,Reset_Callback,sps);
  for (i=0;i<sp->nchans;i++)
    {
      cp=sp->chans[i];
      add_chan_callbacks(cp);
    }
}

static void remove_sound_callbacks(snd_info *sp, XtPointer sps)
{
  snd_context *sx;
  Widget *sw;
  snd_state *ss;
  ss = (snd_state *)(sp->state);
  sx = sp->sgx;
  sw = sx->snd_widgets;
  XtRemoveCallback(sw[W_snd_play],XmNvalueChangedCallback,Play_button_Callback,sps);
  XtRemoveCallback(sw[W_snd_srate_arrow],XmNvalueChangedCallback,Play_arrow_Callback,sps);
  XtRemoveCallback(sw[W_snd_expand_button],XmNvalueChangedCallback,Expand_button_Callback,sps);
  XtRemoveCallback(sw[W_snd_contrast_button],XmNvalueChangedCallback,Contrast_button_Callback,sps);
  XtRemoveCallback(sw[W_snd_reverb_button],XmNvalueChangedCallback,Reverb_button_Callback,sps);
  XtRemoveCallback(sw[W_snd_filter_button],XmNvalueChangedCallback,Filter_button_Callback,sps);
  XtRemoveCallback(sw[W_snd_info],XmNactivateCallback,Info_activate_Callback,sps);
  XtRemoveCallback(sw[W_snd_info],XmNfocusCallback,textfield_focus_Callback,ss);
  XtRemoveCallback(sw[W_snd_info],XmNlosingFocusCallback,textfield_unfocus_Callback,ss);
  XtRemoveCallback(sw[W_snd_filter_order],XmNactivateCallback,Filter_Order_activate_Callback,sps);
  XtRemoveCallback(sw[W_snd_filter_order],XmNfocusCallback,textfield_focus_Callback,ss);
  XtRemoveCallback(sw[W_snd_filter_order],XmNlosingFocusCallback,textfield_unfocus_Callback,ss);
  XtRemoveCallback(sw[W_snd_filter],XmNactivateCallback,Filter_activate_Callback,sps);
  XtRemoveCallback(sw[W_snd_filter],XmNfocusCallback,textfield_focus_Callback,ss);
  XtRemoveCallback(sw[W_snd_filter],XmNlosingFocusCallback,textfield_unfocus_Callback,ss);
  XtRemoveCallback(sw[W_snd_sync],XmNvalueChangedCallback,Sync_button_Callback,sps);
  if (sp->nchans > 1) XtRemoveCallback(sw[W_snd_combine],XmNvalueChangedCallback,Combine_button_Callback,sps);
  XtRemoveCallback(sw[W_snd_record],XmNvalueChangedCallback,Record_Callback,sps);
  XtRemoveCallback(sw[W_snd_replay],XmNvalueChangedCallback,Replay_Callback,sps);
  XtRemoveCallback(sw[W_snd_apply],XmNvalueChangedCallback,Apply_Callback,sps);
  XtRemoveCallback(sw[W_snd_set],XmNactivateCallback,Set_Callback,sps);
  XtRemoveCallback(sw[W_snd_reset],XmNactivateCallback,Reset_Callback,sps);
}


static char TextTrans1[] =
       "Ctrl <Key>b:            backward-character()\n\
        Alt <Key>b:             backward-word()\n\
        Ctrl <Key>a:            beginning-of-line()\n\
        Shift Ctrl <Key>a:      beginning-of-line(extend)\n\
        Ctrl <Key>d:            delete-next-character()\n\
        Ctrl <Key>osfInsert:    copy-clipboard()\n\
        Shift <Key>osfDelete:   cut-clipboard()\n\
        Shift <Key>osfInsert:   paste-clipboard()\n\
        Ctrl <Key>e:            end-of-line()\n\
        Shift Ctrl <Key>e:      end-of-line(extend)\n\
        Ctrl <Key>f:            forward-character()\n\
        Ctrl Alt <Key>f:        forward-word()\n\
        Ctrl Meta <Key>f:       forward-word()\n\
        Ctrl <Key>g:            activate()\n\
        Ctrl <Key>h:            activate()\n\
        Ctrl <Key>s:            activate()\n\
        Ctrl <Key>r:            activate()\n\
        <Key>Return:            activate()\n";
static XtTranslations transTable1 = NULL;

/* bitmaps for the playback direction arrow */
static unsigned char speed_r_bits[] = {
   0x00, 0x04, 0x10, 0x08, 0x00, 0x10, 0x04, 0x20, 0x00, 0x40, 0xa5, 0xbf,
   0x00, 0x40, 0x04, 0x20, 0x00, 0x10, 0x10, 0x08, 0x00, 0x04, 0x00, 0x00};
static unsigned char speed_l_bits[] = {
   0x20, 0x00, 0x10, 0x08, 0x08, 0x00, 0x04, 0x20, 0x02, 0x00, 0xfd, 0xa5,
   0x02, 0x00, 0x04, 0x20, 0x08, 0x00, 0x10, 0x08, 0x20, 0x00, 0x00, 0x00};

#ifdef SGI
  #define LINE_MARGIN 10
  #define CONTROLS_MARGIN 0
#endif
#ifdef NEXT
  #define LINE_MARGIN 10
  #define CONTROLS_MARGIN 0
#endif
#ifdef LINUX
  #define LINE_MARGIN 4
  #define CONTROLS_MARGIN 1
#endif

#define CHANNEL_SASH_INDENT -10

snd_info *add_sound_window (char *filename, snd_state *state)
{  
  snd_info *sp,*osp;
  file_info *hdr;
  Widget *sw;
  XmString s1;
  int snd_slot,nchans,make_widgets,i,k,need_colors,n,old_chans;
  Arg args[24];
  char *old_name;
  Dimension app_y,app_dy,screen_y,chan_min_y;
  /* these dimensions are used to try to get a reasonable channel graph size without falling off the screen bottom */

  Pixmap rb,lb;
  XGCValues v;
  int depth;
  int samples_per_channel;
  Widget form;
  snd_context *sx;
  snd_slot = find_free_sound_slot(state); /* expands sound list if needed */
  snd_IO_error = snd_no_error;
  errno = 0;
  hdr = make_file_info(filename,state);
  if (!hdr) 
    {
      if (snd_IO_error != snd_pending_open)
	{
	  if (snd_IO_error != snd_no_error)
	    {
	      if (errno != 0)
		sprintf(snd_txt_buf,"%s: %s (%s)",filename,strerror(errno),snd_error_name(snd_IO_error));
	      else sprintf(snd_txt_buf,"%s: %s",filename,snd_error_name(snd_IO_error));
	    }
	  else sprintf(snd_txt_buf,snd_string_cant_find,filename);
	  snd_printf(state,snd_txt_buf);
	}
      return(NULL);
    }
  if (state->pending_change) 
    {
      old_name = filename;
      filename = state->pending_change;
      state->pending_change = NULL;
    }
  nchans = hdr->chans;
  samples_per_channel = hdr->samples/nchans;

  XtVaGetValues(main_SHELL(state),XmNy,&app_y,XmNheight,&app_dy,NULL);
  screen_y = DisplayHeight(main_DISPLAY(state),main_SCREEN(state));
  app_dy = (screen_y - app_y - app_dy - 20*nchans);
  chan_min_y = (Dimension)(app_dy/nchans);
  if (chan_min_y > state->channel_min_height) chan_min_y = state->channel_min_height; else if (chan_min_y < 5) chan_min_y = 5;
  if (state->sounds[snd_slot]) /* we're trying to re-use an old, inactive set of widgets and whatnot */
    {
      osp = state->sounds[snd_slot];
      old_chans = osp->allocated_chans;
    }
  else old_chans = 0;
  make_widgets = (state->sounds[snd_slot] == NULL);
  state->sounds[snd_slot] = make_snd_info(state->sounds[snd_slot],state,filename,hdr,snd_slot);
  sp = state->sounds[snd_slot];
  sp->inuse = 1;
  sx = sp->sgx;
  sw = sx->snd_widgets;
  if ((!make_widgets) && (old_chans < nchans))
    {
      for (i=old_chans;i<nchans;i++) add_channel_window(sp,i,state,chan_min_y,1,NULL);
    }

  if (make_widgets)
    {
      need_colors = (!(state->using_schemes));

      n=0;      
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNallowResize,TRUE); n++;
      XtSetArg(args[n],XmNsashIndent,CHANNEL_SASH_INDENT); n++;
      sw[W_snd_pane] = XtCreateManagedWidget("snd-pane",xmPanedWindowWidgetClass,main_PANE(state),args,n);
      XtAddCallback(sw[W_snd_pane],XmNhelpCallback,W_name_Help_Callback,state);

      n=0;      
      for (i=0;i<nchans;i++) add_channel_window(sp,i,state,chan_min_y,0,NULL);
      
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNpaneMinimum,state->ctrls_height); n++;
      sw[W_snd_ctrls] = XtCreateManagedWidget ("snd-ctrls",xmFormWidgetClass,sw[W_snd_pane],args,n);

      n=0;
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      sw[W_snd_name_form] = XtCreateManagedWidget("snd-name-form",xmFormWidgetClass,sw[W_snd_ctrls],args,n);
      XtAddCallback(sw[W_snd_name_form],XmNhelpCallback,W_name_Help_Callback,state);

      n=0;      
      s1=XmStringCreateLtoR(sp->shortname,XmFONTLIST_DEFAULT_TAG);
      if (need_colors) n = background_high_color(args,n,state);
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNlabelString,s1); n++;
      sw[W_snd_name] = XtCreateManagedWidget ("snd-name",xmLabelWidgetClass,sw[W_snd_name_form],args,n);
      XtAddCallback(sw[W_snd_name],XmNhelpCallback,W_name_Help_Callback,state);
      XmStringFree(s1);

      n=0;      
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNorientation,XmVERTICAL); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNbottomWidget,sw[W_snd_name]); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,sw[W_snd_name]); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNwidth,20); n++; /* was 40 */
      XtSetArg(args[n],XmNseparatorType,XmSHADOW_ETCHED_IN); n++;
      sw[W_snd_info_sep] = XtCreateManagedWidget ("snd-info-sep",xmSeparatorWidgetClass,sw[W_snd_name_form],args,n);
      XtAddCallback(sw[W_snd_info_sep],XmNhelpCallback,W_info_sep_Help_Callback,state);

      n=0;
      s1=XmStringCreate("     ","button_font");
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNbottomWidget,sw[W_snd_info_sep]); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,sw[W_snd_info_sep]); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNlabelString,s1); n++;
      XtSetArg(args[n],XmNfontList,button_FONT(state)); n++;
      sw[W_snd_info_label] = XtCreateManagedWidget ("snd-info-label",xmLabelWidgetClass,sw[W_snd_name_form],args,n);
      XtAddCallback(sw[W_snd_info_label],XmNhelpCallback,W_info_Help_Callback,state);
      XmStringFree(s1);

      n=0;
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,sw[W_snd_info_label]); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNfontList,bold_button_FONT(state)); n++;
      XtSetArg(args[n],XmNresizeWidth,TRUE); n++;
      XtSetArg(args[n],XmNmarginHeight,1); n++;
      XtSetArg(args[n],XmNshadowThickness,0); n++;
      XtSetArg(args[n],XmNcolumns,30); n++;
      XtSetArg(args[n],XmNhighlightThickness,0); n++;
      sw[W_snd_info] = XtCreateManagedWidget ("snd-info",xmTextFieldWidgetClass,sw[W_snd_name_form],args,n);
      if (!transTable1) transTable1 = XtParseTranslationTable(TextTrans1);
      XtOverrideTranslations(sw[W_snd_info],transTable1);
      XtAddCallback(sw[W_snd_info],XmNhelpCallback,W_info_Help_Callback,state);

      n=0;
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
#ifdef LINUX
      XtSetArg(args[n],XmNmarginHeight,0); n++;
      XtSetArg(args[n],XmNmarginTop,0); n++;
#endif
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNfontList,button_FONT(state)); n++;
      XtSetArg(args[n],XmNrecomputeSize,FALSE); n++;
      if (!(state->using_schemes)) {XtSetArg(args[n],XmNselectColor,(state->sgx)->text); n++;}
      sw[W_snd_play] = XtCreateManagedWidget(snd_string_play,xmToggleButtonWidgetClass,sw[W_snd_name_form],args,n);
      XtAddCallback(sw[W_snd_play],XmNhelpCallback,W_play_Help_Callback,state);

      n=0;
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
#ifdef LINUX
      XtSetArg(args[n],XmNmarginHeight,0); n++;
      XtSetArg(args[n],XmNmarginTop,0); n++;
#endif
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNrightWidget,sw[W_snd_play]); n++;
      XtSetArg(args[n],XmNfontList,button_FONT(state)); n++;
      if (!(state->using_schemes)) {XtSetArg(args[n],XmNselectColor,(state->sgx)->text); n++;}
      sw[W_snd_sync] = XtCreateManagedWidget(snd_string_sync,xmToggleButtonWidgetClass,sw[W_snd_name_form],args,n);
      XtAddCallback(sw[W_snd_sync],XmNhelpCallback,W_sync_Help_Callback,state);

      n=0;
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNbottomWidget,sw[W_snd_sync]); n++;
#ifdef LINUX
      XtSetArg(args[n],XmNmarginHeight,0); n++;
      XtSetArg(args[n],XmNmarginTop,0); n++;
#endif
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNrightWidget,sw[W_snd_sync]); n++;
      XtSetArg(args[n],XmNfontList,button_FONT(state)); n++;
      if (!(state->using_schemes)) {XtSetArg(args[n],XmNselectColor,(state->sgx)->text); n++;}
      sw[W_snd_combine] = XtCreateManagedWidget(snd_string_unite,xmToggleButtonWidgetClass,sw[W_snd_name_form],args,n);
      XtAddCallback(sw[W_snd_combine],XmNhelpCallback,W_combine_Help_Callback,state);

      n=0;
      XtVaSetValues(sw[W_snd_ctrls],XmNskipAdjust,TRUE,NULL);

      /* tried a dial widget here, but it didn't seem to fit and was harder to manipulate and read than a scale */
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_name]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNmargin,LINE_MARGIN); n++;
      XtSetArg(args[n],XmNheight,LINE_MARGIN); n++;
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      sw[W_snd_amp_separator] = XtCreateManagedWidget ("snd-amp-sep",xmSeparatorWidgetClass,sw[W_snd_ctrls],args,n);
      XtAddCallback(sw[W_snd_amp_separator],XmNhelpCallback,W_amp_Help_Callback,state);

      n=0;      
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_amp_separator]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      sw[W_snd_amp_form] = XtCreateManagedWidget ("snd-amp",xmFormWidgetClass,sw[W_snd_ctrls],args,n);

      n=0;      
      /* AMP */
      s1=XmStringCreateLtoR(snd_string_amp,XmFONTLIST_DEFAULT_TAG);
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNlabelString,s1); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++;
      XtSetArg(args[n],XmNrecomputeSize,FALSE); n++;
      sw[W_snd_amp_label] = XtCreateManagedWidget ("amp-label",xmLabelWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_amp_label],XmNhelpCallback,W_amp_Help_Callback,state);
      XmStringFree(s1);

      n=0;
      s1=XmStringCreateLtoR(number_one,XmFONTLIST_DEFAULT_TAG);
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_amp_label]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,sw[W_snd_amp_label]); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNlabelString,s1); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++;
      XtSetArg(args[n],XmNrecomputeSize,FALSE); n++;
      sw[W_snd_amp_number] = XtCreateManagedWidget ("amp-number",xmLabelWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_amp_number],XmNhelpCallback,W_amp_Help_Callback,state);
      XmStringFree(s1);

      n=0;      
      if (need_colors) n = background_scale_color(args,n,state);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_amp_label]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNheight,16); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,sw[W_snd_amp_number]); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNmaximum,SCROLLBAR_MAX); n++;
      XtSetArg(args[n],XmNvalue,(int)(SCROLLBAR_MAX*0.5)); n++;
      XtSetArg(args[n],XmNdragCallback,make_callback_list(W_amp_Drag_Callback,(XtPointer)sp)); n++;
      XtSetArg(args[n],XmNvalueChangedCallback,make_callback_list(W_amp_ValueChanged_Callback,(XtPointer)sp)); n++;
      sw[W_snd_amp] = XtCreateManagedWidget("amp",xmScrollBarWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_amp],XmNhelpCallback,W_amp_Help_Callback,state);

      n=0;
      /* SRATE */
      s1=XmStringCreateLtoR(snd_string_speed,XmFONTLIST_DEFAULT_TAG);
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_amp_label]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNlabelString,s1); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++; 
      XtSetArg(args[n],XmNrecomputeSize,FALSE); n++;
      sw[W_snd_srate_label] = XtCreateManagedWidget ("srate-label",xmLabelWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_srate_label],XmNhelpCallback,W_srate_Help_Callback,state);
      XmStringFree(s1);

      n=0;
      s1=XmStringCreateLtoR(number_one,XmFONTLIST_DEFAULT_TAG);
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_srate_label]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,sw[W_snd_srate_label]); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNlabelString,s1); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++; 
      XtSetArg(args[n],XmNrecomputeSize,FALSE); n++;
      sw[W_snd_srate_number] = XtCreateManagedWidget ("srate-number",xmLabelWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_srate_number],XmNhelpCallback,W_srate_Help_Callback,state);
      XmStringFree(s1);

      n=0;
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_srate_label]); n++;
      XtSetArg(args[n],XmNindicatorOn,FALSE); n++;
      XtSetArg(args[n],XmNlabelType,XmPIXMAP); n++;
      XtSetArg(args[n],XmNmarginHeight,0); n++;
      XtSetArg(args[n],XmNmarginWidth,0); n++;
      XtSetArg(args[n],XmNmarginTop,0); n++;
      XtSetArg(args[n],XmNtopOffset,0); n++;
      sw[W_snd_srate_arrow] = XtCreateManagedWidget("dir",xmToggleButtonWidgetClass,sw[W_snd_amp_form],args,n);
      form = sw[W_snd_srate_arrow];
      rb = XCreateBitmapFromData(XtDisplay(form),RootWindowOfScreen(XtScreen(form)),speed_r_bits,16,12);
      lb = XCreateBitmapFromData(XtDisplay(form),RootWindowOfScreen(XtScreen(form)),speed_l_bits,16,12);
      XtVaGetValues(form,XmNforeground,&v.foreground,XmNbackground,&v.background,XmNdepth,&depth,NULL);
      sx->speed_gc = XtGetGC(form,GCForeground | GCBackground,&v);
      sx->speed_r = XCreatePixmap(XtDisplay(form),RootWindowOfScreen(XtScreen(form)),16,12,depth);
      sx->speed_l = XCreatePixmap(XtDisplay(form),RootWindowOfScreen(XtScreen(form)),16,12,depth);
      XCopyPlane(XtDisplay(form),rb,sx->speed_r,sx->speed_gc,0,0,16,12,0,0,1);
      XCopyPlane(XtDisplay(form),lb,sx->speed_l,sx->speed_gc,0,0,16,12,0,0,1);
      XFreePixmap(XtDisplay(form),rb);
      XFreePixmap(XtDisplay(form),lb);
      XtVaSetValues(form,XmNselectPixmap,sx->speed_l,XmNlabelPixmap,sx->speed_r,NULL);
      /* pretty damn tedious -- we can't use the bare pixmap because X dies sputtering incomprehensible jargon */
      XtAddCallback(sw[W_snd_srate_arrow],XmNhelpCallback,W_srate_arrow_Help_Callback,state);

      n=0;
      if (need_colors) n = background_scale_color(args,n,state);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_srate_label]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,sw[W_snd_srate_number]); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNrightWidget,sw[W_snd_srate_arrow]); n++;
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNmaximum,1000); n++;
      XtSetArg(args[n],XmNvalue,450); n++;
      XtSetArg(args[n],XmNheight,16); n++;
      XtSetArg(args[n],XmNdragCallback,make_callback_list(W_srate_Drag_Callback,(XtPointer)sp)); n++;
      XtSetArg(args[n],XmNvalueChangedCallback,make_callback_list(W_srate_ValueChanged_Callback,(XtPointer)sp)); n++;
      sw[W_snd_srate] = XtCreateManagedWidget("srate",xmScrollBarWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_srate],XmNhelpCallback,W_srate_Help_Callback,state);

      n=0;
      /* EXPAND */
      s1=XmStringCreateLtoR(snd_string_expand,XmFONTLIST_DEFAULT_TAG);
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_srate_label]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNlabelString,s1); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++;
      XtSetArg(args[n],XmNrecomputeSize,FALSE); n++;
      sw[W_snd_expand_label] = XtCreateManagedWidget ("expand-label",xmLabelWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_expand_label],XmNhelpCallback,W_expand_Help_Callback,state);
      XmStringFree(s1);
      
      n=0;
      s1=XmStringCreateLtoR(number_one,XmFONTLIST_DEFAULT_TAG);
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_expand_label]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,sw[W_snd_expand_label]); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNlabelString,s1); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++;
      XtSetArg(args[n],XmNrecomputeSize,FALSE); n++;
      sw[W_snd_expand_number] = XtCreateManagedWidget ("expand-number",xmLabelWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_expand_number],XmNhelpCallback,W_expand_Help_Callback,state);
      XmStringFree(s1);
      
      n=0;
      s1=XmStringCreateLtoR("",XmFONTLIST_DEFAULT_TAG);
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_expand_label]); n++;
      XtSetArg(args[n],XmNheight,16); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++;
      XtSetArg(args[n],XmNmarginWidth,0); n++;
      XtSetArg(args[n],XmNtopOffset,1); n++;
      XtSetArg(args[n],XmNspacing,0); n++;
      XtSetArg(args[n],XmNlabelString,s1); n++;
      if (!(state->using_schemes)) {XtSetArg(args[n],XmNselectColor,(state->sgx)->text); n++;}
      sw[W_snd_expand_button] = XtCreateManagedWidget("expoff",xmToggleButtonWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_expand_button],XmNhelpCallback,W_expand_button_Help_Callback,state);
      XmStringFree(s1);

      n=0;
      if (need_colors) n = background_main_color(args,n,state); /* was scale */
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_expand_label]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,sw[W_snd_expand_number]); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNrightWidget,sw[W_snd_expand_button]); n++;
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNmaximum,1000); n++;
      XtSetArg(args[n],XmNvalue,450); n++;
      XtSetArg(args[n],XmNheight,16); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++;
      XtSetArg(args[n],XmNdragCallback,make_callback_list(W_expand_Drag_Callback,(XtPointer)sp)); n++;
      XtSetArg(args[n],XmNvalueChangedCallback,make_callback_list(W_expand_ValueChanged_Callback,(XtPointer)sp)); n++;
      sw[W_snd_expand] = XtCreateManagedWidget("",xmScrollBarWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_expand],XmNhelpCallback,W_expand_Help_Callback,state);

      /* CONTRAST */
      n=0;
      s1=XmStringCreateLtoR(snd_string_contrast,XmFONTLIST_DEFAULT_TAG);
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_expand_label]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNlabelString,s1); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++;
      XtSetArg(args[n],XmNrecomputeSize,FALSE); n++;
      sw[W_snd_contrast_label] = XtCreateManagedWidget ("contrast-label",xmLabelWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_contrast_label],XmNhelpCallback,W_contrast_Help_Callback,state);
      XmStringFree(s1);
      
      n=0;
      s1=XmStringCreateLtoR(number_zero,XmFONTLIST_DEFAULT_TAG);
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_contrast_label]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,sw[W_snd_contrast_label]); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNlabelString,s1); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++;
      XtSetArg(args[n],XmNrecomputeSize,FALSE); n++;
      sw[W_snd_contrast_number] = XtCreateManagedWidget ("contrast-number",xmLabelWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_contrast_number],XmNhelpCallback,W_contrast_Help_Callback,state);
      XmStringFree(s1);
      
      n=0;
      s1=XmStringCreateLtoR("",XmFONTLIST_DEFAULT_TAG);
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_contrast_label]); n++;
      XtSetArg(args[n],XmNheight,16); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++;
      XtSetArg(args[n],XmNmarginWidth,0); n++;
      XtSetArg(args[n],XmNtopOffset,1); n++;
      XtSetArg(args[n],XmNlabelString,s1); n++;
      XtSetArg(args[n],XmNspacing,0); n++;
      if (!(state->using_schemes)) {XtSetArg(args[n],XmNselectColor,(state->sgx)->text); n++;}
      sw[W_snd_contrast_button] = XtCreateManagedWidget("conoff",xmToggleButtonWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_contrast_button],XmNhelpCallback,W_contrast_button_Help_Callback,state);
      XmStringFree(s1);

      n=0;
      if (need_colors) n = background_main_color(args,n,state); /* was scale */
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_contrast_label]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,sw[W_snd_contrast_number]); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNrightWidget,sw[W_snd_contrast_button]); n++;
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNheight,16); n++;
      XtSetArg(args[n],XmNvalue,0); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++;
      XtSetArg(args[n],XmNdragCallback,make_callback_list(W_contrast_Drag_Callback,(XtPointer)sp)); n++;
      XtSetArg(args[n],XmNvalueChangedCallback,make_callback_list(W_contrast_ValueChanged_Callback,(XtPointer)sp)); n++;
      sw[W_snd_contrast] = XtCreateManagedWidget("",xmScrollBarWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_contrast],XmNhelpCallback,W_contrast_Help_Callback,state);

      /* REVERB */
      /* REVSCL */
      n=0;
      s1=XmStringCreateLtoR(snd_string_reverb,XmFONTLIST_DEFAULT_TAG);
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_contrast_label]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNlabelString,s1); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++;
      XtSetArg(args[n],XmNrecomputeSize,FALSE); n++;
      sw[W_snd_revscl_label] = XtCreateManagedWidget ("revscl-label",xmLabelWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_revscl_label],XmNhelpCallback,W_revscl_Help_Callback,state);
      XmStringFree(s1);
      
      n=0;
      s1=XmStringCreateLtoR(number_long_zero,XmFONTLIST_DEFAULT_TAG);
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_revscl_label]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,sw[W_snd_revscl_label]); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNlabelString,s1); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++;
      XtSetArg(args[n],XmNrecomputeSize,FALSE); n++;
      sw[W_snd_revscl_number] = XtCreateManagedWidget ("revscl-number",xmLabelWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_revscl_number],XmNhelpCallback,W_revscl_Help_Callback,state);
      XmStringFree(s1);
      
      n=0;
      if (need_colors) n = background_main_color(args,n,state); /* was scale */
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_revscl_label]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,sw[W_snd_revscl_number]); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_POSITION); n++;
      XtSetArg(args[n],XmNrightPosition,60); n++;
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNheight,16); n++;
      XtSetArg(args[n],XmNvalue,0); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++;
      XtSetArg(args[n],XmNdragCallback,make_callback_list(W_revscl_Drag_Callback,(XtPointer)sp)); n++;
      XtSetArg(args[n],XmNvalueChangedCallback,make_callback_list(W_revscl_ValueChanged_Callback,(XtPointer)sp)); n++;
      sw[W_snd_revscl] = XtCreateManagedWidget("",xmScrollBarWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_revscl],XmNhelpCallback,W_revscl_Help_Callback,state);

      /* REVOFF */
      n=0;
      s1=XmStringCreateLtoR("",XmFONTLIST_DEFAULT_TAG);
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_revscl_label]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNrightWidget,sw[W_snd_contrast_button]); n++;
      XtSetArg(args[n],XmNheight,16); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++;
      XtSetArg(args[n],XmNmarginWidth,0); n++;
      XtSetArg(args[n],XmNtopOffset,1); n++;
      XtSetArg(args[n],XmNspacing,0); n++;
      XtSetArg(args[n],XmNlabelString,s1); n++;
      if (!(state->using_schemes)) {XtSetArg(args[n],XmNselectColor,(state->sgx)->text); n++;}
      sw[W_snd_reverb_button] = XtCreateManagedWidget("revoff",xmToggleButtonWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_reverb_button],XmNhelpCallback,W_reverb_button_Help_Callback,state);
      XmStringFree(s1);


      /* REVLEN */
      n=0;
      s1=XmStringCreateLtoR(snd_string_len,XmFONTLIST_DEFAULT_TAG);
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_revscl]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_POSITION); n++;
      XtSetArg(args[n],XmNleftPosition,60); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNlabelString,s1); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++;
      XtSetArg(args[n],XmNrecomputeSize,FALSE); n++;
      sw[W_snd_revlen_label] = XtCreateManagedWidget ("revlen-label",xmLabelWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_revlen_label],XmNhelpCallback,W_revlen_Help_Callback,state);
      XmStringFree(s1);

      n=0;
      s1=XmStringCreateLtoR(number_one,XmFONTLIST_DEFAULT_TAG);
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_revlen_label]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,sw[W_snd_revlen_label]); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNlabelString,s1); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++;
      XtSetArg(args[n],XmNrecomputeSize,FALSE); n++;
      sw[W_snd_revlen_number] = XtCreateManagedWidget ("revlen-number",xmLabelWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_revlen_number],XmNhelpCallback,W_revlen_Help_Callback,state);
      XmStringFree(s1);

      n=0;
      if (need_colors) n = background_main_color(args,n,state); /* was scale */
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_revlen_label]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,sw[W_snd_revlen_number]); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNrightWidget,sw[W_snd_reverb_button]); n++;
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNheight,16); n++;
      XtSetArg(args[n],XmNvalue,20); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++;
      XtSetArg(args[n],XmNdragCallback,make_callback_list(W_revlen_Drag_Callback,(XtPointer)sp)); n++;
      XtSetArg(args[n],XmNvalueChangedCallback,make_callback_list(W_revlen_ValueChanged_Callback,(XtPointer)sp)); n++;
      sw[W_snd_revlen] = XtCreateManagedWidget("",xmScrollBarWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_revlen],XmNhelpCallback,W_revlen_Help_Callback,state);


      /* FILTER */
      n=0;
      s1=XmStringCreateLtoR(snd_string_filter_order,XmFONTLIST_DEFAULT_TAG);
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_revscl_label]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNlabelString,s1); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++;
      XtSetArg(args[n],XmNrecomputeSize,FALSE); n++;
      sw[W_snd_filter_label] = XtCreateManagedWidget ("filter-label",xmLabelWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_filter_label],XmNhelpCallback,W_filter_Help_Callback,state);
      XmStringFree(s1);

      /* filter order */
      n=0;
      if (need_colors) n = background_main_color(args,n,state);
      /* XtSetArg(args[n],XmNfontList,bold_button_FONT(state)); n++; */
      XtSetArg(args[n],XmNresizeWidth,FALSE); n++;
      XtSetArg(args[n],XmNcolumns,3); n++;
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_filter_label]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,sw[W_snd_filter_label]); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++;
      XtSetArg(args[n],XmNrecomputeSize,FALSE); n++;
      sw[W_snd_filter_order] = XtCreateManagedWidget ("filter-order",xmTextFieldWidgetClass,sw[W_snd_amp_form],args,n);
      XtOverrideTranslations(sw[W_snd_filter_order],transTable1);
      XmTextSetString(sw[W_snd_filter_order],"  2");
      XtAddCallback(sw[W_snd_filter_order],XmNhelpCallback,W_filter_order_Help_Callback,state);

      n=0;
      s1=XmStringCreateLtoR(snd_string_env_p,XmFONTLIST_DEFAULT_TAG);
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_filter_order]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,sw[W_snd_filter_order]); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNlabelString,s1); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++;
      XtSetArg(args[n],XmNrecomputeSize,FALSE); n++;
      sw[W_snd_filter_env] = XtCreateManagedWidget ("filter-env",xmLabelWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_filter_env],XmNhelpCallback,W_filter_envelope_Help_Callback,state);
      XmStringFree(s1);

      n=0;
      s1=XmStringCreateLtoR("",XmFONTLIST_DEFAULT_TAG);
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_filter_env]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNrightWidget,sw[W_snd_reverb_button]); n++;
      XtSetArg(args[n],XmNheight,16); n++;
      XtSetArg(args[n],XmNmarginWidth,0); n++;
      XtSetArg(args[n],XmNtopOffset,2); n++;
      XtSetArg(args[n],XmNspacing,0); n++;
      XtSetArg(args[n],XmNlabelString,s1); n++;
      if (!(state->using_schemes)) {XtSetArg(args[n],XmNselectColor,(state->sgx)->text); n++;}
      sw[W_snd_filter_button] = XtCreateManagedWidget("fltoff",xmToggleButtonWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_filter_button],XmNhelpCallback,W_filter_button_Help_Callback,state);
      XmStringFree(s1);

      n=0;
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNfontList,bold_button_FONT(state)); n++;
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_filter_env]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,sw[W_snd_filter_env]); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNrightWidget,sw[W_snd_filter_button]); n++;
      XtSetArg(args[n],XmNmarginHeight,CONTROLS_MARGIN); n++;
      sw[W_snd_filter] = XtCreateManagedWidget ("filter-window",xmTextFieldWidgetClass,sw[W_snd_amp_form],args,n);
      XtOverrideTranslations(sw[W_snd_filter],transTable1);
      XtAddCallback(sw[W_snd_filter],XmNhelpCallback,W_filter_envelope_Help_Callback,state);

      /* RECORD SEPARATOR */
      n=0;
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_filter]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNmargin,LINE_MARGIN); n++;
      XtSetArg(args[n],XmNheight,LINE_MARGIN); n++;
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      sw[W_snd_record_sep] = XtCreateManagedWidget("snd-rec-sep",xmSeparatorWidgetClass,sw[W_snd_amp_form],args,n);

      #define LEFT_SPACE 3
      #define INTER_SPACE 30

      /* RECORD */
      n=0;
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_record_sep]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_POSITION); n++;
      XtSetArg(args[n],XmNleftPosition,LEFT_SPACE); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      if (!(state->using_schemes)) {XtSetArg(args[n],XmNselectColor,(state->sgx)->text); n++;}
      sw[W_snd_record] = XtCreateManagedWidget(snd_string_Record,xmToggleButtonWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_record],XmNhelpCallback,W_record_Help_Callback,state);

      /* REPLAY */
      n=0;
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_record]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_POSITION); n++;
      XtSetArg(args[n],XmNleftPosition,LEFT_SPACE + INTER_SPACE); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      if (!(state->using_schemes)) {XtSetArg(args[n],XmNselectColor,(state->sgx)->text); n++;}
      sw[W_snd_replay] = XtCreateManagedWidget(snd_string_Replay,xmToggleButtonWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_replay],XmNhelpCallback,W_replay_Help_Callback,state);

      /* APPLY */
      n=0;
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_replay]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_POSITION); n++;
      XtSetArg(args[n],XmNleftPosition,LEFT_SPACE + 2*INTER_SPACE); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      if (!(state->using_schemes)) {XtSetArg(args[n],XmNselectColor,(state->sgx)->text); n++;}
      sw[W_snd_apply] = XtCreateManagedWidget(snd_string_Apply,xmToggleButtonWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_apply],XmNhelpCallback,W_apply_Help_Callback,state);

      /* SET */
      n=0;
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNfontList,button_FONT(state)); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_record_sep]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_POSITION); n++;
      XtSetArg(args[n],XmNleftPosition,90); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      if (!(state->using_schemes)) {XtSetArg(args[n],XmNselectColor,(state->sgx)->text); n++;}
      sw[W_snd_set] = XtCreateManagedWidget(snd_string_s,xmPushButtonWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_set],XmNhelpCallback,W_set_Help_Callback,state);

      /* RESET */
      n=0;
      if (need_colors) n = background_main_color(args,n,state);
      XtSetArg(args[n],XmNfontList,button_FONT(state)); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sw[W_snd_record_sep]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_POSITION); n++;
      XtSetArg(args[n],XmNleftPosition,95); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      if (!(state->using_schemes)) {XtSetArg(args[n],XmNselectColor,(state->sgx)->text); n++;}
      sw[W_snd_reset] = XtCreateManagedWidget(snd_string_r,xmPushButtonWidgetClass,sw[W_snd_amp_form],args,n);
      XtAddCallback(sw[W_snd_reset],XmNhelpCallback,W_reset_Help_Callback,state);
      n=0;

    } /* new sound state */
  else
    { /* re-manage currently inactive chan */
      XtVaSetValues(sw[W_snd_ctrls],XmNpaneMaximum,state->ctrls_height,XmNpaneMinimum,state->ctrls_height,NULL);
      for (i=0;i<NUM_SND_WIDGETS;i++)
	{
	  if ((sw[i]) && (!XtIsManaged(sw[i])))
	    {
	      XtManageChild(sw[i]);
	      switch (i)
		{
		case W_snd_pane: for (k=0;k<nchans;k++) add_channel_window(sp,k,state,chan_min_y,0,NULL); break;
		case W_snd_name: make_name_label(sw[i],sp->shortname); break;
		case W_snd_amp: XtVaSetValues(sw[i],XmNvalue,(int)(SCROLLBAR_MAX*0.5),NULL); break;
		case W_snd_srate: XtVaSetValues(sw[i],XmNvalue,450,NULL); break;
		case W_snd_expand: XtVaSetValues(sw[i],XmNvalue,450,NULL); break;
		case W_snd_contrast: XtVaSetValues(sw[i],XmNvalue,0,NULL); break;
		case W_snd_revscl: XtVaSetValues(sw[i],XmNvalue,0,NULL); break;
		case W_snd_revlen: XtVaSetValues(sw[i],XmNvalue,20,NULL); break;
		case W_snd_amp_number: reset_amp_number(sw[i]); break;
		case W_snd_srate_number: reset_srate_number(sw[i]); break;
		case W_snd_expand_number: reset_expand_number(sw[i]); break;
		case W_snd_contrast_number: reset_contrast_number(sw[i]); break;
		case W_snd_revscl_number: reset_revscl_number(sw[i]); break;
		case W_snd_revlen_number: reset_revlen_number(sw[i]); break;
		  /* set combine button dependent on sp->combining? */
		default: break;
		}
	    }
	}
      XtVaSetValues(sw[W_snd_ctrls],XmNpaneMaximum,1000,XmNpaneMinimum,1,NULL);
    }
  if (sp->nchans == 1) 
    {
      XmToggleButtonSetState(snd_widget(sp,W_snd_combine),FALSE,FALSE);
      XtUnmanageChild(snd_widget(sp,W_snd_combine));
    }
  add_sound_data(filename,sp,state);
  add_sound_callbacks(sw,(XtPointer)sp);
  if (samples_per_channel > 100000) start_amp_env(sp,0);
  state->pending_open = NULL;
  if (state->pending_change)
    {
      sprintf(snd_txt_buf,"(translated %s)",old_name);
      report_in_minibuffer(sp,snd_txt_buf);
    }
  return(sp);
}

void snd_info_cleanup(snd_info *sp)
{
  Widget w;
  int i;
  if ((sp) && (sp->sgx))
    {
      XtVaSetValues(snd_widget(sp,W_snd_sync),XmNset,FALSE,NULL);
      XtVaSetValues(snd_widget(sp,W_snd_expand_button),XmNset,FALSE,NULL);
      XtVaSetValues(snd_widget(sp,W_snd_contrast_button),XmNset,FALSE,NULL);
      XtVaSetValues(snd_widget(sp,W_snd_srate_arrow),XmNset,FALSE,NULL);
      XtVaSetValues(snd_widget(sp,W_snd_filter_button),XmNset,FALSE,NULL);
      XtVaSetValues(snd_widget(sp,W_snd_reverb_button),XmNset,FALSE,NULL);
      XmToggleButtonSetState(snd_widget(sp,W_snd_combine),FALSE,FALSE);
      sp->combining = 0;
      remove_sound_callbacks(sp,(XtPointer)sp);
      for (i=NUM_SND_WIDGETS-1;i>=0;i--)
	{
	  w = snd_widget(sp,i);
	  if ((w) && (XtIsManaged(w))) XtUnmanageChild(w);
	}
    }
}

void unlock_ctrls(snd_info *sp) {XtVaSetValues(snd_widget(sp,W_snd_ctrls),XmNpaneMinimum,1,NULL);}

static void replay_amp_callback(XtPointer clientData, XtIntervalId *id) {replay_amp((void *)clientData);}
void call_replay_amp(snd_state *ss, XtPointer ap,int time) {XtAppAddTimeOut(main_APP(ss),time,replay_amp_callback,ap);}

static void replay_speed_callback(XtPointer clientData, XtIntervalId *id) {replay_speed((void *)clientData);}
void call_replay_speed(snd_state *ss, XtPointer ap,int time) {XtAppAddTimeOut(main_APP(ss),time,replay_speed_callback,ap);}

static void replay_expand_callback(XtPointer clientData, XtIntervalId *id) {replay_expand((void *)clientData);}
void call_replay_expand(snd_state *ss, XtPointer ap,int time) {XtAppAddTimeOut(main_APP(ss),time,replay_expand_callback,ap);}

static void replay_contrast_callback(XtPointer clientData, XtIntervalId *id) {replay_contrast((void *)clientData);}
void call_replay_contrast(snd_state *ss, XtPointer ap,int time) {XtAppAddTimeOut(main_APP(ss),time,replay_contrast_callback,ap);}

static void replay_reverb_callback(XtPointer clientData, XtIntervalId *id) {replay_reverb((void *)clientData);}
void call_replay_reverb(snd_state *ss, XtPointer ap,int time) {XtAppAddTimeOut(main_APP(ss),time,replay_reverb_callback,ap);}

static void replay_direction_callback(XtPointer clientData, XtIntervalId *id) {replay_direction((void *)clientData);}
void call_replay_direction(snd_state *ss, XtPointer ap,int time) {XtAppAddTimeOut(main_APP(ss),time,replay_direction_callback,ap);}

void set_apply_button(snd_info *sp, int val) {XmToggleButtonSetState(snd_widget(sp,W_snd_apply),val,FALSE);}


void normalize_sound(snd_state *ss, snd_info *sp, snd_info *osp, chan_info *ncp)
{
  /* make sp look ok, squeezing others if needed; called only if ss->auto_open */
  /* if there's already enough (i.e. ss->channel_min_height), just return */
  /* this is used in goto_next_graph and goto_previous_graph (snd-chn.c) to open windows that are currently squeezed shut */
  int chans,i;
  float val,size,high,low;
  Dimension chan_y,snd_y,snd_ctrl,height,screen_y,adjustment;
  int *wid;
  Widget *cw,*sw;
  Widget wcp;
  chan_info *cp;
  chan_context *cx;
  snd_context *sx;
  snd_info *nsp;

  if ((!ss) || (!sp) || (!(ss->auto_open))) return;
  if ((!osp) || (!sp->combining) || (osp != sp)) /* new sound selected -- ignore scroll possibility */
    {
      adjustment = 0;
      if (!ncp) /* separate_sound in snd-xchn == cancel unite */
	{
	  for (i=0;i<sp->nchans;i++)
	    {
	      cp = sp->chans[i];
	      cx = cp->cgx;
	      cw = cx->chan_widgets;
	      XtVaGetValues(cw[W_chn_main_window],XmNheight,&chan_y,NULL);
	      if (chan_y < (ss->channel_min_height>>1)) adjustment += ((ss->channel_min_height>>1) - chan_y);
	    }
	}
      else 
	{
	  cx = ncp->cgx;
	  cw = cx->chan_widgets;
	  XtVaGetValues(cw[W_chn_main_window],XmNheight,&chan_y,NULL);
	  if (chan_y < (ss->channel_min_height>>1)) adjustment = ((ss->channel_min_height>>1) - chan_y);
	}
      if (adjustment)
	{
	  /* now find how much space is available */
	  height = sp->nchans * ss->channel_min_height;
	  wid = (int *)calloc(1,sizeof(int));
	  sx = sp->sgx;
	  sw = sx->snd_widgets;
	  XtVaGetValues(sw[W_snd_pane],XmNheight,&snd_y,NULL);
	  XtVaGetValues(sw[W_snd_ctrls],XmNheight,&snd_ctrl,NULL);
	  if (((snd_y-snd_ctrl) >= height) || (ss->active_sounds == 1))
	    {
	      /* redivide space in current snd pane */
	      if (ss->active_sounds == 1)
		chan_y = ss->channel_min_height;
	      else chan_y = (snd_y-snd_ctrl)/sp->nchans;
	      (*wid) = chan_y;
	      map_over_sound_chans(sp,channel_lock_pane,(void *)wid);
	      map_over_sound_chans(sp,channel_unlock_pane,NULL);
	    }
	  else
	    {
	      /* not space in current sound pane -- grab enough, then reset others */
	      snd_y = height+10;
	      height = get_window_height(main_PANE(ss));
	      screen_y = DisplayHeight(main_DISPLAY(ss),main_SCREEN(ss));
	      if (height > screen_y) height = screen_y;
	      (*wid) = snd_y/sp->nchans;
	      map_over_sound_chans(sp,channel_lock_pane,(void *)wid);
	      map_over_sound_chans(sp,channel_unlock_pane,NULL);
	      chans = active_channels(ss,0) - sp->nchans;
	      (*wid) = (height - snd_y)/chans;
	      for (i=0;i<ss->max_sounds;i++)
		{
		  if (nsp=((snd_info *)(ss->sounds[i])))
		    {
		      if (nsp != sp)
			{
			  map_over_sound_chans(nsp,channel_lock_pane,(void *)wid);
			  map_over_sound_chans(nsp,channel_unlock_pane,NULL);
			}
		    }
		}
	    }
	  free(wid);
	}
    }
  else /* sp->combining, current sound, channel changed, so auto-scroll display to show current channel */
    {
      cp = any_selected_channel(sp);
      high = (float)(sp->nchans - cp->chan)/(float)sp->nchans;
      low = high - 1.0/(float)sp->nchans;
      cp = sp->chans[0];
      wcp = chan_widget(cp,W_chn_gsy);
      val = (float)get_raw_value(wcp)/(float)(SCROLLBAR_MAX);
      size = (float)get_raw_size(wcp)/(float)(SCROLLBAR_MAX);
      if ((val > low) || ((val+size) < high))
	{
	  val = low;
	  if ((val+size) > 1.0) val = 1.0 - size;
	  set_raw_value(wcp,val*SCROLLBAR_MAX);
	  gsy_changed(val*SCROLLBAR_MAX,cp);
	}
    }
}
