#include "snd.h"

#ifdef NEXT
  #include <sys/dir.h>
  #include <sys/dirent.h>
#else
  #include <dirent.h>
#endif

#define INIT_FILE_NAME "~/.snd"
#define BACKUP_FILE_NAME ".snd-backup"
#define EPS_FILE_NAME "snd.eps"

/* our default basic colors (resource file can override these): */
#define LIGHTEST_COLOR  "ivory1"
#define BASIC_COLOR     "ivory2"
#define DARK_COLOR      "ivory3"
#define DARKEST_COLOR   "ivory4"
#define TEXT_COLOR      "lightsteelblue1"
#define CURSOR_COLOR    "cyan"
#define SELECTION_COLOR "ivory4"
#define DEFAULT_SONOGRAM_COLOR 0     /* index into table of choices -- see snd.xchn.c and snd-xmenu.c */
#define DEFAULT_SPECTROGRAM_COLOR -1 /* -1 = just lines or index */

/* we assume later that we can always find these fonts (if resource file gives bogus entry, we fall back on these) */
#ifdef SGI
  #define BUTTON_FONT "-*-times-medium-r-*-*-14-*-*-*-*-*-iso8859-1"
  #define BOLD_BUTTON_FONT "-*-times-bold-r-*-*-14-*-*-*-*-*-iso8859-1"
  #define AXIS_LABEL_FONT "-*-times-medium-r-normal-*-20-*-*-*-*-*-iso8859-1"
  #define AXIS_NUMBERS_FONT "-*-courier-medium-r-normal-*-14-*-*-*-*-*-iso8859-1"
  #define HELP_TEXT_FONT "9x15"
  #define CLOSED_CTRLS_HEIGHT 22
  #define OPEN_CTRLS_HEIGHT 185
#endif
#ifdef LINUX
  #define BUTTON_FONT "-*-times-medium-r-*-*-12-*-*-*-*-*-iso8859-1"
  #define BOLD_BUTTON_FONT "-*-times-bold-r-*-*-12-*-*-*-*-*-iso8859-1"
  #define AXIS_LABEL_FONT "-*-times-medium-r-normal-*-16-*-*-*-*-*-iso8859-1"
  #define AXIS_NUMBERS_FONT "-*-courier-medium-r-normal-*-12-*-*-*-*-*-iso8859-1"
  #define HELP_TEXT_FONT "8x13"
  #define CLOSED_CTRLS_HEIGHT 18
  #define OPEN_CTRLS_HEIGHT 155
#endif
#ifdef NEXT
  #define BUTTON_FONT "-*-times-medium-r-*-*-14-*-*-*-*-*-iso8859-1"
  #define BOLD_BUTTON_FONT "-*-times-bold-r-*-*-14-*-*-*-*-*-iso8859-1"
  #define AXIS_LABEL_FONT "-*-times-medium-r-normal-*-20-*-*-*-*-*-iso8859-1"
  #define AXIS_NUMBERS_FONT "-*-courier-medium-r-normal-*-14-*-*-*-*-*-iso8859-1"
  #define HELP_TEXT_FONT "9x15"
  #define CLOSED_CTRLS_HEIGHT 22
  #define OPEN_CTRLS_HEIGHT 185
#endif
#define CHANNEL_MIN_HEIGHT 150  /* open size (set to 5 immediately thereafter) */
                                /* paned window default setup is not very smart, so we force these to be this size to begin with */
                                /* this number is only a first approximation -- we try not to expand below the screen */
                                /* if too small (i.e. 100), the scrollbars are sometimes messed up on the initial layout */
#define MAIN_SASH_SIZE 14
#define MAIN_SASH_INDENT -6

/* /usr/lib/X11/rgb.txt, /usr/lib/X11/fonts/Type1/fonts.dir, /usr/lib/X11/fonts/misc */
/* cursor color is actually XOR relative -- cyan -> red */
/* schemes are in /usr/lib/X11/schemes */

#define HELP_ROWS 6
#define HELP_COLUMNS 48

typedef struct {
  char *lightest_color;
  char *basic_color;
  char *dark_color;
  char *darkest_color;
  char *text_color;
  char *cursor_color;
  char *selection_color;
  char *use_schemes;
  char *button_font;
  char *bold_button_font;
  char *axis_label_font;
  char *axis_numbers_font;
  char *help_text_font;
  char *init_file_name;
  char *backup_file_name;
  char *eps_file_name;
  int default_output_type;
  int sonogram_color;
  int spectrogram_color;
  int overwrite_check;
} sndres;

static XtResource resources[] = {
  {"lightestcolor","Lightestcolor",XmRString,sizeof(char *),XtOffset(sndres *,lightest_color),XmRString,LIGHTEST_COLOR},
  {"basiccolor","Basiccolor",XmRString,sizeof(char *),XtOffset(sndres *,basic_color),XmRString,BASIC_COLOR},
  {"darkcolor","Darkcolor",XmRString,sizeof(char *),XtOffset(sndres *,dark_color),XmRString,DARK_COLOR},
  {"darkestcolor","Darkestcolor",XmRString,sizeof(char *),XtOffset(sndres *,darkest_color),XmRString,DARKEST_COLOR},
  {"textcolor","Textcolor",XmRString,sizeof(char *),XtOffset(sndres *,text_color),XmRString,TEXT_COLOR},
  {"cursorcolor","Cursorcolor",XmRString,sizeof(char *),XtOffset(sndres *,cursor_color),XmRString,CURSOR_COLOR},
  {"selectioncolor","Selectioncolor",XmRString,sizeof(char *),XtOffset(sndres *,selection_color),XmRString,SELECTION_COLOR},
  {"useSchemes","UseSchemes",XmRString,sizeof(char *),XtOffset(sndres *,use_schemes),XmRString,"none"},
  {"buttonFont","ButtonFont",XmRString,sizeof(char *),XtOffset(sndres *,button_font),XmRString,BUTTON_FONT},
  {"boldbuttonFont","BoldbuttonFont",XmRString,sizeof(char *),XtOffset(sndres *,bold_button_font),XmRString,BOLD_BUTTON_FONT},
  {"axisLabelFont","AxisLabelFont",XmRString,sizeof(char *),XtOffset(sndres *,axis_label_font),XmRString,AXIS_LABEL_FONT},
  {"axisNumbersFont","AxisNumbersFont",XmRString,sizeof(char *),XtOffset(sndres *,axis_numbers_font),XmRString,AXIS_NUMBERS_FONT},
  {"helpTextFont","HelpTextFont",XmRString,sizeof(char *),XtOffset(sndres *,help_text_font),XmRString,HELP_TEXT_FONT},
  {"initFile","InitFile",XmRString,sizeof(char *),XtOffset(sndres *,init_file_name),XmRString,INIT_FILE_NAME},
  {"backupFile","BackupFile",XmRString,sizeof(char *),XtOffset(sndres *,backup_file_name),XmRString,BACKUP_FILE_NAME},
  {"epsFile","EpsFile",XmRString,sizeof(char *),XtOffset(sndres *,eps_file_name),XmRString,EPS_FILE_NAME},
  {"defaultOutputType","DefaultOutputType",XmRInt,sizeof(int),XtOffset(sndres *,default_output_type),XmRImmediate,(XtPointer)DEFAULT_OUTPUT_TYPE},
  {"sonogramColor","SonogramColor",XmRInt,sizeof(int),XtOffset(sndres *,sonogram_color),XmRImmediate,(XtPointer)DEFAULT_SONOGRAM_COLOR},
  {"spectrogramColor","SpectrogramColor",XmRInt,sizeof(int),XtOffset(sndres *,spectrogram_color),XmRImmediate,(XtPointer)DEFAULT_SPECTROGRAM_COLOR},
  {"overwriteCheck","OverwriteCheck",XmRInt,sizeof(int),XtOffset(sndres *,overwrite_check),XmRImmediate,(XtPointer)0}
};

static task_manager *make_task_manager(snd_state *ss, Widget shell, Display *dpy)
{
  task_manager *tm;
  tm = (task_manager *)calloc(1,sizeof(task_manager));
  tm->slice = 0;
  tm->ss = ss;
  tm->shell = shell;
  tm->dpy = dpy;
  return(tm);
}

static void FinalCleanupCB(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_exit_cleanly((snd_state *)clientData);
}

static void Save_Yourself_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  task_manager *tm = (task_manager *)clientData;
  snd_state *ss;
  ss = tm->ss;
  save_snd_state(ss);
}

static char **auto_open_file_names = NULL;
static int auto_open_files = 0;

static Boolean startup_funcs(XtPointer clientData)
{
  task_manager *tm = (task_manager *)clientData;
  Widget menu;
  Atom wm_delete_window,wm_save_yourself;
  snd_info *sp;
  snd_state *ss;
  chan_info *cp;
  axis_info *ap;
  DIR *dp;
  int i,files;
  static auto_open_ctr = 0;
  ss = tm->ss;
  switch (tm->slice)
    {
    case 0:
      menu = add_menu(ss);
      XtVaSetValues(menu,XmNpositionIndex,0,XmNpaneMinimum,MENU_HEIGHT,XmNpaneMaximum,MENU_HEIGHT,XmNmarginHeight,0,NULL);
      break;
    case 1:
      create_popup_menu(ss);
      break;
    case 2:
      intern_atoms(ss);
      break;
    case 3:
      InitializeDrop(ss);
      break;
    case 4:  
#ifndef NEXT
      /* trap outer-level Close for cleanup check */
      wm_delete_window = XmInternAtom(tm->dpy,"WM_DELETE_WINDOW",FALSE);
      XmAddWMProtocolCallback(tm->shell,wm_delete_window,FinalCleanupCB,(XtPointer)ss);
      wm_save_yourself = XmInternAtom(tm->dpy,"WM_SAVE_YOURSELF",FALSE);
      XmAddWMProtocols(tm->shell,&wm_save_yourself,1);
      XmAddWMProtocolCallback(tm->shell,wm_save_yourself,Save_Yourself_Callback,(XtPointer)tm); 
#endif
      break;
    case 5: 
      if (auto_open_files > 0)
	{
	  if (auto_open_ctr == 0) remove_backup_file(ss);
	  if (auto_open_file_names[auto_open_ctr])
	    { /* wanted to use "-d" and "-i" (or "-s") but they're in use */
	      if (strcmp("-p",auto_open_file_names[auto_open_ctr]) == 0)
		{
		  /* preload sound files in dir (can be ., should be unquoted) */
		  auto_open_ctr++;
		  add_directory_to_prevlist(ss,auto_open_file_names[auto_open_ctr]);
		  auto_open_ctr++;
		}
	      else
		{
		  if (strcmp("-l",auto_open_file_names[auto_open_ctr]) == 0)
		    {
		      /* grab session name -- if arg is "." grab latest on this directory */
		      auto_open_ctr++;
		      set_session_name(ss,auto_open_file_names[auto_open_ctr]);
		      auto_open_ctr++;
		    }
		  else
		    {
		      sp = snd_open_file(auto_open_file_names[auto_open_ctr],ss);
		      if (!sp) {tm->slice++; return(FALSE);} /* geez what could have gone wrong... */
		      auto_open_ctr++;
		      if (auto_open_ctr < auto_open_files) 
			{
			  if (strcmp("-s",auto_open_file_names[auto_open_ctr]) == 0)
			    {
			      /* start up info follows -- [sx,sy,zx,zy] ... (per chan) */
			      auto_open_ctr++;
			      for (i=0;i<sp->nchans;i++)
				{
				  cp = sp->chans[i];
				  ap = cp->axis;
				  sscanf(auto_open_file_names[auto_open_ctr],"%f,%f,%f,%f",&ap->sx,&ap->sy,&ap->zx,&ap->zy);
				  set_xy_bounds(cp,ap);
				  auto_open_ctr++;
				}
			    }
			}
		    }
		}
	    }
	  if (auto_open_ctr < auto_open_files) return(FALSE); /* i.e. come back to this branch */
	}
      break;
    case 6: 
      /* this stupid thing (which I can't customize without major hassles) takes forever on large directories */
      files = 0;
      if ((dp=opendir(".")) != NULL)
	{
	  while ((files < 400) && (readdir(dp) != NULL)) files++;
	}
      closedir(dp);
      if (files < 400) CreateOpenDialog(tm->shell,(XtPointer)ss);
      break;
    case 7:
      make_graph_cursor(tm);
      break;
    case 8:
      CLM_connect(ss);
      break;
    case 9:
      snd_add_locals(ss);
      break;
    case 10: 
      snd_load_init_file(ss);
      break;
    case 11:
      restore_state(ss,(ss->session_name) ? ss->session_name : ss->backup_file); /* could I trust C's || operator here? */
      break;
    case 12: return(TRUE); break;
    }
  tm->slice++;
  return(FALSE);
}

void snd_doit(snd_state *state,int argc, char **argv)
{
  XtAppContext app;     
  Widget shell;
  Display *dpy;
  Arg args[32];
  int n;
  sndres snd_rs;
  state_context *sx;
  int scr;
  Colormap cmap;
  XColor tmp_color,ignore;
#ifdef NEXT
  Widget toplevel;
#endif

  XtSetLanguageProc(NULL,NULL,NULL);

  state->ctrls_height = CLOSED_CTRLS_HEIGHT;
  state->channel_min_height = CHANNEL_MIN_HEIGHT;

#ifdef NEXT
  toplevel = XtInitialize(argv[0],"Snd",NULL,0,&argc,argv);
  app = XtCreateApplicationContext();
  n=0;
  XtSetArg(args[n],XmNallowShellResize,TRUE); n++;
  shell = XtCreateApplicationShell("mainshell",shellWidgetClass,args,n);
#else
  shell = XtVaOpenApplication(&app,"mainshell",NULL,0,&argc,argv,NULL,applicationShellWidgetClass,XmNallowShellResize,TRUE,NULL);
#endif

  auto_open_files = argc-1;
  if (argc > 1) auto_open_file_names = (char **)(argv+1);

  dpy=XtDisplay(shell);
  scr=DefaultScreen(dpy);
  XtGetApplicationResources(shell,&snd_rs,resources,XtNumber(resources),NULL,0);
  state->using_schemes = (strcmp(snd_rs.use_schemes,"all") == 0);
  state->sgx = (state_context *)calloc(1,sizeof(state_context));
  sx = state->sgx;

  cmap=DefaultColormap(dpy,scr);
  sx->white = WhitePixel(dpy,scr);
  sx->black = BlackPixel(dpy,scr);
  state->place_scroll_size = 13;
  if (!XAllocNamedColor(dpy,cmap,snd_rs.lightest_color,&tmp_color,&ignore)) XtAppWarning(app,"no light color"); else sx->high = tmp_color.pixel;
  if (!XAllocNamedColor(dpy,cmap,snd_rs.basic_color,&tmp_color,&ignore)) XtAppWarning(app,"no basic color"); else sx->main = tmp_color.pixel;
  if (!XAllocNamedColor(dpy,cmap,snd_rs.dark_color,&tmp_color,&ignore)) XtAppWarning(app,"no dark color"); else sx->scale = tmp_color.pixel;
  if (!XAllocNamedColor(dpy,cmap,snd_rs.darkest_color,&tmp_color,&ignore)) XtAppWarning(app,"no darkest color"); else sx->zoom = tmp_color.pixel;
  if (!XAllocNamedColor(dpy,cmap,snd_rs.text_color,&tmp_color,&ignore)) XtAppWarning(app,"no back color"); else sx->text = tmp_color.pixel;
  if (!XAllocNamedColor(dpy,cmap,snd_rs.cursor_color,&tmp_color,&ignore)) XtAppWarning(app,"no cursor color"); else sx->cursor = tmp_color.pixel;
  if (!XAllocNamedColor(dpy,cmap,snd_rs.selection_color,&tmp_color,&ignore)) XtAppWarning(app,"no selection color"); else sx->gray = tmp_color.pixel;

  sx->mainapp = app;
  sx->mainshell = shell;
  sx->mdpy = dpy;

  sx->button_fontstruct = XLoadQueryFont(dpy,snd_rs.button_font);
  if (!(sx->button_fontstruct)) 
    {
      fprintf(stderr,snd_string_cant_find_font,snd_rs.button_font);
      sx->button_fontstruct = XLoadQueryFont(dpy,BUTTON_FONT);
    }
  sx->button_fontlist = XmFontListCreate(sx->button_fontstruct,"buttonset");
  XmFontListEntryCreate("button_font",XmFONT_IS_FONT,sx->button_fontstruct);

  sx->bold_button_fontstruct = XLoadQueryFont(dpy,snd_rs.bold_button_font);
  if (!(sx->bold_button_fontstruct)) 
    {
      fprintf(stderr,snd_string_cant_find_font,snd_rs.bold_button_font);
      sx->bold_button_fontstruct = XLoadQueryFont(dpy,BOLD_BUTTON_FONT);
    }
  sx->bold_button_fontlist = XmFontListCreate(sx->bold_button_fontstruct,"boldbuttonset");
  XmFontListEntryCreate("bold_button_font",XmFONT_IS_FONT,sx->bold_button_fontstruct);

  sx->axis_label_fontstruct = XLoadQueryFont(dpy,snd_rs.axis_label_font);
  if (!(sx->axis_label_fontstruct)) 
    {
      fprintf(stderr,snd_string_cant_find_font,snd_rs.axis_label_font);
      sx->axis_label_fontstruct = XLoadQueryFont(dpy,AXIS_LABEL_FONT);
    }

  sx->axis_numbers_fontstruct = XLoadQueryFont(dpy,snd_rs.axis_numbers_font);
  if (!(sx->axis_numbers_fontstruct)) 
    {
      fprintf(stderr,snd_string_cant_find_font,snd_rs.axis_numbers_font);
      sx->axis_numbers_fontstruct = XLoadQueryFont(dpy,AXIS_NUMBERS_FONT);
    }

  sx->help_text_fontstruct = XLoadQueryFont(dpy,snd_rs.help_text_font);
  if (!(sx->help_text_fontstruct)) 
    {
      fprintf(stderr,snd_string_cant_find_font,snd_rs.help_text_font);
      sx->help_text_fontstruct = XLoadQueryFont(dpy,HELP_TEXT_FONT);
    }
  sx->help_text_fontlist = XmFontListCreate(sx->help_text_fontstruct,"helptextset");
  XmFontListEntryCreate("help_text_font",XmFONT_IS_FONT,sx->help_text_fontstruct);

  if (!(state->using_schemes)) XtVaSetValues(shell,XmNbackground,sx->main,NULL);
  state->init_file = snd_rs.init_file_name;
  state->backup_file = snd_rs.backup_file_name;
  state->eps_file = snd_rs.eps_file_name;
  state->default_output_type = snd_rs.default_output_type;
  state->sonogram_color = snd_rs.sonogram_color;
  state->spectrogram_color = snd_rs.spectrogram_color;
  state->ask_before_overwrite = snd_rs.overwrite_check;

  n=0;
  if (!(state->using_schemes)) n = background_main_color(args,n,(snd_state *)state);
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNallowResize,TRUE); n++;
  XtSetArg(args[n],XmNsashHeight,MAIN_SASH_SIZE); n++;
  XtSetArg(args[n],XmNsashWidth,MAIN_SASH_SIZE); n++;
  XtSetArg(args[n],XmNsashIndent,MAIN_SASH_INDENT); n++;
  sx->mainpane = XtCreateManagedWidget("mainpane",xmPanedWindowWidgetClass,shell,args,n);

  XtRealizeWidget(shell);
  XtAppAddWorkProc(app,startup_funcs,make_task_manager(state,shell,dpy));
  XtAppMainLoop(app);
}


/* ---------------- HELP MONOLOG ---------------- */

typedef struct {Widget help; Widget text;} help_win;
static char help_window_label[64];

static void help_help_callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Help",
"You can get help within Snd either from\n\
the Help Menu items, or by clicking on\n\
some portion of the Snd display while the\n\
cursor is '?'.  See Click for Help in the\n\
Help Menu.\n\
");
}

static void create_help_monolog(snd_state *ss)
{
  /* create scrollable but not editable text window */
  help_win *hlp;
  Arg args[20];
  int n;

  hlp = (help_win *)calloc(1,sizeof(help_win));
  n=0;
  /* this window should be resizable by the user (i.e. have the resize bars), but not resize itself */
  XtSetArg(args[n],XmNresizePolicy,XmRESIZE_GROW); n++;
#ifdef LINUX
  XtSetArg(args[n],XmNnoResize,FALSE); n++;
#endif
  hlp->help = XmCreateMessageDialog(main_PANE(ss),"snd-help",args,n);
  XtUnmanageChild(XmMessageBoxGetChild(hlp->help,XmDIALOG_SYMBOL_LABEL));
  XtUnmanageChild(XmMessageBoxGetChild(hlp->help,XmDIALOG_CANCEL_BUTTON));
  XtAddCallback(hlp->help,XmNhelpCallback,help_help_callback,ss);

  n=0;
  XtSetArg(args[n],XmNeditMode,XmMULTI_LINE_EDIT); n++;
  XtSetArg(args[n],XmNeditable,FALSE); n++;
  XtSetArg(args[n],XmNcolumns,HELP_COLUMNS); n++;
  XtSetArg(args[n],XmNrows,HELP_ROWS); n++;
  XtSetArg(args[n],XmNfontList,help_text_FONT(ss)); n++;
  hlp->text = XmCreateScrolledText(hlp->help,"help-text",args,n);
  XtManageChild(hlp->text);
  if (!(ss->using_schemes))
    {
      map_over_children(hlp->help,set_main_color_of_widget,(void *)ss);
      XtVaSetValues(hlp->text,XmNbackground,(ss->sgx)->white,NULL);
    }
  ss->help_monolog = hlp;
}

void snd_help(snd_state *ss, char *subject, char *helpstr)
{
  /* place help string in scrollable help window */
  /* if window is already active, add this help at the top and reposition */
  help_win *h;
  XmString xstr1;
  if (!(ss->help_monolog)) create_help_monolog(ss);
  h = (help_win *)(ss->help_monolog);
  sprintf(help_window_label,"%s help",subject);
  xstr1 = XmStringCreate(help_window_label,XmFONTLIST_DEFAULT_TAG);
  XtVaSetValues(h->help,XmNmessageString,xstr1,NULL);
  XmTextSetString(h->text,helpstr);
  if (!XtIsManaged(h->help)) XtManageChild(h->help);
  XmStringFree(xstr1);
}

void add_snd_help(snd_state *ss, char *str)
{
  int pos;
  help_win *h;
  if (!(ss->help_monolog)) create_help_monolog(ss);
  h = (help_win *)(ss->help_monolog);
  pos = XmTextGetLastPosition(h->text);
  if (pos == 0) XmTextSetString(h->text,str);
  else XmTextInsert(h->text,pos,str);
  if (!XtIsManaged(h->help)) XtManageChild(h->help);
}

void ssnd_help(snd_state *ss, char *subject, ...)
{
  help_win *h;
  XmString xstr1;
  va_list ap;
  char *helpstr;
  int pos;
  va_start(ap,subject);
  if (!(ss->help_monolog)) create_help_monolog(ss);
  h = (help_win *)(ss->help_monolog);
  sprintf(help_window_label,"%s help",subject);
  xstr1 = XmStringCreate(help_window_label,XmFONTLIST_DEFAULT_TAG);
  XtVaSetValues(h->help,XmNmessageString,xstr1,NULL);
  pos = 0;
  while (helpstr = va_arg(ap,char *))
    {
      if (pos == 0) XmTextSetString(h->text,helpstr);
      else XmTextInsert(h->text,pos,helpstr);
      pos = XmTextGetLastPosition(h->text);
    }
  va_end(ap);
  if (!XtIsManaged(h->help)) XtManageChild(h->help);
  XmStringFree(xstr1);
}  

int snd_window_width(snd_state *ss)
{
  Dimension width;
  XtVaGetValues(main_SHELL(ss),XmNwidth,&width,NULL);
  return(width);
}

int snd_window_height(snd_state *ss)
{
  Dimension height;
  XtVaGetValues(main_SHELL(ss),XmNheight,&height,NULL);
  return(height);
}

void set_snd_window_width(snd_state *ss,int width)
{
  XtVaSetValues(main_SHELL(ss),XmNwidth,width,NULL);
}

void set_snd_window_height(snd_state *ss,int height)
{
  XtVaSetValues(main_SHELL(ss),XmNheight,height,NULL);
}


/* 
 * miniaturize via XIconifyWindow(dpy,w,screen)
 * reopen via XMapWindow(dpy,w) and XMapRaised()
 * size changed via XChangeProperty
 * maximize button is MWM_DECOR_MAXIMIZE (also MWM_DECOR_MINIMIZE)
 * function is 'f.maximize'
 */


void sound_show_ctrls(snd_info *sp)
{
  XtUnmanageChild(snd_widget(sp,W_snd_ctrls));
  XtVaSetValues(snd_widget(sp,W_snd_ctrls),XmNpaneMaximum,OPEN_CTRLS_HEIGHT,NULL);
  XtVaSetValues(snd_widget(sp,W_snd_ctrls),XmNpaneMinimum,OPEN_CTRLS_HEIGHT,NULL);
  XtManageChild(snd_widget(sp,W_snd_ctrls));
  XtVaSetValues(snd_widget(sp,W_snd_ctrls),XmNpaneMaximum,1000,NULL);
  XtVaSetValues(snd_widget(sp,W_snd_ctrls),XmNpaneMinimum,1,NULL);
}

void sound_hide_ctrls(snd_info *sp)
{
  XtUnmanageChild(snd_widget(sp,W_snd_ctrls));
  XtVaSetValues(snd_widget(sp,W_snd_ctrls),XmNpaneMaximum,CLOSED_CTRLS_HEIGHT,NULL);
  XtVaSetValues(snd_widget(sp,W_snd_ctrls),XmNpaneMinimum,CLOSED_CTRLS_HEIGHT,NULL);
  XtManageChild(snd_widget(sp,W_snd_ctrls));
  XtVaSetValues(snd_widget(sp,W_snd_ctrls),XmNpaneMaximum,1000,NULL);
  XtVaSetValues(snd_widget(sp,W_snd_ctrls),XmNpaneMinimum,1,NULL);
}

void show_controls(snd_state *ss)
{
  snd_info *sp;
  int i;
  ss->ctrls_height = OPEN_CTRLS_HEIGHT;
  set_view_ctrls_label(snd_string_Hide_controls);
  for (i=0;i<ss->max_sounds;i++)
    {
      sp = ss->sounds[i];
      if ((sp) && (sp->inuse)) sound_show_ctrls(sp);
    }
}

void hide_controls(snd_state *ss)
{
  snd_info *sp;
  int i;
  ss->ctrls_height = CLOSED_CTRLS_HEIGHT;
  set_view_ctrls_label(snd_string_Show_controls);
  for (i=0;i<ss->max_sounds;i++)
    {
      sp = ss->sounds[i];
      if ((sp) && (sp->inuse)) sound_hide_ctrls(sp);
    }
}

