#include "snd.h"

void normalize_all_sounds(snd_state *ss)
{
  /* normalize: get size, #chans, #snds, set pane minima, force remanage(?), unlock */
  int sounds,chans,chan_y,height,screen_y;
  int *wid;
  height = get_window_height(main_PANE(ss));
  screen_y = DisplayHeight(main_DISPLAY(ss),main_SCREEN(ss));
  if (height > screen_y) height = screen_y;
  sounds = ss->active_sounds;
  chans = active_channels(ss,0);
  if (chans > 1)
    {
      /* now we try to make room for the sound ctrl bar, each channel, perhaps the menu */
      chan_y = (height-(sounds*ss->ctrls_height)-MENU_HEIGHT)/chans;
      wid=(int *)calloc(1,sizeof(int));
      (*wid) = chan_y;
      map_over_sounds(ss,sound_lock_ctrls,NULL);
      map_over_separate_chans(ss,channel_lock_pane,(void *)wid);
      free(wid);
      map_over_separate_chans(ss,channel_unlock_pane,NULL);
      map_over_sounds(ss,sound_unlock_ctrls,NULL);
    }
}

void add_sound_data(char *filename, snd_info *sp, snd_state *ss)
{
  int i;
  for (i=0;i<sp->nchans;i++) add_channel_data(filename,sp->chans[i],sp->hdr,ss);
}

typedef struct {
  int samps_per_bin; 
  int amp_env_size;
  int amp_ctr;
  int *data_max; 
  int *data_min;
} env_info;

typedef struct {
  int amp_env_size;
  int slice; 
  int loc; 
  int samp; 
  int true_samp; 
  int samps_per_bin; 
  int samples;  
  env_info **eps; 
  int chans; 
  int fd; 
  int **ibufs;
  snd_fd **sfs;
} env_state;

#define AMP_BUFFER_SIZE 1024

static void free_env_info(env_info *ep)
{
  /* can be either during channel close, or premature work proc removal */
  if (ep)
    {
      /* fprintf(stderr,"free ep: %p %p %p ",(void *)ep,(void *)(ep->data_min),(void *)(ep->data_max)); */
      if (ep->data_max) free(ep->data_max);
      if (ep->data_min) free(ep->data_min);
      free(ep);
    }
}

static void free_env_state(snd_info *sp, int in_progress)
{ /* if in_progress, we're quitting prematurely, so everything must go */
  env_state *ep;
  env_info *es;
  int *buf;
  int i;
  if ((sp) && (sp->sgx) && ((sp->sgx)->env_data))
    {
      /* assume someone else is removing the work proc via remove_amp_env(sp) */
      ep = ((sp->sgx)->env_data);
      if (ep->ibufs)
	{
	  for (i=0;i<ep->chans;i++)
	    {
	      buf = ep->ibufs[i];
	      if (buf) free(buf);
	      ep->ibufs[i] = NULL;
	    }
	  free(ep->ibufs);
	  ep->ibufs = NULL;
	}
      if (ep->eps)
	{
	  if (in_progress)
	    {
	      for (i=0;i<ep->chans;i++)
		{
		  es = ep->eps[i];
		  if (es) free_env_info(es);
		  ep->eps[i] = NULL;
		}
	    }
	  free(ep->eps);
	  ep->eps = NULL;
	}
      if (in_progress)
	{
	  if (ep->fd != -1)
	    {
	      clm_close(ep->fd);
	    }
	}
      if (ep->sfs)
	{
	  for (i=0;i<ep->chans;i++)
	    {
	      if (ep->sfs[i]) free_snd_fd(ep->sfs[i]);
	    }
	  free(ep->sfs);
	  ep->sfs = NULL;
	}
      free(ep);
      (sp->sgx)->env_data = NULL;
      if (!in_progress) (sp->sgx)->env_in_progress = 0; /* assume someone else does this in the premature-quit case */
    }
}

void free_chan_env(chan_info *cp)
{
  if (cp->amp_env) 
    {
      if (cp->amp_env == cp->original_env) cp->original_env = NULL;
      free_env_info(cp->amp_env);
      cp->amp_env = NULL;
    }
  if (cp->original_env) 
    {
      free_env_info(cp->original_env);
      cp->original_env = NULL;
    }
}

void free_sound_env(snd_info *sp, int in_progress)
{
  free_env_state(sp,in_progress);
}


static void allocate_env_state(snd_info *sp, snd_context *sgx, int samples, int file)
{
  int i,val;
  env_info *ep;
  env_state *es;
  env_info **e;
  int **bufs;
  snd_fd **sfs;
  sgx->env_data = (env_state *)calloc(1,sizeof(env_state));
  es = sgx->env_data;
  es->samples = samples/sp->nchans;
  /* we want samps_per_bin to be useful over a wide range of file sizes */
  val = (float)(es->samples)/100000.0;
  if (val < 5) es->amp_env_size = 2048;
  else if (val < 10) es->amp_env_size = 4096;
  else if (val < 40) es->amp_env_size = 8192;
  else if (val < 100) es->amp_env_size = 16384;
  else if (val < 1000) es->amp_env_size = 32768;
  else es->amp_env_size = 65536; /* i.e. more than 10^8 samples/chan */
  es->samps_per_bin = ceil((float)(es->samples)/(float)(es->amp_env_size));
  es->chans = sp->nchans;
  es->slice = 0;
  es->loc = -1;
  es->samp = es->samps_per_bin;
  es->true_samp = 0;
  es->fd = -1;
  if (file) 
    {
      es->ibufs = (int **)calloc(sp->nchans,sizeof(int *));
      bufs = es->ibufs;
      es->sfs = NULL;
    }
  else
    {
      sfs = (snd_fd **)calloc(sp->nchans,sizeof(snd_fd *));
      es->sfs = sfs;
      es->ibufs = NULL;
    }
  es->eps = (env_info **)calloc(sp->nchans,sizeof(env_info *));
  e = es->eps;
  for (i=0;i<sp->nchans;i++)
    {
      e[i] = (env_info *)calloc(1,sizeof(env_info));
      ep = e[i];
      ep->amp_env_size = es->amp_env_size;
      ep->data_max = (int *)calloc(es->amp_env_size,sizeof(int));
      ep->data_min = (int *)calloc(es->amp_env_size,sizeof(int));
      /* fprintf(stderr,"alloc ep: %p %p %p ",(void *)ep,(void *)(ep->data_min),(void *)(ep->data_max)); */
      ep->samps_per_bin = es->samps_per_bin;
      if (file)
	bufs[i] = (int *)calloc(AMP_BUFFER_SIZE,sizeof(int));
      else sfs[i] = init_sample_read(0,sp->chans[i],1);
    }
}
	
/* using file reads here for optimization (slightly faster than edit-tree traversal below) */
/* in a one hour 44KHz stereo file, this takes about 1.5 mins to get the overall amp env */

static int run_original_env(snd_info *sp)
{
  /* calculate the amp env of each channel */
  /* this is an optimization for large sound display */
  int i,j,k,loc0,loc1,ymin,ymax,samp0,samp1,val;
  env_info *ep;
  env_state *es;
  int *buf;
  file_info *hdr;
  chan_info *cp;
  snd_context *sgx;
  sgx = sp->sgx;
  if (!(sgx->env_data)) 
    {
      hdr = sp->hdr; 
      allocate_env_state(sp,sgx,hdr->samples,1);
      return(FALSE);
    }
  es = sgx->env_data;
  switch (es->slice)
    {
    case 0:
      /* open the file, set up all pointers, grab first buffer */
      /* start_time(); */
      es->fd = clm_open_read(sp->fullname);
      hdr = sp->hdr;
      open_clm_file_descriptors(es->fd,hdr->format,c_snd_datum_size(hdr->format),hdr->data_location);
      clm_seek(es->fd,hdr->data_location,0);
      clm_read(es->fd,0,AMP_BUFFER_SIZE-1,es->chans,es->ibufs);
      es->slice++;
      return(FALSE);
      break;
    case 1: 
      /* spin down n points doing the cur min/max shuffle */
      /* when file end reached, increment slice */
      /* es->loc is the bin we're working on now */
      /* es->samp is the current sample within that bin (-1 = time to reset) */
      /* es->true_samp is where we are in the input file */
      /* es->samples is how many there are (per channel) */
      loc0 = es->loc;
      loc1 = loc0;
      samp0 = es->samp;
      samp1 = samp0;
      for (j=0;j<es->chans;j++)
	{
	  samp1 = samp0;
	  loc1 = loc0;
	  ep = es->eps[j];
	  buf = es->ibufs[j];
	  ymin = ep->data_min[loc1];
	  ymax = ep->data_max[loc1];
	  for (i=0;i<AMP_BUFFER_SIZE;i++)
	    {
	      if (samp1 == es->samps_per_bin)
		{
		  if (loc1>=0)
		    {
		      ep->data_max[loc1] = ymax;
		      ep->data_min[loc1] = ymin;
		    }
		  loc1++;
		  ymin = buf[i];
		  ymax = buf[i];
		  samp1 = 1;
		}
	      else
		{
		  val = buf[i];
		  if (ymin > val) ymin = val;
		  if (ymax < val) ymax = val;
		  samp1++;
		}
	    }
	  ep->data_max[loc1] = ymax; /* save for next invocation */
	  ep->data_min[loc1] = ymin;
	}
      es->loc = loc1;
      es->samp = samp1;
      es->true_samp += AMP_BUFFER_SIZE;
      if ((es->true_samp) >= (es->samples))
	{
	  es->slice++;
	  /* check for round-off errors */
	  if (es->loc < (es->amp_env_size-1))
	    {
	      for (j=0;j<es->chans;j++)
		{
		  ep = es->eps[j];
		  for (k=(es->loc + 1);k<es->amp_env_size;k++)
		    {
		      ep->data_max[k] = 0;
		      ep->data_min[k] = 0;
		    }
		}
	    }
	}
      else clm_read(es->fd,0,AMP_BUFFER_SIZE-1,es->chans,es->ibufs);
      return(FALSE);
      break;
    case 2:
      /* assign env_infos to the respective chan_info pointers */
      /* free and nullify all the rest of the env_state */
      /* return true to remove work proc */
      /* fprintf(stderr,"done in %d",run_time()); */
      clm_close(es->fd);
      for (i=0;i<es->chans;i++)
	{
	  cp = sp->chans[i];
	  cp->amp_env = es->eps[i];
	  if (cp->edit_ctr == 0) cp->original_env = cp->amp_env;
	  ((env_info *)(cp->amp_env))->amp_ctr = cp->edit_ctr;
	  es->eps[i] = NULL;
	}
      free_env_state(sp,0);
      signal_amp_env_done(sp);
      return(TRUE);
      break;
    }
}

static int run_another_env(snd_info *sp)
{
  /* calculate the amp env of each channel using edit-tree, not stored file data */
  int i,j,k,loc0,loc1,ymin,ymax,samp0,samp1,val;
  env_info *ep;
  env_state *es;
  chan_info *cp;
  snd_context *sgx;
  sgx = sp->sgx;
  if (!(sgx->env_data))
    {
      allocate_env_state(sp,sgx,(current_ed_samples(sp->chans[0]))*sp->nchans,0);
      return(FALSE);
    }
  es = sgx->env_data;
  switch (es->slice)
    {
    case 0:
      loc0 = es->loc;
      loc1 = loc0;
      samp0 = es->samp;
      samp1 = samp0;
      for (j=0;j<es->chans;j++)
	{
	  samp1 = samp0;
	  loc1 = loc0;
	  ep = es->eps[j];
	  ymin = ep->data_min[loc1];
	  ymax = ep->data_max[loc1];
	  for (i=0;i<AMP_BUFFER_SIZE;i++)
	    {
	      if (samp1 == es->samps_per_bin)
		{
		  if (loc1>=0)
		    {
		      ep->data_max[loc1] = ymax;
		      ep->data_min[loc1] = ymin;
		    }
		  loc1++;
		  NEXT_SAMPLE(val,es->sfs[j]);
		  ymin = val;
		  ymax = val;
		  samp1 = 1;
		}
	      else
		{
		  NEXT_SAMPLE(val,es->sfs[j]);
		  if (ymin > val) ymin = val;
		  if (ymax < val) ymax = val;
		  samp1++;
		}
	    }
	  ep->data_max[loc1] = ymax; /* save for next invocation */
	  ep->data_min[loc1] = ymin;
	}
      es->loc = loc1;
      es->samp = samp1;
      es->true_samp += AMP_BUFFER_SIZE;
      if ((es->true_samp) >= (es->samples))
	{
	  es->slice++;
	  /* check for round-off errors */
	  if (es->loc < (es->amp_env_size-1))
	    {
	      for (j=0;j<es->chans;j++)
		{
		  ep = es->eps[j];
		  for (k=(es->loc + 1);k<es->amp_env_size;k++)
		    {
		      ep->data_max[k] = 0;
		      ep->data_min[k] = 0;
		    }
		}
	    }
	}
      return(FALSE);
      break;
    case 1:
      for (i=0;i<es->chans;i++)
	{
	  cp = sp->chans[i];
	  cp->amp_env = es->eps[i];
	  if (cp->edit_ctr == 0) cp->original_env = cp->amp_env;
	  ((env_info *)(cp->amp_env))->amp_ctr = cp->edit_ctr;
	  es->eps[i] = NULL;
	}
      free_env_state(sp,0);
      signal_amp_env_done(sp);
      sp->env_anew = 0;
      return(TRUE);
      break;
    }
}

int run_amp_env(snd_info *sp)
{
  if (sp->env_anew)
    return(run_another_env(sp));
  else return(run_original_env(sp));
}

void clobber_amp_env(chan_info *cp)
{
  env_info *ep;
  snd_info *sp;
  sp = cp->sound;
  if (sp->env_anew)
    {
      remove_amp_env(sp); /* if work proc in progress, it's no longer needed (or trustworthy) */
      free_env_state(sp,0);
    }
  ep = cp->amp_env;
  if ((ep) && (cp->edit_ctr <= ep->amp_ctr) && (ep->amp_ctr != 0)) 
    {
      if (ep != cp->original_env) free_env_info(ep);
      cp->amp_env = NULL;
    }
}

int amp_env_maxamp_ok(chan_info *cp)
{
  env_info *ep;
  ep = cp->amp_env;
  if ((ep) && (cp->edit_ctr == ep->amp_ctr)) return(1);
  if ((cp->edit_ctr == 0) && (ep = (cp->original_env)))
    {
      if ((cp->amp_env) && (cp->amp_env != ep)) clobber_amp_env(cp);
      cp->amp_env = ep;
      return(1);
    }
  return(0);
}

float amp_env_maxamp(chan_info *cp)
{
  env_info *ep;
  int i,ymax;
  ymax = 0;
  ep = cp->amp_env;
  for (i=0;i<ep->amp_env_size;i++)
    {
      if (ymax < ep->data_max[i]) ymax = ep->data_max[i];
      if (ymax < -ep->data_min[i]) ymax = -ep->data_min[i];
    }
  return(ymax*clm_sndflt);
}

int amp_env_usable(chan_info *cp,float samples_per_pixel) 
{
  env_info *ep;
  snd_info *sp;
  chan_info *ncp;
  int i;
  snd_context *sgx;
  ep = cp->amp_env;
  if ((ep) && (cp->edit_ctr == ep->amp_ctr))
    return(samples_per_pixel >= ep->samps_per_bin);
  if ((cp->edit_ctr == 0) && (ep = (cp->original_env)))
    {
      if ((cp->amp_env) && (cp->amp_env != ep)) clobber_amp_env(cp);
      cp->amp_env = ep;
      return(samples_per_pixel >= ep->samps_per_bin);
    }
  sp = cp->sound;
  sgx = sp->sgx;
  if (sgx->env_in_progress) return(0); /* i.e. not ready yet */
  /* either there is no original env, or we haven't scanned at the current edit_ctr */
  if (current_ed_samples(cp) > 100000) 
    {
      for (i=0;i<sp->nchans;i++)
	{
	  ncp = sp->chans[i];
	  if ((ep = (ncp->amp_env)) && (ep != ncp->original_env)) 
	    {
	      free_env_info(ep); 
	      ncp->amp_env = NULL;
	    }
	}
      start_amp_env(sp,1);
    }
  return(0);
}

int amp_env_graph(chan_info *cp,float samples_per_pixel) 
{
  float beg,step,x,xf,xk,pinc;
  int ymin,ymax,i,j,xi,k,kk;
  env_info *ep;
  axis_info *ap;
  ap = cp->axis;
  ep = cp->amp_env;
  step = samples_per_pixel/(float)(ep->samps_per_bin);
  beg = (float)(ap->losamp)/(float)(ep->samps_per_bin);
  j=0;
  x=ap->x0;
  xi=grf_x(x,ap);
  xf = beg;
  i = ap->losamp;
  xk = i;
  /* fprintf(stderr,"\nsamps: %f, step: %f, beg: %f, bin: %d ",samples_per_pixel,step,beg,ep->samps_per_bin); */
  ymin=32767;
  ymax=-32768;
  if (cp->printing) pinc = (float)samples_per_pixel/(float)snd_SRATE(cp);
  while (i<=ap->hisamp)
    {
      k=xf;
      xf+=step;
      kk=xf;
      if (kk>=ep->amp_env_size) kk=ep->amp_env_size-1;
      /* fprintf(stderr,"[%d to %d] ",k,kk); */
      for (;k<=kk;k++)
	{
	  if (ep->data_min[k] < ymin) ymin=ep->data_min[k];
	  if (ep->data_max[k] > ymax) ymax=ep->data_max[k];
	}
      xk += samples_per_pixel;
      i = xk;
      set_grf_points(xi,j,grf_y(clm_sndflt * ymin,ap),grf_y(clm_sndflt * ymax,ap));
      if (cp->printing) {x+=pinc; ps_set_grf_points(x,j,ymin*clm_sndflt,ymax*clm_sndflt);}
      xi++;
      j++;
      ymin=32767;
      ymax=-32768;
    }
  return(j);
}

void snd_add_locals (snd_state *ss)
{
  sop_add_snd_id("explen",0);
  sop_add_snd_id("exprmp",1);
  sop_add_snd_id("exphop",2);
  sop_add_snd_id("revfb",3);
  sop_add_snd_id("y_max",4);
  sop_add_snd_id("revlp",5);
  sop_add_snd_id("x",6);
  sop_add_snd_id("samples",7);
  sop_add_snd_id("srate",8);
  sop_add_snd_id("x_max",9);
  sop_add_snd_id("sono_max",10);
  sop_add_snd_id("color_scale",11);
  sop_add_snd_id("revdecay",12);
  sop_add_snd_id("tickle_clm",13); /* debugging etc */
  sop_add_snd_id("line_size",14);
  sop_add_snd_id("contrast_amp",15);
  sop_add_snd_id("fft_size",16);
  sop_add_snd_id("spectro_hop",17);
  sop_add_snd_id("auto_open",18);
  sop_add_snd_id("spectro_zscl",19);
  sop_add_snd_id("dot_size",20);
  sop_add_snd_id("spectro_xangle",21);
  sop_add_snd_id("spectro_yangle",22);
  sop_add_snd_id("spectro_zangle",23);
  sop_add_snd_id("spectro_xscl",24);
  sop_add_snd_id("spectro_yscl",25);
  sop_add_snd_id("wavo_hop",26);
  sop_add_snd_id("sono_color",27);
  sop_add_snd_id("spectro_color",28);
  sop_add_snd_id("color_inverted",29);
  sop_add_snd_id("color_cutoff",30);
  sop_add_snd_id("wavo_trace",31);
}

int snd_built_in_locals(void) {return(32);}

float get_snd_var(int var)
{
  snd_info *sp;
  chan_info *cp;
  axis_info *ap;
  snd_state *ss;
  cp = (chan_info *)snd_search_fd->cp;
  sp = cp->sound;
  ss = cp->state;
  switch (var)
    {
    case 0: return(sp->exp_seglen); break;
    case 1: return(sp->exp_ramp); break;
    case 2: return(sp->exp_hop); break;
    case 3: return(sp->revfb); break;
    case 4: 
      ap = cp->axis;
      return(ap->ymax);
      break;
    case 5: return(sp->revlp); break;
    case 6: return((float)(cp->cursor)); break;
    case 7: return((float)current_ed_samples(cp)); break;
    case 8: return((float)snd_SRATE(sp)); break;
    case 10: return(ss->sonogram_cutoff); break;
    case 11: return(ss->color_scale); break;
    case 12: return(ss->reverb_decay); break;
    case 13: tickle_clm(cp->state); break;
    case 14: return(ss->editor_line_size); break;
    case 15: return(sp->contrast_amp); break;
    case 16: return(ss->global_fft_size); break;
    case 17: return(ss->spectro_hop); break;
    case 18: return(ss->auto_open); break;
    case 19: return(ss->spectro_zscl); break;
    case 20: return(ss->dot_size); break;
    case 21: return(ss->spectro_xangle); break;
    case 22: return(ss->spectro_yangle); break;
    case 23: return(ss->spectro_zangle); break;
    case 24: return(ss->spectro_xscl); break;
    case 25: return(ss->spectro_yscl); break;
    case 26: return(ss->wavo_hop); break;
    case 27: return(ss->sonogram_color); break;
    case 28: return(ss->spectrogram_color); break;
    case 29: return(ss->color_inverted); break;
    case 30: return(ss->color_cutoff); break;
    case 31: return(ss->wavo_trace); break;
    }
  return(0.0);
}

static int sono_update(chan_info *cp, void *ptr)
{
  make_sonogram_axes(cp);
  update_graph(cp,NULL);
  return(0);
}

void set_sonogram_cutoff(snd_state *ss,float val)
{
  ss->sonogram_cutoff = val; 
  if (ss->fft_style != normal_fft)
    map_over_chans(ss,sono_update,NULL);
}

static int snd_set_fft_size (chan_info *cp, void *fd)
{
  fft_info *fp;
  int flen;
  if (cp->fft) 
    {
      flen = (int)fd;
      fp=cp->fft;
      if (fp->size < flen) fp->ok = 0;
      fp->size = flen;
    }
  return(0);
}

float set_global_fft_size(snd_state *ss,float val)
{
  int flen;
  flen = pow(2.0,ceil(log(val)/log(2.0)));
  ss->global_fft_size = flen;
  map_over_chans(ss,snd_set_fft_size,(void *)flen);
  map_over_chans(ss,calculate_fft,NULL);
  return(val);
}

int set_dot_size(snd_state *ss, int val)
{
  ss->dot_size = val;
  if (ss->graph_style == graph_dots) map_over_chans(ss,update_graph,NULL);
  return(val);
}

float set_snd_var(int var, float val)
{
  snd_info *sp;
  chan_info *cp;
  snd_state *ss;
  cp = (chan_info *)snd_search_fd->cp;
  sp = cp->sound;
  ss = cp->state;
  switch (var)
    {
    case 0: return(sp->exp_seglen = val); break;
    case 1: return(sp->exp_ramp = val); break;
    case 2: return(sp->exp_hop = val); break;
    case 3: return(sp->revfb = val); break;
    case 4: set_y_limits(cp,cp->axis,val); break;
    case 5: return(sp->revlp = val); break;
    case 6: cursor_moveto(cp,val); update_graph(cp,NULL); return(val); break;
    case 10: set_sonogram_cutoff(ss,val); return(val); break;
    case 11: return(ss->color_scale = val); map_over_chans(ss,update_graph,NULL); return(val); break;
    case 12: return(ss->reverb_decay = val); break;
    case 14: return(ss->editor_line_size = val); break;
    case 15: return(sp->contrast_amp = val); break;
    case 16: return(set_global_fft_size(ss,val)); break;
    case 17: ss->spectro_hop = val; map_over_chans(ss,update_graph,NULL); return(val); break;
    case 18: return(ss->auto_open = val); break;
    case 19: ss->spectro_zscl = val; map_over_chans(ss,update_graph,NULL); return(val); break;
    case 20: return(set_dot_size(ss,val)); break;
    case 21: ss->spectro_xangle = val; map_over_chans(ss,update_graph,NULL); return(val); break;
    case 22: ss->spectro_yangle = val; map_over_chans(ss,update_graph,NULL); return(val); break;
    case 23: ss->spectro_zangle = val; map_over_chans(ss,update_graph,NULL); return(val); break;
    case 24: ss->spectro_xscl = val; map_over_chans(ss,update_graph,NULL); return(val); break;
    case 25: ss->spectro_yscl = val; map_over_chans(ss,update_graph,NULL); return(val); break;
    case 26: ss->wavo_hop = val; map_over_chans(ss,update_graph,NULL); return(val); break;
    case 27: ss->sonogram_color = val; map_over_chans(ss,update_graph,NULL); return(val); break;
    case 28: ss->spectrogram_color = val; map_over_chans(ss,update_graph,NULL); return(val); break;
    case 29: ss->color_inverted = val; map_over_chans(ss,update_graph,NULL); return(val); break;
    case 30: ss->color_cutoff = val; map_over_chans(ss,update_graph,NULL); return(val); break;
    case 31: ss->wavo_trace = val; map_over_chans(ss,update_graph,NULL); return(val); break;
    }
  return(0.0);
}

