#include "snd.h"
#include <errno.h>

/* -------------------------------- EDIT LISTS -------------------------------- *
 *
 * each channel has a list of lists containing the current edit history and the associated sound temp files or buffers
 * undo: back up current list position
 * redo: push position foward
 * No actual changes are flushed out to the file system until the file is saved, so this implements unlimited undo.
 *
 * the three editing possibilities are insert, change, delete.  All input goes through these lists.
 * currently revert (as opposed to undo) clears the mark list -- dubious!
 */

/* ed_list struct definition is above chan_info definition */

static char edit_buf[128];

/* each block in an edit-list list describes one fragment of the current sound */
#define ED_OUT 0
#define ED_SND 1
#define ED_BEG 2
#define ED_END 3
#define ED_SIZE 4
#define EDIT_LIST_END_MARK -2
#define EDIT_ALLOC_SIZE 128

static void copy_ed_blocks(int *new_list, int *old_list, int new_beg, int old_beg, int num_lists)
{
  /* beg and num_lists here are in terms of blocks, not ints */
  int i,end,k;
  if (num_lists > 0)
    {
      end = (old_beg+num_lists)*ED_SIZE;
      for (k=new_beg*ED_SIZE,i=old_beg*ED_SIZE;i<end;i++,k++) {new_list[k] = old_list[i];}
    }
}

static ed_list *make_ed_list(int size)
{
  ed_list *ed;
  ed=(ed_list *)calloc(1,sizeof(ed_list));
  ed->s_type = make_snd_pointer_type(ED_LIST);
  ed->size = size;
  ed->fragments = (int *)calloc(size*ED_SIZE,sizeof(int));
  return(ed);
}

static ed_list *free_ed_list(ed_list *ed)
{
  if (ed)
    {
      if (ed->fragments) free(ed->fragments);
      free(ed);
    }
  return(NULL);
}

void free_edit_list(chan_info *cp)
{
  int i;
  if (cp)
    {
      if (cp->edits)
	{
	  for (i=0;i<cp->edit_size;i++)
	    {
	      /* cp->edit_ctr follows current edit state (redo/undo) */
	      if (cp->edits[i]) free_ed_list(cp->edits[i]);
	    }
	  free(cp->edits);
	}
      cp->edits = NULL;
      cp->edit_ctr = -1;
      cp->edit_size = 0;
    }
}

ed_list *initial_ed_list(int beg, int end)
{
  ed_list *ed;
  ed = make_ed_list(2);
  ed->fragments[0*ED_SIZE + ED_BEG] = beg;
  ed->fragments[0*ED_SIZE + ED_END] = end;
  ed->fragments[0*ED_SIZE + ED_SND] = 0;
  ed->fragments[0*ED_SIZE + ED_OUT] = 0;
  /* second block is our end-of-tree marker */
  ed->fragments[1*ED_SIZE + ED_BEG] = 0;
  ed->fragments[1*ED_SIZE + ED_END] = 0;
  ed->fragments[1*ED_SIZE + ED_SND] = EDIT_LIST_END_MARK;
  ed->fragments[1*ED_SIZE + ED_OUT] = end+1;
  return(ed);
}

static int find_split_loc (int samp, ed_list *current_state)
{
  int i,k;
  if (!current_state) {fprintf(stderr,"current state missing!"); abort();}
  for (i=0,k=0;i<current_state->size;i++,k+=ED_SIZE)
    {
      if (current_state->fragments[k+ED_OUT] >= samp) return(i);
    }
  fprintf(stderr,"find failed at %d!",samp); abort();
}

#define SND_DATA_FILE 0
#define SND_DATA_BUFFER 1

snd_data *make_snd_data_file(char *name, int *io, int *data, file_info *hdr, int temp)
{
  snd_data *sf;
  sf = (snd_data *)calloc(1,sizeof(snd_data));
  sf->s_type = make_snd_pointer_type(SND_DATA);
  sf->type = SND_DATA_FILE;
  sf->data = data;
  sf->io = io;
  sf->filename = (char *)calloc(strlen(name)+1,sizeof(char));
  strcpy(sf->filename,name);
  sf->hdr = hdr;
  sf->temporary = temp;
  return(sf);
}

snd_data *make_snd_data_buffer(int *data, int len)
{
  snd_data *sf;
  int i;
  sf = (snd_data *)calloc(1,sizeof(snd_data));
  sf->s_type = make_snd_pointer_type(SND_DATA);
  sf->type = SND_DATA_BUFFER;
  sf->data = (int *)calloc(len,sizeof(int));
  for (i=0;i<len;i++) sf->data[i] = data[i];
  return(sf);
}

static snd_data *free_snd_data(snd_data *sf)
{
  /* in the snd file case, these pointers are dealt with elsewhere */
  if (sf)
    {
      if ((sf->type == SND_DATA_BUFFER) && (sf->data)) free(sf->data);
      sf->data = NULL;
      if (sf->hdr) free_file_info(sf->hdr);
      sf->hdr = NULL;
      if (sf->io)
	{
	  clm_close(sf->io[io_fd]);
	  sf->io = free_file_state(sf->io);
	  if (sf->temporary) 
	    {
	      remove(sf->filename); 
	    }
	  free(sf->filename);
	}
      free(sf);
    }
  return(NULL);
}

void free_sound_list (chan_info *cp)
{
  int i;
  if (cp)
    {
      if (cp->sounds)
	{
	  for (i=0;i<cp->sound_size;i++)
	    {
	      if (cp->sounds[i]) cp->sounds[i] = free_snd_data(cp->sounds[i]);
	    }
	  free(cp->sounds);
	  cp->sounds = NULL;
	}
      cp->sound_ctr = -1;
      cp->sound_size = 0;
    }
}
  
static void prepare_sound_list (chan_info *cp)
{
  int i;
  cp->sound_ctr++;
  if (cp->sound_ctr >= cp->sound_size)
    {
      cp->sound_size += EDIT_ALLOC_SIZE;
      cp->sounds = (snd_data **)realloc(cp->sounds,cp->sound_size * sizeof(snd_data *));
      for (i=cp->sound_ctr;i<cp->sound_size;i++) cp->sounds[i] = NULL;
    }
  if (cp->sounds[cp->sound_ctr]) cp->sounds[cp->sound_ctr] = free_snd_data(cp->sounds[cp->sound_ctr]);
}

static int add_sound_buffer_to_edit_list(chan_info *cp, int *data, int len)
{
  prepare_sound_list(cp);
  cp->sounds[cp->sound_ctr] = make_snd_data_buffer(data, len);
  return(cp->sound_ctr);
}

static int add_sound_file_to_edit_list(chan_info *cp, char *name, int *io, int *data, file_info *hdr, int temp)
{
  prepare_sound_list(cp);
  cp->sounds[cp->sound_ctr] = make_snd_data_file(name,io,data,hdr,temp);
  return(cp->sound_ctr);
}

static void ripple_out(int *list,int beg,int num, int len)
{
  int i,k;
  for (i=beg,k=beg*ED_SIZE;i<len;i++,k+=ED_SIZE) list[k+ED_OUT] += num;
}

static void prepare_edit_list(chan_info *cp,int len)
{
  int i;
  cp->edit_ctr++;
  if (cp->edit_ctr >= cp->edit_size)
    {
      cp->edit_size += EDIT_ALLOC_SIZE;
      if (!cp->edits) cp->edits = (ed_list **)calloc(cp->edit_size,sizeof(ed_list *));
      else cp->edits = (ed_list **)realloc(cp->edits,cp->edit_size*sizeof(ed_list *));
      for (i=cp->edit_ctr;i<cp->edit_size;i++) cp->edits[i] = NULL;
      if (!cp->samples) cp->samples = (int *)calloc(cp->edit_size,sizeof(int));
      else cp->samples = (int *)realloc(cp->samples,cp->edit_size*sizeof(int));
    }
  if (cp->edits[cp->edit_ctr]) 
    {
      cp->edits[cp->edit_ctr] = free_ed_list(cp->edits[cp->edit_ctr]);
      release_pending_marks(cp,cp->edit_ctr);
    }
  cp->samples[cp->edit_ctr] = len;
}

int current_ed_samples(chan_info *cp)
{
  if (cp) return(cp->samples[cp->edit_ctr]);
}

static void display_edits(chan_info *cp)
{
  int eds,i,len,j;
  ed_list *ed;
  eds = cp->edit_ctr;
  fprintf(stderr,"\nEDITS: %d\n",eds);
  for (i=0;i<=eds;i++)
    {
      ed = cp->edits[i];
      if (!ed) {fprintf(stderr,"ctr is %d, but [%d] is nil!",eds,i); abort();}
      len=ed->size;
      fprintf(stderr,"\n  [%d:%d]:",i,len);
      for (j=0;j<len;j++)
	{
	  fprintf(stderr," (%d %d %d %d)",
		  ed->fragments[j*4+ED_OUT],ed->fragments[j*4+ED_SND],ed->fragments[j*4+ED_BEG],ed->fragments[j*4+ED_END]);
	}
    }
}

void reflect_sample_change_in_axis(chan_info *cp)
{
  axis_info *ap;
  ap = cp->axis;
  if (ap)
    {
      ap->xmax = (double)(current_ed_samples(cp))/(double)snd_SRATE(cp);
      if (ap->x1 > ap->xmax) ap->x1 = ap->xmax;
      set_x_bounds(ap);
    }
}

static ed_list *insert_samples_1 (int samp, int num, int* vals, ed_list *current_state, chan_info *cp, int **cb_back)
{
  int len,k,old_beg,old_end,old_snd,old_out;
  ed_list *new_state;
  int *cb,*cbback,*ed;
  len = current_state->size;
  ed = current_state->fragments;
  k=find_split_loc(samp,current_state);
  cb = (int *)(ed + k*ED_SIZE);
  if ((samp == cb[ED_OUT]) || (samp == (cb[ED_OUT] - 1)))
    {
      new_state = make_ed_list(len+1);
      copy_ed_blocks(new_state->fragments,ed,0,0,k);
      copy_ed_blocks(new_state->fragments,ed,k+1,k,len-k);
      len++;
    }
  else
    {
      cbback = (int *)(ed + (k-1)*ED_SIZE);
      old_beg = cbback[ED_BEG];
      old_end = cbback[ED_END];
      old_snd = cbback[ED_SND];
      old_out = cbback[ED_OUT];
      new_state = make_ed_list(len+2);
      copy_ed_blocks(new_state->fragments,ed,0,0,k);
      copy_ed_blocks(new_state->fragments,ed,k+2,k,len-k);
      cb = (int *)(new_state->fragments + (k+1)*ED_SIZE);  /* old after split */
      cbback = (int *)(new_state->fragments + (k-1)*ED_SIZE); /* old before split */
      cb[ED_SND] = old_snd;
      cb[ED_OUT] = samp;
      cb[ED_BEG] = old_beg+samp-old_out;
      cb[ED_END] = old_end;
      cbback[ED_END] = old_beg+samp-old_out-1;
      len += 2;
    }
  cb = (int *)(new_state->fragments + k*ED_SIZE); /* new */
  cb[ED_BEG] = 0;
  cb[ED_END] = num-1;
  cb[ED_OUT] = samp;
  if (vals) cb[ED_SND] = add_sound_buffer_to_edit_list(cp,vals,num); else (*cb_back) = cb;
  ripple_out(new_state->fragments,k+1,num,len);
  ripple_marks(cp,samp,num);
  ripple_selection(cp,samp,num);
  reflect_sample_change_in_axis(cp);
  clobber_amp_env(cp);
  return(new_state);
}

void file_insert_samples(int beg, int num, char *tempfile, chan_info *cp, int chan, int auto_delete)
{
  int k;
  int *cb;
  int fd;
  int *datai;
  file_info *hdr;
  k=cp->edit_ctr;
  prepare_edit_list(cp,current_ed_samples(cp)+num);
  cp->edits[cp->edit_ctr] = insert_samples_1(beg,num,NULL,cp->edits[k],cp,&cb);
  hdr = make_file_info(tempfile,cp->state);
  fd = clm_open_read(tempfile);
  open_clm_file_descriptors(fd,hdr->format,c_snd_datum_size(hdr->format),hdr->data_location);
  datai = make_file_state(fd,hdr,io_in_f,chan);
  cb[ED_SND] = add_sound_file_to_edit_list(cp,tempfile,datai,(int *)(datai[io_dats+6+chan]),hdr,auto_delete);
}

void insert_samples(int beg, int num, int *vals, chan_info *cp)
{
  int k,len;
  int *cb;
  len = current_ed_samples(cp);
  if (beg < len)
    {
      k=cp->edit_ctr;
      prepare_edit_list(cp,len+num);
      cp->edits[cp->edit_ctr] = insert_samples_1(beg,num,vals,cp->edits[k],cp,&cb);
    }
}

static ed_list *delete_samples_1(int beg, int num, ed_list *current_state, chan_info *cp)
{
  int len,k,need_to_delete,curbeg,old_out,cbi,start_del,len_fixup;
  int *cb,*temp_cb;
  ed_list *new_state;
  len=current_state->size;
  len_fixup = -1;
  k=find_split_loc(beg,current_state);
  need_to_delete = num;
  start_del = k;
  curbeg = beg;
  cb = (int *)(current_state->fragments + k*ED_SIZE);
  if (cb[ED_OUT]>beg) start_del--;
  new_state = make_ed_list(len+1);
  copy_ed_blocks(new_state->fragments,current_state->fragments,0,0,start_del);
  cbi=start_del;
  temp_cb = (int *)(current_state->fragments + start_del*ED_SIZE);
  old_out = temp_cb[ED_OUT];
  if (beg>old_out)
    {
      cb = (int *)(new_state->fragments + start_del*ED_SIZE);
      cb[ED_OUT] = old_out;
      cb[ED_SND] = temp_cb[ED_SND];
      cb[ED_BEG] = temp_cb[ED_BEG];
      cb[ED_END] = temp_cb[ED_BEG]+beg-old_out-1;
      start_del++;
      len_fixup++;
    }
  while (need_to_delete > 0)
    {
      old_out = (current_state->fragments[(cbi+1)*ED_SIZE+ED_OUT]);
      need_to_delete -= (old_out-curbeg);
      if (need_to_delete > 0)
	{
	  cbi++;
	  curbeg = old_out;
	}
    }
  if (need_to_delete < 0)
    {
      temp_cb = (int *)(current_state->fragments+cbi*ED_SIZE);
      cb = (int *)(new_state->fragments+start_del*ED_SIZE);
      cb[ED_OUT] = beg;
      cb[ED_SND] = temp_cb[ED_SND];
      cb[ED_BEG] = temp_cb[ED_END]+1+need_to_delete;
      cb[ED_END] = temp_cb[ED_END];
      start_del++;
      len_fixup++;
    }
  cbi++;
  copy_ed_blocks(new_state->fragments,current_state->fragments,start_del,cbi,len-cbi); /* ??? */
  ripple_out(new_state->fragments,start_del,-num,len+len_fixup);
  ripple_marks(cp,beg,-num);
  ripple_selection(cp,beg,-num);
  reflect_sample_change_in_axis(cp);
  clobber_amp_env(cp);
  new_state->size = len+len_fixup; /* don't propogate useless trailing blocks */
  return(new_state);
}    

void delete_samples(int beg, int num, chan_info *cp)
{
  int k,len;
  len = current_ed_samples(cp);
  if (beg < len)
    {
      if ((beg+num) > len) num = len-beg;
      k=cp->edit_ctr;
      prepare_edit_list(cp,len-num);
      cp->edits[cp->edit_ctr] = delete_samples_1(beg,num,cp->edits[k],cp);
    }
}

static ed_list *change_samples_1(int beg, int num, int *vals, ed_list *current_state, chan_info *cp, int **cb_back)
{
  int len,k,start_del,cbi,curbeg,len_fixup,need_to_delete,old_out;
  ed_list *new_state;
  int *cb,*temp_cb;
  len = current_state->size;
  len_fixup = -1;
  k=find_split_loc(beg,current_state);
  need_to_delete = num;
  start_del = k;
  curbeg = beg;
  cbi=0;
  cb = (int *)(current_state->fragments + k*ED_SIZE);
  if (cb[ED_OUT]>beg) start_del--;
  new_state = make_ed_list(len+2);
  copy_ed_blocks(new_state->fragments,current_state->fragments,0,0,start_del);
  cbi=start_del;
  temp_cb = (int *)(current_state->fragments + start_del*ED_SIZE);  
  old_out = temp_cb[ED_OUT];
  if (beg>old_out)
    {
      cb = (int *)(new_state->fragments + start_del*ED_SIZE);
      cb[ED_OUT] = old_out;
      cb[ED_SND] = temp_cb[ED_SND];
      cb[ED_BEG] = temp_cb[ED_BEG];
      cb[ED_END] = temp_cb[ED_BEG]+beg-old_out-1;
      start_del++;
      len_fixup++;
    }
  while (need_to_delete > 0)
    {
      old_out = (current_state->fragments[(cbi+1)*ED_SIZE+ED_OUT]);
      need_to_delete -= (old_out-curbeg);
      if (need_to_delete > 0)
	{
	  cbi++;
	  curbeg = old_out;
	}
    }
  cb = (int *)(new_state->fragments+start_del*ED_SIZE);
  if (vals) cb[ED_SND] = add_sound_buffer_to_edit_list(cp,vals,num); else (*cb_back) = cb;
  cb[ED_OUT] = beg;
  cb[ED_BEG] = 0;
  cb[ED_END] = num-1;
  start_del++;
  len_fixup++;
  if (need_to_delete < 0)
    {
      temp_cb = (int *)(current_state->fragments+cbi*ED_SIZE);
      cb = (int *)(new_state->fragments+start_del*ED_SIZE);
      cb[ED_OUT] = beg+num;
      cb[ED_SND] = temp_cb[ED_SND];
      cb[ED_BEG] = temp_cb[ED_END]+1+need_to_delete;
      cb[ED_END] = temp_cb[ED_END];
      start_del++;
      len_fixup++;
    }
  cbi++;
  copy_ed_blocks(new_state->fragments,current_state->fragments,start_del,cbi,len-cbi);
  new_state->size = len+len_fixup; /* don't propogate useless trailing blocks */
  ripple_marks(cp,0,0);
  clobber_amp_env(cp);
  return(new_state);
}    

void file_change_samples(int beg, int num, char *tempfile, chan_info *cp, int chan, int auto_delete)
{
  int k,len;
  int *cb;
  int fd;
  int *datai;
  file_info *hdr;
  len = current_ed_samples(cp);
  if (beg < len)
    {
      if ((beg+num) > len) num = len-beg;
      k=cp->edit_ctr;
      prepare_edit_list(cp,len);
      cp->edits[cp->edit_ctr] = change_samples_1(beg,num,NULL,cp->edits[k],cp,&cb);
      hdr = make_file_info(tempfile,cp->state);
      fd = clm_open_read(tempfile);
      open_clm_file_descriptors(fd,hdr->format,c_snd_datum_size(hdr->format),hdr->data_location);
      datai = make_file_state(fd,hdr,io_in_f,chan);
      cb[ED_SND] = add_sound_file_to_edit_list(cp,tempfile,datai,(int *)(datai[io_dats+6+chan]),hdr,auto_delete);
    }
}

void file_override_samples(int num, char *tempfile, chan_info *cp, int chan, int auto_delete)
{
  /* change_samples assumes the file stays the same length, but in the case of apply, this is not always the case */
  /* so for the newly created temp file received here, we create a new fragment list reflecting the new size */
  int fd;
  ed_list *e;
  int *datai;
  file_info *hdr;
  prepare_edit_list(cp,num);
  hdr = make_file_info(tempfile,cp->state);
  if (num == -1) num = (hdr->samples/hdr->chans);
  fd = clm_open_read(tempfile);
  open_clm_file_descriptors(fd,hdr->format,c_snd_datum_size(hdr->format),hdr->data_location);
  datai = make_file_state(fd,hdr,io_in_f,chan);
  e = initial_ed_list(0,num-1);
  cp->edits[cp->edit_ctr] = e;
  e->fragments[0 + ED_SND] = add_sound_file_to_edit_list(cp,tempfile,datai,(int *)(datai[io_dats+6+chan]),hdr,auto_delete);
  reflect_sample_change_in_axis(cp);
  ripple_marks(cp,0,0);
  clobber_amp_env(cp);
  check_for_first_edit(cp);
  update_graph(cp,NULL);
}

void file_mix_samples(int beg, int num, char *tempfile, chan_info *cp, int chan)
{
  /* open tempfile, current data, write to new temp file mixed, close others, open and use new as change case */
  /* used for clip-region/ufun temp file incoming and C-q in snd-chn.c (i.e. mix in file) so sync not relevant */
  snd_fd *csf;
  snd_state *ss;
  snd_info *sp;
  int ofd,ifd;
  char *ofile;
  int **data;
  int *chandata;
  int i,size,j,val,lengthen,cursamps;
  file_info *ihdr,*ohdr;
  sp = cp->sound;
  ss = cp->state;
  ofile = tempnam(ss->temp_dir,"snd_");
  ohdr = make_temp_header(ofile,sp->hdr,0);
  ohdr->chans = 1;
  ofd = open_temp_file(ofile,1,ohdr,ss);
  lengthen = ((beg+num) > current_ed_samples(cp));
  if (lengthen)
    csf = init_sample_read(0,cp,1);
  else csf = init_sample_read(beg,cp,1);
  ihdr = make_file_info(tempfile,ss);
  ifd = clm_open_read(tempfile);
  open_clm_file_descriptors(ifd,ihdr->format,c_snd_datum_size(ihdr->format),ihdr->data_location);
  if (num < MAX_BUFFER_SIZE) size = num; else size = MAX_BUFFER_SIZE;
  data = (int **)calloc(ihdr->chans,sizeof(int *));
  data[chan] = (int *)calloc(size,sizeof(int));
  chandata = data[chan];
  clm_seek(ofd,ohdr->data_location,0);
  clm_seek(ifd,ihdr->data_location,0);
  if (lengthen)
    { /* the problem here is that subsequent edits need true edit trees, so we can't fake up the file_change case */
      for (i=0;i<beg;i+=MAX_BUFFER_SIZE)
	{
	  cursamps = beg-i;
	  if (cursamps > MAX_BUFFER_SIZE) cursamps = MAX_BUFFER_SIZE;
	  for (j=0;j<cursamps;j++)
	    {
	      NEXT_SAMPLE(chandata[j],csf);
	    }
	  clm_write(ofd,0,cursamps-1,1,&chandata);
	}
    }
  clm_read_chans(ifd,0,size-1,ihdr->chans,data,(int *)data);
  for (i=0,j=0;i<num;i++,j++)
    {
      if (j == size)
	{
	  clm_write(ofd,0,size-1,1,&chandata);
	  clm_read_chans(ifd,0,size-1,ihdr->chans,data,(int *)data);
	  j = 0;
	}
      NEXT_SAMPLE(val,csf);
      chandata[j] += val;
    }
  if (j > 0)
    {
      clm_write(ofd,0,j-1,1,&chandata);
    }
  if (lengthen) num += beg;
  close_temp_file(ofd,ohdr,num*c_snd_datum_size(ohdr->format),sp);
  clm_close(ifd);
  free_snd_fd(csf);
  free(data[chan]);
  free(data);
  free_file_info(ihdr);
  free_file_info(ohdr);
  if (lengthen)
    file_override_samples(num,ofile,cp,0,1);
  else file_change_samples(beg,num,ofile,cp,0,1);
}

void change_samples(int beg, int num, int *vals, chan_info *cp)
{
  int k,len;
  len = current_ed_samples(cp);
  if (beg < len)
    {
      if ((beg+num) > len) num = len-beg;
      k=cp->edit_ctr;
      prepare_edit_list(cp,len);
      cp->edits[cp->edit_ctr] = change_samples_1(beg,num,vals,cp->edits[k],cp,NULL);
    }
}

void add_samples(int beg, int num, int *vals, chan_info *cp)
{
  snd_fd *sf;
  int i,val;
  sf = init_sample_read(beg,cp,1);
  for (i=0;i<num;i++)
    {
      NEXT_SAMPLE(val,sf);
      vals[i] += val;
    }
  free_snd_fd(sf);
  change_samples(beg,num,vals,cp);
}

int *load_samples(int beg, int num, chan_info *cp)
{
  snd_fd *sf;
  int *data;
  int i;
  data = (int *)calloc(num,sizeof(int));
  sf = init_sample_read(beg,cp,1);
  for (i=0;i<num;i++) {NEXT_SAMPLE(data[i],sf);}
  free_snd_fd(sf);
  return(data);
}

static int snd_file_read (snd_data *sd, int index)
{
  if ((index < sd->io[io_beg]) || (index > sd->io[io_end])) clm_file_reset(index,sd->io,sd->io);
  return(sd->data[index - sd->io[io_beg]]);
}

static int sample_1 (int samp, chan_info *cp)
{ /* slow access */
  ed_list *current_state;
  snd_data *sd;
  int len,i,cb,true_cb,index;
  int *data;
  current_state = cp->edits[cp->edit_ctr];
  data = current_state->fragments;
  len = current_state->size;
  for (i=0,cb=0;i<len;i++,cb+=ED_SIZE)
    {
      if (samp < data[cb+ED_OUT])
	{
	  true_cb = cb-ED_SIZE;
	  if (data[true_cb+ED_SND] == EDIT_LIST_END_MARK) return(0);
	  index = data[true_cb+ED_BEG]+samp-data[true_cb+ED_OUT];
	  sd = cp->sounds[data[true_cb+ED_SND]];
	  if (sd->type == SND_DATA_BUFFER)
	    return(sd->data[index]);
	  else return(snd_file_read(sd,index));
	}
    }
  fprintf(stderr,"yow!"); abort();
  return(0);
}

float sample (int samp, chan_info *cp) {return(clm_sndflt * sample_1(samp,cp));}

/* now for optimized sample access -- since everything goes through these lists, we want the access to be fast */

static snd_fd *make_snd_fd (void)
{
  snd_fd *sf;
  sf=(snd_fd *)calloc(1,sizeof(snd_fd));
  sf->s_type = make_snd_pointer_type(SND_FD);
  return(sf);
}

snd_fd *free_snd_fd(snd_fd *sf)
{
  if (sf) free(sf);
  return(NULL);
}

snd_fd *copy_snd_fd(snd_fd *sf)
{
  snd_fd *nsf;
  nsf = (snd_fd *)calloc(1,sizeof(snd_fd));
  nsf->s_type = sf->s_type;
  nsf->current_state = sf->current_state;
  nsf->cb = sf->cb;
  nsf->cbi = sf->cbi;
  nsf->eof = sf->eof;
  nsf->first = sf->first;
  nsf->last = sf->last;
  nsf->data = sf->data;
  nsf->sounds = sf->sounds;
  nsf->beg = sf->beg;
  nsf->end = sf->end;
  return(nsf);
}

static void massage_snd_file(int ind0, int ind1, int indx, snd_fd *sf, snd_data *cur_snd)
{
  /* need to track in-core buffer and file-relative index */
  if ((indx < cur_snd->io[io_beg]) || (indx > cur_snd->io[io_end])) 
    clm_file_reset(indx,cur_snd->io,cur_snd->io);
  sf->data = (int *)(cur_snd->data + indx - cur_snd->io[io_beg]);
  /* only indx is guaranteed to be within the current in-core buffer */

  if (ind0 >= cur_snd->io[io_beg])
    sf->first = (int *)(cur_snd->data + ind0 - cur_snd->io[io_beg]);
  else sf->first = cur_snd->data;

  if (ind1 <= cur_snd->io[io_end]) 
    {
      sf->last = (int *)(cur_snd->data + ind1 - cur_snd->io[io_beg]);
      sf->eof = 1;
    }
  else 
    {
      sf->last = (int *)(cur_snd->data + cur_snd->io[io_bufsiz] -1);
      sf->eof = 0;
    }
  sf->beg = cur_snd->io[io_beg];
  sf->end = cur_snd->io[io_end];
}

static void massage_snd_file_back(int ind0, int ind1, int indx, snd_fd *sf, snd_data *cur_snd)
{
  if ((indx > cur_snd->io[io_end]) || (indx < cur_snd->io[io_beg])) 
    clm_file_reset(indx-cur_snd->io[io_bufsiz]+1,cur_snd->io,cur_snd->io);
  sf->data = (int *)(cur_snd->data + indx - cur_snd->io[io_beg]);

  if (ind1 <= cur_snd->io[io_end])
    sf->last = (int *)(cur_snd->data + ind1 - cur_snd->io[io_beg]);
  else sf->last = (int *)(cur_snd->data + cur_snd->io[io_bufsiz] -1);

  if (ind0 >= cur_snd->io[io_beg]) 
    {
      sf->first = (int *)(cur_snd->data + ind0 - cur_snd->io[io_beg]);
      sf->eof = 1;
    }
  else 
    {
      sf->first = cur_snd->data;
      sf->eof = 0;
    }
  sf->beg = cur_snd->io[io_beg];
  sf->end = cur_snd->io[io_end];
}

snd_fd *init_sample_read (int samp, chan_info *cp, int direction)
{
  snd_fd *sf;
  ed_list *ed;
  int len,i,k,ind0,ind1,indx;
  int *cb;
  snd_data *first_snd;
  sf = make_snd_fd();
  sf->direction = 0;
  sf->cp = cp;
  ed = (ed_list *)(cp->edits[cp->edit_ctr]);
  sf->current_state = ed;
  len = ed->size;
  for (i=0,k=0;i<len;i++,k+=ED_SIZE)
    {
      cb = (int *)(ed->fragments+k);
      if ((cb[ED_OUT] > samp) || (cb[ED_SND] == EDIT_LIST_END_MARK))
	{
	  sf->cb = (int *)(ed->fragments+k-ED_SIZE);
	  sf->cbi = i-1;
	  ind0 = sf->cb[ED_BEG];
	  indx = sf->cb[ED_BEG]+samp-sf->cb[ED_OUT];
	  ind1 = sf->cb[ED_END];
	  sf->sounds = (snd_data **)(cp->sounds);
	  first_snd = sf->sounds[sf->cb[ED_SND]];
	  if (first_snd->type == SND_DATA_FILE)
	    {
	      if (direction > 0)
		massage_snd_file(ind0,ind1,indx,sf,first_snd);
	      else massage_snd_file_back(ind0,ind1,indx,sf,first_snd);
	    }
	  else 
	    {
	      sf->data = (int *)(first_snd->data+indx);
	      sf->first = (int *)(first_snd->data+ind0);
	      sf->last = (int *)(first_snd->data+ind1);
	      sf->eof = 1;
	    }
	  sf->current_value = (*sf->data);
	  return(sf);
	}
    }
  display_edits(cp);
  fprintf(stderr,"big trouble"); abort();
  return(NULL);
 }

int previous_sound (snd_fd *sf) 
{
  int ind0,ind1,indx;
  snd_data *prev_snd;
  if (sf->eof)
    {
      if (sf->cbi == 0) return(0); /* can't back up any further */
      sf->cbi--;
      /* now start in the final portion of this block (if a file) */
      sf->cb = (int *)((sf->current_state)->fragments+sf->cbi*ED_SIZE);
      ind0 = sf->cb[ED_BEG];
      ind1 = sf->cb[ED_END];
      prev_snd = sf->sounds[sf->cb[ED_SND]];
      if (prev_snd->type == SND_DATA_FILE)
	{
	  massage_snd_file_back(ind0,ind1,ind1,sf,prev_snd);
	}
      else 
	{
	  sf->data = (int *)(prev_snd->data+ind1);
	  sf->first = (int *)(prev_snd->data+ind0);
	  sf->last = sf->data;
	  sf->eof = 1;
	}
    }
  else
    {
      /* back up in current file */
      ind0 = sf->cb[ED_BEG];
      ind1 = sf->cb[ED_END];
      indx = sf->beg-1;
      massage_snd_file_back(ind0,ind1,indx,sf,sf->sounds[sf->cb[ED_SND]]);
    }
  return(sf->current_value = *sf->data--);
}

int next_sound (snd_fd *sf)
{
  int ind0,ind1,indx;
  snd_data *nxt_snd;
  if (sf->eof) /* a convenience -- we could figure this out from various pointers */
    {
      if (sf->last == (int *)0) return(0);
      if (!(sf->cb)) return(0);
      sf->cbi++;
      sf->cb = (int *)((sf->current_state)->fragments+sf->cbi*ED_SIZE);
      if (sf->cb[ED_SND] == EDIT_LIST_END_MARK) 
	{
          sf->data = (int *)1;
	  sf->last = (int *)0; /* can I get away with this?? */
	  return(0);
	}
      ind0 = sf->cb[ED_BEG];
      ind1 = sf->cb[ED_END];
      nxt_snd = sf->sounds[sf->cb[ED_SND]];
      if (nxt_snd->type == SND_DATA_FILE)
	{
	  massage_snd_file(ind0,ind1,ind0,sf,nxt_snd);
	}
      else 
	{
	  sf->data = (int *)(nxt_snd->data+ind0);
	  sf->first = sf->data;
	  sf->last = (int *)(nxt_snd->data+ind1);
	  sf->eof = 1;
	}
    }
  else
    { 
      ind0 = sf->cb[ED_BEG];
      ind1 = sf->cb[ED_END];
      indx = sf->end+1;
      massage_snd_file(ind0,ind1,indx,sf,sf->sounds[sf->cb[ED_SND]]);
    }
  return(sf->current_value = *sf->data++);
}

int next_sample_1 (snd_fd *sf)
{
  if (sf->data > sf->last) return(next_sound(sf));
  return(sf->current_value = *sf->data++);
}

int previous_sample_1(snd_fd *sf)
{
  if (sf->data < sf->first) return(previous_sound(sf));
  return(sf->current_value = *sf->data--);
}

float any_sample(snd_fd *sf, int off)
{
  int true_off,val,i;
  int *sadr;
  snd_fd *nsf;
  /* for search routines, depends on current_value, direction etc -- not a general function! */
  if (off == 0) return(clm_sndflt * (sf->current_value));
  /* postincrement/decrement assumed here -- that is don't ask for offset value immediately after init_sample_read */
  true_off = off-sf->direction;
  /* now look for current block bounds -- maybe we'll luck out */
  sadr = (int *)(sf->data + true_off);
  if ((sadr >= sf->first) && (sadr <= sf->last))
    return(clm_sndflt * (*sadr));
  /* now a hard case -- we have to walk the edit-tree sample by sample */
  nsf = copy_snd_fd(sf); /* repeats current sample */
  if (true_off > 0)
    {
      for (i=0;i<=true_off;i++) val=next_sample_1(nsf);
    }
  else
    {
      for (i=true_off;i<=0;i++) val=previous_sample_1(nsf);
    }
  free_snd_fd(nsf);
  return(clm_sndflt * val);
}


int next_sub_sound (snd_fd *sf, int inc)
{
  int i,val,val1;
  NEXT_SAMPLE(val1,sf);
  for (i=0;i<inc-1;i++) NEXT_SAMPLE(val,sf);
  return(val1);
}

float next_sample (snd_fd *sf) {return(clm_sndflt * next_sample_1(sf));}
float previous_sample (snd_fd *sf) {return(clm_sndflt * previous_sample_1(sf));}

#define NEXT_SAMPLES(data,num,sf) {int i; for (i=0;i<num;i++) NEXT_SAMPLE(data[i],sf);}

int read_sample_eof (snd_fd *sf)
{
  if (sf->cb)
    return((sf->cb[ED_SND] == EDIT_LIST_END_MARK) || ((sf->cbi == 0) && (sf->data < sf->first)));
  else return(sf->data >= sf->last);
}


/* -------------------------------- EDITS -------------------------------- */


int snd_IO_error;

int lisp_call(int hi) {snd_IO_error = hi; return(0);}


static char *snd_error_names[] = {
  snd_string_no_error,snd_string_cant_write_header,snd_string_cant_open_file,
  snd_string_cant_allocate_IO_buffers,snd_string_cant_write_data,
  snd_string_cant_remove_file,snd_string_cant_rename_file,snd_string_cant_read_header,
  snd_string_unsupported_file_type,snd_string_cant_find_file,snd_string_unsupported_data_format};

char *snd_error_name(int i) {return(snd_error_names[i]);}

int open_temp_file(char *ofile, int chans, file_info *hdr, snd_state *ss)
{
  int ofd;
  if (!(output_type_and_format_ok(hdr->type,hdr->format)))
    {
      hdr->type = ss->default_output_type;
      if (hdr->type != RIFF_sound_file)
	hdr->format = snd_16_linear;
      else hdr->format = snd_16_linear_little_endian;
    }
  if ((c_write_header(ofile,hdr->type,hdr->srate,chans,hdr->data_location,0,hdr->format,"",0)) == -1) return(-1);
  if ((ofd = clm_reopen_write(ofile)) == -1) return(-1);
  hdr->data_location = c_snd_header_data_location(); /* header might have changed size (aiff extras) */
  open_clm_file_descriptors(ofd,hdr->format,c_snd_datum_size(hdr->format),hdr->data_location);
  clm_seek(ofd,hdr->data_location,0);
  return(ofd);
}

int close_temp_file(int ofd, file_info *hdr, long bytes, snd_info *sp)
{
  int kleft;
  c_update_header_with_fd(ofd,hdr->type,bytes);
  kleft = disk_kspace(ofd);
  if ((bytes>>10) > kleft) report_in_minibuffer(sp,snd_string_were_getting_short_on_disk_space);
  clm_close(ofd);
  return(0);
}


static int make_file(char *ofile, int chans, file_info *hdr, snd_fd **sfs, int *length, snd_state *ss)
{
  /* create ofile, fill it by following sfs, use hdr for srate/type/format decisions */
  int ofd;
  int i,j,len,datumb;
  int **obufs;
  ofd = open_temp_file(ofile,chans,hdr,ss);
  if (ofd == -1) return(snd_cannot_open_temp_file);
  datumb = c_snd_datum_size(hdr->format);
  obufs = (int **)calloc(chans,sizeof(int *));
  for (i=0;i<chans;i++)
    {
      obufs[i] = (int *)calloc(FILE_BUFFER_SIZE,sizeof(int));
    }
  j=0;
  len = 0;
  while (!(read_sample_eof(sfs[0]))) /* assume all channels are same length (?) */
    {
      for (i=0;i<chans;i++)
	{
	  NEXT_SAMPLE(obufs[i][j],sfs[i]);
	}
      j++;
      len++;
      if (j == FILE_BUFFER_SIZE)
	{
	  clm_write(ofd,0,j-1,chans,obufs);
	  if (snd_IO_error) return(snd_cannot_write_data);
	  j=0;
	}
    }
  if (j > 0)
    {
      clm_write(ofd,0,j-1,chans,obufs);
      if (snd_IO_error) return(snd_cannot_write_data);
    }
  close_temp_file(ofd,hdr,len*chans*datumb,any_selected_sound(ss));
  alert_new_file();
  for (i=0;i<chans;i++) free(obufs[i]);
  free(obufs);
  (*length) = len;
  return(0);
}

char *only_save_edits(snd_info *sp, int *err)
{
  char *ofile;
  snd_state *ss;
  int i,samples;
  snd_fd **sf;
  ss = sp->state;
  samples = 0;
  ofile = tempnam(ss->temp_dir,"snd_");
  sf = (snd_fd **)calloc(sp->nchans,sizeof(snd_fd *));
  for (i=0;i<sp->nchans;i++) 
    {
      sf[i] = init_sample_read(0,sp->chans[i],1);
    }
  (*err) = make_file(ofile,sp->nchans,sp->hdr,sf,&samples,ss);
  for (i=0;i<sp->nchans;i++) free_snd_fd(sf[i]);
  free(sf);
  return(ofile); /* should be freed by caller */
}

int save_edits_1(snd_info *sp, char *new_name)
{
  /* open temp, write current state, rename to old, reopen and clear all state */
  /* can't overwrite current because we may have cut/paste backpointers scattered around the current edit list */
  /* have to decide here what header/data type to write as well -- original? */
  /* if latter, must be able to write all headers! -- perhaps warn user and use snd/aiff/riff/ircam */
  char *ofile;
  int err;
  snd_state *ss;
  int i,samples,call_free;
  chan_info *cp;
  axis_info *ap;
  snd_fd **sf;
  float *axis_data;
  ss = sp->state;
  samples = 0;
  snd_IO_error = 0;
  if (new_name) {ofile = new_name; call_free=0;} else {ofile = tempnam(ss->temp_dir,"snd_"); call_free=1;}
  /* this will use user's TMPDIR if ss->temp_dir is not set, else stdio.h's P_tmpdir else /tmp */
  sf = (snd_fd **)calloc(sp->nchans,sizeof(snd_fd *));
  for (i=0;i<sp->nchans;i++) 
    {
      sf[i] = init_sample_read(0,sp->chans[i],1);
    }
  sprintf(edit_buf,snd_string_saving,sp->shortname);
  report_in_minibuffer(sp,edit_buf);
  snd_IO_error = make_file(ofile,sp->nchans,sp->hdr,sf,&samples,ss);
  if (snd_IO_error != snd_no_error) 
    {
      for (i=0;i<sp->nchans;i++) free_snd_fd(sf[i]);
      free(sf);
      return(snd_IO_error);
    }
  ((file_info *)(sp->hdr))->samples = samples*sp->nchans;
  collapse_marks(sp);
  for (i=0;i<sp->nchans;i++)
    {
      cp = sp->chans[i];
      if (cp->edits) free_edit_list(cp);
      if (cp->sounds) free_sound_list(cp);
      free_snd_fd(sf[i]);
    }
  free(sf);
  if (!new_name)
    {
      if (err = (rename(ofile,sp->fullname)))
	{
	  /* rename can't go across device boundaries, among other things, so it it fails
	   * here for some obvious reason, we'll try to copy before giving up
	   */
	  if (errno == EXDEV) /* cross device error */
	    {
	      err = copy_file(ofile,sp->fullname,sp);
	      if (!err) err = remove(ofile);
	    }
	  if (err) return(snd_cannot_rename_file);
	}
      sp->write_date = file_write_date(sp->fullname);
      add_sound_data(sp->fullname,sp,ss);
    }
  else 
    {
      axis_data = (float *)calloc(4*sp->nchans,sizeof(float));
      for (i=0;i<sp->nchans;i++)
	{
	  cp = sp->chans[i];
	  ap = cp->axis;
	  axis_data[(i*4)+0]=ap->x0;
	  axis_data[(i*4)+1]=ap->x1;
	  axis_data[(i*4)+2]=ap->y0;
	  axis_data[(i*4)+3]=ap->y1;
	}
      snd_close_file(sp,ss);
      alert_new_file();
      sp = snd_open_file(new_name,ss);
      for (i=0;i<sp->nchans;i++)
	{
	  cp = sp->chans[i];
	  set_axes(cp,axis_data[(i*4)+0],axis_data[(i*4)+1],axis_data[(i*4)+2],axis_data[(i*4)+3]);
	}
      free(axis_data);
    }
  if ((call_free) && (ofile)) {free(ofile); ofile=NULL;}
  reflect_file_revert_in_label(sp);
  reflect_file_save_in_menu(ss);
  sprintf(edit_buf,snd_string_wrote,sp->fullname);
  report_in_minibuffer(sp,edit_buf);
  return(snd_no_error);
}

int save_edits_2(snd_info *sp, char *new_name, int type, int format, int srate)
{
  file_info *hdr;
  hdr = sp->hdr;
  hdr->format = format;
  hdr->srate = srate;
  hdr->type = type;
  return(save_edits_1(sp,new_name)); 
  /* calls make_file -> open_temp_file which follows header where possible */
  /* true input header info is stashed away in the clm-style array, so no-one notices these changes internally */
}

int chan_save_edits(chan_info *cp, char *ofile)
{
  /* channel extraction -- does not cause reversion of edits, or change of in-window file, etc */
  snd_info *sp;
  snd_fd **sf;
  int samples;
  sp = cp->sound;
  snd_IO_error = 0;
  sf = (snd_fd **)calloc(1,sizeof(snd_fd *));
  sf[0] = init_sample_read(0,cp,1);
  snd_IO_error = make_file(ofile,1,sp->hdr,sf,&samples,cp->state);
  free_snd_fd(sf[0]);
  free(sf);
  return(snd_IO_error);
}

int save_edits(snd_info *sp, void *ptr)
{
  int i,need_save,err,current_write_date;
  chan_info *cp;
  if (!sp->read_only)
    {
      need_save = 0;
      for (i=0;i<sp->nchans;i++)
	{
	  cp = sp->chans[i];
	  if (cp->edit_ctr > 0) 
	    {
	      need_save = 1;
	      break;
	    }
	}
      if (need_save)
	{
	  errno = 0;
	  /* check for change to file while we were editing it */
	  current_write_date = file_write_date(sp->fullname);
	  if (current_write_date != sp->write_date)
	    {
	      sprintf(edit_buf,snd_string_changed_on_disk_p,sp->shortname);
	      err = snd_yes_or_no_p(sp->state,edit_buf);
	      if (err == 0) return(0);
	    }
	  err = save_edits_1(sp,NULL);
	  if (err)
	    {
	      sprintf(edit_buf,"%s: %s (%s)",sp->fullname,strerror(errno),snd_error_name(err));
	      report_in_minibuffer(sp,edit_buf);
	    }
	}
    }
  else
    {
      sprintf(edit_buf,snd_string_cant_write,sp->shortname);
      strcat(edit_buf," ("); strcat(edit_buf,snd_string_read_only); strcat(edit_buf,")");
      text_set_string(snd_widget(sp,W_snd_info),edit_buf);
    }
  return(0);
}

int revert_edits(chan_info *cp, void *ptr)
{
  cp->edit_ctr = 0;
  reflect_sample_change_in_axis(cp);
  update_graph(cp,NULL);
  return(0);
}



/* -------- UNDO -------- */

static void undo_edit(chan_info *cp, int count)
{
  if ((cp) && (cp->edit_ctr > 0) && (count != 0))
    {
      cp->edit_ctr -= count; 
      if (cp->edit_ctr < 0) cp->edit_ctr = 0;
      reflect_sample_change_in_axis(cp);
      reflect_undo_in_menu();
      if (cp->edit_ctr == 0)
	{
	  reflect_file_revert_in_label(cp->sound);
	  reflect_file_revert_in_menu(cp->state);
	}
      update_graph(cp,NULL);
    }
}

void undo_EDIT(void *ptr, int count, int syncd)
{
  chan_info *cp;
  snd_info *sp;
  int i;
  sync_info *si;
  si = NULL;
  cp = current_channel(ptr);
  if (syncd)
    {
      sp = cp->sound;
      if (sp->syncing) si = snd_sync(cp->state);
    }
  if (si)
    {
      for (i=0;i<si->chans;i++) undo_edit(si->cps[i],count);
      free_sync_info(si);
    }
  else undo_edit(cp,count);
}

static void reflect_file_change_in_label (chan_info *cp)
{
  snd_info *sp = cp->sound;
  char *starred_name;
  int len;
  len = strlen(sp->shortname) + 2;
  if (sp->read_only) len+=2;
  starred_name = (char *)calloc(len,sizeof(char));
  strcpy(starred_name,sp->shortname);
  if (sp->read_only)
    {
      starred_name[len-4]='(';
      starred_name[len-3]='*';
      starred_name[len-2]=')';
    }
  else starred_name[len-2]='*';
  starred_name[len-1]='\0';
  make_name_label(snd_widget(sp,W_snd_name),starred_name);
  make_a_big_star_outa_me(sp->state,sp->shortname,1);
  free(starred_name);
}

static void redo_edit(chan_info *cp, int count)
{
  if (cp)
    {
      if (cp->edit_ctr == 0) 
	{
	  reflect_file_change_in_label(cp);
	  reflect_redo_in_menu();
	}
      cp->edit_ctr += count; 
      while ((cp->edit_ctr >= cp->edit_size) || (!(cp->edits[cp->edit_ctr]))) {cp->edit_ctr--;}
      if (((cp->edit_ctr+1) == cp->edit_size) || (!(cp->edits[cp->edit_ctr+1]))) 
	reflect_no_more_redo_in_menu(cp->sound);
      reflect_sample_change_in_axis(cp);
      update_graph(cp,NULL);
    }
}

void redo_EDIT(void *ptr, int count, int syncd)
{
  chan_info *cp;
  snd_info *sp;
  int i;
  sync_info *si;
  si = NULL;
  cp = current_channel(ptr);
  if (syncd)
    {
      sp = cp->sound;
      if (sp->syncing) si = snd_sync(cp->state);
    }
  if (si)
    {
      for (i=0;i<si->chans;i++) redo_edit(si->cps[i],count);
      free_sync_info(si);
    }
  else redo_edit(cp,count);
}


void check_for_first_edit(chan_info *cp)
{
  if (cp->edit_ctr == 1) /* first edit on this file (?) */
    {
      reflect_file_change_in_menu();
      reflect_file_change_in_label(cp);
    }
}


