/* sigscr.c
 * Screen handling for ncurses SigGEN program
 * Jim Jackson    Dec 96
 */

/*
 * Copyright (C) 1997 Jim Jackson                    jj@scs.leeds.ac.uk
 *                    School of Computer Studies,
 *                    The University of Leeds,
 *                    Leeds, LS2 9JT, UK
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with this program - see the file COPYING; if not, write to 
 *  the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
 *  MA 02139, USA.
 */

#include <time.h>
#include "ncfio.h"
#include "siggen.h"

/* Main screen character strings............
 */

char *hl[]={
   "     /====\\            /====\\  ======  |\\    |  ",
   "     |                 |       |       | \\   |    ",
   "     \\====\\  o  /--\\   |   __  |===    |  \\  | ",
   "          |  |  |  |   |    |  |       |   \\ |    ",
   "     \\====/  |  \\--|   \\====/  ======  |    \\| ",
   "                   |  ",
   "       Digital  \\--/  Signal Generator ", 
   " -----------------------------------------------------------------------------",
   NULL };

char *oh[]={
   "      /\\         ",
   "   --/--\\--/--   ", 
   "         \\/      ",
   NULL };

char *sig=" Jim Jackson <jj@scs.leeds.ac.uk> ";

char Mchan1[]="  Channel 1 ",
     Mchan2[]="  Channel 2 ",
     Mstarting[]=" Starting up .......... ",
     Mwavef[]="  Waveform: ",
     Mfreq[]= " Frequency: ",
     Mgain[]= " Hz   Gain: ",
     Mratio[]=" M/S Ratio: ",
     Moffs[]= "Phaseshift: ",
     Msampl[]="  Samplerate:       /sec",
     Minfo[]=" : ",
     Mtime[16]="",
     Mstereo[]="STEREO",
     Mmono[]=  " MONO ",
     M16b[]=  " 16 bit samples ",
     M8b[]=   "  8 bit samples ",
     Mnoplay[]=
     "Setup Parameters.     LEFT/RIGHT arrow moves fields, 'Q' quit, 'ESC' play.",
     Mplay[]=
     "Playing                                            'ESC' to stop  playing.",
     Moff[]="OFF", Msin[]="SINE", Mcos[]="COSINE", Msqu[]="SQUARE",
     Mtri[]="TRIANGLE", Msaw[]="SAWTOOTH", Mpulse[]="PULSE", 
     Mnoise[]="NOISE";

char *WAVEFORMS[]={ Moff, Msin, Mcos, Msqu, Mpulse, Msaw, Mtri, Mnoise, NULL };

struct NCField
 FsamP= { NCF_string, 9, 0, 0, Msampl, 80, 0 },
 FsamV= { NCF_integer, 9, 14, 0, &samplerate, 6, 0 },
 Fbits= { NCF_string, 9, 30, 0, NULL, 16, 0 },
 Fchans= { NCF_string, 9, 50, 0, NULL, 6, 0 },    
 FinfoP= { NCF_string, 21, 0, 0, Minfo, 80, 0 },
 FinfoP2= { NCF_string, 22, 0, 0, Minfo, 80, 0 },
 Finfo= { NCF_string, 21, 3, 0, Mstarting, 75, 0 },
 Finfo2= { NCF_string, 22, 3, 0, "", 75, 0 },
 Ftime= { NCF_string, 9, 66, 0, Mtime, 10, 0 },

 Fchan1= { NCF_string,  11, 0, 0, Mchan1, 12, 0 },
 Fwav1P= { NCF_string,  12, 2, 0, Mwavef, 12, 0 },
 Fwav1V= { NCF_option,  12, 14, 0, WAVEFORMS, 10, 0 },
 Ffrq1P= { NCF_string,  12, 24, 0, Mfreq, 12, 0 },
 Ffrq1V= { NCF_integer, 12, 36, 0, &freq, 6, 0 },
 Fgn1P=  { NCF_string,  12, 42, 0, Mgain, 12, 0 },
 Fgn1V=  { NCF_integer, 12, 54, 0, &Gain, 5, 0 },
    
 Fchan2= { NCF_string,  15, 0, 0, Mchan2, 12, 0 },
 Fwav2P= { NCF_string,  16, 2, 0, Mwavef, 12, 0 },
 Fwav2V= { NCF_option,  16, 14, 0, WAVEFORMS, 10, 0 },
 Ffrq2P= { NCF_string,  16, 24, 0, Mfreq, 12, 0 },
 Ffrq2V= { NCF_integer, 16, 36, 0, &freq2, 6, 0 },
 Fgn2P=  { NCF_string,  16, 42, 0, Mgain, 12, 0 },
 Fgn2V=  { NCF_integer, 16, 54, 0, &Gain2, 5, 0 } 
;

struct NCField 
 *CHhdrs[]= { &Fchan1, &Fwav1P, &Ffrq1P, &Fgn1P,
              &Fchan2, &Fwav2P, &Ffrq2P, &Fgn2P, NULL },
 *CHvals[]= { &Fwav1V, &Ffrq1V, &Fgn1V,
              &Fwav2V, &Ffrq2V, &Fgn2V, NULL },
 *scrP[]= { &FsamP, &FinfoP, &FinfoP2, NULL, },
 *scrV[]= { &Finfo, &Finfo2, &FsamV, &Fbits, &Fchans, &Ftime, NULL };
    
/* Help strings for second info line.
 */

char Hwav[]="DOWN (or SPACE) & UP ARROW to select Function",
     Hnum[]="Enter Digits, DEL or <- to delete, UP/DOWN arrow for Inc/Decrement";

char *Hstr[]={ Hwav, Hnum, Hnum, 
                Hwav, Hnum, Hnum, NULL };

/* WinGen()   The main interactive screen
 */

WinGen(c)
int c;
{
   int att_h1,att_h2,att_h3,att_h4,att_h5,att_h6;
   int i,n,st;
   static int first_time=1;
   
   if (first_time) {
      initscr();
      cbreak(); noecho();
      nonl(); intrflush(stdscr,FALSE); keypad(stdscr,TRUE);
      
      if (has_colors()) {
	 start_color();
	 init_pair(1,COLOR_BLUE,COLOR_GREEN);
	 init_pair(COLOR_PAIRS-1,COLOR_GREEN,COLOR_BLUE);
	 init_pair(2,COLOR_CYAN,COLOR_RED);
	 init_pair(COLOR_PAIRS-2,COLOR_RED,COLOR_CYAN);
	 init_pair(3,COLOR_GREEN,COLOR_RED);
	 init_pair(COLOR_PAIRS-3,COLOR_RED,COLOR_GREEN);
	 init_pair(4,COLOR_YELLOW,COLOR_RED);
	 init_pair(COLOR_PAIRS-4,COLOR_RED,COLOR_YELLOW);
	 init_pair(5,COLOR_WHITE,COLOR_BLUE);
	 init_pair(COLOR_PAIRS-5,COLOR_BLUE,COLOR_WHITE);
	 init_pair(6,COLOR_RED,COLOR_BLUE);
	 init_pair(COLOR_PAIRS-6,COLOR_BLUE,COLOR_RED);
	 att_h1=COLOR_PAIR(1);
	 att_h2=COLOR_PAIR(2)+A_BOLD;
	 att_h3=COLOR_PAIR(3);
	 att_h4=COLOR_PAIR(4)+A_BOLD;
	 att_h5=COLOR_PAIR(5);
	 att_h6=COLOR_PAIR(6);
      } else {
	 att_h1=A_REVERSE; att_h2=att_h4=A_BOLD; att_h3=A_NORMAL;
      }

      Fbits.val.s=(afmt==8)?M8b:M16b;
      Fchans.val.s=(stereo)?Mstereo:Mmono;
      
      dofatt(scrV,att_h4); dofatt(scrP,att_h3);
      dofatt(CHhdrs,att_h5);
      dofatt(CHvals,att_h6);
   
      dohdr(0,att_h1,att_h2,att_h3);
      n=8;
      for (i=0; i<n; i++) {
	 mvputfstr(i+Fchan1.row-1,0,att_h5,"",80);
      }

      dofields(scrP); dofields(scrV);
      dofields(CHhdrs);
      dofields(CHvals);
      refresh();
      first_time=0;
   }

   for (st=0; st==0; ) {

      doinfo2(Mnoplay,0);
      c=doinput(CHvals,Hstr);
      if (c==KEY_END || c=='q' || c=='Q') {
	 move(COLS-1,0);
	 erase();
	 endwin();
	 return(c);
      }
    
      if ((st=generate(Fwav1V.val.sp[Fwav1V.adj],sbuf,sbuf_size,
		       freq,Gain,samplerate,ratio,afmt))==0) {
	 doinfo2("Problem Generating Channel 1 waveform.",5);
      } else {
	 if ((st=generate(Fwav2V.val.sp[Fwav2V.adj],sbuf2,sbuf_size,
			  freq2,Gain2,samplerate,ratio2,afmt))==0) {
	    doinfo2("Problem Generating Channel 2 waveform.",5);
	 }
      }
   }

   if (stereo) {
      /* stereo - mix left, right samples into stereo playbuffer */
      mixplaybuf(plbuf,sbuf,sbuf2,samplerate,afmt); 
   } else {
      /* mono - digitally mix the two channels into mono play buffer */
      if (Fwav1V.adj && Fwav2V.adj) {
	 chanmix(plbuf,sbuf,Gain,sbuf2,Gain2,samplerate,afmt);
      } else if (Fwav1V.adj) {
	 memcpy(plbuf,sbuf,sbuf_size);
      } else {
	 memcpy(plbuf,sbuf2,sbuf_size);
      }
   }      
   doinfo("",0);
   doinfo2(Mplay,0);
   
   return(0);
}

doinput(aa,H)
struct NCField *aa[];
char **H;
{
   int i,c;
   
   for ( i=0; ; ) {
      doinfo(H[i],0);
      c=getfield(aa[i]);
      if (c==27 || c==KEY_END || c=='q' || c=='Q') break;
      if (c==KEY_LEFT) {
	 if (i) i--;
	 else {
	    for ( ; aa[i+1]!=NULL; i++) { }
	 }
      } else if (c==KEY_RIGHT || c=='\r' || c=='\n' || c=='\t') {
	 i++; if (aa[i]==NULL) i=0;
      }
   }
   return(c);
}

/* dohdr(y,a1,a2,a3)  write header out starting at line y using attributes
 *   a1, s2, and a3
 */

dohdr(y,a1,a2,a3)
int y,a1,a2,a3;
{
   int i;

   for ( i=0; hl[i]!=NULL; i++) {
      mvputfstr(y+i,0,a1,hl[i],80);
   }
   for ( i=0; oh[i]!=NULL; i++) {
      mvputfstr(y+i+1,53,a2,oh[i],17);
   }
   mvputfstr(y+6,44,a3,sig,strlen(sig));
   mvputfstr(24,0,a1,VERSION,80);
/*   dotime(time(NULL)); */
}

/* Display string s in the Info field of the screen, and pause for w
 * seconds, if w is non-zero
 */

doinfo(s,w)
char *s;
int w;
{
   Finfo.val.s=s;
   putfield(&Finfo);
   refresh();
   if (w) sleep(w);
}

/* Display string s in the 2nd Info field of the screen, and pause for w
 * seconds, if w is non-zero
 */

doinfo2(s,w)
char *s;
int w;
{
   Finfo2.val.s=s;
   putfield(&Finfo2);
   refresh();
   if (w) sleep(w);
}


dotime(t)
int t;
{
   sprintf(Mtime," %2d:%02d:%02d ",(t/3600)%24,(t/60)%60,t%60);
   putfield(&Ftime);
   refresh();
}

dofields(aa)
struct NCField *aa[];
{
   int i;
   
   for (i=0; aa[i]!=NULL; i++) {
      putfield(aa[i]);
   }
}

dofatt(aa,at)
struct NCField *aa[];
int at;
{
   int i;
   
   for (i=0; aa[i]!=NULL; i++) {
      aa[i]->attr=at;
   }
}

mergefields(aa1,aa2)
struct NCField *aa1[],*aa2[];
{
   int i,j;
   
   for (i=0; aa1[i]!=NULL; i++) {
   }
   for (j=0; aa2[j]!=NULL; i++,j++) {
      aa1[i]=aa2[j];
   }
   aa1[i]=aa2[j];      /* copy the terminating null */
}

   

key_block()
{
   timeout(-1);
}

no_key_block()
{
   timeout(20);
}

