/* siggen.c
 * Ncurses based Signal generator, 16/8 bit, 2 channels if stereo card,
 * Emulates usual functions of a lab sig. gen. generating sine, square,
 *  triangle, sawtooth, pulse, noise waveforms.
 *
 * Linux Version
 */

/*
 * Copyright (C) 1997 Jim Jackson                    jj@scs.leeds.ac.uk
 *                    School of Computer Studies,
 *                    The University of Leeds,
 *                    Leeds, LS2 9JT, UK
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with this program - see the file COPYING; if not, write to 
 *  the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
 *  MA 02139, USA.
 */

/*
 * siggen :
 * 
 * A one second's worth buffer of samples is maintained (samplerate samples)
 * for each channel. These are mixed into a 1 second play buffer
 * that in play mode is played in a circular fashion. By keeping freq.
 * as an integer number of Hertz there is always an exact number of full
 * cycles in 1 sec, so the 1 sec play buffer start and finish match
 * seamlessly. If only mono card then the two channels are mixed into
 * one playing channel.
 * 
 * ToDo:  Still not got offset/ratio input sorted on screen.
 *        Mono Mixing into one buffer needs to take account of gains.
 *        Continous playing to do.
 * 
 * History:
 *  --Feb97 V1.1   Various screen changes - help line, some tweeking
 *                 of ncfio key handling etc.
 *  --Jan97 V1.0   2 channels - if stereo support, then each chan.
 *                 is fed to a seperate stereo output. If only mono
 *                 is possible then the two channels are mixed into the
 *                 one output.
 *                 While entering details of function required
 *                 the program ceases playing any previously generated
 *                 samples.
 */

#include <stdio.h>
#include <strings.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/soundcard.h>
#include <math.h>
#include "siggen.h"
#include <ncurses/ncurses.h>

#define DAC_FILE "/dev/dsp"
#define SAMPLERATE 22050
#define EFAIL -1

#define MAXPRM 32
#define chkword(a,b) ((n>=a)&&(strncmp(p,b,n)==0))

int vflg,dflg;

unsigned int samplerate;         /* Samples/sec        */
unsigned int stereo;             /* stereo mono */
unsigned int afmt;               /* format for DSP  */
    /* channel 1  ... */
char wf[32]="sine";              /* waveform type */
unsigned int freq=440;           /* signal frequency */
unsigned int ratio=0;            /* used in pulse, sweep etc */
unsigned int Gain=100;           /* Amplification factor */
unsigned char *sbuf;             /* waveform sample buffer */
    /* channel 2  ... */
char wf2[32]="sine";             /* waveform type */
unsigned int freq2=440;          /* signal frequency */
unsigned int ratio2=0;           /* used in pulse, sweep etc */
unsigned int Gain2=100;          /* Amplification factor */
unsigned char *sbuf2;            /* waveform sample buffer */

unsigned int sbuf_size;          /* size of 1 sec, 1 chan, waveform buffer */

unsigned char *plbuf;            /* play buffer - mixed left and right bufs */
int plbuf_size;                  /* size of playback buffer */

char *sys;

void siggen_timer(int);

help(e)
int e;
{  fputs(VERSION,stderr);
   fputs("\nUsage: \n 1: siggen [flags] [waveform [freq [param]]]\n",stderr);
   fputs("      waveform is sine, square, triangle, sawtooth, noise, pulse\n",stderr);
   fputs("      for pulse param is Mark/Space ratio as a %.\n\n",stderr);
   fputs("Defaults: output to /dev/dsp, 22050 samples/sec, stereo, 16 bit\n",stderr);
   fputs("          samples if possible, else 8 bit and/or mono. \n\n",stderr);
   fputs("flags: -s samples    generate with samplerate of samples/sec\n",stderr);
   fputs(" -8/-16 or -b 8|16   force 8 bit or 16 bit mode.\n",stderr);
   fputs("       -1,-2         force mono or stereo (1 or 2 channels)\n",stderr);
   fputs("       -A n          scale samples by n/100, def. n is 100\n",stderr);
   return(e);
}

/* main
 *
 */
 
main(argc,argv)
int argc;
char **argv;
{
   unsigned int v[MAXPRM],maxv,i,j,k,l,m,n,N;
   FILE *f;
   char *p,bf[130];
   int fd;
   int c;
   unsigned int t;
   
   if ((p=strrchr(sys=*argv++,'/'))!=NULL) sys=++p;
   argc--;
   
   samplerate=SAMPLERATE; afmt=AFMT_QUERY;
   stereo=-1;
   vflg=dflg=0;
   
   while (argc && **argv=='-') {          /* all flags and options must come */
      n=strlen(p=(*argv++)+1); argc--;    /* before paramters                */
      if (chkword(1,"samplerate")) {
	 if (argc && isdigit(**argv)) { samplerate=atoi(*argv++); argc--; }
      }
      else if (chkword(2,"16")) { afmt=AFMT_S16_LE; }
      else if (chkword(1,"bits")) {
	 i=0;
	 if (argc) { 
	    i=atoi(*argv++); argc--;
	 }
	 if (i==8) afmt=AFMT_U8;
	 else if (i==16) afmt=AFMT_S16_LE;
	 else exit(err_rpt(EINVAL,"must be '-b 8' or '-b 16'."));
      }
      else if (chkword(1,"A")) {
	 if (argc && isdigit(**argv)) { 
	    Gain=atoi(*argv++); argc--;
	 }
      }
      else {                              /* check for single char. flags    */
	 for (; *p; p++) {
	    if (*p=='h') exit(help(EFAIL));
	    else if (*p=='1') stereo=0;
	    else if (*p=='2') stereo=1;
	    else if (*p=='8') afmt=AFMT_U8;
	    else if (*p=='d') dflg=1;
	    else if (*p=='v') vflg=1;
	    else {
	       *bf='-'; *(bf+1)=*p; *(bf+2)=0;
	       exit(help(err_rpt(EINVAL,bf)));
	    }
	 }
      }
   }
   
   if (argc) {
      strncpy(wf,*argv++,32); wf[31]=0; argc--;    /* waveform type */
      strcpy(wf2,wf);
      if (argc) {
	 freq2=freq=atoi(*argv++); argc--;
	 if (argc) { ratio=atoi(*argv++); argc--; }
	 if (argc) exit(help(err_rpt(EINVAL,"Too many parameters")));
      }
   }
   
      
   /* if no format specified then try 16 bit */
   i=afmt;
   n=stereo;
   if ((fd=DACopen(DAC_FILE,"w",&samplerate,&i,&n))<0) {
      exit(err_rpt(errno,"Opening DSP for output."));
   }
   if (((afmt!=AFMT_QUERY) && (i!=afmt)) ||
       ((stereo!=-1) && (n!=stereo))) {
      exit(err_rpt(EINVAL,"Sound card doesn't support format requested."));
   }
   afmt=i; stereo=n;
       
   if (freq > samplerate/2) {
      fputd(freq,stderr); 
      fputs(" Hz is more than half the sampling rate\n",stderr);
      exit(err_rpt(EINVAL,"Frequency setting too great"));
   }

   sbuf_size=samplerate;                  /* buf sizes if 8 bit samples */
   if (afmt==AFMT_S16_LE) sbuf_size<<=1;  /* double size if 16 bit      */
   else if (afmt!=AFMT_U8) {
      exit(err_rpt(EINVAL,"Only unsigned 8 and signed 16 bit, supported."));
   }
	
   sbuf=(unsigned char *)malloc(sbuf_size);
   sbuf2=(unsigned char *)malloc(sbuf_size);
   plbuf_size=sbuf_size<<((stereo)?1:0);
   plbuf=(unsigned char *)malloc(plbuf_size);
   if ((sbuf==NULL) || (sbuf2==NULL) || (plbuf==NULL)) {
      exit(err_rpt(ENOMEM,"Attempting to get buffers"));
   }
   if (vflg) {
      fputs((stereo)?"Stereo, ":"Mono, ",stdout);
      fputs((afmt==AFMT_S16_LE)?"16":"8",stdout);
      fputs(" bit samples being generated.\n",stdout);
      fputs("Samples scaled by a factor of ",stdout);
      fputd(Gain,stdout); fputs("/100.\n",stdout);
      fputd(sbuf_size,stdout); 
      fputs(" byte buffer allocated per channel.\n",stdout);
   }

/*   start_timer();  */
   for (c=0;;) {
      if (WinGen(c)) break;
      c=playloop(fd,plbuf,plbuf_size);
   }
   
   free(sbuf);
   if (stereo) {
      free(sbuf2); free(plbuf);
   }
   
   close(fd);
   exit(0);
}  

#ifdef INC_TIMER
/* siggen_timer(int i)  periodic timer - executed every 50msec or so.
 */

void siggen_timer(i)
int i;
{
   int t;
   static int tim=0;

/* check if we can draw on screen and if so if we need to update time */

   if (WaitingKey) {
      t=time(NULL);
      if (tim!=t) {
         dotime(tim=t);
      }
   }
 
/* check if we need to output more to DSP ???? */

/* reset sig handler.... */
   signal(SIGALRM,siggen_timer);
}
     
/* start_timer()   set off a 50msec timer along with the routine to get
 *                 when the signal arrives.
 */

start_timer()
{
   static struct itimerval period,rem_period;
   
   period.it_interval.tv_sec=0;
   period.it_value.tv_sec=1;
   period.it_interval.tv_usec=period.it_value.tv_usec=50000;
   signal(SIGALRM,siggen_timer);
   setitimer(ITIMER_REAL,&period,&rem_period);
}
#endif   /* INC_TIMER */

/*
 *  playloop(fd,bf,bfn)   bf is a sample buffer with bfn samples.
 *      playloop plays this continuously to file descr fd
 */

playloop(fd,bf,bfn)
int fd;
unsigned char *bf;
int bfn;
{
   int i,c,bn,x;
   unsigned char *p;
   
   no_key_block();
   bn=bfn/20;
   for (;;) {
      for ( p=bf,i=0; (bfn-i)>=bn; i+=bn) { 
	 if (write(fd,p+i,bn) < 0) return(0);
	 /* check here if need to supend output or handle keypresses etc*/
	 if ((c=getch())==27) {
	    ioctl(fd,SNDCTL_DSP_RESET,0);  /* tell driver it will be pausing */
	    key_block();
	    return(c);
	 }
      }
      if ((i!=bfn) && (write(fd,p+i,bfn-i)<0)) return(0);
   }
}

is16bit(fd)
int fd;
{
   int fmts;
   
   if (ioctl(fd, SNDCTL_DSP_GETFMTS , &fmts) < 0) {
      return(0);
   }
   return(fmts&AFMT_S16_LE);
}
   
/*
 * DACopen(char *fnm, char *mode, int *samples, int *fmt, int *stereo)
 * 
 *   open dspfile for read "r" or write "w", and set samples per sec
 *    as sampling rate - note we get pointer for samples so we can 
 *    return the actual samplerate set. 
 *    If stereo mode is unspecified (-1) then set stereo if possible,
 *    if fmt is unspecified (AFMT_QUERY) then set 16 bit if possible
 *    otherwise we set mono and/or 8 bit.
 *    Actual settings are returned in fmt and stereo.
 *   return file descriptor or -1 on error.
 */

DACopen(fnm,mode,samples,fmt,stereo)
char *fnm;
char *mode;
int *samples,*fmt,*stereo;
{
   int fd;
   int m,i;
   
   if (*mode=='r') m=O_RDONLY;
   else if (*mode=='w') m=O_WRONLY;
   else {
      errno=EINVAL; return(-1);
   }
   if ((fd = open (fnm,m,0)) >= 0) {  /* params must be set in this order */
      if ((*fmt==AFMT_QUERY) && is16bit(fd)) *fmt=AFMT_S16_LE;
      if (ioctl(fd, SNDCTL_DSP_SETFMT, fmt)>=0) { 
	 if (*stereo==-1) {
	    *stereo=1;
	    if (ioctl(fd, SNDCTL_DSP_STEREO, stereo)<0) *stereo=0;
	 }
	 if (ioctl(fd, SNDCTL_DSP_STEREO, stereo)>=0) {
	    if (ioctl(fd, SNDCTL_DSP_SPEED, samples)>=0) { 
	       return(fd);
	    }
	 }
      }
   }
   return(-1);
}

