/* ncfio.c
 * Structured field functions using ncurses......
 * Jim Jackson  Nov 96
 */

/*
 * Copyright (C) 1997 Jim Jackson                    jj@scs.leeds.ac.uk
 *                    School of Computer Studies,
 *                    The University of Leeds,
 *                    Leeds, LS2 9JT, UK
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with this program - see the file COPYING; if not, write to 
 *  the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
 *  MA 02139, USA.
 */

/*
 * getfint(att,ip,w) input an integer on screen -  w max digits
 * putfint(att,n,w)  display integer n on screen -  w max digits
 * getfstr(att,s,w)  input a string on screen - w max characters
 * putfstr(att,s,w)  display a string on screen - w max characters
 * getfreal(att,rp,w,dp) input a real on screen - w max chars, dp dec places
 * putfreal(att,r,w,dp) display a real on screen - w max chars, dp dec places
 * getfopt(att,sp,ip,w) input one of options char *sp[] set *ip to selected
 * putfopt(att,sp,i,w) display char *sp[i] in field of width w
 * 
 * getfield(NCFp)    input field defined by NCField *NCFp
 * putfield(NCFp)    display field defined by NCField *NCFp
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <signal.h>
#include <ctype.h>
#include <time.h>
#include "ncfio.h"        /* local header file */

/* reverse_attr(a)  calculate the reverse attribute of attribute a
 *    if its not a color pair then simply toggle A_REVERSE on a
 *    if it is a color pair then make it COLOR_PAIRS-pair_number
 */

reverse_attr(a)
int a;
{
   int r;

   if (PAIR_NUMBER(a)) r=(a&(~A_COLOR))+(COLOR_PAIR(COLOR_PAIRS-PAIR_NUMBER(a)));
   else r=((a+A_REVERSE)&A_REVERSE)+(a&~A_REVERSE);
   return(r);
}

/* getfint(att,ip,w)   Set an input field of width w using att reversed.
 *           reset to unrev on exit. Input digits to make an integer
 *           until a non-graphical char is pressed. DEL or backspace erases
 *           least sig digit. ip points to where to store the integer.
 *      all printable characters (in range x21 to x7e) that are not digits
 *      are ignored. First non-printable character (not del, bs) causes
 *      exit from routine and is returned.
 */

getfint(att,ip,w)
int att,*ip,w;
{
   int c,y,x,ratt;
   int f[COLS];
   
   ratt=reverse_attr(att);
   getyx(stdscr,y,x);
   if (w>=COLS) w=COLS-1;
   putfint(ratt,*ip,w);
   move(y,x+w-1);
   refresh();
   for ( ; ; ) {
      c=getch();
      if (c=='\b' || c==KEY_BACKSPACE || c==127 || c==KEY_DC) {
	 *ip/=10;
      } else if (isdigit(c)) {
	 *ip=(*ip)*10+c-'0';
      } else if (c==KEY_UP || c=='+') {
         (*ip)++;
      } else if ((c==KEY_DOWN || c=='-') && *ip) {
         (*ip)--;
      } else break;
      mvputfint(y,x,ratt,*ip,w);
      move(y,x+w-1);
      refresh();
   }
   mvputfint(y,x,att,*ip,w);
   return(c);
}


/* putfint(att,n,w)   display integer n in a field
 *           of width w using attribute att.
 */

putfint(att,n,w)
int att,n,w;
{
   int c,i,j;
   int f[COLS];
   
   if (w>=COLS) w=COLS-1;
   for (c='0', i=w, j=n; i; i--) {
      f[i]=j%10+att+c; j/=10;
      if (!j) c=' ';
   }
   addchnstr(f+1,w);
}

/* getfstr(att,s,w)   Set an input field of width w using attribute att in rev
 *           reset to unrev on exit. Input chars to make a string
 *           until an unknown non-graphical char is pressed.
 *           DELETE deletes char over cursor,
 *           <- or ^H deletes char to left of cursor,
 *           Left and right arrows move cursor,
 *           HOME and END move cursor to start, end of string,
 *      first unknown non-graphical char causes exit from routine
 *      and is returned.
 */

getfstr(att,s,w)
int att;
char *s;
int w;
{
   int i,c,l,y,x,ratt;
   char f[COLS+1],*q,*p;
   
   ratt=reverse_attr(att);
   getyx(stdscr,y,x);
   if (w>=COLS-x) w=COLS-x;
   putfstr(ratt,s,w);
   refresh();
   for (q=s, i=0; i<w; i++) {
      if (*q) c=*q++; else c=' ';
      f[i]=c;
   }
   f[i]=0;
   for ( p=f ; ; ) {
      c=getch();
      if (c=='\b' || c==KEY_BACKSPACE || c==127) {
	 if (p>f) {
	    for ( q=(--p) ; *q=*(q+1); q++) {  }
	    *q=' ';
	 }
      } else if (isprint(c)) {
	 for ( q=p ; *q; q++) {
	    l=*q; *q=c; c=l;
	 }
	 if ((p-f)<w) p++;
      } else if (c==KEY_DC) {
	 for ( q=p ; *q=*(q+1); q++) {  }
	 *q=' ';
      } else if (c==KEY_LEFT) {
	 if (p>f) p--; 
      } else if (c==KEY_RIGHT) {
	 if ((p-f)<w) p++;
      } else if (c==KEY_HOME) {
	 p=f;
      } else if (c==KEY_END) {
	 p=f+w-1;
      } else break;
      mvputfstr(y,x,ratt,f,w);
      move(y,x+p-f);
      refresh();
   }
   mvputfstr(y,x,att,f,w);
   strcpy(s,f);
   return(c);
}

/* putfstr(att,s,w)   display string s in a field
 *           of width w using attribute att.
 */

putfstr(att,s,w)
int att;
char *s;
int w;
{
   int c,i;
   int f[COLS],*p;
   
   if (w>=COLS) w=COLS-1;
   for ( p=f, i=0 ; i<w ; i++) {
      if (*s) c=*s++; else c=' ';
      *p++=att+c;
   }
   *p=0;
   addchnstr(f,w);
}

/* getfopt(att,sp,ip,w)   Set an input field of width w using attr. att in rev
 *           reset to unrev on exit. Select on of char *sp[], set *ip to index
 *           of selected option.
 *           Left and right arrows move back and forward an option.
 *           SPACE moves forward thru' options
 *           HOME and END move to first and last option
 *      first unknown char causes exit from routine
 *      and is returned.
 */

getfopt(att,sp,ip,w)
int att;
char **sp;
int *ip,w;
{
   int i,c,l,y,x,ratt;
   char f[COLS+1],*q,*p;
   
   ratt=reverse_attr(att);
   getyx(stdscr,y,x);
   if (w>=COLS-x) w=COLS-x;
   putfstr(ratt,sp[*ip],w);
   refresh();

   for ( ; ; ) {
      c=getch();
      if (c=='\b' || c==KEY_BACKSPACE || c==127 || c==KEY_UP) {
	 if (*ip) (*ip)--;
	 else c=KEY_END;
      }  /* no else here on purpose */
      if (c==KEY_DOWN || c==' ') {
	 (*ip)++;
	 if (sp[*ip]==NULL) *ip=0;
      } else if (c==KEY_HOME) {
	 *ip=0;
      } else if (c==KEY_END) {
	 for ( ; sp[*ip+1]!=NULL; (*ip)++) { }
      } else break;
      putfstr(ratt,sp[*ip],w);
      refresh();
   }
   putfstr(att,sp[*ip],w);
   return(c);
}

/* putfield(fp)   display field *fp 
 */

putfield(fp)
struct NCField *fp;
{
   int t;
   
   t=(fp->type)&NCF_type;
   if (t==NCF_integer) mvputfint(fp->row,fp->col,fp->attr,*(fp->val.i),fp->W);
   else if (t==NCF_string) mvputfstr(fp->row,fp->col,fp->attr,fp->val.s,fp->W);
   else if (t==NCF_option) 
     mvputfstr(fp->row,fp->col,fp->attr,(fp->val.sp)[fp->adj],fp->W);
   else return(1);
   return(0);
}


/* getfield(fp)   update value of field *fp 
 */

getfield(fp)
struct NCField *fp;
{
   int t,c;
   
   if ((fp->type)&NCF_fixed) return(-1);
   t=(fp->type)&NCF_type;
   if (t==NCF_integer) c=mvgetfint(fp->row,fp->col,fp->attr,fp->val.i,fp->W);
   else if (t==NCF_string) 
     c=mvgetfstr(fp->row,fp->col,fp->attr,fp->val.s,fp->W);
   else if (t==NCF_option) 
     c=mvgetfopt(fp->row,fp->col,fp->attr,fp->val.sp,&(fp->adj),fp->W);
   else return(-1);
   return(c);
}

 
