#!/bin/sh
#
# NAME
#   make-bindist.sh -- make a Rosegarden binary distribution
#
# SYNOPSIS
#   make-bindist.sh [-h] [-f makefile] [version-id]
#
# DESCRIPTION
#   make-bindist.sh is a tool to assist in creating binary
#   distributions of Rosegarden and preparing them for upload to an
#   ftp site.  The Rosegarden developers use this script to prepare
#   their original binary distributions; if you have successfully
#   built the applications on a system not supported by the original
#   development team, you might consider packaging up a distribution
#   and sending it to the Rosegarden ftp site yourself.  Contact
#   cannam@zands.demon.co.uk for details of where and how to upload.
#
#   make-bindist.sh builds and packs up a binary distribution of
#   the Rosegarden program suite, named according to the architecture
#   of the host machine, using the current Makefile settings.
#
#   This involves running "make clean" and "make all", creating a
#   temporary directory (under /tmp) and moving the executables, help
#   files and test files from their places in the existing source
#   tree into the correct places in the new directory tree.  Files in
#   the current directory called "README", "ANNOUNCEMENT",
#   "README.<system>" and "CAVEATS.<system>" will also be included,
#   where <system> is whatever you get by running "uname -s".  The new
#   directory is then tar'd and gzipped up ready for uploading.
#
#   You should only run make-bindist.sh from the top-level directory
#   of a Rosegarden source distribution.  If you have changed any
#   source files or help, test or other support files from the
#   original distributions for the version of Rosegarden you're
#   building, you should think very carefully before you do anything.
#
#   You must have "gzip" in your PATH in order to run this script
#   successfully.
#
#
# ARGUMENTS
#   version-id    Optional ID string for the version of Rosegarden
#                 you're building.  In this release of make-bindist.sh,
#                 the ID string will default to "2.0.1".  Change
#                 this if and only if you have changed the source code.
#
# OPTIONS
#   -h            Help.  Print this length and rambling explanation.
#
#   -f makefile   Name of the file to tell make(1) to use to build the
#                 system.  The default is whatever make's default is.
#                 Instead of changing this, why not create a symbolic
#                 link from your system-specific makefile to "Makefile"
#                 in the same way as the original source distribution?
#
# CAVEATS
#   Unless you are prepared to put your name to a new binary
#   distribution of Rosegarden, you probably have no use for this
#   script.
#
#   make-bindist.sh constructs the distribution file name from the
#   results of calling "uname" with various options.  This might
#   produce strange results on some systems, though mostly it's
#   probably not important.
#
#   This script is a quick hack, won't work well in anything other
#   than rather obvious situations, and contains no error checking.
#   Improvements welcome (and we'll probably do some more work on
#   it soon ourselves).
#
# AUTHOR
#   Chris Cannam, July 1996

rosehome=`pwd`
myfilename=$0
makearg=
product=rosegarden
versionid="2.0.1"

explain()
{
  cat $myfilename | sed '/^$/q' | tail +2 | sed 's/^#//' | \
   sed "s/make-bindist.sh/`basename $myfilename`/"
  exit 0
}

complain()
{
  echo "For help, run "$myfilename" -h"
  exit 2
}

while getopts fh c
do
  case $c in
    f) makearg=" -f "$OPTARG ;;
    h) explain ;;
    \?) complain ;;
  esac
done
found=$OPTIND
shift `expr $found - 1`

if [ "$1" ] ; then versionid=$1 ; shift ; fi
if [ "$1" ] ; then complain ; fi

if uname -p 1>/dev/null 2>/dev/null
then
  distname="$product"-"$versionid"-`uname -p`-`uname -s``uname -r`
else
  distname="$product"-"$versionid"-`uname -s`-`uname -r`
fi

tmpdir=./"$product"_`whoami`_`date +%H%M%S`

echo
echo Distribution will be called "$distname".tgz
echo
sleep 2

echo Building from sources...
echo

make $makearg all

if [ -d $tmpdir ]; then
  echo Temporary directory $tmpdir already exists -- stopping
  exit 1
fi

echo
echo Stripping symbol information
echo
( set -x
strip bin/rosegarden bin/editor bin/sequencer
)

echo
echo Making directory tree in temporary directory "$tmpdir"
echo
( set -x
mkdir $tmpdir
)
tmpdir=$tmpdir/$distname
( set -x
mkdir $tmpdir
mkdir $tmpdir/bin
mkdir $tmpdir/lib
mkdir $tmpdir/lib/music
mkdir $tmpdir/lib/rosegarden
)

echo
echo Copying binary files
echo
( set -x
cp bin/rosegarden $tmpdir/bin
cp bin/editor bin/sequencer $tmpdir/lib/rosegarden
chmod +x $tmpdir/bin/rosegarden $tmpdir/lib/rosegarden/editor \
 $tmpdir/lib/rosegarden/sequencer
)

echo
echo Copying help files, synth initialisation file, example files and resources
echo
( set -x
cp common/help/rosehelp.info common/help/rosehelp.hnx \
 common/synth-patches/std.sb \
 $tmpdir/lib/rosegarden
cp common/music/glazunov.rose common/music/sonataC.rose \
 $tmpdir/lib/music
cp Rosegarden $tmpdir
)

echo
echo Copying ANNOUNCEMENT, README and CAVEATS files
echo
readmes=
if [ -f ANNOUNCEMENT ] ; then readmes="$readmes ANNOUNCEMENT" ; fi
if [ -f README ] ; then readmes="$readmes README" ; fi
if [ -f README.`uname -s` ] ; then readmes="$readmes README.`uname -s`" ; fi
if [ -f CAVEATS ] ; then readmes="$readmes CAVEATS" ; fi
if [ -f CAVEATS.`uname -s` ] ; then readmes="$readmes CAVEATS.`uname -s`" ; fi
if [ t"$readmes" != "t" ]
then
  ( set -x
  cp $readmes $tmpdir
  )
fi

echo
echo Packing up distribution
echo
cd $tmpdir
cd ..
tar cvf - $distname | gzip -c > ../$distname.tgz

echo
echo Destroying the evidence
echo
cd ..
rm -rf $tmpdir
rmdir `dirname $tmpdir`

echo
echo Done
echo
