#include "stdio.h"
#ifndef mips
#include "stdlib.h"
#endif
#include "xlisp.h"
#include "sound.h"

#include "falloc.h"
#include "osc.h"

void osc_free();


typedef struct osc_susp_struct {
    snd_susp_node susp;
    long terminate_cnt;

    double table_len;
    double ph_incr;
    sample_type *table_ptr;
    double phase;
} osc_susp_node, *osc_susp_type;


void osc__fetch(register osc_susp_type susp, snd_list_type snd_list)
{
    int cnt = 0; /* how many samples computed */
    int togo;
    int n;
    sample_block_type out;
    register sample_block_values_type out_ptr;

    register sample_block_values_type out_ptr_reg;

    register double table_len_reg;
    register double ph_incr_reg;
    register sample_type * table_ptr_reg;
    register double phase_reg;
    falloc_sample_block(out, "osc__fetch");
    out_ptr = out->samples;
    snd_list->block = out;

    while (cnt < max_sample_block_len) { /* outer loop */
	/* first compute how many samples to generate in inner loop: */
	/* don't overflow the output sample block: */
	togo = max_sample_block_len - cnt;

	/* don't run past terminate time */
	if (susp->terminate_cnt != UNKNOWN &&
	    susp->terminate_cnt <= susp->susp.current + cnt + togo) {
	    togo = susp->terminate_cnt - (susp->susp.current + cnt);
	    if (togo == 0) break;
	}

	n = togo;
	table_len_reg = susp->table_len;
	ph_incr_reg = susp->ph_incr;
	table_ptr_reg = susp->table_ptr;
	phase_reg = susp->phase;
	out_ptr_reg = out_ptr;
	if (n) do { /* the inner sample computation loop */

	    long table_index = phase_reg;
	    double x1 = table_ptr_reg[table_index];
	    *out_ptr_reg++ = x1 + (phase_reg - table_index) * 
			  (table_ptr_reg[table_index + 1] - x1);
	    phase_reg += ph_incr_reg;
	    while (phase_reg > table_len_reg) phase_reg -= table_len_reg;
;
	} while (--n); /* inner loop */

	susp->phase = phase_reg;
	out_ptr += togo;
	cnt += togo;
    } /* outer loop */

    /* test for termination */
    if (togo == 0 && cnt == 0) {
	snd_list_terminate(snd_list);
    } else {
	snd_list->block_len = cnt;
	susp->susp.current += cnt;
    }
} /* osc__fetch */


void osc_free(osc_susp_type susp)
{
    free(susp->table_ptr);
    ffree_generic(susp, sizeof(osc_susp_node), "osc_free");
}


void osc_print_tree(osc_susp_type susp, int n)
{
}


sound_type snd_make_osc(sound_type s, double step, rate_type sr, double hz, time_type t0, time_type d, double phase)
{
    register osc_susp_type susp;
    /* sr specified as input parameter */
    /* t0 specified as input parameter */
    sample_type scale_factor = 1.0;
    long togo;
    long lsc;

    falloc_generic(susp, osc_susp_node, "snd_make_osc");
    susp->table_len = 0;
    susp->ph_incr = 0;
    susp->table_ptr = sound_to_table(s, &(susp->table_len));
    susp->phase = compute_phase(phase, step, susp->table_len,
	s->sr, sr, hz, &susp->ph_incr);
    susp->susp.fetch = osc__fetch;

    susp->terminate_cnt = (d) * sr + 0.5;
    /* initialize susp state */
    susp->susp.free = osc_free;
    susp->susp.sr = sr;
    susp->susp.t0 = t0;
    susp->susp.mark = NULL;
    susp->susp.print_tree = osc_print_tree;
    susp->susp.name = "osc";
    susp->susp.log_stop_cnt = UNKNOWN;
    susp->susp.current = 0;
    return sound_create((snd_susp_type)susp, t0, sr, scale_factor);
}


sound_type snd_osc(sound_type s, double step, rate_type sr, double hz, time_type t0, time_type d, double phase)
{
    sound_type s_copy = sound_copy(s);
    return snd_make_osc(s_copy, step, sr, hz, t0, d, phase);
}
