// rescaler.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include "rescaler.h"
#include "data.h"
#include "processfun.h"
#include "request.h"
#include "requester.h"

class RescaleRequester : public TitledRequester {
	friend Rescaler;
protected:
	RescaleRequester(Rescaler* r)
		: TitledRequester("Rescale Values to Fit New Range:"), client(r) {}
	redefined void configureRequest(Request *);
private:
	Rescaler* client;
};

void
RescaleRequester::configureRequest(Request* request) {
	request->appendValue("New Maximum Value:", &client->newRange.max_());
	request->appendValue("New Minimum Value:", &client->newRange.min_());
	if(client->target()->limits().includesZero())
		request->appendChoice(
			"Check this to scale symmetrically around zero",
			"|Preserve zero crossings|",
			&client->preserveZero,
			/* exclusive states = */ false
		);
}


//********

Rescaler::Rescaler(Data* d)
	: DataModifier(d), newRange(0.0, 1.0), preserveZero(false) {}
		
Rescaler::Rescaler(Data* d, const Range& newrange)
		: DataModifier(d), newRange(newrange), preserveZero(false) {
	initialize();
}

Requester *
Rescaler::createRequester() {
	return new RescaleRequester(this);
}

int
Rescaler::doApply(Data *data) {
	oldRange = data->limits();
	double factor = preserveZero ?
		newRange.absoluteMax() / oldRange.absoluteMax()
		: newRange.spread() / oldRange.spread();
	data->deferUpdate(true);
	data->scaleSelf(factor);	// first scale min-to-max range
	if(!preserveZero) {
		Range currentRange = data->limits();
		double offset = newRange.min() - currentRange.min();
		Offset o(data, offset);
		o.apply();		// then adjust offset to match new min
	}
	data->deferUpdate(false);
	return true;
}

void
Rescaler::initialize() {
	newRange.check();
	Super::initialize();
}

Modifier *
Rescaler::createUnDo() {
	return new Rescaler(target(), oldRange);
}
