// graphplot.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// GraphPlot and its subclasses encapsulate the persistant state of the
// actual displayed graph line(s) for any single channel of data.  They are
// created and contained by the Graph subclass instances, and are intimately
// linked to them due to the necessity of sharing graphic state, hence the 
// 'graph' member in the base class.

#ifndef GRAPHPLOT_H
#ifdef __GNUG__
#pragma interface
#endif
#define GRAPHPLOT_H

#include <InterViews/resource.h>

class Canvas;
class Painter;
class Perspective;
class Margin;
class Graph;
class Data;

class GraphPlot : public Resource {
public:
	GraphPlot(Graph* gr, int points) : graph(gr), maxPoints(points) {}
	virtual ~GraphPlot() {}
	virtual void load(Data *, EnvelopeType=PeakToPeakAmplitude) = 0;
	virtual void plot() = 0;
	virtual void setPlotLength(int) = 0;
	virtual void draw(Canvas *, Painter *) = 0;
	virtual boolean contains(Coord l, Coord b, Coord r, Coord t) = 0;
protected:
	Graph* graph;			// reference to associated graph
	int maxPoints;
};

class SimplePlot : public GraphPlot {
	enum Traces { MaxTraces = 2 };
public:
	SimplePlot(Graph *gr, int points);
	virtual ~SimplePlot();
	redefined void load(Data *d, EnvelopeType type);
	redefined void plot();
	redefined void setPlotLength(int newlen);
	redefined boolean contains(Coord l, Coord b, Coord r, Coord t);
protected:
	int currentTraces() { return nTraces; }
	Data *plotdata;				// internal plot storage
	Coord *xlocs[MaxTraces], *ylocs[MaxTraces];
	Coord maxVertLoc, minVertLoc;
	int nPoints, maxTraces, nTraces;
};

// LinePlot draws a graph line.  If more than one value is represented by a
// single point, two lines are drawn, one for the max's and one for the min's

class LinePlot : public SimplePlot {
public:
	LinePlot(Graph *gr, int points) : SimplePlot(gr, points) {}
	virtual ~LinePlot() {}
	redefined void draw(Canvas *, Painter *);
};

// BarPlot draws a solid bar-graph style representation of the data values

class BarPlot : public SimplePlot {
public:
	BarPlot(Graph *gr, int points) : SimplePlot(gr, points + 1) {}
	virtual ~BarPlot() {}
	redefined void plot();
	redefined void draw(Canvas *, Painter *);
};

// SolidPlot draws a "mountain range" style, filled graph for use in cascade-
// style multi-graphs

class SolidPlot : public SimplePlot {
public:
	SolidPlot(Graph *gr, int points) : SimplePlot(gr, points + 2) {}
	virtual ~SolidPlot() {}
	redefined void plot();
	redefined void draw(Canvas *, Painter *);
};

#endif
