// datarep.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// The various derived classes of DataRep are identical except for the 
// underlying raw type of the samples (frames) which they contain.  These are
// made into separate classes to optimize the setting and retrieving of values
// (all of which are cast from and to double).  Instances of DataRep subclasses
// never exist independently;  they are always a member of a Data class or 
// subclass.  DataReps have two private internal classes:  the Array class,
// a wrapper for the actual pointer to the raw char* data pointer, and Handle,
// an auxillary class used to keep track of the current sample location during
// iteration.

#ifndef DATAREP_H
#ifdef __GNUG__
#pragma interface
#endif
#define DATAREP_H

#include <InterViews/resource.h>
#include "localdefs.h"

class DataFile;
class Header;
class Range;
class SimpleFunction;
class OutputFunction;
class InputOutputFunction;
template <class Rep> class Cloned;
class SignedCharDataRep;
class UnsignedCharDataRep;
class MuLawDataRep;
class ShortDataRep;
class IntDataRep;
class FloatDataRep;
class DoubleDataRep;

class DataRep : public Resource {
protected:
	class Array : public Resource {
	public:
		Array(int size, int len);	// size is element size in bytes
		~Array();
		operator char * () const { return (char *) memPtr; }
		operator short * () const { return (short *) memPtr; }
		operator int * () const { return (int *) memPtr; }
		operator float * () const { return (float *) memPtr; }
		operator double * () const { return (double *) memPtr; }
		addr operator [] (int offset) const { return memPtr + offset; }
		boolean operator == (const Array &rhs) const {
			return rhs.memPtr == memPtr;
		}
		boolean operator != (const Array &rhs) const {
			return rhs.memPtr != memPtr;
		}
		unsigned changeSize(int size, int newlen);
		unsigned size() { return bytes; }
	private:
		addr memPtr;
		unsigned bytes;
	};
	class Handle {
	public:
		Handle(void *pp) : p(pp) {}
	
		operator char *() const {return (char *)p;}
		operator short *() const {return (short *)p;}
		operator int *() const {return (int *)p;}
		operator float *() const {return (float *)p;}
		operator double *() const {return (double *)p;}
		int operator -(Handle rhs) {return (char *)p - (char *)rhs.p;}
		int operator -(addr rhs) {return (char *)p - (char *)rhs;}
	private:
		void *p;
	};
public:
	static DataRep* create(DataFile *, DataType, int chans);
	static DataRep* create(DataType, int length, int chans);
	virtual DataRep* clone(const Range &);
	virtual DataRep* clone(const Range&, const Range&) = 0;
	virtual DataRep* newRep();
	virtual DataRep* newLengthRep(int length) = 0;
	virtual ~DataRep();

	int realLength() const { return len; }	    // not to be overridden
	int realChannels() const { return nchans; } // not to be overridden
	virtual int dataSize() const { return data_size; };
	virtual int length() const { return len; }
	virtual int channels() const { return nchans; }
	virtual DataType dataType() const = 0;
	virtual int sizeInBytes() const;
	virtual addr getAddressOfContiguousData() const { return *a; }
	virtual int checkLength(int newend);
	virtual Handle getHandle(int pos=0, int chan=0) const = 0;

	// the many and various sample or frame iteration methods
	virtual Handle setAndShift(Handle h, double val, int incr=1) = 0;
	virtual Handle getAndShift(Handle h, double *val, int incr=1) const = 0;
	virtual Handle setAndAdvance(Handle h, double val);
	virtual Handle getAndAdvance(Handle h, double *val) const;
	virtual double get(int pos, int chan=0) const = 0;
	virtual void set(double val, int pos, int chan) = 0;
	virtual void getFrame(DataRep*, int frameno, int offset, int incr=1) const;
	virtual void getFrame(float*, int arraySize, int frameno) const;
	virtual void getFrame(double*, int arraySize, int frameno) const;
	virtual void setFrame(const DataRep*, int frameno, int incr=1);
	virtual void setFrame(float*, int arraySize, int frameno);
	virtual void setFrame(double*, int arraySize, int frameno);
	int getArray(int*, int size, int channel) const;
	int getArray(float*, int size, int channel) const;
	int getArray(double*, int size, int channel) const;
	int getArray(double*, int size, int channel, int offset) const;
	int setFromArray(float*, int size, int channel, int offset);
	int setFromArray(double*, int size, int channel, int offset);

	int getEnvelope(DataRep *evp, int chan, EnvelopeType et, double ref) const;
	virtual void copyFrom(const DataRep*);
	virtual void copyRescaledFrom(const DataRep*, double, boolean dither=true);
	virtual void add(const DataRep *src);
	virtual int spliceIn(const Range &);
	virtual int spliceOut(const Range &);
	virtual void reverse() = 0;
	virtual void swapBytes() {}

	int applyFunction(SimpleFunction& fun, int start=0, int at_chan=0);
	RunStatus getSamplesFrom(OutputFunction &fun, int start=0, int at_chan=0);
	int giveSamplesTo(InputOutputFunction& fun) const;

	virtual int shiftOver(int start, int shift);
	virtual double valueLimit() const = 0;
	double rmsAmplitude(int, int, int, int) const;
	virtual void valueRange(Range &, int, int, int, int, boolean real=false) const =0;
	virtual int zeroValue() const { return 0; }
	virtual int read(DataFile *f);
	virtual int write(DataFile *f) const;
	boolean operator == (const DataRep &rhs) const { return *rhs.a == *a; }
	boolean operator != (const DataRep &rhs) const { return *rhs.a != *a; }
protected:	
	DataRep(int size=1, int length=0, int chans=1); 
	DataRep(const DataRep* rep);			// special ctor for cloning only
	Array& array() const { return *a; }
	addr arrayOffset() const { return (*a)[offset]; }
	void zero(addr, int bytes);
	virtual int changeLength(int newlen);
	virtual void changeNChans(int newchans);
protected:	
	enum Length { MinLength = 8 };
	short data_size;
	int len;
	int nchans;
	int offset;
	typedef DataRep* (*Constructor) (int, int);
	static Constructor ctor_table[];
private:
	Array *a;	// Array instance contains actual ptr to data
	double get(double pos, int chan=0) const;
	void valueRange(Range &) const;
	int offsetPointer(int o) { int old = offset; offset += o; return old; }	
	void setLength(int newlen) { len = newlen; }
	int frameOffset();
	DataRep(const DataRep&);
	friend class Data;
	friend class Cloned<SignedCharDataRep>;
	friend class Cloned<UnsignedCharDataRep>;
	friend class Cloned<MuLawDataRep>;
	friend class Cloned<ShortDataRep>;
	friend class Cloned<IntDataRep>;
	friend class Cloned<FloatDataRep>;
	friend class Cloned<DoubleDataRep>;
};

inline DataRep::Handle
DataRep::setAndAdvance(DataRep::Handle h, double val) {
	return setAndShift(h, val, nchans);
}

inline DataRep::Handle
DataRep::getAndAdvance(DataRep::Handle h, double *val) const {
	return getAndShift(h, val, nchans);
}

//********

class CharDataRep : public DataRep {
	typedef DataRep Super;
public:
	CharDataRep(int length, int chans=1, boolean doZero=true);
	redefined void reverse();
	redefined Handle getHandle(int pos=0, int chan=0) const;
	redefined double valueLimit() const { return 127; }
protected:
	CharDataRep(const CharDataRep* r) : DataRep(r) {}
	redefined int changeLength(int newlen);		// redefined here only
private:
	CharDataRep(const CharDataRep&);
};

inline DataRep::Handle
CharDataRep::getHandle(int pos, int chan) const {
	return (char *) arrayOffset() + chan + (pos*nchans); 
}

//********

class SignedCharDataRep : public CharDataRep {
	typedef CharDataRep Super;
public:
	SignedCharDataRep(int length, int chans=1);
	static DataRep* new_Rep(int len, int chans);
	// "virtual" constructors
	redefined DataRep *newLengthRep(int newlen);
	redefined DataRep *clone(const Range&, const Range&);

	redefined DataType dataType() const { return SignedCharData; }
	redefined void set(double val, int pos, int chan);
	redefined double get(int pos, int chan=0) const;
	redefined Handle setAndShift(Handle h, double val, int incr);
	redefined Handle getAndShift(Handle h, double *val, int incr) const;
	redefined void valueRange(Range&, int, int, int, int, boolean=false) const;
protected:
	SignedCharDataRep(const SignedCharDataRep* r) : CharDataRep(r) {}
private:
	SignedCharDataRep(const SignedCharDataRep&);
};

//********

class UnsignedCharDataRep : public CharDataRep {
	typedef CharDataRep Super;
public:
	UnsignedCharDataRep(int length, int chans=1);
	static DataRep* new_Rep(int len, int chans);
	// "virtual" constructors
	redefined DataRep *newLengthRep(int newlen);
	redefined DataRep *clone(const Range&, const Range&);

	redefined DataType dataType() const { return UnsignedCharData; }
	redefined void set(double val, int pos, int chan);
	redefined double get(int pos, int chan=0) const;
	redefined Handle setAndShift(Handle h, double val, int incr);
	redefined Handle getAndShift(Handle h, double *val, int incr) const;
	redefined void valueRange(Range&, int, int, int, int, boolean=false) const;
protected:
	UnsignedCharDataRep(const UnsignedCharDataRep* r) : CharDataRep(r) {}
	redefined int zeroValue() const;
private:
	UnsignedCharDataRep(const UnsignedCharDataRep&);
};

//********

class MuLawDataRep : public CharDataRep {
	typedef CharDataRep Super;
public:
	MuLawDataRep(int length, int chans=1);
	static DataRep* new_Rep(int len, int chans);
	// "virtual" constructors
	redefined DataRep *newLengthRep(int newlen);
	redefined DataRep *clone(const Range&, const Range&);
	redefined DataType dataType() const { return MuLawData; }
	redefined void set(double val, int pos, int chan);
	redefined double get(int pos, int chan=0) const;
	redefined Handle setAndShift(Handle h, double val, int incr);
	redefined Handle getAndShift(Handle h, double *val, int incr) const;
	redefined void valueRange(Range&, int, int, int, int, boolean=false) const;
	redefined double valueLimit() const { return 32767; }
protected:
	MuLawDataRep(const MuLawDataRep* rep) : CharDataRep(rep) {}
	redefined int zeroValue() const;
private:
	MuLawDataRep(const MuLawDataRep&);
};

//********

class ShortDataRep : public DataRep {
	typedef DataRep Super;
public:
	ShortDataRep(int length, int chans=1)
		: DataRep(sizeof(short), length, chans) {}
	static DataRep* new_Rep(int len, int chans);
	redefined DataRep *newLengthRep(int newlen);
	redefined DataRep *clone(const Range&, const Range&);

	redefined DataType dataType() const { return ShortData; }
	redefined Handle getHandle(int pos=0, int chan=0) const;
	redefined Handle setAndShift(Handle h, double val, int incr);
	redefined Handle getAndShift(Handle h, double *val, int incr) const;
	redefined void valueRange(Range&, int, int, int, int, boolean=false) const;
	redefined double valueLimit() const { return 32767; }
	redefined void set(double val, int pos, int chan);
	redefined double get(int pos, int chan=0) const;
	redefined void reverse();
	redefined void swapBytes();
protected:
	ShortDataRep(const ShortDataRep* rep) : DataRep(rep) {}
private:
	ShortDataRep(const ShortDataRep&);
};

inline DataRep::Handle 
ShortDataRep::getHandle(int pos, int chan) const {
	return (short *) arrayOffset() + chan + (pos*nchans); 
}

//********

class IntDataRep : public DataRep {
	typedef DataRep Super;
public:
	IntDataRep(int length, int chans=1)
		: DataRep(sizeof(int), length, chans) {}
	static DataRep* new_Rep(int len, int chans);
	redefined DataRep *newLengthRep(int newlen);
	redefined DataRep *clone(const Range&, const Range&);

	redefined DataType dataType() const { return IntData; }
	redefined Handle getHandle(int pos=0, int chan=0) const;
	redefined Handle setAndShift(Handle h, double val, int incr);
	redefined Handle getAndShift(Handle h, double *val, int incr) const;
	redefined void valueRange(Range&, int, int, int, int, boolean=false) const;
	redefined double valueLimit() const { return 2147483647; }
	redefined void set(double val, int pos, int chan);
	redefined double get(int pos, int chan=0) const;
	redefined void reverse();
	redefined void swapBytes();
protected:
	IntDataRep(const IntDataRep* rep) : DataRep(rep) {}
private:
	IntDataRep(const IntDataRep&);
};

inline DataRep::Handle 
IntDataRep::getHandle(int pos, int chan) const {
	return (int *) arrayOffset() + chan + (pos*nchans); 
}

//********

class FloatDataRep : public DataRep {
	typedef DataRep Super;
public:
	FloatDataRep(int length, int chans=1)
		: DataRep(sizeof(float), length, chans) {}
	static DataRep *new_Rep(int len, int chans);
	redefined DataRep *newLengthRep(int newlen);
	redefined DataRep *clone(const Range&, const Range&);
	
	redefined DataType dataType() const { return FloatData; }
	redefined Handle getHandle(int pos=0, int chan=0) const;
	redefined Handle setAndShift(Handle h, double val, int incr);
	redefined Handle getAndShift(Handle h, double *val, int incr) const;
	redefined void set(double val, int pos, int chan);
	redefined double get(int pos, int chan=0) const;
	redefined void valueRange(Range&, int, int, int, int, boolean=false) const;
	redefined double valueLimit() const { return 1.0; }
	redefined void reverse();
	redefined void swapBytes();
protected:
	FloatDataRep(const FloatDataRep* rep) : DataRep(rep) {}
private:
	FloatDataRep(const FloatDataRep&);
};

inline DataRep::Handle 
FloatDataRep::getHandle(int pos, int chan) const {
	return (float *) arrayOffset() + chan + (pos*nchans); 
}

//********

class DoubleDataRep : public DataRep {
	typedef DataRep Super;
public:
	DoubleDataRep(int length, int chans=1)
		: DataRep(sizeof(double), length, chans) {}
	static DataRep* new_Rep(int len, int chans);
	redefined DataRep *newLengthRep(int newlen);
	redefined DataRep *clone(const Range&, const Range&);
	
	redefined DataType dataType() const { return DoubleData; }
	redefined Handle getHandle(int pos=0, int chan=0) const;
	redefined Handle setAndShift(Handle h, double val, int incr);
	redefined Handle getAndShift(Handle h, double *val, int incr) const;
	redefined void valueRange(Range&, int, int, int, int, boolean=false) const;
	redefined double valueLimit() const { return 1.0; }
	redefined void set(double val, int pos, int chan);
	redefined double get(int pos, int chan=0) const;
	redefined void reverse();
	redefined void swapBytes();
protected:
	DoubleDataRep(const DoubleDataRep* rep) : DataRep(rep) {}
private:
	DoubleDataRep(const DoubleDataRep&);
};

inline DataRep::Handle
DoubleDataRep::getHandle(int pos, int chan) const {
	return (double *) arrayOffset() + chan + (pos*nchans); 
}

#endif
