// formantsynth.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <math.h>

#include "application.h"
#include "data.h"
#include "formantsynth.h"
#include "lpcdata.h"
#include "lpcfilter.h"
#include "request.h"
#include "query.h"
#include "synthfunction.h"

static unsigned seed = 3;

FormantSynthesizer::FormantSynthesizer(Data* output, LPCData* data,
	double freq, int nHarms, double amp)
		: LPC_Filter(data, output->length(), false),
		  Buzz(output, freq, nHarms, amp),
		  noise(seed), gain(1.0), voicedFactor(3.0),
		  hiThreshold(1.0), lowThreshold(0.0) {
    ok(LPC_Filter::good());
}

FormantSynthesizer::FormantSynthesizer(Data* output, LPCData* data)
		: LPC_Filter(data, output->length(), false), Buzz(output),
		  noise(seed), gain(1.0), voicedFactor(3.0),
		  hiThreshold(1.0), lowThreshold(0.0) {
    ok(LPC_Filter::good());
}

void
FormantSynthesizer::initialize() {
	LPC_Filter::initialize();
	Buzz::initialize();
	srOverTwo = sampRate()/2.0;
	restoreState();
}

void
FormantSynthesizer::restoreState() {
	LPC_Filter::reset();
    Buzz::restoreState();
	noise.setSeed(seed);
}

double
FormantSynthesizer::doSynthesis() {
	RunStatus status;
	float amp = 0.0;
	float error = 0.0;
	float pitch = 0.0;
	float* coeffs = getFrame(&amp, &error, &pitch);
	int nharm = int(srOverTwo/pitch) - 2;
	double sqerr = sqrt(error);
	double noiseAmp = (sqerr - lowThreshold) / (hiThreshold - lowThreshold);
	noiseAmp = max(0.0, min(1.0, noiseAmp));	// 0 <= noiseAmp <= 1.0
	double voicedAmp = 1.0 - noiseAmp;			// complement of that
	set(pitch, nharm, voicedAmp * voicedFactor);
	noise.setGain(noiseAmp);
	double signal = unvoiced() + voiced();
	return filter(coeffs, amp * signal * gain);
}

const QueryInfo *
FormantSynthesizer::requestInfo() {
	static QueryLabelInfo labelInfo[] = {
		{ "Resynthesize LPC Data into Selected Region:" },
		{ nil }
	};	
	static QueryValueInfo valueInfo[] = {
		{ "Gain Factor:", "1.00", CharCheck::numsOnly },
		{ "Unvoiced threshold:", "1.0", nil, "0.0|1.0" },
		{ "Voiced threshold:", "0.0", nil, "0.0|1.0" },
		{ "Voiced/Unvoiced amp factor:", "3.0", nil, "0.0|10.0" },
//		{ "Warp Factor:", "0.00", nil, "0.0|1.0" },
		{ nil }
	};	
	static QueryInfo info[] = {
		{
			labelInfo,
			"Noise component will be scaled based on the two thresholds.",
			valueInfo
		},
		{ nil }
	};
	return info;
}

boolean
FormantSynthesizer::setValues(Request &request) {
	QueryValue v[4];
	request.retrieveValues(v, 4);
	gain = v[0];
	hiThreshold = sqrt(v[1]);
	lowThreshold = sqrt(v[2]);
	voicedFactor = v[3];
//	warp = v[4];
	return true;
}

boolean
FormantSynthesizer::confirmValues(Controller *) {
	boolean status = true;
	if(sampRate() != lpcSamplingRate()) {
		char msg[120];
		sprintf(msg,
			"Warning: LPC data samprate (%d) != selection samprate (%d).",
				lpcSamplingRate(), sampRate());
		status = Application::confirm(msg, "The formant peaks will be skewed.",
			"Continue anyway?", Cancel);
	}
	return status;
}

