// combfilter.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "combfilter.h"
#include "delay.h"
#include "query.h"
#include "request.h"
#include "data.h"

CombFilter::CombFilter(Data* input, Data *output, double delaytime,
		double ringtime) : SimpleInputOutputFunction(input, output),
		myDelay(nil), delayTime(delaytime), ringTime(ringtime) {
	initialize();
}

CombFilter::CombFilter(Data* input, Data *output)
	: SimpleInputOutputFunction(input, output), myDelay(nil), delayTime(0),
	  ringTime(0) {}

CombFilter::~CombFilter() {
	delete myDelay;
}

void
CombFilter::clear() {
	myDelay->restoreState();
}

void
CombFilter::initialize() {
	int size = round(sampRate() * delayTime);
	myDelay = new Delay(target(), size);
	feedBack = pow(.001, (delayTime/ringTime));
	Super::initialize();
}

void
CombFilter::restoreState() {
	Super::restoreState();
	clear();
}

boolean
CombFilter::areCompatibleLengths(int oldlen, int newlen) {
	return oldlen >= newlen;
}

int
CombFilter::newLength(int oldLength) {
	return roundUp(oldLength + ringTime * sampRate());
}

int
CombFilter::doProcessing() {
	BUG("CombFilter::doProcessing()");
	int status = true;
	int flushing = needToFlush();
	while((!inQueueEmpty() || flushing) && status == true) {
		status = processInput(flushing);
	}
	moreOutputIf(true);		// let this run until output eof
	return status;
}

double
CombFilter::doSimpleProcess(double input) {
	double output = myDelay->getPast();
	myDelay->push(input + output * feedBack);
	myDelay->increment();
	return output;
}

const QueryInfo *
CombFilter::requestInfo() {
	static QueryLabelInfo labelInfo[] = {
		{ "Comb-filter Selected Region:" },
		{ nil }
	};	
	static QueryValueInfo valueInfo[] = {
		{ "Fundamental Frequency/Loop Time:", "1000",
			CharCheck::posNumsOnly },
		{ "Ring Time (seconds):", "0.10", CharCheck::posNumsOnly },
		{ nil }
	};	
	static QueryChoiceInfo choiceInfo[] = {
		{ "Filter Value Mode:",
		  "|Fundamental Frequency|Delay Loop Time|", Frequency, Yes },
		{ nil }
	};
	static QueryInfo info[] = {
		{ labelInfo, "", valueInfo, choiceInfo },
		{ nil }
	};
	return info;
}

boolean
CombFilter::setValues(Request& request) {
	const int nvals = 2;
	QueryValue vr[nvals];
	request.retrieveValues(vr, nvals);
	double factor = vr[0];
	ringTime = vr[1];
	QueryChoice choice;
	request.retrieveChoices(choice);
	if(factor == 0.0) {
		char msg[64];
		sprintf(msg, "%s%s", choice == Frequency ?
			"Fundamental Frequency" : "Delay Loop Time",
			" must be a nonzero value.");
		Application::alert(msg);
		return false;
	}
	switch(choice) {
	case Frequency:
		delayTime = 1.0/min(factor, sampRate()/2.0);	// Nyquist is max
		break;
	case LoopTime:
		delayTime = factor;
		break;
	default:
		Application::die("CombFilter::queryForValues: invalid choice");
		break;
	}
	return true;
}
