// graph.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include <InterViews/painter.h>
#include <InterViews/brush.h>
#include <InterViews/perspective.h>
#include <InterViews/sensor.h>
#include <InterViews/event.h>
#include <InterViews/shape.h>
#include "graph.h"
#include "data.h"
#include "scale.h"
#include "controller.h"

Graph::Graph(Controller *c, Data *data, const Range &range) 
		: ScaledArea(range, nil),
		  controller(c), graphData(data), view(nil), configured(false) {
	Init();
}

Graph::~Graph() {
	Resource::unref(view);
}

void
Graph::Init() {
	SetClassName("Graph");
	input = new Sensor(updownEvents);
	input->Catch(EnterEvent);
	input->Catch(LeaveEvent);
	input->Catch(KeyEvent);
	input->Catch(MotionEvent);
	viewIsChanged = true;
}

void
Graph::Adjust(Perspective &np) {
	BUG("Graph::Adjust()");
	if(*perspective != np) {
		viewIsChanged = isViewChanged(np);
		*perspective = np;
		doAdjust();
		Draw();
	}
}

void
Graph::Reconfig() {
	BUG("Graph::Reconfig()");
	checkViewPerspective();
	ScaledArea::Reconfig();
	if(!isConfigured()) {	// only the first time Reconfig() called
		const char *a = GetAttribute("PlotWidth");
		shape->width = (a != nil) ? atoi(a) : defaultGraphWidth;
		a = GetAttribute("PlotHeight");
		shape->height = (a != nil) ? atoi(a) : defaultGraphHeight;
		setShape();
	}
	configured = true;
}

void
Graph::Resize() {
	BUG("Graph::Resize()");
	ScaledArea::Resize();
	doResize();		// this is where subclasses take care of specialized stuff
	if(widthChanged()) {
		createPlot();		
		loadPlot();	// horiz size changed (not initialized), so rescan data
	}
	if(isResized())
		doPlot();	// re-plot if any dimension has changed
	resetDimensions();
}

void
Graph::Update() {
	BUG("Graph::Update()");
	checkViewPerspective();
	setReferenceRange();
	if(vscale) vscale->setRange(getReferenceRange());
	viewIsChanged = true;	// force re-reading of data
	ScaledArea::Update();
}

void
Graph::Handle(Event &e) {
	// these are the only events we are interested in
	if(e.meta_is_down())
		return;		// ignore all window manager events	
	if(e.leftmouse || e.rightmouse || e.middlemouse || e.type() == Event::key)
		controller->handleEvent(e);	// send it back to top
}

// this is invoked by the DataView containing this graph

boolean
Graph::handleEvent(Event &e) {
	boolean handled = true;
	switch(e.type()) {
	case Event::down:
		grab(e);
		break;
	case Event::up:
		select();
		break;
	case Event::motion:
		if(e.eventType == MotionEvent)
			track(e);
		else if(e.eventType == EnterEvent) {
			if(e.shift_is_down() && (e.left_is_down() || e.middle_is_down()))
				track(e);
		}
		else if(e.eventType == LeaveEvent) {
			track(e);
			select();
		}
		break;
	case Event::key:
		processKeyEvent(e);
		break;
	default:
		handled = false;
		break;
	}
	return handled;
}

void
Graph::doAdjust() {
	BUG("Graph::doAdjust()");
	ScaledArea::doAdjust();
	if(IsMapped()) {
		if(viewIsChanged) {
			loadPlot();
			viewIsChanged = false;		// reset
		}
		doPlot();
	}
}

const char *
Graph::getTitle() {
	return vscale ? vscale->getLabel(): "";
}

// Graphs needs a pointer to the master perspective to allow them to keep
// current without the need for a call to Adjust()

void
Graph::setViewPerspective(Perspective *p) {
	if(p != nil && view != p) {
		view = p;
		view->ref();
	}
}

// make sure that internal perspective matches master
// this is used during Update()

void
Graph::checkViewPerspective() {
	if(view != nil)
		*perspective = *view;
}

// ********

#ifdef undef

#if defined(iv2_6_compatible)
static float defaultDashWidth = 0;
#else
static Coord defaultDashWidth = 0;
#endif

static const int MaxGridLines = 100;

Grid::Grid(int nvl, int nhl) : nVertLines(nvl), nHorizLines(nhl) {
	static int pattern[] = {2, 2};
	gridBrush = new Brush(pattern, 2, defaultDashWidth);
	gridBrush->ref();
}

Grid::~Grid() {
	Resource::unref(gridBrush);
}

void
Grid::draw(Painter *p, Canvas *c) {
	Brush *oldbrush = p->GetBrush();	// save it
	oldbrush->ref();
	int width = c->Width();
	int height = c->Height();
	int horizspacing = width/nVertLines;
	int vertspacing = height/nHorizLines;
	p->SetBrush(gridBrush);
	register int i, currentloc = 0;
	for(i=0; i<nVertLines; i++) {
		currentloc = width * i / nVertLines;
		p->Line(c, currentloc, 0, currentloc, height);
	}
	currentloc = 0;
	for(i=0; i<nHorizLines; i++) {
		currentloc = width * i / nHorizLines;
		p->Line(c, 0, currentloc, width, currentloc);
	}
	p->SetBrush(oldbrush);
	Resource::unref(oldbrush);
}
#endif /* undef */
