/************************* MPEG-2 NBC Audio Decoder **************************
 *                                                                           *
"This software module was originally developed by 
AT&T, Dolby Laboratories, Fraunhofer Gesellschaft IIS and edited by
Yoshiaki Oikawa (Sony Corporation),
Mitsuyuki Hatanaka (Sony Corporation)
in the course of development of the MPEG-2 NBC/MPEG-4 Audio standard ISO/IEC 13818-7, 
14496-1,2 and 3. This software module is an implementation of a part of one or more 
MPEG-2 NBC/MPEG-4 Audio tools as specified by the MPEG-2 NBC/MPEG-4 
Audio standard. ISO/IEC  gives users of the MPEG-2 NBC/MPEG-4 Audio 
standards free license to this software module or modifications thereof for use in 
hardware or software products claiming conformance to the MPEG-2 NBC/MPEG-4
Audio  standards. Those intending to use this software module in hardware or 
software products are advised that this use may infringe existing patents. 
The original developer of this software module and his/her company, the subsequent 
editors and their companies, and ISO/IEC have no liability for use of this software 
module or modifications thereof in an implementation. Copyright is not released for 
non MPEG-2 NBC/MPEG-4 Audio conforming products.The original developer
retains full right to use the code for his/her  own purpose, assign or donate the 
code to a third party and to inhibit third party from using the code for non 
MPEG-2 NBC/MPEG-4 Audio conforming products. This copyright notice must
be included in all copies or derivative works." 
Copyright(c)1996.
 *                                                                           *
 ****************************************************************************/

#include "all.h"
#include "sony_local.h" /* 971117 YT */
#define RAND_MAX 1 /* 971117 YT temporarily added. not defined anywhere? */

typedef int DINT_DATATYPE ;

static void
deinterleave(DINT_DATATYPE inptr[], DINT_DATATYPE outptr[], short ngroups,
             short nsubgroups[], int ncells[], short cellsize[])
{
  int i, j, k, l;
  DINT_DATATYPE *start_inptr, *start_subgroup_ptr, *subgroup_ptr;
  short cell_inc, subgroup_inc;

  start_subgroup_ptr = outptr;

  for (i = 0; i < ngroups; i++)
    {
      cell_inc = 0;
      start_inptr = inptr;

      /* Compute the increment size for the subgroup pointer */

      subgroup_inc = 0;
      for (j = 0; j < ncells[i]; j++) {
        subgroup_inc += cellsize[j];
      }

      /* Perform the deinterleaving across all subgroups in a group */

      for (j = 0; j < ncells[i]; j++) {
        subgroup_ptr = start_subgroup_ptr;

        for (k = 0; k < nsubgroups[i]; k++) {
          outptr = subgroup_ptr + cell_inc;
          for (l = 0; l < cellsize[j]; l++) {
            *outptr++ = *inptr++;
          }
          subgroup_ptr += subgroup_inc;
        }
        cell_inc += cellsize[j];
      }
      start_subgroup_ptr += (inptr - start_inptr);
    }
}

static void 
calc_gsfb_table(Info *info, byte *group)
{
  int group_offset;
  int group_idx;
  int offset;
  short * group_offset_p;
  int sfb,len;
  /* first calc the group length*/
  if (info->islong){
    return;
  } else {
    group_offset = 0;
    group_idx =0;
    do  {
      info->group_len[group_idx]=group[group_idx]-group_offset;
      group_offset=group[group_idx];
      group_idx++;
    } while (group_offset<8);
    info->num_groups=group_idx;
    group_offset_p = info->bk_sfb_top;
    offset=0;
    for (group_idx=0;group_idx<info->num_groups;group_idx++){
      len = info->group_len[group_idx];
      for (sfb=0;sfb<info->sfb_per_sbk[group_idx];sfb++){
        offset += info->sfb_width_short[sfb] * len;
        *group_offset_p++ = offset;
#if 0
        PRINT(SE,"%d %d: %d\n",  group_idx, sfb, offset);
#endif	  
      }
    }
  }
}

void
getgroup(Info *info, byte *group)
{
  int i, j, first_short;

  if( debug['g'] ) PRINT(SE,"Grouping: 0");
  first_short=1;
  for (i = 0; i < info->nsbk; i++) {
    if (info->bins_per_sbk[i] > SN2) {
      /* non-short windows are always their own group */
      *group++ = i+1;
    }
    else {
      /* only short-window sequences are grouped! */
      if (first_short) {
        /* first short window is always a new group */
        first_short=0;
      }
      else {
        if((j = getbits(1)) == 0) {
          *group++ = i;
        }
        if( debug['g'] ) PRINT(SE,"%1d", j);
      }
    }
  }
  *group = i;
  if( debug['g'] ) PRINT(SE,"\n");
}

/*
 * read a synthesis mask
 *  uses EXTENDED_MS_MASK
 *  and grouped mask 
 */
int
getmask(Info *info, byte *group, byte max_sfb, byte *mask)
{
  int b, i, mp;

  mp = getbits(LEN_MASK_PRES);
  if( debug['m'] )
    PRINT(SE,"\nExt. Mask Present: %d\n",mp);

  /* special EXTENDED_MS_MASK cases */
  if(mp == 0) { /* no ms at all */
    return 0;
  }
  if(mp == 2) {/* MS for whole spectrum on, mask bits set to 1 */
    for(b = 0; b < info->nsbk; b = *group++)
      for(i = 0; i < info->sfb_per_sbk[b]; i ++)
        *mask++ = 1;
    return 2;
  }

  /* otherwise get mask */
  for(b = 0; b < info->nsbk; b = *group++){
    if( debug['m'] ) PRINT(SE," gr%1i:",b);
    for(i = 0; i < max_sfb; i ++) {
      *mask = getbits(LEN_MASK);
      if( debug['m'] )PRINT(SE,"%1i",*mask);
      mask++;
    }
    for( ; i < info->sfb_per_sbk[b]; i++){
      *mask = 0;
      if( debug['m'] ) PRINT(SE,"%1i",*mask);
      mask++;
    }
  }
  if( debug['m'] ) PRINT(SE,"\n");
  return 1;
}

void
clr_tns( Info *info, TNS_frame_info *tns_frame_info )
{
  int s;

  tns_frame_info->n_subblocks = info->nsbk;
  for (s=0; s<tns_frame_info->n_subblocks; s++)
    tns_frame_info->info[s].n_filt = 0;
}

int
get_tns( Info *info, TNS_frame_info *tns_frame_info )
{
  int                       f, t, top, res, res2, compress;
  int                       short_flag, s;
  short                     *sp, tmp, s_mask, n_mask;
  TNSfilt                   *tns_filt;
  TNSinfo                   *tns_info;
  static short              sgn_mask[] = { 
    0x2, 0x4, 0x8     };
  static short              neg_mask[] = { 
   (short) 0xfffc, (short)0xfff8, (short)0xfff0     };


  short_flag = (!info->islong);
  tns_frame_info->n_subblocks = info->nsbk;

  for (s=0; s<tns_frame_info->n_subblocks; s++) {
    tns_info = &tns_frame_info->info[s];

    if (!(tns_info->n_filt = getbits( short_flag ? 1 : 2 )))
      continue;
	    
    tns_info -> coef_res = res = getbits( 1 ) + 3;
    top = info->sfb_per_sbk[s];
    tns_filt = &tns_info->filt[ 0 ];
    for (f=tns_info->n_filt; f>0; f--)  {
      tns_filt->stop_band = top;
      top = tns_filt->start_band = top - getbits( short_flag ? 4 : 6 );
      tns_filt->order = getbits( short_flag ? 3 : 5 );

      if (tns_filt->order)  {
        tns_filt->direction = getbits( 1 );
        compress = getbits( 1 );

        res2 = res - compress;
        s_mask = sgn_mask[ res2 - 2 ];
        n_mask = neg_mask[ res2 - 2 ];

        sp = tns_filt -> coef;
        for (t=tns_filt->order; t>0; t--)  {
          tmp = getbits( res2 );
          *sp++ = (tmp & s_mask) ? (tmp | n_mask) : tmp;
        }
      }
      tns_filt++;
    }
  }   /* subblock loop */
  return 1;
}

/*
 * NEC noiseless coding
 */
struct Nec_Info
{
  int pulse_data_present;
  int number_pulse;
  int pulse_start_sfb;
  int pulse_position[NUM_NEC_LINES];
  int pulse_offset[NUM_NEC_LINES];
  int pulse_amp[NUM_NEC_LINES];
};
static struct Nec_Info nec_info;

static void
get_nec_nc(struct Nec_Info *nec_info)
{
  int i;
  nec_info->number_pulse = getbits(LEN_NEC_NPULSE);
  nec_info->pulse_start_sfb = getbits(LEN_NEC_ST_SFB);
  for(i=0; i<nec_info->number_pulse; i++) {
    nec_info->pulse_offset[i] = getbits(LEN_NEC_POFF);
    nec_info->pulse_amp[i] = getbits(LEN_NEC_PAMP);
  }
}

static void
nec_nc(int *coef, struct Nec_Info *nec_info)
{
  int i, k;
    
  k = only_long_info.sbk_sfb_top[0][nec_info->pulse_start_sfb];
    
  for(i=0; i<=nec_info->number_pulse; i++) {
    k += nec_info->pulse_offset[i];
    if (coef[k]>0) coef[k] += nec_info->pulse_amp[i];
    else coef[k] -= nec_info->pulse_amp[i];
  }
}



int
getics(Info *info, int common_window, byte *win, byte *wshape, 
       byte *group, byte *max_sfb, PRED_TYPE pred_type, int *lpflag, int *prstflag, 
       byte *cb_map, Float *coef, short *global_gain, 
       short *factors,NOK_LT_PRED_STATUS *nok_ltp, TNS_frame_info *tns
#ifdef  SRS
       , BsBitStream *gc_streamCh 
#endif
       , enum AAC_BIT_STREAM_TYPE bitStreamType )
{
  int nsect, i, cb, top, bot, tot_sfb;
  byte sect[ 2*(MAXBANDS+1) ];

  /*
   * global gain
   */
  *global_gain = getbits(LEN_SCL_PCM);
  if (debug['f'])
    PRINT(SE,"global gain: %3d\n", *global_gain);
  if ( bitStreamType != SCALEABLE ){
    memcpy(info, winmap[*win], sizeof(Info));
  }
  if (( !common_window ) && ( bitStreamType != SCALEABLE )) {
    get_ics_info(win, wshape, group, max_sfb, pred_type, lpflag, prstflag, bitStreamType);
    memcpy(info, winmap[*win], sizeof(Info));
  } else {
    if( info->nsbk == 1 ) {
      group[0] = 1;
    } else {
      int sum = 0;
      for( i=0; i<info->num_groups; i++ ) {
        sum  += info->group_len[i];
        group[i]  = sum;
      }
    }
  }

  /* calculate total number of sfb for this grouping */
  if (*max_sfb == 0) {
    tot_sfb = 0;
  }
  else {
    i=0;
    tot_sfb = info->sfb_per_sbk[0];
    if (debug['f'])PRINT(SE,"tot sfb %d %d\n", i, tot_sfb);
    while (group[i++] < info->nsbk) {
      tot_sfb += info->sfb_per_sbk[0];
      if (debug['f'])PRINT(SE,"tot sfb %d %d\n", i, tot_sfb);
    }
  }

  /* 
   * section data
   */
  nsect = huffcb(sect, info->sectbits, tot_sfb, info->sfb_per_sbk[0], *max_sfb);
  if(nsect==0 && *max_sfb>0)
    return 0;

  /* generate "linear" description from section info
   * stored as codebook for each scalefactor band and group
   */
  if (nsect) {
    bot = 0;
    for (i=0; i<nsect; i++) {
      cb = sect[2*i];
      top = sect[2*i + 1];
      for (; bot<top; bot++)
        *cb_map++ = cb;
      bot = top;
    }
  }  else {
    for (i=0; i<MAXBANDS; i++)
      cb_map[i] = 0;
  }

  /* calculate band offsets
   * (because of grouping and interleaving this cannot be
   * a constant: store it in info.bk_sfb_top)
   */
  calc_gsfb_table(info, group);

  /*
   * scale factor data
   */
  if(!hufffac(info, group, nsect, sect, *global_gain, factors))
    return 0;

  /*
   * NEC noiseless coding
   */
  if ((nec_info.pulse_data_present = getbits(LEN_PULSE_PRES))) {
    if (info->islong) {
      get_nec_nc(&nec_info);
    }
    else {
      CommonExit(1, "Pulse data not allowed for short blocks\n");    
    }
  }
  if (bitStreamType  !=  SCALEABLE){  
    /*
     * Nokia long term prediction
     */
    if (pred_type == NOK_LTP)
      nok_lt_decode((WINDOW_TYPE)win, *max_sfb, nok_ltp->sbk_prediction_used,
		    nok_ltp->sfb_prediction_used, &nok_ltp->weight,
		    nok_ltp->delay);
    /*
     * tns data
     */
    if (getbits(LEN_TNS_PRES)) {
      get_tns(info, tns);
    }
    else {
      clr_tns(info, tns);
    }
    
    /*
     * Sony gain control
     */
    if (getbits(LEN_GAIN_PRES)) {
#ifdef  SRS
	get_gcBuf(*win,gc_streamCh);
#else
      
      CommonExit(1, "Gain control not implemented\n");
#endif
    }
  }
  return huffspec(info, nsect, sect, factors, coef);
}

/*
 * read the codebook and boundaries
 */
int
huffcb(byte *sect, int *sectbits, int tot_sfb, int sfb_per_sbk, byte max_sfb)
{
  int nsect, n, base, bits, len;

  if (debug['s']) {
    PRINT(SE,"total sfb %d\n", tot_sfb);
    PRINT(SE,"sect, top, cb\n");
  }
  bits = sectbits[0];
  len = (1 << bits) - 1;
  nsect = 0;
  for(base = 0; base < tot_sfb && nsect < tot_sfb; ){
    *sect++ = getbits(LEN_CB);

    n = getbits(bits);
    if (n==0)
      CommonWarning("\n a section length of 0 ");
    while(n == len && base < tot_sfb){
      base += len;
      n = getbits(bits);
    }
    base += n;
    *sect++ = base;
    nsect++;
    if (debug['s'])
      PRINT(SE," %6d %6d %6d \n", nsect, sect[-1], sect[-2]);

    /* insert a zero section for regions above max_sfb for each group */
    if ((sect[-1] % sfb_per_sbk) == max_sfb) {
      base += (sfb_per_sbk - max_sfb);
      *sect++ = 0;
      *sect++ = base;
      nsect++;
      if (debug['s'])
	PRINT(SE,"(%6d %6d %6d)\n", nsect, sect[-1], sect[-2]);
    }
  }

  if(base != tot_sfb || nsect > tot_sfb)
    return 0;
  return nsect;
}

/* 
 * get scale factors
 */
int
hufffac(Info *info, byte *group, int nsect, byte *sect,
        short global_gain, short *factors)
{
  Hcb *hcb;
  Huffman *hcw;
  int i, b, bb, t, n, sfb, top, fac, is_pos;
  int factor_transmitted[MAXBANDS], *fac_trans;
  int noise_pcm_flag = 1;
  int noise_nrg;

  /* clear array for the case of max_sfb == 0 */
  intclr(factor_transmitted, MAXBANDS);
  shortclr(factors, MAXBANDS);

  sfb = 0;
  fac_trans = factor_transmitted;
  for(i = 0; i < nsect; i++){
    top = sect[1];		/* top of section in sfb */
    t = sect[0];		/* codebook for this section */
    sect += 2;
    for(; sfb < top; sfb++) {
      fac_trans[sfb] = t;
    }
  }

  /* scale factors are dpcm relative to global gain
   * intensity positions are dpcm relative to zero
   */
  fac = global_gain;
  is_pos = 0;
  noise_nrg = global_gain;

  /* get scale factors */
  hcb = &book[BOOKSCL];
  hcw = hcb->hcw;
  bb = 0;
  if (debug['f'])
    PRINT(SE,"scale factors\n");
  for(b = 0; b < info->nsbk; ){
    n = info->sfb_per_sbk[b];
    b = *group++;
    for(i = 0; i < n; i++){
      switch (fac_trans[i]) {
      case ZERO_HCB:	    /* zero book */
        break;
      default:		    /* spectral books */
        /* decode scale factor */
        t = decode_huff_cw(hcw);
        fac += t - MIDFAC;    /* 1.5 dB */

        if (debug['f'])
          PRINT(SE,"%3d:%3d", i, fac);
        if(fac >= 2*maxfac || fac < 0)
          return 0;
        factors[i] = fac;
        break;
      case BOOKSCL:	    /* invalid books */
        return 0;
      case INTENSITY_HCB:	    /* intensity books */
      case INTENSITY_HCB2:
        /* decode intensity position */
        t = decode_huff_cw(hcw);
        is_pos += t - MIDFAC;
        factors[i] = is_pos;
        break;
      case NOISE_HCB:	    /* noise books */
        /* decode noise energy */
        if (noise_pcm_flag) {
          noise_pcm_flag = 0;
          t = getbits( NOISE_PCM_BITS ) - NOISE_PCM_OFFSET;
        }
        else
          t = decode_huff_cw(hcw) - MIDFAC;
        noise_nrg += t;
        if (debug['f'])
          PRINT(SE,"\n%3d %3d (noise, code %d)", i, noise_nrg, t);
        factors[i] = noise_nrg;
        break;

      }
      if (debug['f'])
        PRINT(SE,"%3d: %3d %3d\n", i, fac_trans[i], factors[i]);
    }
    if (debug['f'])
      PRINT(SE,"\n");

    /* expand short block grouping */
    if (!(info->islong)) {
      for(bb++; bb < b; bb++) {
        for (i=0; i<n; i++) {
          factors[i+n] = factors[i];
        }
        factors += n;
      }
    }
    fac_trans += n;
    factors += n;
  }
  return 1;
}

/* rm2 inverse quantization
 * escape books need ftn call
 * other books done via macro
 */
#define iquant( q ) ( q >= 0 ) ? \
(Float)iq_exp_tbl[ q ] : (Float)(-iq_exp_tbl[ - q ])

     Float
esc_iquant(int q)
{
  if (q > 0) {
    if (q < MAX_IQ_TBL) {
      return((Float)iq_exp_tbl[q]);
    }
    else {
      return(pow(q, 4./3.));
    }
  }
  else {
    q = -q;
    if (q < MAX_IQ_TBL) {
      return((Float)(-iq_exp_tbl[q]));
    }
    else {
      return(-pow(q, 4./3.));
    }
  }
}


#define MEAN_NRG 1.5625e+18      /* Theory: (2^31)^2 / 3 = 1.537228e+18 */

long
random2( long *seed )
{
#if 0
  *seed = (*seed * 1103515245L) + 12345L;    /* VM */
#else
  *seed = (1664525L * *seed) + 1013904223L;  /* Numerical recipes */
#endif
  return(long)(*seed);
}

void
gen_rand_vector( float *spec, int size, int ctrl )  
     /* Noise generator, generating vector with unity energy */
{
  int i;
  int mode = 3;

  if (mode==0)  {       /* Dummy 0:  zero vector */
    for (i=0; i<size; i++)
      spec[i] = 0.0;
  }
  else if (mode==1)  {  /* Dummy 1:  zero vector with one spectral component */
    for (i=0; i<size; i++)
      spec[i] = 0.0;
    spec[3] = 1.0;
  }
  else if (mode==2) {    /* Dummy 2:  random vector */
    float s, nrg=0.0;

    for (i=0; i<size; i++) {
      spec[i] = s = (float)rand()/RAND_MAX - 0.5;
      nrg += s * s;
    }
    s = 1.0 / sqrt( nrg );
    for (i=0; i<size; i++) {
      spec[i] *= s;
      /*fprintf(stderr,"\nRAND[%d] = %f", i, spec[i]);*/
    }
  }

  else if (mode==3) {    /* Mode 3:  real random number generator */
    static long  seed = 0L;
    float s, norm, nrg= 0.0;

    norm = 1.0 / sqrt( size * MEAN_NRG );

    for (i=0; i<size; i++) {
      spec[i] = (float)(random2( &seed ) * norm);   
      nrg += spec[i] * spec[i];
    }

    s = 1.0 / sqrt( nrg );
    for (i=0; i<size; i++)
      spec[i] *= s;      
  }

}


int
huffspec(Info *info, int nsect, byte *sect, short *factors, Float *coef)
{
  Hcb *hcb;
  Huffman *hcw;
  int i, j, k, table, step, temp, stop, bottom, top;
  short *bands, *bandp;
  int quant[LN2], *qp;	    /* probably could be short */

  int idx,kstart;
  int* quantDebug;
  int tmp_spec[LN2];

  int noise_flag[ 200 ], *np = noise_flag;

  intclr(noise_flag, 200);

  intclr(quant, LN2);

  bands = info->bk_sfb_top;
  bottom = 0;
  k = 0;
  bandp = bands;
  for(i = nsect; i; i--) {
    table = sect[0];
    top = sect[1];
    sect += 2;
    quantDebug=quant;
    if(table == NOISE_HCB  &&  !info->islong)
      CommonExit(-1,"No noise substitution supported for short blocks");

    for (j=bottom; j<top; j++)
      *np++ = (table == NOISE_HCB);
    if( (table == 0) ||
        (table == NOISE_HCB) ||
        (table == INTENSITY_HCB) ||
        (table == INTENSITY_HCB2) ) {
      bandp = bands+top;
      k = bandp[-1];
      bottom = top;
      continue;
    }
    if(table < BY4BOOKS+1) {
      step = 4;
    }
    else {
      step = 2;
    }
    hcb = &book[table];
    hcw = hcb->hcw;
    qp = quant+k;
    kstart=k;
    for(j=bottom; j<top; j++) {
      stop = *bandp++;
      while(k < stop) {
        temp = decode_huff_cw(hcw);
        unpack_idx(qp, temp, hcb);

        if (!hcb->signed_cb)
          get_sign_bits(qp, step);
        if(table == ESCBOOK){
          qp[0] = getescape(qp[0]);
          qp[1] = getescape(qp[1]);
        }
        qp += step;
        k += step;
      }
      if(debug['q']){
        PRINT(SE,"\nsect %d %d\n", table, kstart);
        for (idx=kstart ;idx<k;idx+=2) {
          PRINT(SE,"%d %d  ",quantDebug[idx],quantDebug[idx+1]);
        }
        PRINT(SE,"\n");
      }
    }
    bottom = top;
  }

  /* NEC noisless coding reconstruction */
  if ( (info->islong) && (nec_info.pulse_data_present) )
    nec_nc(quant, &nec_info);

  if (!info->islong) {
    deinterleave (quant,tmp_spec,
                  info->num_groups,   
                  info->group_len,
                  info->sfb_per_sbk,
                  info->sfb_width_short);
    memcpy(quant,tmp_spec,sizeof(tmp_spec));
  }

  /* inverse quantization */
  for (i=0; i<info->bins_per_bk; i++) {
    coef[i] = esc_iquant(quant[i]);
  }

  /* Noise substitution */
  if (info->islong) {   /* Code only works for long blocks */

    int sfb, nsfb, top;

    nsfb = info->sfb_per_sbk[0];
    k=0;
    for (sfb=0; sfb<nsfb; sfb++) {
      top = info->sbk_sfb_top[0][sfb];
      if (noise_flag[sfb]) {
        /* Compensate for noise factor offset */
        factors[sfb] += SF_OFFSET-90;           /* <<<< NOISE_OFFSET */

        /* Generate noise vector */
        gen_rand_vector( coef+k, top-k, 0 );
#if 0
        fprintf(stderr, "NS: NRG[%d] = %d\n", sfb, factors[sfb]-SF_OFFSET);
#endif
      }

      k = top;
    }
  }

  /* rescaling */
  {
    int sbk, nsbk, sfb, nsfb, fac, top;
    Float *fp, scale;

    i = 0;
    fp = coef;
    nsbk = info->nsbk;
    for (sbk=0; sbk<nsbk; sbk++) {
      nsfb = info->sfb_per_sbk[sbk];
      k=0;
      for (sfb=0; sfb<nsfb; sfb++) {
        top = info->sbk_sfb_top[sbk][sfb];
        fac = factors[i++]-SF_OFFSET;

        if (fac >= 0 && fac < TEXP) {
          scale = exptable[fac];
        }
        else {
          if (fac == -SF_OFFSET) {
            scale = 0;
          }
          else {
            scale = pow( 2.0,  0.25*fac );
          }
        }
        for ( ; k<top; k++) {
          *fp++ *= scale;
        }
      }
    }
  }
  return 1;
}
static BsBitStream *vmBitBuffer;

  
/* -------------------------------------- */
/* -------------------------------------- */
/* -----  Set the current Bitbuffer ----- */
/* -------------------------------------- */
/* -------------------------------------- */
void
setHuffdec2BitBuffer(BsBitStream *fixed_stream)
{
  vmBitBuffer = fixed_stream;
} 

int
byte_align(void)
{
  int i=0;
#if 1
  int curBitPos;
  curBitPos=BsCurrentBit(vmBitBuffer);
  if (curBitPos & ((1<<3)-1)) {
    CommonExit(-1,"start of frame not byte aligned, huffdec2.c line 775");
  }
#else    
  if(nbits <= 16){
    cword = (cword<<16) | getshort();
    nbits += 16;
  }

  if (debug['a'])
    PRINT(SE,"align: %lx %d\n", cword, nbits);

  while (nbits & ((1<<3)-1)) {
    numUsedBits++;
    nbits -= 1;
    i += 1;
  }
#endif
  return(i);
}
#if 1
long
getbits(int n)
{
  unsigned long value;
  if (n!=0){
    numUsedBits+=n;
    BsGetBit(vmBitBuffer,&value,n);

    if (debug['b'])
      PRINT(SE, "  AAC: getbits: val=%5ld      num=%5d\n", value, n );
  }
  return value;
}
#else
long
getbits(int n)
{
  if (n!=0){
    numUsedBits+=n;
    if(nbits <= 16){
      cword = (cword<<16) | getshort();
      nbits += 16;
    }
    
    nbits -= n;
    
    if (debug['b'])
      PRINT(SE, "  AAC: getbits: val=%5d      num=%5d\n", (cword>>nbits) & ((1<<n)-1), n );
  }
  return (cword>>nbits) & ((1<<n)-1);
}
#endif
int
getescape(int q)
{
  int i, off, neg;

  if(q < 0){
    if(q != -16)
      return q;
    neg = 1;
  } else{
    if(q != +16)
      return q;
    neg = 0;
  }

  for(i=4;; i++){
    if(getbits(1) == 0)
      break;
  }

  if(i > 16){
    off = getbits(i-16) << 16;
    off |= getbits(16);
  } else
    off = getbits(i);

  i = off + (1<<i);
  if(neg)
    i = -i;
  return i;
}

#ifdef SRS

static int
get_gcBuf(int window_sequence, BsBitStream *gc_streamCh)
{
    int bd, wd, ad;
    unsigned long   max_band, natks, ltmp;
    int loc;

    loc = gc_streamCh->currentBit;
    max_band = getbits(NBANDSBITS);
    BsPutBit(gc_streamCh, max_band, NBANDSBITS);/*   0 < max_band <= 3 */

    switch (window_sequence) {
    case ONLY_LONG_SEQUENCE:
        for (bd = 1; bd <= max_band; bd++) {
            for (wd = 0; wd < 1; wd++) {
                natks = getbits(NATKSBITS);
                BsPutBit(gc_streamCh, natks, NATKSBITS);
                for (ad = 0; ad < natks; ad++) {
                    ltmp = getbits(IDGAINBITS);
                    BsPutBit(gc_streamCh, ltmp, IDGAINBITS);
                    ltmp = getbits(ATKLOCBITS);
                    BsPutBit(gc_streamCh, ltmp, ATKLOCBITS);
                }
            }
        }
        break;
    case LONG_START_SEQUENCE:
        for (bd = 1; bd <= max_band; bd++) {
            for (wd = 0; wd < 2; wd++) {
                natks = getbits(NATKSBITS);
                BsPutBit(gc_streamCh, natks, NATKSBITS);
                for (ad = 0; ad < natks; ad++) {
                    ltmp = getbits(IDGAINBITS);
                    BsPutBit(gc_streamCh, ltmp, IDGAINBITS);
                    if (wd == 0) {
                        ltmp = getbits(ATKLOCBITS_START_A);
                        BsPutBit(gc_streamCh, ltmp, ATKLOCBITS_START_A);
                    }
                    else {
                        ltmp = getbits(ATKLOCBITS_START_B);
                        BsPutBit(gc_streamCh, ltmp, ATKLOCBITS_START_B);
                    }
                }
            }
        }
        break;
    case EIGHT_SHORT_SEQUENCE:
        for (bd = 1; bd <= max_band; bd++) {
            for (wd = 0; wd < 8; wd++) {
                natks = getbits(NATKSBITS);
                BsPutBit(gc_streamCh, natks, NATKSBITS);
                for (ad = 0; ad < natks; ad++) {
                    ltmp = getbits(IDGAINBITS);
                    BsPutBit(gc_streamCh, ltmp, IDGAINBITS);
                    ltmp = getbits(ATKLOCBITS_SHORT);
                    BsPutBit(gc_streamCh, ltmp, ATKLOCBITS_SHORT);
                }
            }
        }
        break;
    case LONG_STOP_SEQUENCE:
        for (bd = 1; bd <= max_band; bd++) {
            for (wd = 0; wd < 2; wd++) {
                natks = getbits(NATKSBITS);
                BsPutBit(gc_streamCh, natks, NATKSBITS);
                for (ad = 0; ad < natks; ad++) {
                    ltmp = getbits(IDGAINBITS);
                    BsPutBit(gc_streamCh, ltmp, IDGAINBITS);
                    if (wd == 0) {
                        ltmp = getbits(ATKLOCBITS_STOP_A);
                        BsPutBit(gc_streamCh, ltmp, ATKLOCBITS_STOP_A);
                    }
                    else {
                        ltmp = getbits(ATKLOCBITS_STOP_B);
                        BsPutBit(gc_streamCh, ltmp, ATKLOCBITS_STOP_B);
                    }
                }
            }
        }
        break;
    default:
        return  -1;
    }
    return  gc_streamCh->currentBit - loc;
}

#endif

