/****************************************************************************** 
 *
 *  mixview - X Window System based soundfile editor and processor
 *
 *  Copyright 1989 by Douglas Scott
 *
 *  Author:     Douglas Scott 
 *  Date:       May 1, 1989
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The author makes no representations about
 *  the suitability of this software for any purpose.  
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/
/* 
	mixview.c 
*/

#include "main.h"
#include "dialog.h"
#include "windows.h"
#include <X11/cursorfont.h>
#include <math.h>
#include "debug.h"

Window canvas_win, mbar_win; 
Display *dpy;
XFontStruct *mbar_fs, *canvas_fs, *scale_fs;
char large_font[128], tiny_font[128], canvas_font[128];
Pixmap plot_pm, mbar_pm, vscale_pm;
GC edit_gc, dbase_gc, plot_gc, mbar_gc, vscale_gc, hscale_gc;
Canvas *canvas, *menu_bar;
Cursor main_curs, arrow_curs, wait_curs, mbar_curs, draw_curs, cross_curs; 

extern Window mixwin, cwin, mwin, lwin;
extern Window lcwin, lpwin, pframe;
extern Mv_panel *ctrl_panel, *view_panel;

char *progname, **gargv; 
int gargc;
unsigned long myfg, mybg;

FLAG line, box;           /* flags for hairline-drawn, box-drawn, */
FLAG amps;                /* amplitude printed,                   */
FLAG is_buff, yesbuff;    /* cut buffer status,	buffer filled	  */
FLAG insert, region;      /* insert-point-set & region-selected,  */
FLAG chmode, chscale;	  /* cursor mode shift & vert scale shift */
FLAG hairmode;    	  /* crosshair mode			  */
FLAG rdrline, rdrbox; 	  /* flags for redraw after screen shift  */
FLAG is_open, is_new;	  /* soundfile open, soundfile is new     */
FLAG is_saved = 1, is_displayed = 0;
FLAG is_owner, read_only;  /* cut property is owned		  */
FLAG durflag, skipflag, is_visible = 0, is_blocked = 0;
extern FLAG ismapped;
extern int xend;
#if defined(NeXT) || defined(SPARC_DACS)
extern int use_native;
#endif
extern double last_edstart, last_edend;	/* storage for recall */
double ampfac = 1.0;
int xloc = 0, yloc = 0;
int nheaders = 7;
double skiptime, durtime;
int playtag;

sf_struct view, *v;
ebuf_struct edit, *e;
cbuf_struct cut, *c;

void stop_play();
int stop_in_prog = 0;
FLAG start_set = TRUE;

main(argc, argv)
int argc;
char *argv[];
{
	int o, badopt = 0;
	extern char *optarg;
	extern int optind;
	static char dpy_name[16];
	char *font;

        /* set global pointer-to-structure for display information */

        v = &view;
        e = &edit;
	c = &cut;

	strcpy(dpy_name, XDisplayName(""));

	/* Parse some options: */

	while ((o = getopt(argc,argv,"D:s:d:")) != EOF) {
		switch(o) {
			case 'D':
				strcpy(dpy_name, optarg);	
				break;
			case 'd':
				durtime = atof(optarg);
				durflag = 1;
				break;
			case 's':
				skiptime = atof(optarg);
				skipflag = 1;
				break;
			default:
				badopt++;
				break;
		}
	}
        if(argc % 2){
		strcpy(v->sfname, "untitled");
		is_new = 1;
	}
	else if ((argc==optind && (durflag || skipflag)) || badopt) {
		fprintf(stderr,
		  "Usage: mixview [-D DISPLAY][-s SKIP -d DUR soundfile]\n");
		exit(0);
	}
        else strcpy(v->sfname, argv[optind]);

#if defined(debug) && defined(NeXT) /* debugging for calls to calloc */
	malloc_debug(4);
#endif
#if defined(debug) && defined(sun) && defined(sparc)
	malloc_debug(1);
#endif
	/* create global versions of args */

	gargc = argc;
	gargv = argv;

	/* open main display */

        if ((dpy= XOpenDisplay(dpy_name)) == NULL) {
                fprintf(stderr, "%s: can't open display %s\n",
				argv[0], dpy_name);
                exit(1);
        }

	fprintf(stderr, "Welcome to mixview.  Setting up...\n");


	/* load all fonts needed */

	if((font = XGetDefault(dpy, argv[0], "MenuBarFont")) == CNULL)
		strcpy(large_font, LARGE_FONT);
	else strncpy(large_font, font, 127);
        if ((mbar_fs = XLoadQueryFont(dpy, large_font)) == NULL) {
                fprintf(stderr, "%s: display %s doesn't know font %s\n",
			      argv[0], XDisplayString(dpy), large_font);
                exit(1);
        }

	if((font = XGetDefault(dpy, argv[0], "CanvasFont")) == CNULL)
		strcpy(canvas_font, CANVAS_FONT);
	else strncpy(canvas_font, font, 127);
        if ((canvas_fs= XLoadQueryFont(dpy, canvas_font)) == NULL) {
                fprintf(stderr, "%s: display %s doesn't know font %s\n",
			      argv[0], XDisplayString(dpy), canvas_font);
                exit(1);
        }

	if((font = XGetDefault(dpy, argv[0], "ScaleFont")) == CNULL)
		strcpy(tiny_font, TINY_FONT);
	else strncpy(tiny_font, font, 127);
        if ((scale_fs= XLoadQueryFont(dpy, tiny_font)) == NULL) {
                fprintf(stderr, "%s: display %s doesn't know font %s\n",
                                       argv[0], XDisplayString(dpy), tiny_font);
                exit(1);
        }

	/* load all cursors for windows */

        if ((main_curs= XCreateFontCursor(dpy, MAIN_CURS)) == 0) {
                fprintf(stderr, "%s: display %s doesn't know cursorfont %d\n",
                                       argv[0], XDisplayString(dpy), MAIN_CURS);
                exit(1);
        }
        if ((arrow_curs= XCreateFontCursor(dpy, ARROW_CURS)) == 0) {
                fprintf(stderr, "%s: display %s doesn't know cursorfont %d\n",
                                     argv[0], XDisplayString(dpy), ARROW_CURS);
                exit(1);
        }
        if ((wait_curs= XCreateFontCursor(dpy, WAIT_CURS)) == 0) {
                fprintf(stderr, "%s: display %s doesn't know cursorfont %d\n",
                                       argv[0], XDisplayString(dpy), WAIT_CURS);
                exit(1);
        }
        if ((mbar_curs= XCreateFontCursor(dpy, MENU_CURS)) == 0) {
                fprintf(stderr, "%s: display %s doesn't know cursorfont %d\n",
                                       argv[0], XDisplayString(dpy), MENU_CURS);
                exit(1);
        }
        if ((draw_curs= XCreateFontCursor(dpy, DRAW_CURS)) == 0) {
                fprintf(stderr, "%s: display %s doesn't know cursorfont %d\n",
                                       argv[0], XDisplayString(dpy), DRAW_CURS);
                exit(1);
        }
        if ((cross_curs= XCreateFontCursor(dpy, CROSS_CURS)) == 0) {
                fprintf(stderr, "%s: display %s doesn't know cursorfont %d\n",
                                       argv[0], XDisplayString(dpy), CROSS_CURS);
                exit(1);
        }

        progname = *argv;

        mybg = WhitePixel(dpy, DefaultScreen(dpy));
        myfg = BlackPixel(dpy, DefaultScreen(dpy));

/**************************************************************************/

        set_defaults();     /* initialize the sf_struct values & global flags */

	setup_windows();	/* create all framing windows */

        /* setup panels and canvas */

	create_canvas();
        setup_canvas();     
	setup_menus();
        setup_mbar();
        setup_control();
        setup_view();
	setup_curve_window();
	setup_dialogs();
	setup_mesg();

        open_sfile(v->sfname);		/* open soundfile for r/w */

        XMapWindow(dpy, mixwin);	/* map main windows */
        XMapWindow(dpy, pframe);	
        XMapSubwindows(dpy, mixwin);
        XMapSubwindows(dpy, pframe);	
	get_events();		/* main event loop */
	return 0;
}

/***************************************************************************/
/***************************************************************************/

/* The routine to resize the inner windows if the frame is resized */

resize_select(event)
XConfigureEvent *event;
{
	XWindowAttributes xwa;
	unsigned new_width, new_height;
	XEvent evt;
	/* compress events before calling routine */

	while(XEventsQueued(event->display, QueuedAfterReading) > 0)
	{
		XEvent ahead;
		XPeekEvent(event->display, &ahead);
		if(ahead.type != ConfigureNotify) break;
		XNextEvent(event->display, &evt);
		event = (XConfigureEvent *) &evt;
	}
	if(event->window == mixwin)  {
		new_width = event->width;	/* new dimensions for frame */
		new_height = event->height;
	/*	if((new_width == v->width)&&(new_height == v->height)) return;*/
		XResizeWindow(dpy,cwin,new_width, new_height-MFRAME_HEIGHT);
		XResizeWindow(dpy,mwin,new_width,MFRAME_HEIGHT); /* menu bar */

		/* get new screen dimensions */

		XGetWindowAttributes(dpy, cwin, &xwa);
		v->width = xwa.width;
		v->height = xwa.height;
		canvas_win = *((Window *) canvas_get(canvas, LXC_WINDOW));
		plot_pm = *((Pixmap *) canvas_get(canvas, LXC_PIXMAP));
		chscale = 1;
		draw_menubar();
		if(!is_displayed) { /* if wave not currently displayed, do it */
			display_it(1);
			is_displayed = 1;
		}
	}
}

open_sfile(name)	/* opens soundfile for r/w & sets some constants */
char *name;
{
        if(read_sf(name, skiptime, durtime) < 1) {
		is_open = 0;
		is_saved = 1;
		strcpy(v->sfname, "none");
		set_frame_name(v->sfname);
		return;
	}
	is_open = 1;
	chscale = 1;		/* flag for setting vertical scaling */
	reset_screen_time();
	canvas_set(canvas, LXC_ACTIVE, TRUE, LXC_NULL);
	view_panel->active(view_panel);
	ctrl_panel->active(ctrl_panel);
	set_frame_name(v->sfname);
	set_view_display();
	set_file_menu();
}

close_sfile()		/* closes current soundfile and resets menus, etc. */
{
	close(v->sffd);
	cfree(v->sfbuff);
	v->sfdur = 0.0;
	v->peakamp = 1.0;
	is_open = is_new = 0;
	is_saved = 1;
	clear_screen();
	insert = region = 0;
	set_screen_times(1);
	set_menus();
	set_file_menu();
	canvas_set(canvas, LXC_ACTIVE, FALSE, LXC_NULL);
	view_panel->inactive(view_panel);
	ctrl_panel->inactive(ctrl_panel);
}

clear_screen()		/* resets all display flags and clears the pixrect */
{

        line = amps = box = 0;
        canvas_clear(canvas);
}

display_it(readit)	/* calls the waveform display routines */
FLAG readit;
{
	if(!is_open) return;
        plot_pm = *((Pixmap *) canvas_get(canvas, LXC_PIXMAP));
	block_events();
        clear_screen();
        plot_sf(ampfac, readit);  /* draw image into memory pixrect */
	draw_scale();	          /* draw in the time scale */
        canvas_flush(canvas);     /* dump to screen */

        /* redraw insert-line and/or edit box if necessary */

        if(insert & rdrline)
	{
	 	draw_hair(xloc, yloc);
	 	rdrline = 0;
	}
        if(region & rdrbox)
        {
                draw_hair(xloc, yloc);
                draw_box(xend, yloc);
		rdrbox = 0;
        }
	unblock_events();
}

long
Ngrains()
{
	long ngrains = (v->srate/v->grainsize)*v->sfdur;
	ngrains = (ngrains==0)? 1 : ngrains;
	return ngrains;
}

int
Scroll_min()
{
	int scroll_min = 0.5 + 100.0*(v->width-1)/Ngrains();
	scroll_min = (scroll_min < 100) ? scroll_min : 100;
	if(v->sfdur == 0.0) scroll_min = 0;
	return scroll_min;
}

double
get_grainsize(p)  /* calculates current grainsize from slider setting */
Mv_panel *p;
{
        double maxval= *((int *) panelitem_get(p->getPanel(p),
					p->getItem(p, "slider"),
					LXPSLIDER_MAXVAL));
	double factor, grain;
	double minfact = -1.50515;	/* log base 10 of 1/32 */
	double maxfact = 3.91339;	/* log base 10 of 8192 */
        int val= *((int *) panelitem_get(p->getPanel(p),
					p->getItem(p, "slider"),
					LXPSLIDER_VALUE));
	factor = (val / maxval) * (maxfact - minfact);
	grain = pow(10.0, (minfact + factor));
	if(grain >= 1.0) grain = ROUND(grain);	/* truncate when > 1 */
	return grain;
}

double	/* sets slider based on current grainsize after checking bounds */
set_grainsize(grainsize)
double grainsize;
{
        int maxval= *((int *) panelitem_get(ctrl_panel->getPanel(ctrl_panel),
			ctrl_panel->getItem(ctrl_panel, "slider"),
					LXPSLIDER_MAXVAL));
        int minval= *((int *) panelitem_get(ctrl_panel->getPanel(ctrl_panel),
			ctrl_panel->getItem(ctrl_panel, "slider"),
					LXPSLIDER_MINVAL));
	double minfact = -1.50515;	/* log base 10 of 1/32 */
	double maxfact = 3.91339;	/* log base 10 of 8192 */
	int val;

	val = ((log10(grainsize) - minfact)/(maxfact-minfact)) * maxval;
	val = (val < minval) ? minval : (val > maxval) ? maxval : val;
        ctrl_panel->setItem(ctrl_panel, "slider", LXPSLIDER_VALUE, val);
	v->grainsize = (get_grainsize(ctrl_panel));
        v->secgrain = v->grainsize / v->srate;	/* set this here for use */
	return v->grainsize;
}

/***************************************************************************/

setup_canvas()		/* initializes the screen canvas */
{
        unsigned long g_mask;
	unsigned depth;
	long e_mask;
        XGCValues gcv;
        canvas_win = *((Window *) canvas_get(canvas, LXC_WINDOW));
        plot_pm = *((Pixmap *) canvas_get(canvas, LXC_PIXMAP));

	/* Select which events the screen canvas is going to acknowledge */

        e_mask = ExposureMask | ButtonPressMask | ButtonReleaseMask |
			ButtonMotionMask | KeyPressMask | PropertyChangeMask;
        XSelectInput(dpy, canvas_win, e_mask);

	/* GCs for drawing hairs, time values, etc */

        gcv.function = GXxor; 	/* DAS test value */
        gcv.foreground = myfg ^ mybg;
        gcv.background = mybg;
        gcv.font = canvas_fs->fid;	/* small font for time display, etc. */
        g_mask = GCFunction|GCFont|GCForeground|GCBackground;
        edit_gc = XCreateGC(dpy, canvas_win, g_mask, &gcv);
	XSetPlaneMask(dpy, edit_gc, myfg^mybg);
        gcv.font = scale_fs->fid;	/* tiny font for horiz. scale */
        hscale_gc = XCreateGC(dpy, canvas_win, g_mask, &gcv);
	XSetPlaneMask(dpy, hscale_gc, myfg^mybg);

        /* GC for drawing waveform */

        gcv.foreground = myfg;
        gcv.font = scale_fs->fid;
        plot_gc = XCreateGC(dpy, canvas_win, g_mask, &gcv); 
        XSetFunction(dpy, plot_gc, GXcopy);

        /* the GC for drawing baseline (dotted) */

        dbase_gc = XCreateGC(dpy, canvas_win, g_mask, &gcv); 
        XSetLineAttributes(dpy,dbase_gc,0,LineOnOffDash,CapButt,JoinMiter);
        XSetFunction(dpy, dbase_gc, GXxor);

	/* GC and pixmap for vertical scale */

        gcv.font = scale_fs->fid;
	depth = DefaultDepth(dpy, DefaultScreen(dpy));
	vscale_pm = XCreatePixmap(dpy, canvas_win, 
				VSPM_WIDTH, VSPM_HEIGHT, depth);
        vscale_gc = XCreateGC(dpy, vscale_pm, g_mask, &gcv);
	XSetPlaneMask(dpy, vscale_gc, myfg^mybg);
}

create_canvas()
{
	/* the canvas for the view screen */

        if ((canvas= canvas_create(progname, dpy, cwin,
			LXC_CCURSOR, main_curs,
			LXC_MODE, LXC_DYNAMICPRS,
                        LXC_NULL)) == (Canvas *) NULL)
                exit(-1);
}

setup_mbar()
{
	unsigned long g_mask;
	long e_mask;
        XGCValues gcv;
	/* the canvas for the menu bar */

        if ((menu_bar= canvas_create(progname, dpy, mwin,
			LXC_CCURSOR, mbar_curs,
			LXC_MODE, LXC_DYNAMICPRS,
                        LXC_NULL)) == (Canvas *) NULL)
                exit(-1);
        mbar_win = *((Window *) canvas_get(menu_bar, LXC_WINDOW));
        mbar_pm = *((Pixmap *) canvas_get(menu_bar, LXC_PIXMAP));

	/* Select which events the menu bar is going to acknowledge */

        e_mask = ExposureMask | ButtonPressMask | ButtonReleaseMask;
        XSelectInput(dpy, mbar_win, e_mask);

	/* GC for menu bar */

        g_mask = GCFunction | GCFont | GCForeground | GCBackground;
	gcv.function = GXcopy;
        gcv.font = mbar_fs->fid;
	gcv.foreground = myfg;
	gcv.background = mybg;
        mbar_gc = XCreateGC(dpy, mbar_win, g_mask, &gcv);
}

draw_menubar()
{
	static
	char *mbar_headers[]= {
		"FILE",
       		"EDIT",
        	"ALTER",
        	"FILTER",
        	"OPTIONS",
		"HEADER",
        	"ANALYSIS",
        	0
	};
	int loc, offset, len, i;
	int frac = v->width/(nheaders);
	loc = frac/2;

        mbar_pm = *((Pixmap *) canvas_get(menu_bar, LXC_PIXMAP));
	canvas_clear(menu_bar);
	for(i = 0; i < nheaders; i++)
	{
		len = strlen(mbar_headers[i]);
		offset = XTextWidth(mbar_fs, mbar_headers[i], len)*0.7;
		XDrawString(dpy, mbar_pm, mbar_gc, loc-offset, 20, 
					mbar_headers[i], len);
		loc += frac;
	}
	canvas_flush(menu_bar);
}
/***************************************************************************/
/* Routines called by canvas items                                         */
/***************************************************************************/

set_insert_point(x, y)
int x, y;
{
        v->itime = screen_time(x);
        insert = 1; 
	set_menus();
}

set_region(x, y)                /* set the edit start & end times */
int x, y;
{
	int xpos;
        if(!insert) set_insert_point(x, y); /* If insert point not yet set */

	xpos = (x < xloc)? xloc : x;        /* If end pt. before start pt. */
        c->edstart = v->itime;

	/* end time is set to END TIME of last highlighted grain */

        c->edend = screen_time(xpos + 1);
        region = 1; 
        set_view_display();
	set_menus();
}

/**************************************************************************/
/* Routines called by view_panel items                                      */
/**************************************************************************/

void
scroll_it(p, pi) /* The routine for scrolling through the sfile */
Panel *p;
Panel_item *pi;
{
        int loc;

        /* get current value of scrollbar and reset endtime of screen */

        loc = *((int *) panelitem_get(p, pi, LXPSLIDER_VALUE));
	v->egrain = ((double)loc/100.0) * Ngrains();
	v->sgrain = v->egrain - v->width + 1;
        set_screen_times(1); 	/* check and set start grain and time */

        if(insert) set_line_loc();
        if(region) set_box_loc();

        set_view_display();
        display_it(1);
}

page_it(frac) /* routine for paging through the sfile */
double frac;
{
        /* change end time of screen and check for limits */

        int shift = v->width * frac; 	/* no. of screen grains */ 

        v->egrain += shift;
	v->sgrain = v->egrain - v->width + 1;
        set_screen_times(1);

        if(insert) set_line_loc();
        if(region) set_box_loc();

        set_view_display();
        display_it(1);
}

void
qpage_back(p, pi) /* The routine for quarter-paging through the sfile */
Panel *p;
Panel_item *pi;
{
	page_it(-0.25);
}

void
qpage_foward(p, pi) /* The routine for quarter-paging through the sfile */
Panel *p;
Panel_item *pi;
{
	page_it(0.25);
}

set_screen_times(flag) /* sets times after checking scrollbar location */
FLAG flag;
{
        int loc, minimum = Scroll_min();
	FLAG inflag = 1;
	if(v->sgrain < 0)		   /* if underscrolled */
	{
		v->sgrain = 0;
		v->egrain = v->width - 1;
                view_panel->setItem(view_panel, "scroll bar", LXPSLIDER_VALUE,
						 	 minimum);
                inflag = 0;
	}
        else if(v->egrain < (v->width-1))  /* if less than one full page */
        {
                v->egrain = (v->width-1);
                view_panel->setItem(view_panel, "scroll bar", LXPSLIDER_VALUE,
						 	 minimum);
                inflag = 0;
        }
	/* if no problems with endgrain and scroll-set flag is TRUE */
        if(flag & inflag) {
		loc = 0.5 + (v->egrain/(double) Ngrains()) * 100.0;
       		loc = (loc > 100) ? 100 : (loc < minimum) ? minimum : loc;
                view_panel->setItem(view_panel, "scroll bar", LXPSLIDER_VALUE,
						 	 loc);
	}

	/* now set times for display purposes */

	v->stime = v->sgrain * v->secgrain;
        v->etime = (v->sgrain + (v->width-1)) * v->secgrain;
}

reset_screen_time()	/* resets display to beginning of file */
{
	v->sgrain = 0;
	v->egrain = v->width - 1;
	v->stime = 0;
	v->itime = 0;
	set_screen_times(1);
}

void
go_to(p, pi)  /* Routine for jumping to specific locations in the sfile */
Panel *p;
Panel_item *pi;
{
        char *string;
	double time;

        /* get new goto time from go_to string */

	Panel_item *goto_time = view_panel->getItem(view_panel, "time:");
        string = (char *) panelitem_get(p, goto_time, LXPTEXT_VALUE);
	if(strlen(string) == 0) time = v->itime;
        else time = atof(string);

	go_to_time(time);
}

void
recall_edit(m, mi)	/* this is called by the canvas_menu */
Menu *m;
Menu_item *mi;
{
	Time time = *((Time *) menuitem_get(mi, LXMI_TIMESTAMP));
	if(box) undraw_box();
	c->edstart = v->stime = last_edstart;
	c->edend = last_edend;
	v->sgrain = v->stime/v->secgrain;
	v->egrain = v->sgrain + v->width - 1;
	set_screen_times(1);
	clear_events();
	display_it(1);
	xloc = 0;
	draw_hair(xloc, yloc);
	set_insert_point(xloc, yloc);
	xend = screen_loc(last_edend) - 1;
	xend = (xend > (v->width-1)) ? v->width-1 : xend;
	XSetSelectionOwner(dpy, XA_PRIMARY, canvas_win, time);
	is_owner = 1;
	draw_box(xend, yloc);
	region = 1;
	set_view_display();
	set_menus();
}

void
select_all(m, mi)	/* this is called by the canvas menu */
Menu *m;
Menu_item *mi;
{
	v->itime = c->edstart = 0.0;
	c->edend = v->sfdur;
	insert = region = 1;
	set_line_loc();
	set_box_loc();
	set_view_display();
	set_menus();
	display_it(0);
}

void
set_cursor_line(m, mi)	/* sets cursor to be vertical line */
Menu *m;
Menu_item *mi;
{
        hairmode = 0;
        
        if(hairmode & line) amps = 1;       /* flag on for redraw */

        /* Draw over to negate horiz. line or add new one */

        if(line & !box)
        {
		chmode = 1;
		draw_hair(xloc, yloc);
		chmode = 0;
        }
        else if(box)
        {
		chmode = 1;
		draw_hair(xend, yloc);
		chmode = 0;
        }
	if(!hairmode) amps = 0;  /* flag off if amp print erased & off */
	set_item_state(m, "Vert. line", LXMI_INACTIVE);
	set_item_state(m, "Crosshairs", LXMI_ACTIVE);
}

void
set_cursor_cross(m, mi)	/* sets cursor to be crosshairs */
Menu *m;
Menu_item *mi;
{
        hairmode = 1;
        
        if(hairmode & line) amps = 1;       /* flag on for redraw */

        /* Draw over to negate horiz. line or add new one */

        if(line & !box)
        {
		chmode = 1;
		draw_hair(xloc, yloc);
		chmode = 0;
        }
        else if(box)
        {
		chmode = 1;
		draw_hair(xend, yloc);
		chmode = 0;
        }
	if(!hairmode) amps = 0;  /* flag off if amp print erased & off */
	set_item_state(m, "Vert. line", LXMI_ACTIVE);
	set_item_state(m, "Crosshairs", LXMI_INACTIVE);
}

void
reset_screen(p, pi)	/* reset all edit flags and clear display */
Panel *p;
Panel_item *pi;
{
	if(!is_open) return;	/* no action if file closed */
	c->edstart = c->edend = 0.00;
        insert = region = 0;
	ampfac = 1.0;		/* set vertical scale back to default */
	chscale = 1;
        clear_screen();
	set_screen_times(1);
	set_view_display();
	set_menus();
	is_displayed = 0;	/* reset resize routine's flag */
        display_it(1);
}

void
resize_canvas(p, pi)
Panel *p;
Panel_item *pi;
{
	XWindowAttributes xwa;

        XGetWindowAttributes(dpy, cwin, &xwa);
        v->width = xwa.width;
        v->height = xwa.height;
        canvas_win = *((Window *) canvas_get(canvas, LXC_WINDOW));
        plot_pm = *((Pixmap *) canvas_get(canvas, LXC_PIXMAP));
	draw_menubar();
	reset_screen(p, pi);
}

set_view_display()	/* sets all time labels for the screen */
{
        char string[16], string2[16];
	sprintf(string, "%10.5f", v->stime);
	sprintf(string2, "%10.5f", v->etime);
	view_panel->setItem(view_panel, "start time", LXPI_STRING, string);
	view_panel->setItem(view_panel, "end time", LXPI_STRING, string2);
	if(region) {
		sprintf(string, "%10.5f", c->edstart);
		sprintf(string2, "%10.5f", c->edend);
	}
	else {
		sprintf(string, "%10.5f", 0.0);
		sprintf(string2, "%10.5f", 0.0);
	}
	view_panel->setItem(view_panel, "edit start time", LXPI_STRING, string);
	view_panel->setItem(view_panel, "edit end time", LXPI_STRING, string2);
	if(is_visible) {
		view_panel->redisplay(view_panel, "start time");
		view_panel->redisplay(view_panel, "end time"); 
 		view_panel->redisplay(view_panel, "edit start time"); 
		view_panel->redisplay(view_panel, "edit end time"); 
	}
}

/**************************************************************************/
/* Routines called by ctrl_panel items                                 */
/**************************************************************************/

void
set_grain(p, pi)
Panel *p;
Panel_item *pi;
{
        v->grainsize = get_grainsize(ctrl_panel);
	v->secgrain = v->grainsize / v->srate;

        /* set hairloc and time to same place on screen */

        if((insert&line) && v->itime >= v->stime && v->itime < v->etime)
	{
		v->sgrain = (v->itime/v->secgrain) - xloc;
		v->egrain = v->sgrain + v->width - 1;
	}
	else	/* set to same start time on screen */ 
	{
		v->sgrain = v->stime / v->secgrain;
		v->egrain = v->sgrain + v->width - 1;
	}
        set_screen_times(1); /* reset scrollbar & check time values for range */
        if(insert) set_line_loc();
        if(region) set_box_loc();
	set_view_display();
        display_it(1);
}


void
zoom(p, pi)
Panel *p;
Panel_item *pi;
{
	static double oldgrainsize, grainsize;
	static double old_stime, old_etime;
        double grainfac;

        if(region) /* zoom in to display only selected region */
        {
                /* grainsize shifted by relative dimensions of old/new screen */

		oldgrainsize = v->grainsize;	/* very old = current */
                grainfac = (v->width-1)/((c->edend-c->edstart)/v->secgrain);
		set_grainsize(v->grainsize/grainfac); /* check & set */
		grainsize = v->grainsize;	/* old = new */
		old_stime = v->stime;		/* store old screen times */
		old_etime = v->etime;
                v->etime = c->edend;		/* get new screen times */
                v->stime = c->edstart;
                v->egrain = v->etime/v->secgrain;
                v->sgrain = v->stime/v->secgrain;
                set_screen_times(1);		/* check screen boundries */
                region = box = line = insert = 0; /* reset for next selection */
		set_menus();
		set_view_display();    
                display_it(1);
        }
	/* if no change, set to old vals (unzoom) */

	else if(v->grainsize == grainsize) {
		set_grainsize(oldgrainsize);
		oldgrainsize = grainsize;
		grainsize = v->grainsize;
		c->edend = v->etime;
		c->edstart = v->stime;
		region = 1;
		v->stime = old_stime;
		v->etime = old_etime;
		set_grain(p, pi);
	}
        /* otherwise just update grainsize and scrollbar */

        else {
		old_stime = v->stime;
		old_etime = v->etime;
		oldgrainsize = grainsize;
		grainsize = v->grainsize;
		set_grain(p, pi);
	}
}

void
xzoom(p, pi)
Panel *p;
Panel_item *pi;
{
        set_grainsize(v->grainsize/2.0);
	set_grain(p, pi);
}

void
xunzoom(p, pi)
Panel *p;
Panel_item *pi;
{
        set_grainsize(v->grainsize*2.0);
	set_grain(p, pi);
}

void
yzoom(p, pi)
Panel *p;
Panel_item *pi;
{
	ampfac *= 2.0;
        if(insert) set_line_loc();
        if(region) set_box_loc();
	chscale = 1;
	display_it(0);
}

void
yunzoom(p, pi)
Panel *p;
Panel_item *pi;
{
	ampfac /= 2.0;
        if(insert) set_line_loc();
        if(region) set_box_loc();
	chscale = 1;
	display_it(0);
}

#ifdef ADC_DACS
extern int use_adc;
#endif

void
play_it(p, pi)		/* do D to A conversion of selection */
Panel *p;
Panel_item *pi;
{ 
	int nboffset, rmainder;
	double endtime, starttime;
	if(stop_in_prog) return;
	if(!(box && region)) {		/* play screen if no region */
		starttime = v->stime;
		endtime = v->etime;
					/* play from hair if onscreen */
		if(insert && line) starttime = v->itime;
	}
	else {				/* play highlighted region */
		starttime = c->edstart;
		endtime = c->edend;
	}
	e->nchans = v->nchans;
	e->srate = v->srate;
	e->size = v->size;                /* sampsize in bytes    */
	e->is_float = v->is_float;
	e->peakamp = v->peakamp;
	if(starttime >= v->sfdur) {
		mv_alert("Start time is beyond EOF!");
		return;
	}
	endtime = (endtime > v->sfdur) ? v->sfdur : endtime;
	e->nsamps = (endtime - starttime) * v->srate;
	e->bufsize = e->nsamps * e->nchans * e->size;
	nboffset = starttime * v->srate * e->size;
	/* keep this on sample frame border       */
	if(rmainder = (nboffset % e->size)) nboffset -= rmainder;
	nboffset *= e->nchans;
	e->sfbuff = v->sfbuff + nboffset; /* ptr. to beg. of play */
	block_events();
	if(startPlaying(e) > 0) {
#ifdef NeXT
		/* this is done here since startPlaying doesnt always block */
		if(use_native) setStopButton();
#endif
	}
	else unblock_events();
#ifdef ADC_DACS
	if(use_adc) {
		waitForStop();
		setStartButton();
	}
#else
#ifdef SPARC_DACS
	unblock_events();
#endif
#endif /* ADC_DACS */
}

setStopButton()
{
	/* change play button to stop button and block input to screen */
	ctrl_panel->setItem(ctrl_panel, " Play ", LXPI_STRING, " Stop "); 
	ctrl_panel->setItem(ctrl_panel, " Play ", LXPI_PROC, stop_play);
	ctrl_panel->redisplay(ctrl_panel, " Play ");
	start_set = FALSE;
	unblock_events();
	block_menus();
	block_canvas();
}

setStartButton()
{
	if(start_set) return;
	ctrl_panel->setItem(ctrl_panel, " Play ", LXPI_STRING, " Play "); 
	ctrl_panel->setItem(ctrl_panel, " Play ", LXPI_PROC, play_it);
	ctrl_panel->redisplay(ctrl_panel, " Play ");
	unblock_menus();
	unblock_canvas();
/*	XSync(dpy, 0); */
	start_set = TRUE;
}

void
stop_play(p, pi)   /* called from either stop button or by play command end */
	Panel *p;
	Panel_item *pi;
{
	void stopPlaying();
	stop_in_prog++;
	stopPlaying();
	setStartButton();
	playtag = 0;
	stop_in_prog--;
}

void
quit(p, pi)
Panel *p;
Panel_item *pi;
{
	extern Menu *file_menu;
	if(is_open && !is_saved){
		switch(dialog->call(dialog, D_SBQUIT, v->sfname)){
		case YES:
			if(!read_only) save_file(file_menu, (Menu_item *) NULL);
			else saveto_file(file_menu, (Menu_item *) NULL);
			if(!is_saved) return; /* if operation cancelled */
			break;
		case NO:
			break;
		case CANCEL:
			return;
		}
	}
	XFlush(dpy);
        exit(0);
}

block_events()	/* freezes all input while computations are done */
{
	block_canvas();
	block_menus();
	ctrl_panel->block(ctrl_panel);
	view_panel->block(view_panel);
}

block_canvas()
{
	canvas_set(canvas, LXC_ACTIVE, FALSE, LXC_CCURSOR, wait_curs, LXC_NULL);
}

block_menus()
{
	canvas_set(menu_bar, LXC_ACTIVE, FALSE,
		LXC_CCURSOR, wait_curs, LXC_NULL);
}

unblock_events()	/* resume all regular event processing */
{
	unblock_canvas();
	unblock_menus();
	ctrl_panel->unBlock(ctrl_panel);
	view_panel->unBlock(view_panel);
}

unblock_canvas()
{
        long c_mask = ExposureMask | ButtonPressMask | ButtonReleaseMask |
			ButtonMotionMask | KeyPressMask | PropertyChangeMask;
	canvas_set(canvas, LXC_ACTIVE, TRUE, LXC_CCURSOR, main_curs, LXC_NULL);
        XSelectInput(dpy, canvas_win, c_mask);
}

unblock_menus()
{
        long m_mask = ExposureMask | ButtonPressMask | ButtonReleaseMask;
	canvas_set(menu_bar, LXC_ACTIVE, TRUE,
		LXC_CCURSOR, mbar_curs, LXC_NULL);
	XSelectInput(dpy, mbar_win, m_mask);
}

set_frame_name(name)
char *name;
{
	XStoreName(dpy, mixwin, name);
	XSetIconName(dpy, mixwin, name);
	XStoreName(dpy, pframe, name);
	XSetIconName(dpy, pframe, name);
}

faterr(msg)	/* for deathly error conditions */
char  *msg;
{
	fprintf(stderr,"%s\n",msg);
	exit(1);
}
#ifndef NeXT
extern char *sys_errlist[];
char *
strerror(errn)
int errn;
{
	return(sys_errlist[errn]);
}
#endif NeXT

#define MALLOC_MINSIZE 8

char *
mv_alloc(size)
	int size;
{
	char *ptr;
	/* round to minimum of 8 bytes */
	size = size > MALLOC_MINSIZE ? size : MALLOC_MINSIZE;
	if((ptr = (char *) calloc(1, size)) == (char *) NULL) {
		perror("mv_alloc");
#ifdef USE_ABORT
		abort();
#else
		exit(-1);
#endif
	}
	return ptr;
}
