/* 
 * xwave - an interactive audio player, recorder, editor 
 * for the XWindow System
 * 
 * Copyright (C) 1996 Kai Kollmorgen
 * (kkollmor@informatik.uni-rostock.de)
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <dirent.h>

#include <fcntl.h>
#include <unistd.h>

#include <sys/types.h>
#include <math.h>

#include <X11/Intrinsic.h>/* Intrinsics Definitions */
#include <X11/StringDefs.h>/* Standard Name-String definitions */
#include <X11/Shell.h>     /* Shell Definitions */

#include "types.h"
#include "xwave.h"
#include "xwave_widget.h"
#include "misc.h"
#include "graphics.h"
#include "sample_settings.h"
#include "audio_file.h"
#include "effects.h"

extern Main_Data *MD;
extern AppResources app_resources;

static void reverse_8(char *buffer,int channels,int length);
static void reverse_16(short *buffer,int channels,int length);
static void swap_8(char *buffer,int length);
static void swap_16(short *buffer,int length);
static void props_return(Sample_Return *sr);
static void resample_16(int inlength,int infreq,short *inbuf,
			int outlength,int outfreq,short *outbuf);
static void resample_8(int inlength,int infreq,byte *inbuf,
		       int outlength,int outfreq,byte *outbuf);
static void av_return(float mult1,float mult2);
static void ev_return(float fo_time[MAX_CHANNELS],
		      float fo_factor[MAX_CHANNELS],
		      float sup_amp[MAX_CHANNELS]);
static void ec_return(float fo_time[MAX_CHANNELS],
		      float fo_factor[MAX_CHANNELS]);


void reverse_8(char *buffer,int channels,int length)
{
    int i,j,hi,lo;
    static char valbuf[MAX_CHANNELS];
    
    if (channels>MAX_CHANNELS) return;
    
    for (i=0;i<length/2;i+=channels) {
	lo=i;
	hi=length-i-channels;
	j=0;
	while (j<channels) {
	    valbuf[j]=buffer[lo+j];
	    buffer[lo+j]=buffer[hi+j];
	    buffer[hi+j]=valbuf[j];
	    j++;
	}
    }
}

void reverse_16(short *buffer,int channels,int length)
{
    int i,j,hi,lo;
    static short valbuf[MAX_CHANNELS];
    
    if (channels>MAX_CHANNELS) return;
    
    for (i=0;i<length/2;i+=channels) {
	lo=i;
	hi=length-i-channels;
	j=0;
	while (j<channels) {
	    valbuf[j]=buffer[lo+j];
	    buffer[lo+j]=buffer[hi+j];
	    buffer[hi+j]=valbuf[j];
	    j++;
	}
    }
}

void reverse_call(Widget w, XtPointer client_data, XtPointer call_data)
{
    Main_Data *md=(Main_Data*) client_data;
    int length;
    char *buf=(char*)md->wd->buffer;
    
    if (!md->wd->inmem) {
	warn_popup(md->mw->form,app_resources.err_notimplement);
	return;
    }
    
    watch_cursor(True);
    
    length=md->wd->length;
    
    if (md->wd->ismark) {
	buf+=md->wd->markbeg*md->wd->bpspl;
	length=md->wd->marklength*md->wd->bpspl;
    }
    
    if (md->wd->inmem) {
	switch (md->wd->res) {
	 case 8:
	    reverse_8(buf,md->wd->channels,length);
	    break;
	 case 16:
	    reverse_16((short*)buf,md->wd->channels,length/2);
	    break;
	}
	md->cb->update=True;
	md->cb->modified=True;
	update_canvas(md,NULL);
	update_display(md);
    }
    watch_cursor(False);
}

void swap_8(char *buffer,int length)
{
    int i;
    char val;
    
    for (i=0;i<length;i+=2) {
	val=buffer[i];
	buffer[i]=buffer[i+1];
	buffer[i+1]=val;
    }
}

void swap_16(short *buffer,int length)
{
    int i;
    short val;
    
    for (i=0;i<length;i+=2) {
	val=buffer[i];
	buffer[i]=buffer[i+1];
	buffer[i+1]=val;
    }
}

void swap_call(Widget w, XtPointer client_data, XtPointer call_data)
{
    Main_Data *md=(Main_Data*) client_data;
    int length;
    char *buf=(char*)md->wd->buffer;
    
    
    if (md->wd->channels<2) return;
    
    if (!md->wd->inmem) {
	warn_popup(md->mw->form,app_resources.err_notimplement);
	return;
    }
    
    watch_cursor(True);
    
    length=md->wd->length;
    
    if (md->wd->ismark) {
	buf+=md->wd->markbeg*md->wd->bpspl;
	length=md->wd->marklength*md->wd->bpspl;
    }
    
    if (md->wd->inmem) {
	switch (md->wd->res) {
	 case 8:
	    swap_8(buf,length);
	    break;
	 case 16:
	    swap_16((short*) buf,md->wd->length/2);
	    break;
	}
	md->cb->update=True;
	md->cb->modified=True;
	update_canvas(md,NULL);
	update_display(md);
    }
    watch_cursor(False);
}

void pitch_call(Widget w, XtPointer client_data, XtPointer call_data)
{
}


void props_call(Widget w, XtPointer client_data, XtPointer call_data)
{
    Main_Data *md=(Main_Data*) client_data;
    
    if (!md->wd->inmem) {
	warn_popup(md->mw->form,app_resources.err_notimplement);
	return;
    }
    sample_dialog(props_return,md->wd);
}

void props_return(Sample_Return *sr)
{
    int i;
    Main_Data *md=MD;
    Wave_Data *wd=md->wd;
    
    watch_cursor(True);
    
    if (sr->freq!=wd->freq) {
	int newlength;
	char *newbuf,*sbufin=NULL,*sbufout=NULL;
	
	newlength=(int)((double)sr->freq/
			(double)wd->freq*(double)wd->length);
	
	if ((newbuf=malloc(newlength))==NULL) {
	    watch_cursor(False);
	    warn_popup(md->mw->form,app_resources.err_mem);
	    return;
	}
	
	if (wd->channels>1) {
	    if ((sbufin=malloc(wd->length/2))==NULL) {
		watch_cursor(False);
		warn_popup(md->mw->form,app_resources.err_mem);
		free(newbuf);
		return;
	    }
	    if ((sbufout=malloc(newlength/2))==NULL) {
		watch_cursor(False);
		warn_popup(md->mw->form,app_resources.err_mem);
		free(sbufin);
		free(newbuf);
		return;
	    }
	}
	
	switch (wd->res) {
	 case 8:
	    switch (wd->channels) {
	     case 1:
		resample_8(wd->length,wd->freq,wd->buffer,
			   newlength,sr->freq,(byte*) newbuf);
		break;
	     case 2: {
		 /* catch every 2nd sample (channel 1) */
		 for (i=0;i<wd->length/2;i++) sbufin[i]=wd->buffer[i<<1];
		 
		 resample_8(wd->length/2,wd->freq,sbufin,
			    newlength/2,sr->freq,sbufout);
		 for (i=0;i<newlength/2;i++) newbuf[i<<1]=sbufout[i];
		 
		 /* catch every 2nd sample (channel 2) */
		 for (i=0;i<wd->length/2;i++) 
		   sbufin[i]=wd->buffer[(i<<1)+1];
		 
		 resample_8(wd->length/2,wd->freq,(byte*)sbufin,
			    newlength/2,sr->freq,(byte*)sbufout);
		 for (i=0;i<newlength/2;i++) newbuf[(i<<1)+1]=sbufout[i];
		 
		 break;
	     }
	    }
	    break;
	 case 16:
	    switch (wd->channels) {
	     case 1:
		resample_16(wd->length/2,wd->freq,(short*) wd->buffer,
			    newlength/2,sr->freq,(short*)newbuf);
		break;
	     case 2: {
		 short *buf=(short*) wd->buffer;
		 short *inbuf=(short*) sbufin;
		 short *outbuf=(short*) sbufout;
		 short *nbuf=(short*) newbuf;
		 
		 /* catch every 2nd sample (channel 1) */
		 for (i=0;i<wd->length/4;i++) inbuf[i]=buf[i<<1];
		 
		 resample_16(wd->length/4,wd->freq,inbuf,
			     newlength/4,sr->freq,(short*)outbuf);
		 for (i=0;i<newlength/4;i++) nbuf[i<<1]=outbuf[i];
		 
		 /* catch every 2nd sample (channel 2) */
		 for (i=0;i<wd->length/4;i++) inbuf[i]=buf[(i<<1)+1];
		 resample_16(wd->length/4,wd->freq,(short*) inbuf,
			     newlength/4,sr->freq,(short*)outbuf);
		 for (i=0;i<newlength/4;i++) nbuf[(i<<1)+1]=outbuf[i];
		 
		 break;
	     }
	    }
	    break;
	} 
	
	if (wd->channels>1) {
	    free(sbufin);
	    free(sbufout);
	}
	
	free(wd->buffer);
	wd->buffer=(unsigned char*)newbuf;
	wd->length=newlength;
	wd->tlength=newlength/wd->bpspl;
	wd->freq=sr->freq;
	
	md->cb->modified=True;
	
    } /* resample */
    
    if (sr->res!=wd->res) {
	char *obuf=(char*)wd->buffer;
	char *newbuf;
	int newlength;
	
	newlength=(int)((float) sr->res/(float) wd->res * 
			(float) wd->length);
	
	if ((newbuf=malloc(newlength))==NULL) {
	    watch_cursor(False);
	    warn_popup(md->mw->form,app_resources.err_mem);
	    return;
	}
	
	switch (sr->res) {
	 case 8: {
#if defined (linux) || defined (FreeBSD) 
	     for (i=0;i<newlength;i++) newbuf[i]=obuf[(i<<1)+1]+128;
#elif defined (sgi) || defined (sun)
	     for (i=0;i<newlength;i++) newbuf[i]=obuf[(i<<1)]+128;
#endif
	     break;
	 }
	 case 16: {
#if defined (linux) || defined (FreeBSD) 
	     for (i=0;i<wd->length;i++) newbuf[(i<<1)+1]=obuf[i]-128;
#elif defined (sgi) || defined (sun)
	     for (i=0;i<wd->length;i++) newbuf[(i<<1)]=obuf[i]-128;
#endif
	     break;
	 }
	}
	
	free(wd->buffer);
	wd->buffer=(unsigned char*)newbuf;
	wd->length=newlength;
	wd->res=sr->res;
	wd->bpspl=(wd->res*wd->channels)/8;
	wd->tlength=newlength/wd->bpspl;
	md->cb->modified=True;
	
    } /* resolution */
    
    /* at the moment only 2 channels supported 
     * MONO to STEREO and vice versa 
     */
    if (sr->channels!=wd->channels) {
	char *newbuf;
	int newlength;
	
	if (sr->channels==1) newlength=wd->length/2;
	else newlength=wd->length*2;
	
	if ((newbuf=malloc(newlength))==NULL) {
	    watch_cursor(False);
	    warn_popup(md->mw->form,app_resources.err_mem);
	    return;
	}
	
	switch (sr->res) {
	 case 8: {
	     byte *obuf=wd->buffer;
	     
	     switch (sr->channels) {
	      case 1:
		 /* (channel_1 / 2) + (channel_2 / 2) */
		 for (i=0;i<newlength;i++) {
		     newbuf[i]=(obuf[i<<1]>>1) + (obuf[(i<<1)+1]>>1);
		 }
		 break;
	      case 2: 
		 for (i=0;i<wd->length;i++) { 
		     newbuf[i<<1]=obuf[i];
		     newbuf[(i<<1)+1]=obuf[i];
		 }
		 break;
	     }
	     break;
	 } /* 8 bit */
	 case 16: {
	     short *inbuf=(short*) wd->buffer;
	     short *outbuf=(short*) newbuf;
	     
	     switch (sr->channels) {
	      case 1:
		 /* (channel_1 / 2) + (channel_2 / 2) */
		 for (i=0;i<newlength/2;i++) {
		     outbuf[i]=(inbuf[i<<1]>>1) + (inbuf[(i<<1)+1]>>1);
		 }
		 break;
	      case 2: 
		 for (i=0;i<wd->length/2;i++) { 
		     outbuf[i<<1]=inbuf[i];
		     outbuf[(i<<1)+1]=inbuf[i];
		 }
		 break;
	     }
	     break;
	 } /* 16 bit */
	} /* switch (sr->res) */
	
	free(wd->buffer);
	wd->buffer=(unsigned char*)newbuf;
	wd->length=newlength;
	wd->channels=sr->channels;
	wd->bpspl=(wd->res*wd->channels)/8;
	wd->tlength=newlength/wd->bpspl;
	md->cb->modified=True;
    } /* channels */
    
    XtFree((char*) sr);
    
    if (md->cb->modified) {
	md->cb->update=True;
	update_canvas(md,NULL);
	update_display(md);
    }
    
    watch_cursor(False);
}

void resample_16(int inlength,int infreq,short *inbuf,
		 int outlength,int outfreq,short *outbuf)
{
    int lcmrate,inskip,outskip,intot,outtot;
    int len,done,val,last;
    
    lcmrate=lcm(infreq,outfreq);
    inskip=lcmrate/infreq;
    outskip=lcmrate/outfreq;
    intot=outtot=0;
    
    last=*inbuf++;
    *outbuf++=last;
    outtot+=outskip;
    while ((intot + inskip) <= outtot) intot += inskip;
    
    len = (inlength * inskip) / outskip;
    if (len > outlength) len = outlength;
    for(done=1; done < len; done++) {
	val=last;
	val+=((float)((*inbuf)-last)*((float)outtot-intot))/inskip;
	*outbuf = val;
	outbuf++;
	outtot += outskip;
	while ((intot + inskip) <= outtot) {
	    last = *inbuf++;
	    intot += inskip;
	}
    }
}

void resample_8(int inlength,int infreq,byte *inbuf,
		int outlength,int outfreq,byte *outbuf)
{
    int lcmrate,inskip,outskip,intot,outtot;
    int len,done,val,last;
    
    lcmrate=lcm(infreq,outfreq);
    inskip=lcmrate/infreq;
    outskip=lcmrate/outfreq;
    intot=outtot=0;
    
    last=*inbuf++;
    *outbuf++=last;
    outtot+=outskip;
    while ((intot + inskip) <= outtot) intot += inskip;
    
    len = (inlength * inskip) / outskip;
    if (len > outlength) len = outlength;
    for(done=1; done < len; done++) {
	val=last;
	val+=((float)((*inbuf)-last)*((float)outtot-intot))/inskip;
	*outbuf = val;
	outbuf++;
	outtot += outskip;
	while ((intot + inskip) <= outtot) {
	    last = *inbuf++;
	    intot += inskip;
	}
    }
}

void abs_vol_call(Widget w, XtPointer client_data, XtPointer call_data)
{
    Main_Data *md=(Main_Data*) client_data;
    
    if (!md->wd->inmem) {
	warn_popup(md->mw->form,app_resources.err_notimplement);
	return;
    }

    watch_cursor(True);

    find_peak(md->wd);
    
    watch_cursor(False);
    
    absvol_dialog(av_return,md);
}

void av_return(float mult1,float mult2)
{
    int i,new_val,max;
    Main_Data *md=MD;
    Wave_Data *wd=md->wd;
    int length=wd->length;
    char *buf=(char*)wd->buffer;
    
    watch_cursor(True);
    
    max=(int)pow(2.0,(float)wd->res)/2.0;
    
    if (wd->ismark) {
	buf+=wd->markbeg*wd->bpspl;
	length=wd->marklength*wd->bpspl;
    }
    
    switch (wd->res) {
     case 8: {
	 char val;
	 
	 switch (wd->channels) {
	  case 1:
	     for (i=0;i<length;i++) {
		 val=buf[i]^0x80;
		 new_val=(int)((float)val*mult1);
		 if (new_val>max) new_val=max;
		 if (new_val<-max) new_val=-max;
		 buf[i]=new_val^0x80;
	     }
	     break;
	  case 2:
	     for (i=0;i<length;) {
		 val=buf[i]^0x80;
		 new_val=(int)((float)val*mult1);
		 if (new_val>max) new_val=max;
		 if (new_val<-max) new_val=-max;
		 buf[i]=new_val^0x80;
		 i++;
		 val=buf[i]^0x80;
		 new_val=(int)((float)val*mult2);
		 if (new_val>max) new_val=max;
		 if (new_val<-max) new_val=-max;
		 buf[i]=new_val^0x80;
		 i++;
	     }
	     break;
	 }
	 break;
     }
     case 16: {
	 short *sbuf=(short*) buf,val;
	 
	 switch (md->wd->channels) {
	  case 1:
	     for (i=0;i<length/2;i++) {
		 val=sbuf[i];
		 new_val=(int)((float)val*mult1);
		 if (new_val>max-1) new_val=max-1;
		 if (new_val<-max) new_val=-max;
		 sbuf[i]=new_val;
	     }
	     break;
	  case 2:
	     for (i=0;i<length/2;) {
		 val=sbuf[i];
		 new_val=(int)((float)val*mult1);
		 if (new_val>max-1) new_val=max-1;
		 if (new_val<-max) new_val=-max;
		 sbuf[i]=new_val;
		 i++;
		 val=sbuf[i];
		 new_val=(int)((float)val*mult2);
		 if (new_val>max-1) new_val=max;
		 if (new_val<-max) new_val=-max;
		 sbuf[i]=new_val;
		 i++;
	     }
	     break;
	 }
	 break;
     }
    }
    
    md->cb->modified=True;
    md->cb->update=True;
    update_canvas(md,NULL);
    update_display(md);
    
    watch_cursor(False);
}

void env_vol_call(Widget w, XtPointer client_data, XtPointer call_data)
{
    Main_Data *md=(Main_Data*) client_data;

    if (!md->wd->inmem) {
	warn_popup(md->mw->form,app_resources.err_notimplement);
	return;
    }
    
    envvol_dialog(ev_return,md->wd);
}

void ev_return(float fo_time[MAX_CHANNELS],
	       float fo_factor[MAX_CHANNELS],
	       float sup_amp[MAX_CHANNELS])
{
    Main_Data *md=MD;
    Wave_Data *wd=md->wd;
    char *buf;
    float k;
    int i,j,value,env,time_wait,time_index,k_index,max,min_amp;
    int length=wd->length;
    
    watch_cursor(True);
    
    max=(int)pow(2.0,(float)wd->res)/2.0;
    
    switch(wd->res) {
     case 8: {
	 char val;
	 
	 for (j=0;j<wd->channels;j++) {
	     
	     buf=(char*) wd->buffer;
	     if (wd->ismark) {
		 buf+=wd->markbeg*wd->bpspl;
		 length=wd->marklength*wd->bpspl;
	     }
	     env=0;
	     time_index=0;
	     k_index=0;
	     k=1.0;
	     min_amp=(int) ((float)max*sup_amp[j]);
	     time_wait=(int) ((float)(wd->freq*fo_time[j])/1000.0 /*1ms*/);
	     
	     for (i=j;i<length;i+=wd->channels) {
		 val=buf[i]^0x80;
		 value=abs(val);
		 if (value>env) {
		     time_index=0;
		     env=value;
		 } else {
		     if (time_index<time_wait) time_index++;
		     else env=(int) ((float)env*fo_factor[j]);
		 }
		 if (env>min_amp) { 
		     k+=(1-fo_factor[j])*(((float)(max-1)/(float) env)-k);
		     k_index=0;
		 } else {
		     if (k_index<time_wait) k_index++; 
		     else k*=fo_factor[j];
		 }
		 if (value*k>max-1) k=((float)max-1)/((float)value);
		 val*=k;
		 buf[i]=val^0x80;
	     }
	 } /* for (j=0;j<wd->channels;j++) */
	 break;
     }
     case 16: {
	 short val,*buf16;
	 
	 for (j=0;j<wd->channels;j++) {
	     
	     buf=(char*) wd->buffer;
	     if (wd->ismark) {
		 buf+=wd->markbeg*wd->bpspl;
		 length=wd->marklength*wd->bpspl;
	     }
	     buf16=(short*)buf;
	     env=0;
	     time_index=0;
	     k_index=0;
	     k=1.0;
	     min_amp=(int) ((float)max*sup_amp[j]);
	     time_wait=(int) ((float)(wd->freq*fo_time[j])/1000.0 /*1ms*/);
	     
	     for (i=j;i<length/2;i+=wd->channels) {
		 val=buf16[i];
		 value=abs(val);
		 if (value>env) {
		     time_index=0;
		     env=value;
		 } else {
		     if (time_index<time_wait) time_index++;
		     else env=env*fo_factor[j];
		 }
		 if (env>min_amp) { 
		     k+=(1-fo_factor[j])*(((float)(max-1)/(float) env)-k);
		     k_index=0;
		 } else {
		     if (k_index<time_wait) k_index++; 
		     else k*=fo_factor[j];
		 }
		 if (value*k>max-1) k=((float)max-1)/((float)value);
		 val*=k;
		 buf16[i]=val;
	     }
	 } /* for (j=0;j<wd->channels;j++) */
	 break;
     }
    } /* switch (wd->res) */
    md->cb->modified=True;
    md->cb->update=True;
    update_canvas(md,NULL);
    update_display(md);
    
    watch_cursor(False);
}

void echo_call(Widget w, XtPointer client_data, XtPointer call_data)
{
    Main_Data *md=(Main_Data*) client_data;
    
    if (!md->wd->inmem) {
	warn_popup(md->mw->form,app_resources.err_notimplement);
	return;
    }

    echo_dialog(ec_return,md->wd);
}

void ec_return(float fo_time[MAX_CHANNELS],
	       float fo_factor[MAX_CHANNELS])
{
    Main_Data *md=MD;
    Wave_Data *wd=md->wd;
    char *ring,*buf;
    int i,j,echo,ring_length,ring_count;
    int length=wd->length;
    float teiler;
    
    watch_cursor(True);
    
    for (j=0;j<wd->channels;j++) {
	
	buf=(char*) wd->buffer;
	if (wd->ismark) {
	    buf+=wd->markbeg*wd->bpspl;
	    length=wd->marklength*wd->bpspl;
	}
	
	ring_length=(int) (fo_time[j]*(float)wd->freq);
	
	if (ring_length*(wd->res/8)<MAXUSHORT) ring=(char*)md->mg->fbuf;
	else {
	    if ((ring=malloc(ring_length*(wd->res/8)))==NULL) {
		watch_cursor(False);
		warn_popup(md->mw->form,app_resources.err_mem);
		return;
	    }
	}
	for (i=0;i<ring_length*(wd->res/8);i++) ring[i]=0;
	ring_count=0;
	teiler=2.0*fo_factor[j];
	
	switch (wd->res) {
	 case 8: {
	     char val;
	     for (i=j;i<length;i++) {
		 val=buf[i]^0x80;
		 echo=ring[ring_count % ring_length]*fo_factor[j];
		 val+=echo;
		 ring[ring_count]=val/2;
		 val/=teiler;
		 buf[i]=val^0x80;
		 ring_count++;
		 if (ring_count==ring_length) ring_count=0;
	     }
	     break;
	 }
	 case 16: { 
	     short val,*ring16,*buf16;
	     
	     buf16=(short*)buf;
	     ring16=(short*)ring;
	     
	     for (i=j;i<length/2;i++) {
		 val=buf16[i];
		 echo=ring16[ring_count % ring_length]*fo_factor[j];
		 val+=echo;
		 ring16[ring_count]=val/2;
		 buf16[i]=val/teiler;
		 ring_count++;
		 if (ring_count==ring_length) ring_count=0;
	     }
	     break;
	 }
	}
	if (ring_length*(wd->res/8)>=MAXUSHORT) free(ring);
    }
    
    md->cb->modified=True;
    md->cb->update=True;
    update_canvas(md,NULL);
    update_display(md);
    
    watch_cursor(False);
}


void find_peak(Wave_Data *wd)
{
    int i,peak_r=0,peak_l=0;
    char *buffer;
    int begin=0,length=wd->length;
    
    if (wd->ismark) {
	begin=wd->markbeg*wd->bpspl;
	length=wd->marklength*wd->bpspl;
    }
    
    if (wd->inmem) {
	buffer=(char*)(wd->buffer+begin);
	
	switch (wd->res) {
	 case 8: {
	     char val_l,val_r;
	     
	     switch (wd->channels) {
	      case 1:
		 for (i=0;i<length;i++) {
		     val_l=buffer[i]^0x80;
		     if (abs(val_l)>peak_l) {
			 peak_l=abs(val_l);
			 wd->ind_peak_l=i;
		     }
		 }
		 break;
	      case 2:
		 for (i=0;i<length;) {
		     val_l=buffer[i++]^0x80;
		     if (abs(val_l)>peak_l) {
			 peak_l=abs(val_l);
			 wd->ind_peak_l=i;
		     }
		     val_r=buffer[i++]^0x80;
		     if (abs(val_r)>peak_r) {
			 peak_r=abs(val_r);
			 wd->ind_peak_r=i;
		     }
		 }
		 break;
	     } /* switch (wd->channels) */
	     break;
	 } /* case 8 */
	 case 16: {
	     short *buf=(short*) buffer,val_l,val_r;
	     
	     switch (wd->channels) {
	      case 1:
		 for (i=0;i<length/2;i++) {
		     val_l=abs(buf[i]);
		     if (val_l>peak_l) {
			 peak_l=val_l;
			 wd->ind_peak_l=i<<1;
		     }
		 }
		 break;
	      case 2:
		 for (i=0;i<length/2;) {
		     val_l=abs(buf[i++]);
		     if (val_l>peak_l) {
			 peak_l=val_l;
			 wd->ind_peak_l=i<<1;
		     }
		     val_r=abs(buf[i++]);
		     if (val_r>peak_r) {
			 peak_r=val_r;
			 wd->ind_peak_r=i<<1;
		     }
		 }
	     } /* switch (wd->channels) */
	     break;
	 } /* case 16 */
	    break;
	} /* switch (wd->res) */
    } else { /* if (wd->inmem) */
	Audio_File af;
	int toread,r_index=0;
	
	wd2af(wd,&af);
	af_rewind(af);
	af_seek(af,begin,SEEK_CUR);
	
	buffer=(char*)MD->mg->fbuf;
	
	i=length;
	while (i>0) {
	    if (i>MAXUSHORT) toread=MAXUSHORT;
	    else toread=i;
	    toread=af_read(af,buffer,toread);
	    if (toread<=0) break;
	    
	    switch (wd->res) {
	     case 8: {
		 char val_l,val_r;
		 
		 switch (wd->channels) {
		  case 1:
		     for (i=0;i<MAXUSHORT;i++) {
			 val_l=buffer[i]^0x80;
			 if (abs(val_l)>peak_l) {
			     peak_l=abs(val_l);
			     wd->ind_peak_l=i+(r_index*MAXUSHORT);
			 }
		     }
		     break;
		  case 2:
		     for (i=0;i<MAXUSHORT;) {
			 val_l=buffer[i++]^0x80;
			 if (abs(val_l)>peak_l) {
			     peak_l=abs(val_l);
			     wd->ind_peak_l=i+(r_index*MAXUSHORT);
			 }
			 val_r=buffer[i++]^0x80;
			 if (abs(val_r)>peak_r) {
			     peak_r=abs(val_r);
			     wd->ind_peak_r=i+(r_index*MAXUSHORT);
			 }
		     }
		     break;
		 } /* switch (wd->channels) */
		 break;
	     } /* case 8 */
	     case 16: {
		 short *buf=(short*) buffer,val_l,val_r;
		 
		 switch (wd->channels) {
		  case 1:
		     for (i=0;i<MAXUSHORT/2;i++) {
			 val_l=abs(buf[i]);
			 if (val_l>peak_l) {
			     peak_l=val_l;
			     wd->ind_peak_l=(i<<1)+(r_index*MAXUSHORT);
			 }
		     }
		     break;
		  case 2:
		     for (i=0;i<MAXUSHORT/2;) {
			 val_l=abs(buf[i++]);
			 if (val_l>peak_l) {
			     peak_l=val_l;
			     wd->ind_peak_l=(i<<1)+(r_index*MAXUSHORT);
			 }
			 val_r=abs(buf[i++]);
			 if (val_r>peak_r) {
			     peak_r=val_r;
			     wd->ind_peak_r=(i<<1)+(r_index*MAXUSHORT);
			 }
		     }
		 } /* switch (wd->channels) */
		 break;
	     } /* case 16 */
		break;
	    } /* switch (wd->res) */
	    r_index++;
	} /* while (i>0) */
    }
    wd->ind_peak_r+=begin;
    wd->ind_peak_l+=begin;
    wd->peak_l=peak_l;
    wd->peak_r=peak_r;
}


/* merge buf2 into buf1, buf1 must be bigger than buf2 */
void merge_8(char *buf1,char *buf2,int length,float factor1,float factor2)
{
    char val1,val2;
    int i;
    
    if (factor1==factor2) {
	for (i=0;i<length;i++) {
	    val1=buf1[i]^0x80;
	    val2=buf2[i]^0x80;
	    val1=(char) (val1*factor1);
	    val2=(char) (val2*factor1);
	    buf1[i]=(val1 + val2)^0x80;
	}
    } else {
	for (i=0;i<length;) {
	    val1=buf1[i]^0x80;
	    val2=buf2[i]^0x80;
	    val1=(char) (val1*factor1);
	    val2=(char) (val2*factor1);
	    buf1[i]=(val1 + val2)^0x80;
	    i++;
	    val1=buf1[i]^0x80;
	    val2=buf2[i]^0x80;
	    val1=(char) (val1*factor2);
	    val2=(char) (val2*factor2);
	    buf1[i]=(val1 + val2)^0x80;
	    i++;
	}
    }
}

/* merge buf2 into buf1, buf1 must be bigger than buf2 */
void merge_16(short *buf1,short *buf2,int length,float factor1,float factor2)
{
    short val1,val2;
    int i;
    
    if (factor1==factor2) {
	for (i=0;i<length;i++) {
	    val1=buf1[i]*factor1;
	    val2=buf2[i]*factor1;
	    buf1[i]=val1 + val2;
	}
    } else {
	for (i=0;i<length;) {
	    val1=buf1[i]*factor1;
	    val2=buf2[i]*factor1;
	    buf1[i]=val1 + val2;
	    i++;
	    val1=buf1[i]*factor2;
	    val2=buf2[i]*factor2;
	    buf1[i]=val1 + val2;
	    i++;
	}
    }
}
