C
C-----------------------------------------------------------------------
C MAIN PROGRAM: MAXIMALLY FLAT FIR FILTER DESIGN
C AUTHOR:       J. F. KAISER
C               BELL LABORATORIES, MURRAY HILL, NEW JERSEY 07974
C
C INPUT:        BETA IS THE CENTER OF THE TRANSITION REGION,
C                    0 < BETA < 0.5
C               GAMMA IS THE WIDTH OF THE TRANSITION REGION,
C                    0 < GAMMA < MIN( 2*BETA, 1-2*BETA )
C-----------------------------------------------------------------------
C
      DIMENSION A(200), B(200), C(200)
      COMMON IOUTD
C
C SET UP MACHINE CONSTANTS
C
      IND = I1MACH(1)
      IOUTD = I1MACH(2)
C
      LIMIT = 200
      READ (IND,9999) BETA
      READ (IND,9999) GAMMA
9999  FORMAT (E15.7)
      CALL MXFLAT(BETA, GAMMA, NP, A, B, C, LIMIT, IERR)
C
C PRINT RESULTS
C
      IF (IERR.GT.1) WRITE (IOUTD,9998) BETA, GAMMA
9998  FORMAT (10H FOR BETA=, F5.3, 11H AND GAMMA=, F5.3)
      GO TO (10, 20, 30, 40), IERR
  10  CALL PRCOEF(BETA, GAMMA, NP, B, LIMIT)
      STOP
  20  WRITE (IOUTD,9997)
9997  FORMAT (24H BETA NOT IN RANGE 0-0.5)
      STOP
  30  WRITE (IOUTD,9996)
9996  FORMAT (19H GAMMA NOT IN RANGE)
      STOP
  40  WRITE (IOUTD,9995)
9995  FORMAT (30H GAMMA TOO SMALL, MIN IS 0.04+)
      STOP
      END
C
C-----------------------------------------------------------------------
C SUBROUTINE:  PRCOEF
C PRINT THE FILTER COEFFICIENTS
C-----------------------------------------------------------------------
C
      SUBROUTINE PRCOEF(BETA, GAMMA, NP, B, LIMIT)
      DIMENSION B(LIMIT)
      COMMON IOUTD
      WRITE (IOUTD,9999) BETA, GAMMA, NP
9999  FORMAT(10H FOR BETA=, F5.3, 11H AND GAMMA=, F5.3, 13H NO. OF COEFS
     *    , 2H.=, I4)
      DO 10 I=1,NP
        WRITE (IOUTD,9998) I, B(I)
9998    FORMAT (3H B(, I3, 3H) =, F11.8)
  10  CONTINUE
      RETURN
      END
C
C-----------------------------------------------------------------------
C SUBROUTINE:  MXFLAT
C COMPUTE THE COEFFICIENTS OF A MAXIMALLY FLAT FIR LINEAR PHASE FILTER
C WITH ODD NUMBER OF TERMS AND EVEN SYMMETRY IN FILTER COEFFICIENTS
C-----------------------------------------------------------------------
C
      SUBROUTINE MXFLAT(BE,GA,NP,A,B,C,LIMIT,IERR)
C
C INPUT: BE = CENTER OF TRANSITION REGION
C           (BETA), 0<BE<0.5
C        GA = WIDTH OF TRANSITION REGION (GAMMA)
C          WIDTH IS REGION WHERE 5% < MAG < 95%
C           0 < GAMMA < MIN(2*BE, 1-2*BE)
C        LIMIT = DIMENSION OF COEF ARRAY
C OUTPUT: NP = NO. OF FILTER COEFS, = <LIMIT
C           (TOTAL FILTER LENGTH = 2*NP-1)
C         B = ARRAY OF NP FILTER COEFFICIENTS
C          B(1),B(2),...B(NP)
C         IERR = 1, NORMAL RETURN
C         IERR = 2, BETA NOT IN RANGE, 0 TO 0.5
C         IERR = 3, GAMMA NOT IN RANGE
C         IERR = 4, GAMMA TOO SMALL, MIN IS 0.04+
C A = WORKING ARRAY OF SAME SIZE AS B
C C = WORKING ARRAY OF SAME SIZE AS B
C K = NO. OF ZEROS AT NYQUIST FREQ.
C L = NO. OF ZERO DERIVATIVES AT ZERO FREQ.
C NT = FILTER HALF ORDER = NP-1
C
      DIMENSION A(LIMIT), B(LIMIT), C(LIMIT)
      IERR = 1
      NP = 0
      TWOPI = 8.0*ATAN(1.0)
      IF (BE.LE.0. .OR. BE.GE.0.5) GO TO 80
      BM = AMIN1(2.0*BE,1.0-2.0*BE)
      IF (GA.LE.0. .OR. GA.GE.BM) GO TO 90
      NT = INT(1.0/(4.0*GA*GA))
      IF (NT.GT.160) GO TO 100
      AC = (1.0+COS(TWOPI*BE))/2.0
      QLIM = LIMIT
      CALL RATPRX(AC, NT, K, NP, QLIM)
      N = 2*NP - 1
      IF (K.EQ.0) K = 1
C
C COMPUTE MAGNITUDE AT NP PTS.
C
      C(1) = 1.0
      A(1) = 1.0
      LL = NT - K
      L = LL + 1
      DO 40 I=2,NP
        FF = FLOAT(I-1)/FLOAT(N)
        C(I) = COS(TWOPI*FF)
        X = (1.0-C(I))/2.0
        SUM = 1.0
        IF (K.EQ.NT) GO TO 40
        Y = X
        DO 30 J=1,LL
          FJ = J
          JL = K - 1
          Z = Y
          IF (K.EQ.1) GO TO 20
          DO 10 JJ=1,JL
            AJ = JJ
            Z = Z*(1.0+FJ/AJ)
  10      CONTINUE
  20      Y = Y*X
          SUM = SUM + Z
  30    CONTINUE
        A(I) = SUM*(1.0-X)**K
  40  CONTINUE
C
C CALCULATE WEIGHTING COEFS BY
C AN N-POINT IDFT
C
      DO 70 I=1,NP
        B(I) = A(1)/2.0
        DO 60 J=2,NP
          M = MOD((I-1)*(J-1),N)
          IF (M.LE.NT) GO TO 50
          M = N - M
  50      B(I) = B(I) + C(M+1)*A(J)
  60    CONTINUE
        B(I) = 2.0*B(I)/FLOAT(N)
  70  CONTINUE
      RETURN
  80  IERR = 2
      RETURN
  90  IERR = 3
      RETURN
 100  IERR = 4
      RETURN
      END
C
C-----------------------------------------------------------------------
C SUBROUTINE:  RATPRX
C COMPUTE RATIONAL FRACTION APPROXIMATION, K/NP, TO NUMBER A WITHIN
C THE LIMIT OF N <= NP <= 2*N FOR THE DENOMINATOR
C-----------------------------------------------------------------------
C
      SUBROUTINE RATPRX(A, N, K, NP, QLIM)
C
C INPUT: A = DESIRED NUMBER
C N = INTEGER MAX LOWER LIMIT ON NP
C OUTPUT: K = INTEGER NUMERATOR
C NP = INTEGER DENOMINATOR
C N RETURNS AS NP-1
C K/NP IS NEAREST TO A IN THE ALGEBRAIC
C SENSE, N < LIMIT
C
      IF(N .LE. 0)GOTO 3
      AA = ABS(A)
      AI = IFIX(AA)
      AF = AMOD(AA,1.0)
      QMAX = 2*N
      IF(QMAX .GT. QLIM)QMAX = QLIM
      Q = N-1
      EM = 1.0
    1 Q = Q+1.0
      IF(Q .GT. QMAX)GOTO 2
      PS = Q*AF
      IP = PS+0.5
      E = ABS((PS-FLOAT(IP))/Q)
      IF(E .GE. EM)GOTO 1
      EM = E
      PP = IP
      QQ = Q
      GOTO 1
    2 K = SIGN(AI*QQ+PP,A)
      NP = QQ
      N = NP-1
      IF(K .EQ. NP)GOTO 4
      RETURN
    3 K = 0
      N = -1
      NP = 0
      RETURN
    4 NP = QMAX
      K = NP-1
      N = K
      RETURN
      END
