/* %M%	%I%	(CARL)	%G%	%U% */

# include <stdio.h>
# include "aswdaemon.h"
# include "symtab.h"
# include "numlist.h"
# include "ident.h"
# include "var.h"

/*
 * symbol table management routines
 */

/*
 * rather than have separate routines
 * for each symbol table we have an
 * array of symbol tables.
 */
struct symtab {
	int		ss_type;	/* the ``type'' of the symbol table */
	char *		ss_name;	/* the string name (for diags)	*/
	struct hlist **	ss_hlist;	/* the symbol table itself	*/
};

extern struct symtab *getsymtab();
extern struct hlist **makhlist();

/*
 * the array of the various symbol tables.
 */
static struct symtab symtab[] = {
	{ ST_NL_INPUT,	"input",	NULL },
	{ ST_NL_OUTPUT,	"output",	NULL },
	{ ST_IDENT,	"ident",	NULL },
	{ ST_NVAR,	"nvar",		NULL },
	{ ST_SVAR,	"svar",		NULL },
	{ -1,		NULL,		NULL }
};

/*
 * hash value routine. algorithm
 * copied from the ucb vax assembler.
 */
hash(sym)
	register char *sym;
{
	register int hashval;

	for (hashval = 0; *sym != NULL; hashval <<= 2, hashval += *sym++)
		continue;

	hashval += *(sym-1) << 5;
	hashval %= HSIZE;

	if (hashval < 0)
		hashval += HSIZE;

	return(hashval);
}

/*
 * find the symbol sym in the
 * symbol table
 */
struct hlist *
lookup(sym, type)
	register char *sym;
{
	register struct hlist *hl;
	register struct symtab *ss;

	/*
	 * first, get the symbol table
	 */
	if ((ss = getsymtab(type)) == NULL)
		return(NULL);

	/*
	 * regular symbol table search
	 */
	for (hl = ss->ss_hlist[hash(sym)]; hl != NULL; hl = hl->hl_next) {
		if ((hl->hl_sym != NULL) && (strcmp(sym, hl->hl_sym) == 0)) {
			debug(DB_SYMTAB, "lookup: found symbol %s in %s", sym, ss->ss_name);
			return(hl);
		}
	}

	debug(DB_SYMTAB, "lookup: can't find symbol %s in %s", sym, ss->ss_name);

	return(NULL);
}

/*
 * install a symbol in the symbol table.
 * if there is already a definition for
 * this symbol in the symbol table then
 * we clobber the previous definition
 * with this one.
 */
struct hlist *
install(sym, type, val)
	register char *sym;
	char *val;
{
	extern char *calloc();
	extern char *strsave();
	register struct symtab *ss;
	register struct hlist *hl;
	register int hashval;

	if ((hl = lookup(sym, type)) == NULL) {
		if ((hl = (struct hlist *) calloc(1, sizeof(struct hlist))) == NULL) {
			err(ERR_SYMTAB, "install: can't calloc");
			return(NULL);
		}

		if ((hl->hl_sym = strsave(sym)) == NULL) {
			err(ERR_SYMTAB, "install: can't strsave");
			return(NULL);
		}

		if ((ss = getsymtab(type)) == NULL)
			return(NULL);

		debug(DB_SYMTAB, "install: installing new sym %s in %s", sym, ss->ss_name);

		hashval = hash(hl->hl_sym);
		hl->hl_next = ss->ss_hlist[hashval];
		ss->ss_hlist[hashval] = hl;
	}
	else
		debug(DB_SYMTAB, "install: overwriting sym %s", sym);

	/* one way to deal with unions */
	switch (type) {
		case ST_NL_INPUT:
		case ST_NL_OUTPUT:
			if (hl->hl_numlist != NULL)
				free((char *) hl->hl_numlist);

			hl->hl_numlist = (struct numlist *) val;
			break;

		case ST_IDENT:
			hl->hl_ident = (struct ident *) val;
			break;

		case ST_NVAR:
		case ST_SVAR:
			hl->hl_var = (struct var *) val;
			break;

		default:
			err(ERR_SYMTAB, "install: invalid type (%d)", type);
			return(NULL);
			break;
	}

	return(hl);
}

/*
 * routine to copy a string into
 * a new dynamically allocated
 * piece of memory.
 */
char *
strsave(str)
	register char *str;
{
	extern char *malloc();
	register char *sp;

	if ((sp = malloc(strlen(str) + 1)) == NULL)
		return(NULL);

	strcpy(sp, str);

	return(sp);
}

/*
 * routine to make/allocate a symbol table
 */
struct hlist **
makhlist() {
	extern char *calloc();
	register struct hlist **hl;

	if ((hl = (struct hlist **) calloc(HSIZE, sizeof(struct hlist *))) == NULL)
		return(NULL);

	return(hl);
}

/*
 * getsymtab gets a symbol table via a linear
 * search through the array of the symbol tables.
 * it initializes the symbol table the first
 * time it is accessed.
 */
struct symtab *
getsymtab(type) {
	register struct symtab *ss;

	for (ss = &symtab[0]; ss->ss_type != -1; ss++) {
		if (ss->ss_type == type) {
			if (ss->ss_hlist == NULL) {
				if ((ss->ss_hlist = makhlist()) == NULL)
					return(NULL);
			}

			return(ss);
		}
	}

	return(NULL);
}
