/* %M%	%I%	(CARL)	%G%	%U% */

# include <sys/types.h>
# include <sys/time.h>
# include <sys/socket.h>
# include <netinet/in.h>
# include <netdb.h>
# include <signal.h>
# include <stdio.h>
# include "client.h"
# include "aswdaemon.h"

/*
 * stuff for catching, ignoring, etc.
 * signals.
 */

# define NULLVEC	((struct sigvec *) 0)
# define IVTIME		(5*60)			/* 5 minutes */

# define mask(n)	(1 << ((n) - 1))

extern int	reconf();
extern int	die();
extern int	interval();

/*
 * what to do with various
 * signals
 */
struct sigvecs {
	int	sg_signal;
	struct	sigvec sg_sigvec;
};

/*
 * functions to call when the interval timer goes off.
 */
struct ivfuncs {
	char		*(*iv_func)();		/* returns a string	 */
	char		*iv_name;		/* name of the function	 */
	struct ivfuncs	*iv_next;		/* next function to call */
};

/*
 * what to do with various
 * signals
 */
static struct sigvecs sigvecs[] = {
	{ SIGHUP,	{ reconf,	0,	0 } },
	{ SIGTERM,	{ die,		0,	0 } },
	{ SIGALRM,	{ interval,	0,	0 } },
	{ 0,		{ 0,		0,	0 } }
};

static struct ivfuncs	*ivfuncs;
static int		omask;

/*
 * block, ignore, catch, etc.
 * various signals.
 */
setsigs() {
	register struct sigvecs	*sg;
	struct itimerval	ivtime;

	for (sg = &sigvecs[0]; sg->sg_signal != 0; sg++) {
		if (sigvec(sg->sg_signal, &sg->sg_sigvec, NULLVEC) == -1)
			err(ERR_SYSTEM, "setsigs: sigvec(%d): %m", sg->sg_signal);
	}

	/*
	 * set up the interval timer (SIGALRM)
	 */
	timerclear(&ivtime.it_interval);
	ivtime.it_interval.tv_sec = IVTIME;
	timerclear(&ivtime.it_value);
	ivtime.it_value.tv_sec = IVTIME;

	setitimer(ITIMER_REAL, &ivtime, (struct itemerval *) 0);

	sigblock(0);
}

/*
 * reconfigure the daemon
 */
reconf() {
	blocksigs();

	if (shutdown(sock, 2) == -1)
		err(ERR_IPC, "reconf: shutdown(sock): %m");

	if (client.cl_sock != -1) {
		if (shutdown(client.cl_sock, 2) == -1)
			err(ERR_IPC, "reconf: shutdown(cl_sock): %m");
	}

	cfget();

	if (aswsock() == RET_ERR)
		exit(1);

	unblocksigs();
}

/*
 * interval calls each of the functions in the
 * ivfuncs list when we get a SIGALRM signal.
 */
interval() {
	register struct ivfuncs	*iv;

	blocksigs();

	for (iv = ivfuncs; iv != NULL; iv = iv->iv_next) {
		debug(DB_SYSTEM, "interval: calling %s", iv->iv_name);
		(void) (*iv->iv_func)();
	}

	unblocksigs();
}

/*
 * ivadd adds a function to the list of
 * functions to be called by interval.
 */
ivadd(pf, pfname)
	char	*(*pf)();
	char	*pfname;
{
	register struct ivfuncs	*lastiv;
	register struct ivfuncs	*newiv, *tiv;

	blocksigs();

	if ((newiv = (struct ivfuncs *) malloc(sizeof(struct ivfuncs))) == NULL) {
		err(ERR_SYSTEM, "ivadd: malloc failed");
		return(RET_ERR);
	}

	if (ivfuncs == NULL) {
		newiv->iv_func = pf;
		newiv->iv_name = pfname;
		newiv->iv_next = NULL;

		ivfuncs = newiv;
	}
	else {
		/*
		 * find end of list
		 */
		for (tiv = ivfuncs; tiv != NULL; tiv = tiv->iv_next)
			lastiv = tiv;

		/*
		 * add to end
		 */
		newiv->iv_func = pf;
		newiv->iv_name = pfname;
		newiv->iv_next = NULL;

		lastiv->iv_next = newiv;
	}

	unblocksigs();

	return(RET_OK);
}

/*
 * block all signals.
 */
blocksigs() {
	register struct sigvecs	*sg;
	static int		sigbits;

	if (sigbits == 0) {
		for (sg = &sigvecs[0]; sg->sg_signal != 0; sg++)
			sigbits |= mask(sg->sg_signal);
	}

	omask = sigblock(sigbits);
}

/*
 * unblock previously blocked signals.
 */
unblocksigs() {
	(void) sigsetmask(omask);
}

char *
signam(sig) {
	extern char *sys_siglist[];

	if (sig >= NSIG)
		return("unknown signal");

	return(sys_siglist[sig]);
}
