/*
 * CARL MIDI Performance Toolkit @(#)libscale.c	1.1 2/15/87
 * Xavier Chabot
 */

#include <stdio.h>
#include <math.h>
#include <string.h>
#include <malloc.h>

#include "types.h"
#include "scale.h"
static flag_type debug = FALSE ;

/* list of intervals of all scales degrees to basenote*/
interval_list[] = {-1,0,1,1,2,3,3,4,5,4,5,6,6,7,8,8,9,10,10,11,12};
				/* list of note names */
char *note_list[] = {"Cf","C","C#","Df","D","D#","Ef","E","E#",
		      "Ff","F","F#","Gf","G","G#","Af","A","A#","Bf","B","B#"};

SCALE *GetTemperedScale(basefreq,basenote,octave)
char *basenote ;		/* string like "C", "A#", or "Gf" */
double basefreq ;		/* no check is made with  the name basenote */
int octave ;			/* desired octave (middle A 440Hz is A3) */
/* 
 * Returns a pointer to a SCALE structure filled with the desired
 * tempered scale definition data. See the type SCALE in scale.h
 */
{
  register SCALE *scale ;
  register i ;
	int base_midikey ;

  scale = (SCALE *)calloc(1,sizeof(SCALE));
  scale->scale_type = TEMPERED ;
  strcpy(scale->scale_name,"TEMPERED SCALE");
  scale->scale_octave = octave ;
  scale->scale_base_freq = basefreq ;
  if(!(scale->scale_base_note = GetBaseNote(basenote))) return 0 ;
  base_midikey = BASE_MIDIKEY  +
		interval_list[scale->scale_base_note] +
		12 * octave ;
  for(i=0;i<21;i++) {
	scale->scale_ratios[i] = pow(2.,(double)(interval_list[i]/12.)) ;
	scale->scale_midikey[i] = base_midikey + interval_list[i] ;
	scale->scale_pitchbend[i] = 0x2000 ;	/* (pib=0)because wpib divides*/
  }
  return scale ;
}

SCALE *GetJustMinorScale(basefreq,basenote,octave)
char *basenote ;		/* string like "C", "A#", or "Gf" */
double basefreq ;		/* no check is made with  the name basenote */
int octave ;			/* desired octave (middle A 440Hz is A3) */
/* 
 * Returns a pointer to a SCALE structure filled with the desired
 * just minor scale definition data. See the type SCALE in scale.h
 */
{
  register i;
  SCALE *scale ;
#define MINOR_CHRO (25./24.)	/* chromatic half step (sharp) */
  static double minor_ratios[] = {
    1./MINOR_CHRO,1.,MINOR_CHRO, /* C */
    (9./8.)/MINOR_CHRO,9./8.,(9./8.)*MINOR_CHRO, /* D */
    (6./5.),(6./5.)*MINOR_CHRO,(6./5.)*MINOR_CHRO*MINOR_CHRO, /* E flatt */
    (4./3.)/MINOR_CHRO,4./3.,(4./3.)*MINOR_CHRO, /* F */
    (3./2.)/MINOR_CHRO,3./2.,(3./2.)*MINOR_CHRO, /* G */
    (25./15.),(25./15.)*MINOR_CHRO,(25./15.)*MINOR_CHRO*MINOR_CHRO, /*A flatt*/
    (15./8.)/MINOR_CHRO,15./8.,(15./8.)*MINOR_CHRO /* B */
    } ;
  scale = (SCALE *)calloc(1,sizeof(SCALE));
  scale->scale_type = JUST_MINOR ;
  strcpy(scale->scale_name,"JUST MINOR SCALE");
  scale->scale_octave = octave ;
  scale->scale_base_freq = basefreq ;
  if(!(scale->scale_base_note = GetBaseNote(basenote))) return 0 ;

  for(i=0;i<21;i++) {
    scale->scale_ratios[i] = minor_ratios[i] ;
  }
	RatioToPitchBend(scale) ;

  return scale ;
}


SCALE *GetJustMajorScale(basefreq,basenote,octave)
char *basenote ;		/* string like "C", "A#", or "Gf" */
double basefreq ;		/* no check is made with  the name basenote */
int octave ;			/* desired octave (middle A 440Hz is A3) */
/* 
 * Returns a pointer to a SCALE structure filled with the desired
 * just major  scale definition data. See the type SCALE in scale.h
 */
{
  register i;
  SCALE *scale ;
#define MAJOR_CHRO (25./24.)	/* chromatic half step (sharp) */
  static double major_ratios[] = {
    1./MAJOR_CHRO,1.,MAJOR_CHRO, /* C */
    (9./8.)/MAJOR_CHRO,9./8.,(9./8.)*MAJOR_CHRO, /* D */
    (5./4.)/MAJOR_CHRO,5./4.,(5./4.)*MAJOR_CHRO, /* E */
    (4./3.)/MAJOR_CHRO,4./3.,(4./3.)*MAJOR_CHRO, /* F */
    (3./2.)/MAJOR_CHRO,3./2.,(3./2.)*MAJOR_CHRO, /* G */
    (5./3.)/MAJOR_CHRO,5./3.,(5./3.)*MAJOR_CHRO, /* A */
    (15./8.)/MAJOR_CHRO,15./8.,(15./8.)*MAJOR_CHRO /* B */
    } ;
  scale = (SCALE *)calloc(1,sizeof(SCALE));
  scale->scale_type = JUST_MAJOR ;
  strcpy(scale->scale_name,"JUST MAJOR SCALE");
  scale->scale_octave = octave ;
  scale->scale_base_freq = basefreq ;
  if(!(scale->scale_base_note = GetBaseNote(basenote))) return 0 ;

  for(i=0;i<21;i++) {
    scale->scale_ratios[i] = major_ratios[i] ;
  }
	RatioToPitchBend(scale) ;
  return scale ;

}

SCALE *GetPythagoreanScale(basefreq,basenote,octave)
char *basenote ;		/* string like "C", "A#", or "Gf" */
double basefreq ;		/* no check is made with  the name basenote */
int octave ;			/* desired octave (middle A 440Hz is A3) */
/* 
 * Returns a pointer to a SCALE structure filled with the desired
 * pythagorean scale definition data. See the type SCALE in scale.h
 */
{
  register i;
  SCALE *scale ;
#define PYTH_CHRO (2187./2048.)	/* chromatic half step (sharp) */
  static double pyth_ratios[] = {
    1./PYTH_CHRO,1.,PYTH_CHRO, /* C */
    (9./8.)/PYTH_CHRO,9./8.,(9./8.)*PYTH_CHRO, /* D */
    (81./64.)/PYTH_CHRO,81./64.,(81./64.)*PYTH_CHRO, /* E */
    (4./3.)/PYTH_CHRO,4./3.,(4./3.)*PYTH_CHRO, /* F */
    (3./2.)/PYTH_CHRO,3./2.,(3./2.)*PYTH_CHRO, /* G */
    (27./16.)/PYTH_CHRO,27./16.,(27./16.)*PYTH_CHRO, /* A */
    (243./128.)/PYTH_CHRO,243./128.,(243./128.)*PYTH_CHRO /* B */
    } ;
  scale = (SCALE *)calloc(1,sizeof(SCALE));
  scale->scale_type = PYTHAGOREAN ;
  strcpy(scale->scale_name,"PYTHAGOREAN SCALE");
  scale->scale_octave = octave ;
  scale->scale_base_freq = basefreq ;
  if(!(scale->scale_base_note = GetBaseNote(basenote))) return 0 ;

  for(i=0;i<21;i++) {
    scale->scale_ratios[i] = pyth_ratios[i] ;
  }
	RatioToPitchBend(scale) ;
  return scale ;
}
  
PrintScale(scale,minoct,maxoct)
SCALE *scale ;
/*
 * Prints from octmin to octmax octaves of frequencies of the scale
 */
{
  register oct,pitch, i;
  double basefreq, freq ;

  if(!scale) {fprintf(stderr,"bad scale\n"); return -1 ;} ;

  printf("%s on %s\n",scale->scale_name,note_list[scale->scale_base_note]);
  for(oct=minoct;oct<maxoct;oct++) {		/* print octave number */
    printf("\t%d",oct);
  }
  printf("\n\t---------------------------------------------------------------\n");
				/* lowest frequency is flatt basenote */
  basefreq = pow(2.,-(double)(scale->scale_octave - minoct))* 
    scale->scale_base_freq ;
DB PF(basefreq);  
  for(i=0;i<21;i++) {
    pitch = (i + scale->scale_base_note -1)%21 ;
    printf("%s\t",note_list[pitch]) ;
    for(oct=0;oct<maxoct-minoct;oct++) {
      freq = basefreq * scale->scale_ratios[i] * pow(2.,(double)oct) ;
      printf("%.2f\t",freq);
    }
    printf("\n") ;
  }
  return 0;
}

  
PrintSequence(scale,pitchlist,octave)
SCALE *scale ;			/* scale definition */
int pitchlist[] ;		/* list of indexes in note_list[] */
int octave ;			/* octave from which to start */
/*
 * Prints from octmin to octmax octaves of frequencies of the scale
 */
{
  register oct,pitch, i;
  double basefreq, freq, cents ;
  int bend, note ;
  int midikey ;
  if(!scale) {fprintf(stderr,"bad scale\n"); return -1 ;} ;

  printf("%s on %s\n",scale->scale_name,note_list[scale->scale_base_note]);
  printf("---------------------------------------------------------------\n");
  printf("pitch\tfreq(Hz)\tdeviation(cents)\tMIDI key\tPitch Bend\n");
				/* lowest frequency  */
  basefreq = pow(2.,-(double)(scale->scale_octave - octave))* 
    scale->scale_base_freq ;
DB PF(basefreq);  
  i = 0 ;
  while((pitch = pitchlist[i++]) != -1) {
    pitch = (pitch + scale->scale_base_note -1)%21 ;
    oct = pitch / 21 ;
		/* C0 is BASE_MIDIKEY = 24 */
    midikey = scale->scale_midikey[pitch] + 12 * oct; 
    freq = basefreq * scale->scale_ratios[pitch] * pow(2.,(double)oct) ;
    printf("%s%d\t%.2f\t\t%.2f\t\t\t0x%x (%d)\t0x%x\n",
	   note_list[pitch],oct+octave,freq,scale->scale_deviation[pitch],
	midikey,midikey,scale->scale_pitchbend[pitch]*2);
  }
  return 0;
}


GetBaseNote(notename)
char *notename ;
/*
 * Gets a note name, that should be in note_list[], ie string
 * "C" to "B" (notice the uppercase), and returns the index in notelist
 * that willl be used to index the ratio array.
 */
{
  register i ;
  for(i=0;i<21;i++) {
    if(strcmp(note_list[i],notename) == 0) return i ;
  }
    fprintf(stderr,"bad note name %s\n",notename);
    return -1 ;
}


RatioToPitchBend(scale)
SCALE *scale ;
/*
 * Compute deviation in cents from the tempered scale and 
 * MIDI Pitch Bend for each step of the scale defined by SCALE *\f2scale\f1.
 * Expects the scale_ratio[] to be already computed.
 */
{
	register i ;
	int base_midikey  = BASE_MIDIKEY  +
		interval_list[scale->scale_base_note] +
		12 * scale->scale_octave ;

	for(i=0;i<21;i++) {
	if(scale->scale_ratios[i] == 0.) {
		fprintf(stderr,"RatioToPitchBend: null ratio for step %d\n",i);
	} else {
	    scale->scale_deviation[i] = 1200. * log(scale->scale_ratios[i]) /
		log(2.) - 100. * (double)interval_list[i] ;
	    scale->scale_pitchbend[i] = 
		(int16)((((scale->scale_deviation[i]/100.)* .5) 
		+ .5) * 16384. );
	scale->scale_midikey[i] = base_midikey + interval_list[i] ;
	}
	}

	return 0 ;
}
