#include <carl/Player.h>

/* 
 * cpy_args -	copy args into global memory 
 */

cpy_args(ac, av)
	register char **av;
{
	argc = ac;
	argv = av;
}

/* 
 * parse_flags -	read flags from command line
 */

parse_flags(ac, av)
	register char **av;
{
	extern long atol();
	register char ch;

	while ((ch = crack(ac, av, "b|", 1)) != NULL) {
		switch (ch) {
			case 'b':
				if (arg_option != NULL) 
					_Pdebug = atol(arg_option);
				break;
			default:
				exit(1);
		}
	}
	arg_index = 0;	/* reset in case anybody else wants to look */
}

hi_p_ck(x, plen)
	Player *x;
	int plen;
{
	if (x->hipn != 0) {
		if (x->hipn != plen) {
			fprintf(stderr, 
			"Setup_p: P field count mismatch, hipn=%d, plen=%d\n",
				x->hipn, plen);
			return(-1);
		}
	} else {
		x->hipn = plen;
		return(0);
	}
}

Setup_p(x, plen)
	Player *x;
	int plen;
{
	hi_p_ck(x, plen);
	if (x->pn == NULL) {
		x->pn=(double *)calloc(plen, (unsigned) sizeof(double));
		return(0);
	} else {
		fprintf(stderr, "Setup_p, P fields set up already\n");
		return(-1);
	}
}

Setup_ps(x, plen)
	Player *x;
	int plen;
{
	hi_p_ck(x, plen);
	if (x->ps == NULL) {
		x->ps=(char **)calloc(plen, (unsigned) sizeof(char *));
	} else {
		fprintf(stderr, "Setup_ps, Ps fields set up already\n");
		return(-1);
	}
}

Setup_pp(x, plen)
	Player *x;
	int plen;
{
	hi_p_ck(x, plen);
	if (x->pp == NULL) {
		x->pp=(char **)calloc(plen, (unsigned) sizeof(char *));
	} else {
		fprintf(stderr, "Setup_pp, Pp fields set up already\n");
		return(-1);
	}
}

/* 
 * ckmaxp - 	verify references to P fields in Player are valid
 * takes:
 *	i -	Player instance data structure
 *	n -	parameter index
 * diagnostics:
 *	ckmaxp: non-existant P field= FIELD for player PLAYER
 * returns:	0 on success, else -1
 * side effects:
 *	Sets Player_stat to P_NULL_PTR if no Player instance data.
 */

ckmaxp(i, n)
	register Player *i;
	register int n; 
{
	Player_stat = 0;
	if (i == NULL) {
		Player_stat = P_NULL_PTR;
		return(-1);
	}
	if (n > i->plen) {
		fprintf(stderr, "ckmaxp: %s %d for player %s\n",
		"non-existant parameter variable =", n, i->name);
		return(-1);
	}
	return(0);
}

/*
 * P -	fetch P field from another Player
 * takes:
 * 	vp -	pointer to Player instance data
 * 	n -	P field index
 * returns:
 * 	value, if found, otherwise 0.0
 * side effects:
 * 	Sets Player_stat to P_NO_PARAM if no such P field.
 * 	Sets Player_stat to P_NULL_PTR if no such Player.
 */

double 
P(vp, n)
	register Player *vp;
	register int n;
{
	Player_stat = 0;
	if (vp == NULL) {
		Player_stat = P_NULL_PTR;
		return(NULL);
	}
		
	if (ckmaxp(vp, n) == -1) {
		Player_stat = P_NO_PARAM;
		return(0.0);
	}
	return(vp->pn[n]);
}

/*
 * Ps -	fetch Ps field from another Player
 * takes:
 * 	vp -	pointer to Player instance data
 * 	n -	Ps field index
 * returns:
 * 	value, if found, otherwise 0.0
 * side effects:
 * 	Sets Player_stat to P_NO_PARAM if no such Ps field.
 *	Sets Player_stat to P_NULL_PTR if no Player instance data.
 */
		
char *
Ps(vp, n)
	register Player *vp; 
	register int n;
{

	Player_stat = 0;
	if (vp == NULL) {
		Player_stat = P_NULL_PTR;
		return(NULL);
	}
	if (ckmaxp(vp, n) == -1) {
		Player_stat = P_NO_PARAM;
		return(NULL);
	}
	return(vp->ps[n]);
}

/*
 * Pp -	fetch Pp field from another Player
 * takes:
 * 	vp -	pointer to Player instance data
 * 	n -	Pp field index
 * returns:
 * 	value, if found, otherwise 0.0
 * side effects:
 * 	Sets Player_stat to P_NO_PARAM if no such Pp field.
 *	Sets Player_stat to P_NULL_PTR if no Player instance data.
 */
		
char *
Pp(vp, n)
	register Player *vp; 
	register int n;
{
	Player_stat = 0;
	if (vp == NULL) {
		Player_stat = P_NULL_PTR;
		return(NULL);
	}
	if (ckmaxp(vp, n) == -1) {
		Player_stat = P_NO_PARAM;
		return(NULL);
	}
	return(vp->pp[n]);
}

/*
 * Set_p -	set P field in another Player
 * takes:
 * 	vp -	pointer to Player instance data
 * 	n -	P field index
 * 	val -	value
 * returns:
 * 	0 on success, -1 on error
 * side effects:
 * 	Sets Player_stat to P_NO_PARAM if no such P field.
 * 	Sets Player_stat to P_NULL_PTR if no such Player.
 */

Set_p(vp, n, val)
	register Player *vp;
	register int n;
	register float val;
{
	Player_stat = 0;
	if (vp == NULL) {
		Player_stat = P_NULL_PTR;
		return(-1);
	}
		
	if (ckmaxp(vp, n) == -1) {
		Player_stat = P_NO_PARAM;
		return(-1);
	}
	vp->pn[n] = val;
	return(0);
}

/*
 * Set_ps -	set Ps field in another Player
 * takes:
 * 	vp -	pointer to Player instance data
 * 	n -	Ps field index
 *	val -	value
 * returns:
 * 	0 on success, -1 on error
 * side effects:
 * 	Sets Player_stat to P_NO_PARAM if no such Ps field.
 *	Sets Player_stat to P_NULL_PTR if no Player instance data.
 */
		
Set_ps(vp, n, val)
	register Player *vp; 
	register int n;
	register char *val;
{

	Player_stat = 0;
	if (vp == NULL) {
		Player_stat = P_NULL_PTR;
		return(-1);
	}
	if (ckmaxp(vp, n) == -1) {
		Player_stat = P_NO_PARAM;
		return(-1);
	}
	vp->ps[n] = val;
	return(0);
}

/*
 * Set_pp -	set Pp field in another Player
 * takes:
 * 	vp -	pointer to Player instance data
 * 	n -	Pp field index
 * 	val -	string value
 * returns:
 * 	0 on success, -1 otherwise
 * side effects:
 * 	Sets Player_stat to P_NO_PARAM if no such Pp field.
 *	Sets Player_stat to P_NULL_PTR if no Player instance data.
 */
		
Set_pp(vp, n, val)
	register Player *vp; 
	register int n;
	register char *val;
{
	Player_stat = 0;
	if (vp == NULL) {
		Player_stat = P_NULL_PTR;
		return(-1);
	}
	if (ckmaxp(vp, n) == -1) {
		Player_stat = P_NO_PARAM;
		return(-1);
	}
	vp->pp[n] = val;
	return(0);
}

/* 
 * Abs_time -	return actual time from start of piece of current moment 
 * returns:
 *	value of the global variable Now
 * notes:
 *	You might as well just reference the Now variable itself.
 */

double 
Abs_time()
{
	return(Now);
}

/* 
 * Rel_time - return relative elapsed time
 * takes:
 *	x -	Player instance structure pointer
 * returns:
 *	elapsed time, taking the begin time of the player as 0,
 * 	and a duration expressed as a fraction in the [0,1) interval.
 * notes:
 *	As the Player calling Rel_time progresses from its stated
 *	begin time to its end time, the value returned by Rel_time
 *	will increase from [0 to 1).
 *	Note, that if the Player's duration is very large (when
 *	the duration is left out in a Player declaration, the duration
 *	is set to the value HUGE), then the value returned from Rel_time
 *	will grow towards 1 very slowly.
 * side effects:
 *	Sets Player_stat to P_NULL_PTR if no Player instance data.
 */

double 
Rel_time(x)
	register Player *x;
{
	Player_stat = 0;
	if (x == NULL) {
		Player_stat = P_NULL_PTR;
		return(0.0);
	}
	return((x->ptime - x->beg) / x->dur);
}

/* 
 * Beg_time -	return begin time of Player 
 * takes:
 *	x -	Pointer to Player instance structure
 * side effects:
 *	Sets Player_stat to P_NULL_PTR if no Player instance data.
 */

double 
Beg_time(x)
	register Player *x;
{
	Player_stat = 0;
	if (x == NULL) {
		Player_stat = P_NULL_PTR;
		return(0.0);
	}
	return(x->beg);
}

/* 
 * Dur_time -	return duration time of Player 
 * takes:
 *	x -	Pointer to Player instance structure
 * side effects:
 *	Sets Player_stat to P_NULL_PTR if no Player instance data.
 */

double 
Dur_time(x)
	register Player *x;
{
	Player_stat = 0;
	if (x == NULL) {
		Player_stat = P_NULL_PTR;
		return(0.0);
	}
	return(x->dur);
}


/* 
 * End_time -	return end time of Player 
 * takes:
 *	x -	pointer to Player instance structure
 * side effects:
 *	Sets Player_stat to P_NULL_PTR if no Player instance data.
 */

double 
End_time(x)
	register Player *x;
{
	Player_stat = 0;
	if (x == NULL) {
		Player_stat = P_NULL_PTR;
		return(0.0);
	}
	return(x->dur + x->beg);
}

/* 
 * Set_beg_time -	set begin time of player 
 * takes:
 *	x -	pointer to Player instance structure
 *	beg -	begin time to set
 * side effects:
 *	Sets Player_stat to P_NULL_PTR if no Player instance data.
 */

Set_beg_time(x, beg)
	register Player *x;
	register double beg;
{
	Player_stat = 0;
	if (x == NULL) {
		Player_stat = P_NULL_PTR;
		return;
	}
	x->beg = beg;
}


/* 
 * Set_dur_time -	set duration time of player 
 * takes:
 *	x -	pointer to Player instance structure
 *	dur -	duration 
 * side effects:
 *	Sets Player_stat to P_NULL_PTR if no Player instance data.
 */

Set_dur_time(x, dur)
	register Player *x;
	register double dur;
{
	Player_stat = 0;
	if (x == NULL) {
		Player_stat = P_NULL_PTR;
		return;
	}
	x->dur = dur;
}

/*
 * Hipn - set highest referenceable parameter
 * takes:
 *	Player instance, parameter number.
 * returns: 
 *	former value
 * 	returns -1 if attempting to make it bigger.
 * side effects:
 *	Sets Player_stat to P_NULL_PTR if no Player instance data.
 * notes:
 *	if n < 0, just return former value, do not set
 */

Hipn(x, n)
	register Player *x;
	register int n;
{
	Player_stat = 0;
	if (x == NULL) {
		Player_stat = P_NULL_PTR;
		return(-1);
	}
	if (n < 0)
		return(x->hipn);
	if (n <= x->hipn) {
		register int oldn;

		oldn = x->hipn;
		x->hipn = n;
		return(oldn);
	}
	return(-1);
}

/*
 * Chord - return whether chord has been scanned in cyclic list
 * takes:
 * 	Player *x
 * returns:
 * 	x->chordstat
 * notes:
 * 	x->chordstat is TRUE by cyclic list scanner/parsers
 * 		when an '&' character was scanned as
 * 		a field delimiter, otherwise it is FALSE.
 */

Chord(x)
	register Player *x;
{
	return(x->chordstat);
}
