/* mixsnd.c	1.2	(CARL)	8/27/84	13:54:58 */
#include <stdio.h>
#include <carl/carl.h>
#include <carl/sndio.h>

/* to compile: cc -O mixsnd.c -lsf -lfrm -lm */

/*
 * usage:
 * mixsnd is a cheap score-driven csound file mixer. It reads its 
 * standard input for lines of the form:
 * 
 *     p1   p2         p3       p4       p5   p6     p7
 *     note begin_time filename duration seg# offset gain;
 * 
 * where:
 * 	note is the literal string, note,
 * 	begin_time is the time to start mixing in the file,
 * 	filename is a (possibly partial) csound filename specification,
 * 	duration is the length of time to mix in the file,
 * 	seg# is the segment number, if the sound file has segmentation info,
 *		(currently unimplemented)
 * 	offset is an offset into the sound file of where to begin,
 * 	gain is the gain (what can I say?).
 * It ignores all lines that do not begin with "not", so you can
 * include comments, ter; statements and the like which have no effect.
 * For example: 
 * 
 *     note 0 /snd1/frm/av1 4 0 .5 .25;
 *     note 1 twiddle 2 0 0 .25;
 *     note 1 /snd1/frm/av1 4 0 .5 .25;
 *
 * It writes the mixed samples on the standard output.
 *
 * It has flags:
 * 	-v	verbose, print a digested form of the score on stderr,
 *	-bN	make sound file buffer size = N (default 16K)
 *	-s	share open sound file descriptors if two notes play
 *			the same file
 */

char *pn[32];
int thread;
int verbose;

extern int sferror;
extern float sfexpr();

struct note {
	struct note *nxtn, *lstn;
	long begs, ends, offs;
	float gain;
	int segn;
	CSNDFILE *sfd;
} *rootnote;


extern CSNDFILE *rootsfd;

extern char *arg_option;
extern int arg_index;
extern char crack();

main(argc, argv)
	char **argv;
{
	extern char *malloc();
	extern long mixparse();
	int otty = isatty(1), i;
	float output; 	/* output can't be register variable */
	float fsndi(); 
	struct note *n;
	long maxlen; 
	char ch;

	while ((ch = crack(argc, argv, "b|sv", 0)) != NULL) {
		switch (ch) {
			case 'b':	/* set buffer size */
				setsfbuf((int)sfexpr(arg_option, 1.0));
				break;
			case 's':	/* share open sound files */
				thread++;
				break;
			case 'v':	/* verbose */
				verbose++;
				break;
			default:
				mixhelp(0);
		}
	}

	if ((maxlen = mixparse()) <= 0) {
		fprintf(stderr, "mixsnd: mixparse failed\n");
		exit(1);
	}

	for (i = 0; i < maxlen; i++) {
		output = 0.0;
		for (n = rootnote; n != NULL; n = n->nxtn) {
			if (i < n->begs || i >= n->ends 
				    || n->offs >= n->sfd->fs)
				continue;
			output += fsndi(n->sfd, n->offs++) * n->gain;
			if (sferror) { 
				fprintf(stderr, "mixsnd: sferror=%d\n",sferror);
				sfallclose(); 
				exit(1); 
			}
		}
		if (otty) printf("%d\t%f\n", i, output);
		else putfloat(&output);
	}
	if (!otty) flushfloat();
	sfallclose();
}

mixhelp(x)
	int x;
{
fprintf(stderr, "%s%s%s%s%s",
"mixscr [flags] < scorefile\n",
" flags:\n",
" -bN\tset sound file buffer size to N bytes (default=16384)\n",
" -s\tshare already-opened sound files between note statements\n",
" -v\tverbose, print digested form of score on stderr\n"
);
exit(x);
}

#include <ctype.h>

scanbuf(buf)
	char *buf;
{
	register char *c;
	register int trigger = 0, i = 1;	/* pn[0] not used */

	while (isspace(*buf)) buf++;

	for (c = buf; *c != NULL; c++) {
		if (*c == ';') break;
		if (!isspace(*c)) { 
			if (!trigger) { 
				if (*c == ',') {
					pn[i++] = c;	/* null p field */
					continue;
				}
				pn[i++] = c; 
				trigger++; 
			} 
			else {
				if (*c == ',') {
					*c = '\0';
					trigger = 0;
				}
			}
		}
		else { 
			*c = NULL; 
			trigger = 0; 
		}
	}
	return(i);
}

char buf[BUFSIZ]; 

long 
mixparse()
{
	char *rflags="-r", *rsflags="-r -s", *flags = rflags;
	CSNDFILE *opensf();
	long max = 0;
	 
	/*note begin_time filename duration offset gain;*/
	 
	while ((fgets(buf, BUFSIZ, stdin) != NULL)) {
		register struct note *note;
		struct note *newnote();
		double tmp1, tmp2;

		if (strncmp(buf, "not", 3)) 
			continue;
		if (scanbuf(buf) >= 6) { 
			note = newnote();
			if (thread) 	/* look for sfd opened already */
				flags = rsflags;
			if ((note->sfd = opensf(pn[3], flags)) == NULL) { 
				fprintf(stderr, "mixsnd: opensf failed\n");
				sfallclose(); 
				exit(1); 
			}
			tmp1 = note->sfd->sr;
			tmp2 = sfexpr(pn[2], tmp1);	/* p2 - begin */
			note->begs = tmp2;
							/* p6 - offset */
			note->offs = (long) sfexpr(pn[6], note->sfd->sr);
			tmp2 = sfexpr(pn[4], tmp1);	/* p4 - duration */
			if (tmp2 < 0)	/* neg. time means EOF */
				note->ends = note->sfd->fs + note->begs 
					- note->offs;
			else
				note->ends = tmp2 + note->begs;
							/* p5 - seg. # */
			note->segn = sfexpr(pn[5], 1.0);
							/* p7 - gain */
			note->gain = sfexpr(pn[7], 1.0);
			max = max < note->ends ? note->ends : max;
			if (verbose) 
				pnote(note);
		} 
	}
	return(max);
}

struct note *newnote()
{
	struct note *n;
	static struct note *last;

	n = (struct note *) malloc((unsigned) sizeof(struct note));

	if (rootnote == NULL) {
		last = rootnote = n;
		n->lstn = n->nxtn = NULL;
	}
	else {
		last->nxtn = n;
		n->lstn = last;
		last = n;
		n->nxtn = NULL;
		}
	return(n);
}

pnote(n)
	struct note *n;
{
	float sr = n->sfd->sr;
	fprintf(stderr, "note %6.3f [%d], %s %6.3f[%d], %6.3f[%d], %6.3f\n", 
		n->begs/sr, 
		n->begs, 
		n->sfd->sfn, 
		(n->ends - n->begs)/sr, 
		n->ends, 
		n->offs/sr,
		n->offs, 
		n->gain);
}
