/* %M%	%I%	(CARL)	%G%	%U% */
#include <stdio.h>
#include <carl/libsf.h>

/* 
 * checksfd - validate an open sound file.  These conditions must be
 * met before an sfd can be used in a call to sndo() or sndi().   
 * The routine is called by creatsf() and opensf().
 * The routine returns -1 if can't read sfd at all, otherwise it returns
 * an error code, which it also deposits in sfd->err.
 * Fields checked are:
 * sound file name (sfn)
 * /1* number of channels (nc) is no longer checked *1/
 * sampling rate (sr)
 * packing mode (pm)
 * realtime flag (rtflag).
 * Note that the checks for pm and rtflag determine that one mode or
 * the other is set, but not both.
 * Also note that there are a lot of things NOT checked, notably there is
 * no check for cylinder address consistency.
 * There is another check made by rsdf() that makes sure a minimum
 * set of fields has been specified.
 */

checksfd (sfd)
struct sndesc  *sfd;
{
	int     errz = 0;

	if (sfd == NULL) {
		fprintf(stderr, "checksfd: NULL descriptor\n");
		return (-1);	/* now THAT's an error!! */
	}
	if (sfd -> sfn == NULL) {
		fprintf(stderr, "checksfd: NULL filename\n");
		errz |= SFDER;
	}
	if (!strlen (sfd -> sfn)) {
		fprintf(stderr, "checksfd: 0-length filename\n");
		errz |= SFDER;
	}
	if (sfd -> sr <= 0.0) {
		fprintf(stderr, "checksfd: %s invalid sampling rate\n",
			sfd->sfn);
		errz |= SFDER;
	}
	if (!((sfd -> pm == PM16BIT) || (sfd -> pm == PMFLOAT))) {
		fprintf(stderr, "checksfd: %s invalid packing mode\n",
			sfd->sfn);
		errz |= SFDER;
	}
	if (!((sfd -> rtflag == RT) || (sfd -> rtflag == NRT))) {
		fprintf(stderr, "checksfd: %s invalid file mode\n",
			sfd->sfn);
		errz |= RTIOCONFLICT;
	}
	if (ckseq(sfd)) {
		fprintf(stderr, "checksfd: %s non-sequential block list\n",
			sfd->sfn);
		errz |= SFDER;
	}
	return (errz);
}

/*
 * ckseq - check block list sequence
 * block sequence should be -1 if a realtime file containing 1 block, else
 * block sequence #'s start at 1 and should be in increasing monotonic order.
 * No block list is ok: it may be just now created.
 */

ckseq(sfd)
	struct sndesc  *sfd;
{
	struct dskblk  *c;
	int seq = 1;

	for (c = sfd->cp; c != NULL; c = c->next) {
		if (c->seq == -1)
			return(0);
		else if (c->seq != seq++)
			return(seq-1);
	}
	return(0);
}
