/*----------------------------------------------------------------------
  key.c :         Key Management Engine for BSD

  Copyright 1995 by Bao Phan,  Randall Atkinson, & Dan McDonald,
  All Rights Reserved.  All Rights have been assigned to the US
  Naval Research Laboratory (NRL).  The NRL Copyright Notice and
  License governs distribution and use of this software.

  Patents are pending on this technology.  NRL grants a license
  to use this technology at no cost under the terms below with
  the additional requirement that software, hardware, and 
  documentation relating to use of this technology must include
  the note that:
     	This product includes technology developed at and
	licensed from the Information Technology Division, 
	US Naval Research Laboratory.

----------------------------------------------------------------------*/
/*----------------------------------------------------------------------
#	@(#)COPYRIGHT	1.1a (NRL) 17 August 1995

COPYRIGHT NOTICE

All of the documentation and software included in this software
distribution from the US Naval Research Laboratory (NRL) are
copyrighted by their respective developers.

This software and documentation were developed at NRL by various
people.  Those developers have each copyrighted the portions that they
developed at NRL and have assigned All Rights for those portions to
NRL.  Outside the USA, NRL also has copyright on the software
developed at NRL. The affected files all contain specific copyright
notices and those notices must be retained in any derived work.

NRL LICENSE

NRL grants permission for redistribution and use in source and binary
forms, with or without modification, of the software and documentation
created at NRL provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. All advertising materials mentioning features or use of this software
   must display the following acknowledgement:

	This product includes software developed at the Information
	Technology Division, US Naval Research Laboratory.

4. Neither the name of the NRL nor the names of its contributors
   may be used to endorse or promote products derived from this software
   without specific prior written permission.

THE SOFTWARE PROVIDED BY NRL IS PROVIDED BY NRL AND CONTRIBUTORS ``AS
IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO FINISHED SHALL NRL OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation
are those of the authors and should not be interpreted as representing
official policies, either expressed or implied, of the US Naval
Research Laboratory (NRL).

----------------------------------------------------------------------*/

#ifdef linux
#include <netkey/osdep_linux.h>
#else /* linux */
#include <sys/param.h>
#include <sys/proc.h>
#include <sys/mbuf.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/time.h>
#include <sys/kernel.h>
#include <sys/systm.h>

#include <net/raw_cb.h>
#include <net/if.h>
#include <net/if_types.h>
#include <net/if_dl.h>
#include <net/route.h>

#include <netkey/osdep_44bsd.h>

#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/if_ether.h>
#endif /* linux */

#ifdef INET6
#include <netinet6/in6.h>
#include <netinet6/in6_var.h>
#endif /* INET6 */

#include <netkey/key.h>
#ifdef IPSEC
#include <netsec/ipsec.h>
#endif /* IPSEC */

#include <netkey/key_debug.h>

#define MAXHASHKEYLEN (2 * sizeof(int) + 2 * SOCKADDR_MAXSZ)

/*
 *  Not clear whether these values should be 
 *  tweakable at kernel config time.
 */
#define KEYTBLSIZE 61
#define KEYALLOCTBLSIZE 61
#define SO2SPITBLSIZE 61

/*
 *  These values should be tweakable...
 *  perhaps by using sysctl
 */

#define MAXLARVALTIME 240   /* Lifetime of a larval key table entry */ 
#define MAXKEYACQUIRE 1     /* Max number of key acquire messages sent */
                             /*   per destination address               */
#define MAXACQUIRETIME 15   /* Lifetime of acquire message */

/*
 *  Key engine tables and global variables
 */

struct key_tblnode keytable[KEYTBLSIZE];
struct key_allocnode keyalloctbl[KEYALLOCTBLSIZE];
struct key_so2spinode so2spitbl[SO2SPITBLSIZE];

struct keyso_cb keyso_cb;
struct key_tblnode nullkeynode  = { 0, 0, 0, 0, 0 };
struct key_registry *keyregtable;
struct key_acquirelist *key_acquirelist;
u_long maxlarvallifetime = MAXLARVALTIME;
int maxkeyacquire = MAXKEYACQUIRE;
u_long maxacquiretime = MAXACQUIRETIME;

extern SOCKADDR key_addr;

#define ROUNDUP(a) \
  ((a) > 0 ? (1 + (((a) - 1) | (sizeof(long) - 1))) : sizeof(long))
#define ADVANCE(x, n) \
    { x += ROUNDUP(n); }

int my_addr __P((SOCKADDR *));
int key_sendup __P((SOCKET *, struct key_msghdr *, int));

int addrpart_equal __P((SOCKADDR *, SOCKADDR *));
int key_sanitycheck __P((struct key_secassoc *));
int key_gethashval __P((char *, int, int));
int key_createkey __P((char *, u_int, SOCKADDR *, SOCKADDR *, u_int32, u_int));
struct key_so2spinode *key_sosearch __P((u_int, SOCKADDR *, SOCKADDR *, SOCKET *));
void key_sodelete __P((SOCKET *, int));
void key_deleteacquire __P((u_int, SOCKADDR *));
struct key_tblnode *key_search __P((u_int, SOCKADDR *, SOCKADDR *, u_int32, int, struct key_tblnode **));
struct key_tblnode *key_addnode __P((int, struct key_secassoc *));
int key_alloc __P((u_int, SOCKADDR *, SOCKADDR *, SOCKET *, u_int, struct key_tblnode **));
int key_xdata __P((struct key_msghdr *, struct key_msgdata *, int));

static void key_freeassoc DEFARGS((secassoc),
struct key_secassoc *secassoc)
{
    if (secassoc->key)
      KFREE_LEN(secassoc->key, secassoc->keylen + secassoc->ivlen);
    if (secassoc->tp_data)
      KFREE_LEN(secassoc->tp_data, secassoc->tp_len);
    secassoc->state = K_RAISED;
    KFREE(secassoc);
}

static struct key_secassoc *key_allocassoc DEFARGS((src, dst, from),
SOCKADDR *src AND SOCKADDR *dst AND SOCKADDR *from)
{
  int size = sizeof(struct key_secassoc);
  struct key_secassoc *s;

  DPRINTF(IDL_MAJOR_EVENT, (">key_allocassoc(SOCKADDR *src = %08x, SOCKADDR *dst = %08x\n", (unsigned int)src, (unsigned int)dst));
  DPRINTF(IDL_MAJOR_EVENT, (" SOCKADDR *from = %08x)\n", (unsigned int)from));

  if (src)
    size += src->sa_len;
  else
    size += SOCKADDR_MAXSZ;

  if (dst)
    size += dst->sa_len;
  else
    size += SOCKADDR_MAXSZ;

  if (from)
    size += from->sa_len;
  else
    size += SOCKADDR_MAXSZ;

  DP(MAJOR_EVENT, size, d);

  KMALLOC(s, struct key_secassoc *, size);
  if (!s)
    return NULL;

  bzero(s, size);

  s->src = (SOCKADDR *)((caddr_t)s + sizeof(struct key_secassoc));

  if (src) {
    bcopy((caddr_t)src, (caddr_t)s->src, src->sa_len);
    s->dst = (SOCKADDR *)((caddr_t)s->src + src->sa_len);
  } else
    s->dst = (SOCKADDR *)((caddr_t)s->src + SOCKADDR_MAXSZ);

  if (dst) {
    bcopy((caddr_t)dst, (caddr_t)s->dst, dst->sa_len);
    s->from = (SOCKADDR *)((caddr_t)s->dst + dst->sa_len);
  } else
    s->from = (SOCKADDR *)((caddr_t)s->dst + SOCKADDR_MAXSZ);

  if (from)
    bcopy((caddr_t)from, (caddr_t)s->from, from->sa_len);

  DDO(IDL_MAJOR_EVENT, dump_secassoc(s));

  return s;
}

static int key_debug DEFARGS((km),
struct key_msghdr *km)
{
#ifdef KEY_DEBUG
  switch (km->type) {
    case 1:
      key_debug_allocdump();
      return 0;
  }
#endif /* KEY_DEBUG */
  return EINVAL;
}

/*----------------------------------------------------------------------
 * key_secassoc2msghdr(): 
 *      Copy info from a security association into a key message buffer.
 *      Assume message buffer is sufficiently large to hold all security
 *      association information including src, dst, from, key and iv.
 ----------------------------------------------------------------------*/
int key_secassoc2msghdr DEFARGS((secassoc, km, keyinfo),
struct key_secassoc *secassoc AND
struct key_msghdr *km AND
struct key_msgdata *keyinfo)
{
  char *cp;
  DPRINTF(IDL_FINISHED, ("Entering key_secassoc2msghdr\n"));

  if ((km == 0) || (keyinfo == 0) || (secassoc == 0))
    return(-1);

  km->type = secassoc->type;
  km->state = secassoc->state;
  km->label = secassoc->label;
  km->spi = secassoc->spi;
  km->keylen = secassoc->keylen;
  km->ivlen = secassoc->ivlen;
  km->algorithm = secassoc->algorithm;
  km->lifetype = secassoc->lifetype;
  km->lifetime1 = secassoc->lifetime1;
  km->lifetime2 = secassoc->lifetime2;

  /*
   *  Stuff src/dst/from/key/iv in buffer after
   *  the message header.
   */
  cp = (caddr_t)(km + 1);

  DPRINTF(IDL_FINISHED, ("sa2msghdr: 1\n"));
  keyinfo->src = (SOCKADDR *)cp;
  if (secassoc->src->sa_len) {
    bcopy(secassoc->src, cp, secassoc->src->sa_len);
    ADVANCE(cp, secassoc->src->sa_len);
  } else {
    bzero(cp, MAX_SOCKADDR_SZ);
    ADVANCE(cp, MAX_SOCKADDR_SZ);
  }

  DPRINTF(IDL_FINISHED, ("sa2msghdr: 2\n"));
  keyinfo->dst = (SOCKADDR *)cp;
  if (secassoc->dst->sa_len) {
    bcopy(secassoc->dst, cp, secassoc->dst->sa_len);
    ADVANCE(cp, secassoc->dst->sa_len);
  } else {
    bzero(cp, MAX_SOCKADDR_SZ);
    ADVANCE(cp, MAX_SOCKADDR_SZ);
  }

  DPRINTF(IDL_FINISHED, ("sa2msghdr: 3\n"));
  keyinfo->from = (SOCKADDR *)cp;
  if (secassoc->from->sa_len) {
    bcopy(secassoc->from, cp, secassoc->from->sa_len);
    ADVANCE(cp, secassoc->from->sa_len);
  } else {
    bzero(cp, MAX_SOCKADDR_SZ);
    ADVANCE(cp, MAX_SOCKADDR_SZ);
  }

  DPRINTF(IDL_FINISHED, ("sa2msghdr: 4\n"));

  keyinfo->key = cp;
  keyinfo->keylen = secassoc->keylen;
  if (secassoc->keylen) {
    bcopy((caddr_t)(secassoc->key), cp, secassoc->keylen);
    ADVANCE(cp, secassoc->keylen);
  }

  DPRINTF(IDL_FINISHED, ("sa2msghdr: 5\n"));
  keyinfo->iv = cp;
  keyinfo->ivlen = secassoc->ivlen;
  if (secassoc->ivlen) {
    bcopy((caddr_t)(secassoc->iv), cp, secassoc->ivlen);
    ADVANCE(cp, secassoc->ivlen);
  }

  DDO(IDL_FINISHED,printf("msgbuf(len=%d):\n",(caddr_t)cp - (caddr_t)km));
  DDO(IDL_FINISHED,dump_buf((caddr_t)km, (caddr_t)cp - (caddr_t)km));
  DPRINTF(IDL_FINISHED, ("sa2msghdr: 6\n"));
  return(0);
}


/*----------------------------------------------------------------------
 * key_msghdr2secassoc():
 *      Copy info from a key message buffer into a key_secassoc 
 *      structure
 ----------------------------------------------------------------------*/
int key_msghdr2secassoc DEFARGS((secassoc, km, keyinfo),
struct key_secassoc *secassoc AND
struct key_msghdr *km AND
struct key_msgdata *keyinfo)
{
  DPRINTF(IDL_MAJOR_EVENT, (">key_msghdr2secassoc()"));

  if ((km == 0) || (keyinfo == 0) || (secassoc == 0))
    return(-1);

  secassoc->len = sizeof(*secassoc);
  secassoc->type = km->type;
  secassoc->state = km->state;
  secassoc->label = km->label;
  secassoc->spi = km->spi;
  secassoc->algorithm = km->algorithm;
  secassoc->ivlen = km->ivlen;
  secassoc->keylen = km->keylen;
  secassoc->lifetype = km->lifetype;
  secassoc->lifetime1 = km->lifetime1;
  secassoc->lifetime2 = km->lifetime2;

  DDO(IDL_MAJOR_EVENT, dump_secassoc(secassoc));

  if (secassoc->key) {
    KFREE_LEN(secassoc, secassoc->keylen + secassoc->ivlen);
    secassoc->key = secassoc->iv = NULL;
  }

  if (secassoc->keylen || secassoc->ivlen) {
    KMALLOC(secassoc->key, caddr_t, secassoc->keylen + secassoc->ivlen);
    if (!secassoc->key) {
      DPRINTF(IDL_ERROR,("msghdr2secassoc: can't allocate mem for iv\n"));
      return(-1);
    }
    bzero(secassoc->key, secassoc->keylen + secassoc->ivlen);
    secassoc->iv = secassoc->key + secassoc->keylen;
    bcopy((caddr_t)keyinfo->key, (caddr_t)secassoc->key, secassoc->keylen);
    bcopy((caddr_t)keyinfo->iv, (caddr_t)secassoc->iv, secassoc->ivlen);
  }

  DDO(IDL_MAJOR_EVENT, dump_secassoc(secassoc));

  return(0);
}

int key_sanitycheck DEFARGS((s),
struct key_secassoc *s)
{
  if (s->spi < 256) {
    DPRINTF(IDL_ERROR, ("key_sanitycheck: reserved spi (< 256)\n"));
    return -1;
  }

  return 0;
}

/*----------------------------------------------------------------------
 * addrpart_equal():
 *      Determine if the address portion of two sockaddrs are equal.
 *      Currently handles only AF_INET and AF_INET6 address families.
 ----------------------------------------------------------------------*/
int addrpart_equal DEFARGS((sa1, sa2),
SOCKADDR *sa1 AND
SOCKADDR *sa2)
{
  if ((sa1->sa_family != sa2->sa_family) ||
      (sa1->sa_len != sa2->sa_len))
    return 0;

  switch(sa1->sa_family) {
  case AF_INET:
    return (((struct sockaddr_in *)sa1)->sin_addr.s_addr == 
	    ((struct sockaddr_in *)sa2)->sin_addr.s_addr);
#ifdef INET6
  case AF_INET6:
    return (IN6_ADDR_EQUAL(((struct sockaddr_in6 *)sa1)->sin6_addr, 
			   ((struct sockaddr_in6 *)sa2)->sin6_addr));
#endif /* INET6 */
  }
  return(0);
}

/*----------------------------------------------------------------------
 * key_inittables():
 *      Allocate space and initialize key engine tables
 ----------------------------------------------------------------------*/
int key_inittables __P((void))
{
  int i;

  KMALLOC(keyregtable, struct key_registry *, sizeof(struct key_registry));
  if (!keyregtable)
    return -1;
  bzero((caddr_t)keyregtable, sizeof(struct key_registry));
  KMALLOC(key_acquirelist, struct key_acquirelist *, 
	   sizeof(struct key_acquirelist));
  if (!key_acquirelist)
    return -1;
  bzero((caddr_t)key_acquirelist, sizeof(struct key_acquirelist));
  for (i = 0; i < KEYTBLSIZE; i++) 
    bzero((caddr_t)&keytable[i], sizeof(struct key_tblnode));
  for (i = 0; i < KEYALLOCTBLSIZE; i++)
    bzero((caddr_t)&keyalloctbl[i], sizeof(struct key_allocnode));
  for (i = 0; i < SO2SPITBLSIZE; i++)
    bzero((caddr_t)&so2spitbl[i], sizeof(struct key_so2spinode));

  return 0;
}

void key_cleanup __P((void))
{
  KFREE_LEN(keyregtable, sizeof(struct key_registry));
  keyregtable = NULL;
  KFREE_LEN(key_acquirelist, sizeof(struct key_acquirelist));
  key_acquirelist = NULL;
#ifdef KEY_DEBUG
  key_debug_free_all();
#endif /* KEY_DEBUG */
}

/*----------------------------------------------------------------------
 * key_gethashval():
 *      Determine keytable hash value.
 ----------------------------------------------------------------------*/
int key_gethashval DEFARGS((buf, len, tblsize),
char *buf AND
int len AND
int tblsize)
{
  int i, j = 0;

  /* 
   * Todo: Use word size xor and check for alignment
   *       and zero pad if necessary.  Need to also pick 
   *       a good hash function and table size.
   */
  if (len <= 0) {
    DPRINTF(IDL_ERROR,("key_gethashval got bogus len!\n"));
    return(-1);
  }
  for(i = 0; i < len; i++) {
    j ^=  (u_int8)(*(buf + i));
  }
  return (j % tblsize);
}


/*----------------------------------------------------------------------
 * key_createkey():
 *      Create hash key for hash function
 *      key is: type+src+dst if keytype = 1
 *              type+src+dst+spi if keytype = 0
 *      Uses only the address portion of the src and dst sockaddrs to 
 *      form key.  Currently handles only AF_INET and AF_INET6 sockaddrs
 ----------------------------------------------------------------------*/
int key_createkey DEFARGS((buf, type, src, dst, spi, keytype),
     char *buf AND
     u_int type AND
     SOCKADDR *src AND
     SOCKADDR *dst AND
     u_int32 spi AND
     u_int keytype)
{
  char *cp, *p;

  DPRINTF(IDL_MAJOR_EVENT, (">key_createkey(char *buf = %08x, u_int type = %x, SOCKADDR *src = %08x\n", (unsigned int)buf, type, (unsigned int)src));
  DPRINTF(IDL_MAJOR_EVENT, (" SOCKADDR *dst = %08x, u_int32 spi = %08x, u_int keytype=%x\n", (unsigned int)dst, (unsigned int)spi, (unsigned int)keytype));

  if (!buf || !src || !dst)
    return(-1);

  cp = buf;
  bcopy((caddr_t)&type, cp, sizeof(type));
  cp += sizeof(type);

#ifdef INET6
  /*
   * Assume only IPv4 and IPv6 addresses.
   */
#define ADDRPART(a) \
    ((a)->sa_family == AF_INET6) ? \
    (caddr_t)&(((struct sockaddr_in6 *)(a))->sin6_addr) : \
    (caddr_t)&(((struct sockaddr_in *)(a))->sin_addr)

#define ADDRSIZE(a) \
    ((a)->sa_family == AF_INET6) ? sizeof(struct in_addr6) : \
    sizeof(struct in_addr)  
#else /* INET6 */
#define ADDRPART(a) (caddr_t)&(((struct sockaddr_in *)(a))->sin_addr)
#define ADDRSIZE(a) sizeof(struct in_addr)  
#endif /* INET6 */

  DPRINTF(IDL_FINISHED,("src addr:\n"));
  DDO(IDL_FINISHED,dump_smart_sockaddr(src));
  DPRINTF(IDL_FINISHED,("dst addr:\n"));
  DDO(IDL_FINISHED,dump_smart_sockaddr(dst)); 

  p = ADDRPART(src);
  bcopy(p, cp, ADDRSIZE(src));
  cp += ADDRSIZE(src);

  p = ADDRPART(dst);
  bcopy(p, cp, ADDRSIZE(dst));
  cp += ADDRSIZE(dst);

#undef ADDRPART
#undef ADDRSIZE

  if (keytype == 0) {
    bcopy((caddr_t)&spi, cp, sizeof(spi));
    cp += sizeof(spi);
  }

  DPRINTF(IDL_FINISHED,("hash key:\n"));
  DDO(IDL_FINISHED, dump_buf(buf, cp - buf));
  return(cp - buf);
}


/*----------------------------------------------------------------------
 * key_sosearch():
 *      Search the so2spi table for the security association allocated to 
 *      the socket.  Returns pointer to a struct key_so2spinode which can
 *      be used to locate the security association entry in the keytable.
 ----------------------------------------------------------------------*/
struct key_so2spinode *key_sosearch DEFARGS((type, src, dst, so),
     u_int type AND
     SOCKADDR *src AND
     SOCKADDR *dst AND
     SOCKET *so)
{
  struct key_so2spinode *np = 0;

  if (!(src && dst)) {
    DPRINTF(IDL_ERROR,("key_sosearch: got null src or dst pointer!\n"));
    return(NULL);
  }

  for (np = so2spitbl[((u_int32)so) % SO2SPITBLSIZE].next; np; np = np->next) {
    if ((so == np->socket) && (type == np->keynode->secassoc->type)
	&& addrpart_equal(src, np->keynode->secassoc->src)
	&& addrpart_equal(dst, np->keynode->secassoc->dst))
      return(np);
  }  
  return(NULL);
}

/*----------------------------------------------------------------------
 * key_sodelete():
 *      Delete entries from the so2spi table.
 *        flag = 1  purge all entries
 *        flag = 0  delete entries with socket pointer matching socket  
 ----------------------------------------------------------------------*/
void key_sodelete DEFARGS((socket, flag),
     SOCKET *socket AND
     int flag)
{
  struct key_so2spinode *prevnp, *np;
  CRITICAL_DCL

  CRITICAL_START;

  DPRINTF(IDL_FINISHED,("Entering keysodelete w/so=0x%x flag=%d\n",
		     (unsigned int)socket,flag));

  if (flag) {
    int i;

    for (i = 0; i < SO2SPITBLSIZE; i++)
      for(np = so2spitbl[i].next; np; np = np->next) {
	KFREE_LEN(np, sizeof(struct key_so2spinode));
      }
    CRITICAL_END;
    return;
  }

  prevnp = &so2spitbl[((u_int32)socket) % SO2SPITBLSIZE];
  for(np = prevnp->next; np; np = np->next) {
    if (np->socket == socket) {
      struct socketlist *socklp, *prevsocklp;

      (np->keynode->alloc_count)--;

      /* 
       * If this socket maps to a unique secassoc,
       * we go ahead and delete the secassoc, since it
       * can no longer be allocated or used by any other 
       * socket.
       */
      if (np->keynode->secassoc->state & K_UNIQUE) {
	if (key_delete(np->keynode->secassoc) != 0)
	  panic("key_sodelete");
	np = prevnp;
	continue;
      }

      /*
       * We traverse the struct socketlist and remove the entry
       * for this socket
       */
      DPRINTF(IDL_FINISHED,("keysodelete: deleting from socklist..."));
      prevsocklp = np->keynode->solist;
      for (socklp = prevsocklp->next; socklp; socklp = socklp->next) {
	if (socklp->socket == socket) {
	  prevsocklp->next = socklp->next;
	  KFREE_LEN(socklp, sizeof(struct socketlist));
	  break;
	}
	prevsocklp = socklp;
      }
      DPRINTF(IDL_FINISHED,("done\n"));
      prevnp->next = np->next;
      KFREE_LEN(np, sizeof(struct key_so2spinode));
      np = prevnp;
    }
    prevnp = np;  
  }
  CRITICAL_END;
}


/*----------------------------------------------------------------------
 * key_deleteacquire():
 *      Delete an entry from the key_acquirelist
 ----------------------------------------------------------------------*/
void key_deleteacquire DEFARGS((type, target),
     u_int type AND
     SOCKADDR *target)
{
  struct key_acquirelist *ap, *prev;

  prev = key_acquirelist;
  for(ap = key_acquirelist->next; ap; ap = ap->next) {
    if (addrpart_equal(target, ap->target) &&
	(type == ap->type)) {
      DPRINTF(IDL_FINISHED,("Deleting entry from acquire list!\n"));
      prev->next = ap->next;
      if (ap->target)
        KFREE_LEN(ap->target, ap->target->sa_len);
      KFREE_LEN(ap, sizeof(struct key_acquirelist));
      ap = prev;
    }
    prev = ap;
  }
}


/*----------------------------------------------------------------------
 * key_search():
 *      Search the key table for an entry with same type, src addr, dest
 *      addr, and spi.  Returns a pointer to struct key_tblnode if found
 *      else returns null.
 ----------------------------------------------------------------------*/
struct key_tblnode *key_search DEFARGS(
			      (type, src, dst, spi, indx, prevkeynode),
     u_int type AND
     SOCKADDR *src AND
     SOCKADDR *dst AND
     u_int32 spi AND
     int indx AND
     struct key_tblnode **prevkeynode)
{
  struct key_tblnode *keynode, *prevnode;

  if (indx > KEYTBLSIZE || indx < 0)
    return (NULL);
  if (!(&keytable[indx]))
    return (NULL);

#define sec_type keynode->secassoc->type
#define sec_spi keynode->secassoc->spi
#define sec_src keynode->secassoc->src
#define sec_dst keynode->secassoc->dst

  prevnode = &keytable[indx];
  for (keynode = keytable[indx].next; keynode; keynode = keynode->next) {
    if ((type == sec_type) && (spi == sec_spi) && 
	addrpart_equal(src, sec_src)
	&& addrpart_equal(dst, sec_dst))
      break;
    prevnode = keynode;
  }
  *prevkeynode = prevnode;
  return(keynode);
}


/*----------------------------------------------------------------------
 * key_addnode():
 *      Insert a key_tblnode entry into the key table.  Returns a pointer 
 *      to the newly created key_tblnode.
 ----------------------------------------------------------------------*/
struct key_tblnode *key_addnode DEFARGS((indx, secassoc),
     int indx AND
     struct key_secassoc *secassoc)
{
  struct key_tblnode *keynode;

  DPRINTF(IDL_FINISHED,("Entering key_addnode w/indx=%d secassoc=0x%x\n",
			indx, (unsigned int)secassoc));

  if (!(&keytable[indx]))
    return(NULL);
  if (!secassoc) {
    panic("key_addnode: Someone passed in a null secassoc!\n");
  }

  KMALLOC(keynode, struct key_tblnode *, sizeof(struct key_tblnode));
  if (keynode == 0)
    return(NULL);
  bzero((caddr_t)keynode, sizeof(struct key_tblnode));

  KMALLOC(keynode->solist, struct socketlist *, sizeof(struct socketlist));
  if (keynode->solist == 0) {
    KFREE_LEN(keynode, sizeof(struct key_tblnode));
    return(NULL);
  }
  bzero((caddr_t)(keynode->solist), sizeof(struct socketlist));

  keynode->secassoc = secassoc;
  keynode->solist->next = NULL;
  keynode->next = keytable[indx].next;
  keytable[indx].next = keynode;

  return(keynode);
}


/*----------------------------------------------------------------------
 * key_add():
 *      Add a new security association to the key table.  Caller is
 *      responsible for allocating memory for the key_secassoc as  
 *      well as the buffer space for the key and iv.  Assumes the security 
 *      association passed in is well-formed.
 ----------------------------------------------------------------------*/
int
key_add DEFARGS((secassoc),
     struct key_secassoc *secassoc)
{
  char buf[MAXHASHKEYLEN];
  int len, indx;
  int inbound = 0;
  int outbound = 0;
  struct key_tblnode *keynode, *prevkeynode;
  struct key_allocnode *np = NULL;
  CRITICAL_DCL

  DPRINTF(IDL_FINISHED, ("Entering key_add w/secassoc=0x%x\n",
			 (unsigned int)secassoc));

  DDO(IDL_MAJOR_EVENT, dump_secassoc(secassoc));

  if (!secassoc) {
    panic("key_add: who the hell is passing me a null pointer");
  }

  /*
   * Should we allow a null key to be inserted into the table ? 
   * or can we use null key to indicate some policy action...
   */

  if (key_sanitycheck(secassoc))
    return -2;

  /*
   *  Check if secassoc with same spi exists before adding
   */
  bzero((caddr_t)&buf, sizeof(buf));
  len = key_createkey((caddr_t)&buf, secassoc->type, secassoc->src,
		      secassoc->dst, secassoc->spi, 0);
  indx = key_gethashval((caddr_t)&buf, len, KEYTBLSIZE);
  DPRINTF(IDL_FINISHED,("keyadd: keytbl hash position=%d\n", indx));
  keynode = key_search(secassoc->type, secassoc->src, secassoc->dst,
		       secassoc->spi, indx, &prevkeynode);
  if (keynode) {
    DPRINTF(IDL_FINISHED,("keyadd: secassoc already exists!\n"));
    return(-2);
  }

  inbound = my_addr(secassoc->dst);
  outbound = my_addr(secassoc->src);
  DPRINTF(IDL_FINISHED,("inbound=%d outbound=%d\n", inbound, outbound));

  /*
   * We allocate mem for an allocation entry if needed.
   * This is done here instead of in the allocaton code 
   * segment so that we can easily recover/cleanup from a 
   * memory allocation error.
   */
  if (outbound || (!inbound && !outbound)) {
    KMALLOC(np, struct key_allocnode *, sizeof(struct key_allocnode));
    if (np == 0) {
      DPRINTF(IDL_ERROR,("keyadd: can't allocate allocnode!\n"));
      return(-1);
    }
  }

  CRITICAL_START;

  DDO(IDL_MAJOR_EVENT, dump_secassoc(secassoc));
  if ((keynode = key_addnode(indx, secassoc)) == NULL) {
    CRITICAL_END;
    DPRINTF(IDL_ERROR,("keyadd: key_addnode failed!\n"));
    if (np)
      KFREE_LEN(np, sizeof(struct key_allocnode));
    return(-1);
  }
  DPRINTF(IDL_MAJOR_EVENT,("Added new keynode:\n"));
  DDO(IDL_MAJOR_EVENT, dump_keytblnode(keynode));
  DDO(IDL_MAJOR_EVENT, dump_secassoc(keynode->secassoc));
 
  /*
   *  We add an entry to the allocation table for
   *  this secassoc if the interfaces are up and
   *  the secassoc is outbound.  In the case 
   *  where the interfaces are not up, we go ahead
   *  and do it anyways.  This wastes an allocation
   *  entry if the secassoc later turned out to be
   *  inbound when the interfaces are ifconfig up.
   */
  if (outbound || (!inbound && !outbound)) {
    len = key_createkey((caddr_t)&buf, secassoc->type, secassoc->src,
			secassoc->dst, 0, 1);
    indx = key_gethashval((caddr_t)&buf, len, KEYALLOCTBLSIZE);
    DPRINTF(IDL_FINISHED,("keyadd: keyalloc hash position=%d\n", indx));
    np->keynode = keynode;
    np->next = keyalloctbl[indx].next;
    keyalloctbl[indx].next = np;
  }
  if (inbound)
    secassoc->state |= K_INBOUND;
  if (outbound)
    secassoc->state |= K_OUTBOUND;

  key_deleteacquire(secassoc->type, secassoc->dst);

  CRITICAL_END;
  return 0;
}


/*----------------------------------------------------------------------
 * key_get():
 *      Get a security association from the key table.
 ----------------------------------------------------------------------*/
int key_get DEFARGS((type, src, dst, spi, secassoc),
     u_int type AND
     SOCKADDR *src AND
     SOCKADDR *dst AND
     u_int32 spi AND
     struct key_secassoc **secassoc)
{
  char buf[MAXHASHKEYLEN];
  struct key_tblnode *keynode, *prevkeynode;
  int len, indx;

  bzero(&buf, sizeof(buf));
  *secassoc = NULL;
  len = key_createkey((caddr_t)&buf, type, src, dst, spi, 0);
  indx = key_gethashval((caddr_t)&buf, len, KEYTBLSIZE);
  DPRINTF(IDL_FINISHED,("keyget: indx=%d\n",indx));
  keynode = key_search(type, src, dst, spi, indx, &prevkeynode);
  if (keynode) {
    DPRINTF(IDL_FINISHED,("keyget: found it! keynode=0x%x",
			     (unsigned int)keynode));
    *secassoc = keynode->secassoc;
    return(0);
  } else
    return(-1);  /* Not found */
}


/*----------------------------------------------------------------------
 * key_dump():
 *      Dump all valid entries in the keytable to a pf_key socket.  Each
 *      security associaiton is sent one at a time in a pf_key message.  A
 *      message with seqno = 0 signifies the end of the dump transaction.
 ----------------------------------------------------------------------*/
int key_dump DEFARGS((so),
     SOCKET *so)
{
  int len, i, j;
  int seq = 1;
  struct key_msgdata keyinfo;
  struct key_msghdr *km;
  struct key_tblnode *keynode;

  /*
   * Routine to dump the key table to a routing socket
   * Use for debugging only!
   */

  KMALLOC(km, struct key_msghdr *, sizeof(struct key_msghdr) +
	  3 * MAX_SOCKADDR_SZ + MAX_KEY_SZ + MAX_IV_SZ);
  if (!km)
    return(ENOBUFS);

  DPRINTF(IDL_FINISHED,("Entering key_dump()"));
  /* 
   * We need to speed this up later.  Fortunately, key_dump 
   * messages are not sent often.
   */
  for (i = 0; i < KEYTBLSIZE; i++) {
    for (keynode = keytable[i].next; keynode; keynode = keynode->next) {
      /*
       * We exclude dead/larval/zombie security associations for now
       * but it may be useful to also send these up for debugging purposes
       */
      if (keynode->secassoc->state & (K_DEAD | K_LARVAL | K_ZOMBIE))
	continue;

      len = (sizeof(struct key_msghdr) +
	     ROUNDUP(keynode->secassoc->src->sa_len) + 
	     ROUNDUP(keynode->secassoc->dst->sa_len) +
	     ROUNDUP(keynode->secassoc->from->sa_len) +
	     ROUNDUP(keynode->secassoc->keylen) + 
	     ROUNDUP(keynode->secassoc->ivlen));

      if (key_secassoc2msghdr(keynode->secassoc, km, &keyinfo) != 0)
	panic("key_dump");

      km->key_msglen = len;
      km->key_msgvers = KEY_VERSION;
      km->key_msgtype = KEY_DUMP;
      km->key_pid = CURRENT_PID;
      km->key_seq = seq++;
      km->key_errno = 0;

      if ((j = key_sendup(so, km, 0)))
        return ENOBUFS;
    }
  }
  bzero((caddr_t)km, sizeof(struct key_msghdr));
  km->key_msglen = sizeof(struct key_msghdr);
  km->key_msgvers = KEY_VERSION;
  km->key_msgtype = KEY_DUMP;
  km->key_pid = CURRENT_PID;
  km->key_seq = 0;
  km->key_errno = 0;

  if ((j = key_sendup(so, km, 0)))
    return ENOBUFS;
  KFREE(km);
  DPRINTF(IDL_FINISHED,("Leaving key_dump()\n"));  
  return(0);
}

/*----------------------------------------------------------------------
 * key_delete():
 *      Delete a security association from the key table.
 ----------------------------------------------------------------------*/
int key_delete DEFARGS((secassoc),
     struct key_secassoc *secassoc)
{
  char buf[MAXHASHKEYLEN];
  int len, indx;
  struct key_tblnode *keynode = 0;
  struct key_tblnode *prevkeynode = 0;
  struct socketlist *socklp, *deadsocklp;
  struct key_so2spinode *np, *prevnp;
  struct key_allocnode *ap, *prevap;
  CRITICAL_DCL

  DPRINTF(IDL_MAJOR_EVENT,(">key_delete(struct key_secassoc *secassoc=%08x)\n", (unsigned int)secassoc));

  DDO(IDL_MAJOR_EVENT, dump_secassoc(secassoc));

  bzero((caddr_t)buf, sizeof(buf));
  len = key_createkey((caddr_t)buf, secassoc->type, secassoc->src,
		      secassoc->dst, secassoc->spi, 0);
  indx = key_gethashval((caddr_t)buf, len, KEYTBLSIZE);
  DPRINTF(IDL_FINISHED,("keydelete: keytbl hash position=%d\n", indx));
  if (!(keynode = key_search(secassoc->type, secassoc->src, secassoc->dst, 
		       secassoc->spi, indx, &prevkeynode))) {
    DPRINTF(IDL_ERROR, ("key_delete: key_search failed\n"));
    return -1;
  }
 
  CRITICAL_START;
  DPRINTF(IDL_FINISHED,("keydelete: found keynode to delete\n"));
  keynode->secassoc->state |= K_DEAD;
  
  if (keynode->ref_count > 0) {
    DPRINTF(IDL_FINISHED,("keydelete: secassoc still held, marking for deletion only!\n"));
    CRITICAL_END;
    return(0); 
  }

  prevkeynode->next = keynode->next;
    
  /*
   *  Walk the struct socketlist and delete the
   *  entries mapping sockets to this secassoc
   *  from the so2spi table.
   */
  DPRINTF(IDL_FINISHED,("keydelete: deleting socklist..."));
  for(socklp = keynode->solist->next; socklp; ) {
    prevnp = &so2spitbl[((u_int32)(socklp->socket)) % SO2SPITBLSIZE];
    for(np = prevnp->next; np; np = np->next) {
      if ((np->socket == socklp->socket) && (np->keynode == keynode)) {
	prevnp->next = np->next;
	KFREE_LEN(np, sizeof(struct key_so2spinode));
	break; 
	}
      prevnp = np;  
    }
    deadsocklp = socklp;
    socklp = socklp->next;
    KFREE_LEN(deadsocklp, sizeof(struct socketlist));
  }
  DPRINTF(IDL_FINISHED,("done\n"));
  /*
   * If an allocation entry exist for this
   * secassoc, delete it.
   */
  bzero((caddr_t)&buf, sizeof(buf));
  len = key_createkey((caddr_t)&buf, secassoc->type, secassoc->src,
		      secassoc->dst, 0, 1);
  indx = key_gethashval((caddr_t)&buf, len, KEYALLOCTBLSIZE);
  DPRINTF(IDL_FINISHED,("keydelete: alloctbl hash position=%d\n", indx));
  prevap = &keyalloctbl[indx];
  for (ap = prevap->next; ap; ap = ap->next) {
    if (ap->keynode == keynode) {
      prevap->next = ap->next;
      KFREE_LEN(ap, sizeof(struct key_allocnode));
      break; 
    }
    prevap = ap;
  }    
  
  if (keynode->secassoc)
    key_freeassoc(keynode->secassoc);
  if (keynode->solist)
    KFREE_LEN(keynode->solist, sizeof(struct socketlist));
  KFREE_LEN(keynode, sizeof(struct key_tblnode));
  CRITICAL_END;
  return(0);
}


/*----------------------------------------------------------------------
 * key_flush():
 *      Delete all entries from the key table.
 ----------------------------------------------------------------------*/
void key_flush __P((void))
{
  struct key_tblnode *keynode;
  int i;

  DPRINTF(IDL_MAJOR_EVENT, (">key_flush(void)\n"));
  /* 
   * This is slow, but simple.
   */
  for (i = 0; i < KEYTBLSIZE; i++) {
    while ((keynode = keytable[i].next)) {
      DPRINTF(IDL_MAJOR_EVENT, ("key_flush: keynode=%08x\n", (unsigned int)keynode));
      if (key_delete(keynode->secassoc) != 0)
	panic("key_flush");
    }
  }
  DPRINTF(IDL_MAJOR_EVENT, ("<key_flush\n"));
}


/*----------------------------------------------------------------------
 * key_getspi():
 *      Get a unique spi value for a key management daemon/program.  The 
 *      spi value, once assigned, cannot be assigned again (as long as the 
 *      entry with that same spi value remains in the table).
 ----------------------------------------------------------------------*/
int key_getspi DEFARGS((type, src, dst, lowval, highval, spi),
     u_int type AND
     SOCKADDR *src AND
     SOCKADDR *dst AND
     u_int32 lowval AND
     u_int32 highval AND
     u_int32 *spi)
{
  struct key_secassoc *secassoc;
  struct key_tblnode *keynode, *prevkeynode;
  int count, done, len, indx;
  int maxcount = 1000;
  u_int32 val;
  char buf[MAXHASHKEYLEN];
  CRITICAL_DCL
  
  DPRINTF(IDL_FINISHED,("Entering getspi w/type=%d,low=%u,high=%u\n",
			   type, (int)lowval, (int)highval));
  if (!(src && dst))
    return(EINVAL);

  if ((lowval == 0) || (highval == 0))
    return(EINVAL);

  if (lowval > highval) {
    u_int32 temp;
    temp = lowval;
    lowval = highval;
    highval = lowval;
  }

  done = count = 0;
  do {
    count++;
    /* 
     *  This may not be "random enough".
     */
    val = lowval + (random() % (highval - lowval + 1));

    if (lowval == highval)
      count = maxcount;
    DPRINTF(IDL_FINISHED,("%u ",(unsigned int)val));
    if (val) {
      DPRINTF(IDL_FINISHED,("\n"));
      bzero(&buf, sizeof(buf));
      len = key_createkey((caddr_t)&buf, type, src, dst, val, 0);
      indx = key_gethashval((caddr_t)&buf, len, KEYTBLSIZE);
      if (!key_search(type, src, dst, val, indx, &prevkeynode)) {
	if (!(secassoc = key_allocassoc(src, dst, NULL))) {
	  DPRINTF(IDL_ERROR,("key_getspi: can't allocate memory for secassoc\n"));
	  return(ENOBUFS);
	}

	DPRINTF(IDL_FINISHED,("getspi: indx=%d\n",indx));
	secassoc->len = sizeof(struct key_secassoc);
	secassoc->type = type;
	secassoc->spi = val;
	secassoc->state |= K_LARVAL;
	if (my_addr(secassoc->dst))
	  secassoc->state |= K_INBOUND;
	if (my_addr(secassoc->src))
	  secassoc->state |= K_OUTBOUND;

	/* We fill this in with a plausable value now to insure
	   that other routines don't break. These will get
	   overwritten later with the correct values. */
#ifdef INET6
	secassoc->from->sa_family = AF_INET6;
#else /* INET6 */
	secassoc->from->sa_family = AF_INET;
#endif /* INET6 */
	secassoc->from->sa_len = SOCKADDR_MAXSZ;

	/* 
	 * We need to add code to age these larval key table
	 * entries so they don't linger forever waiting for
	 * a KEY_UPDATE message that may not come for various
	 * reasons.  This is another task that key_reaper can
	 * do once we have it coded.
	 */
	secassoc->lifetime1 += TIME_SECONDS + maxlarvallifetime;

	CRITICAL_START;
	if (!(keynode = key_addnode(indx, secassoc))) {
	  CRITICAL_END;
	  DPRINTF(IDL_ERROR,("key_getspi: can't add node\n"));
	  return(ENOBUFS);
	}
	CRITICAL_END;
	DPRINTF(IDL_FINISHED,("key_getspi: added node 0x%x\n",
			      (unsigned int)keynode));
	done++;
      }
    }
  } while ((count < maxcount) && !done);
  DPRINTF(IDL_FINISHED,("getspi returns w/spi=%u,count=%d\n",(unsigned int)val,count));
  if (done) {
    *spi = val;
    return(0);
  } else {
    *spi = 0;
    return(EADDRNOTAVAIL);
  }
}


/*----------------------------------------------------------------------
 * key_update():
 *      Update a keytable entry that has an spi value assigned but is 
 *      incomplete (e.g. no key/iv).
 ----------------------------------------------------------------------*/
int key_update DEFARGS((secassoc),
     struct key_secassoc *secassoc)
{
  struct key_secassoc *old;
  struct key_tblnode *keynode, *prevkeynode;
  struct key_allocnode *np = 0;
  u_int8 newstate;
  int len, indx, inbound, outbound;
  char buf[MAXHASHKEYLEN];
  CRITICAL_DCL

  bzero(&buf, sizeof(buf));
  len = key_createkey((caddr_t)&buf, secassoc->type, secassoc->src,
		      secassoc->dst, secassoc->spi, 0);
  indx = key_gethashval((caddr_t)&buf, len, KEYTBLSIZE);
  if(!(keynode = key_search(secassoc->type, secassoc->src, secassoc->dst, 
			    secassoc->spi, indx, &prevkeynode))) {  
    return(ESRCH);
  }
  if (keynode->secassoc->state & K_DEAD)
    return(ESRCH);

  /* Should we also restrict updating of only LARVAL entries ? */

  inbound = my_addr(secassoc->dst);
  outbound = my_addr(secassoc->src);

  newstate = keynode->secassoc->state;
  newstate &= ~K_LARVAL;

  if (inbound)
    newstate |= K_INBOUND;
  if (outbound)
    newstate |= K_OUTBOUND;

  if (outbound || (!inbound && !outbound)) {
    KMALLOC(np, struct key_allocnode *, sizeof(struct key_allocnode));
    if (np == 0) {
      DPRINTF(IDL_ERROR,("keyupdate: can't allocate allocnode!\n"));
      return(ENOBUFS);
    }
  }

  secassoc->state = newstate;

  /*
   * Should we allow a null key to be inserted into the table ? 
   * or can we use null key to indicate some policy action...
   */

  if (key_sanitycheck(secassoc))
    return(EINVAL) ;

  CRITICAL_START;

  old = keynode->secassoc;
  keynode->secassoc = secassoc;

  /*
   *  We now add an entry to the allocation table for this 
   *  updated key table entry.
   */
  if (outbound || (!inbound && !outbound)) {
    len = key_createkey((caddr_t)buf, secassoc->type, secassoc->src,
			secassoc->dst, 0, 1);
    indx = key_gethashval((caddr_t)buf, len, KEYALLOCTBLSIZE);
    DPRINTF(IDL_FINISHED,("keyupdate: keyalloc hash position=%d\n", indx));
    np->keynode = keynode;
    np->next = keyalloctbl[indx].next;
    keyalloctbl[indx].next = np;
  }

  key_deleteacquire(secassoc->type, secassoc->dst);

  CRITICAL_END;

  key_freeassoc(old);

  return(0);
}

/*----------------------------------------------------------------------
 * key_register():
 *      Register a socket as one capable of acquiring security associations
 *      for the kernel.
 ----------------------------------------------------------------------*/
int key_register DEFARGS((socket, type),
     SOCKET *socket AND
     u_int type)
{
  struct key_registry *p, *new;
  CRITICAL_DCL

  CRITICAL_START;

  DPRINTF(IDL_FINISHED,("Entering key_register w/so=0x%x,type=%d\n",
		     (unsigned int)socket,type));

  if (!(keyregtable && socket))
    panic("key_register");
  
  /*
   * Make sure entry is not already in table
   */
  for(p = keyregtable->next; p; p = p->next) {
    if ((p->type == type) && (p->socket == socket)) {
      CRITICAL_END;
      return(EEXIST);
    }
  }

  KMALLOC(new, struct key_registry *, sizeof(struct key_registry));  
  if (new == 0) {
    CRITICAL_END;
    return(ENOBUFS);
  }
  new->type = type;
  new->socket = socket;
  new->next = keyregtable->next;
  keyregtable->next = new;
  CRITICAL_END;
  return(0);
}

/*----------------------------------------------------------------------
 * key_unregister():
 *      Delete entries from the registry list.
 *         allflag = 1 : delete all entries with matching socket
 *         allflag = 0 : delete only the entry matching socket and type
 ----------------------------------------------------------------------*/
void key_unregister DEFARGS((socket, type, allflag),
     SOCKET *socket AND
     u_int type AND
     int allflag)
{
  struct key_registry *p, *prev;
  CRITICAL_DCL

  CRITICAL_START;

  DPRINTF(IDL_FINISHED,("Entering key_unregister w/so=0x%x,type=%d,flag=%d\n",
		     (unsigned int)socket, type, allflag));

  if (!(keyregtable && socket))
    panic("key_register");
  prev = keyregtable;
  for(p = keyregtable->next; p; p = p->next) {
    if ((allflag && (p->socket == socket)) ||
	((p->type == type) && (p->socket == socket))) {
      prev->next = p->next;
      KFREE_LEN(p, sizeof(struct key_registry));
      p = prev;
    }
    prev = p;
  }
  CRITICAL_END;
}


/*----------------------------------------------------------------------
 * key_acquire():
 *      Send a key_acquire message to all registered key mgnt daemons 
 *      capable of acquire security association of type type.
 *
 *      Return: 0 if succesfully called key mgnt. daemon(s)
 *              -1 if not successfull.
 ----------------------------------------------------------------------*/
int key_acquire DEFARGS((type, src, dst),
     u_int type AND
     SOCKADDR *src AND
     SOCKADDR *dst)
{
  struct key_registry *p;
  struct key_acquirelist *ap, *prevap;
  int success = 0, created = 0;
  u_int etype;
  struct key_msghdr *km = NULL;
  int len = 0;

  DPRINTF(IDL_MAJOR_EVENT, (">key_acquire(u_int type=%x, SOCKADDR *src=%08x, SOCKADDR *dst=%08x)\n", (unsigned int)type, (unsigned int)src, (unsigned int)dst));

  if (!keyregtable) {
    DPRINTF(IDL_ERROR, ("key_acquire: keyregtable == NULL!\n"));
    return (-1);
  }

  if (!src || !dst) {
    DPRINTF(IDL_ERROR, ("key_acquire: passed a NULL address\n"));
    return -1;
  }

  /*
   * We first check the acquirelist to see if a key_acquire
   * message has been sent for this destination.
   */
  etype = type;
  prevap = key_acquirelist;
  for(ap = key_acquirelist->next; ap; ap = ap->next) {
    if (addrpart_equal(dst, ap->target) &&
	(etype == ap->type)) {
      DPRINTF(IDL_FINISHED,("acquire message previously sent!\n"));
      if (ap->expiretime < TIME_SECONDS) {
	DPRINTF(IDL_FINISHED,("acquire message has expired!\n"));
	ap->count = 0;
	break;
      }
      if (ap->count < maxkeyacquire) {
	DPRINTF(IDL_FINISHED,("max acquire messages not yet exceeded!\n"));
	break;
      }
      return(0);
    } else if (ap->expiretime < TIME_SECONDS) {
      /*
       *  Since we're already looking at the list, we may as
       *  well delete expired entries as we scan through the list.
       *  This should really be done by a function like key_reaper()
       *  but until we code key_reaper(), this is a quick and dirty
       *  hack.
       */
      DPRINTF(IDL_FINISHED,("found an expired entry...deleting it!\n"));
      prevap->next = ap->next;
      if (ap->target)
        KFREE_LEN(ap->target, ap->target->sa_len);
      KFREE_LEN(ap, sizeof(struct key_acquirelist));
      ap = prevap;
    }
    prevap = ap;
  }

  /*
   * Scan registry and send KEY_ACQUIRE message to 
   * appropriate key management daemons.
   */  
  for(p = keyregtable->next; p; p = p->next) {
    if (p->type != type) 
      continue;

    if (!created) {      
      len = sizeof(struct key_msghdr) + ROUNDUP(src->sa_len) + 
	ROUNDUP(dst->sa_len);
      KMALLOC(km, struct key_msghdr *, len);
      if (!km) {
	DPRINTF(IDL_ERROR,("key_acquire: no memory\n"));
	return(-1);
      }
      DPRINTF(IDL_FINISHED,("key_acquire/created: 1\n"));
      bzero((caddr_t)km, len);
      km->key_msglen = len;
      km->key_msgvers = KEY_VERSION;
      km->key_msgtype = KEY_ACQUIRE;
      km->type = type;
      DPRINTF(IDL_FINISHED,("key_acquire/created: 2\n"));
      /*
       * This is inefficient and slow.
       */

      /*
       * We zero out sin_zero here for AF_INET addresses because
       * ip_output() currently does not do it for performance reasons.
       */
      if (src->sa_family == AF_INET)
	bzero((caddr_t)(((struct sockaddr_in *)src)->sin_zero),
	      sizeof(((struct sockaddr_in *)src)->sin_zero));
      if (dst->sa_family == AF_INET)
	bzero((caddr_t)(((struct sockaddr_in *)dst)->sin_zero), 
	      sizeof(((struct sockaddr_in *)dst)->sin_zero));

      bcopy((caddr_t)src, (caddr_t)(km + 1), src->sa_len);
      bcopy((caddr_t)dst, (caddr_t)((int)(km + 1) + ROUNDUP(src->sa_len)),
	    dst->sa_len);
      DPRINTF(IDL_FINISHED,("key_acquire/created: 3\n"));
      created++; 
    }
    if (!key_sendup(p->socket, km, 0))
      success++;
  }

  if (km)
    KFREE_LEN(km, len);
      
  /*
   *  Update the acquirelist 
   */
  if (success) {
    if (!ap) {
      DPRINTF(IDL_FINISHED,("Adding new entry in acquirelist\n"));
      KMALLOC(ap, struct key_acquirelist *, sizeof(struct key_acquirelist));
      if (!ap)
	return (success ? 0 : -1);

      bzero((caddr_t)ap, sizeof(struct key_acquirelist));

      KMALLOC(ap->target, SOCKADDR *, dst->sa_len);
      if (!ap->target)
	return(success ? 0 : -1);

      bcopy((caddr_t)dst, (caddr_t)ap->target, dst->sa_len);
      ap->type = etype;
      ap->next = key_acquirelist->next;
      key_acquirelist->next = ap;
    }
    DPRINTF(IDL_FINISHED,("Updating acquire counter and expiration time\n"));
    ap->count++;
    ap->expiretime = TIME_SECONDS + maxacquiretime;
  }
  DPRINTF(IDL_FINISHED,("key_acquire: done! success=%d\n",success));
  return(success ? 0 : -1);
}

/*----------------------------------------------------------------------
 * key_alloc():
 *      Allocate a security association to a socket.  A socket requesting 
 *      unique keying (per-socket keying) is assigned a security assocation
 *      exclusively for its use.  Sockets not requiring unique keying are
 *      assigned the first security association which may or may not be
 *      used by another socket.
 ----------------------------------------------------------------------*/
int key_alloc DEFARGS((type, src, dst, socket, unique_key, keynodep),
     u_int type AND
     SOCKADDR *src AND
     SOCKADDR *dst AND
     SOCKET *socket AND
     u_int  unique_key AND
     struct key_tblnode **keynodep)
{
  struct key_tblnode *keynode;
  char buf[MAXHASHKEYLEN];
  struct key_allocnode *np, *prevnp;
  struct key_so2spinode *newnp;
  int len;
  int indx;

  DPRINTF(IDL_FINISHED,("Entering key_alloc w/type=%u!\n",type));
  if (!(src && dst)) {
    DPRINTF(IDL_ERROR,("key_alloc: received null src or dst!\n"));
    return(-1);
  }

  /*
   * Search key allocation table
   */
  bzero((caddr_t)&buf, sizeof(buf));
  len = key_createkey((caddr_t)&buf, type, src, dst, 0, 1);
  indx = key_gethashval((caddr_t)&buf, len, KEYALLOCTBLSIZE);  

#define np_type np->keynode->secassoc->type
#define np_state np->keynode->secassoc->state
#define np_src np->keynode->secassoc->src
#define np_dst np->keynode->secassoc->dst
  
  prevnp = &keyalloctbl[indx];
  for (np = keyalloctbl[indx].next; np; np = np->next) {
    if ((type == np_type) && addrpart_equal(src, np_src) &&
	addrpart_equal(dst, np_dst) &&
	!(np_state & (K_LARVAL | K_DEAD | K_UNIQUE))) {
      if (!(unique_key))
	break;
      if (!(np_state & K_USED)) 
	break;
    }
    prevnp = np;
  }

  if (np) {
    struct socketlist *newsp;
    CRITICAL_DCL

    CRITICAL_START;

    DPRINTF(IDL_FINISHED,("key_alloc: found node to allocate\n"));
    keynode = np->keynode;

    KMALLOC(newnp, struct key_so2spinode *, sizeof(struct key_so2spinode));
    if (newnp == 0) {
      DPRINTF(IDL_ERROR,("key_alloc: Can't alloc mem for so2spi node!\n"));
      CRITICAL_END;
      return(ENOBUFS);
    }
    KMALLOC(newsp, struct socketlist *, sizeof(struct socketlist));
    if (newsp == 0) {
      DPRINTF(IDL_ERROR,("key_alloc: Can't alloc mem for struct socketlist!\n"));
      if (newnp)
	KFREE_LEN(newnp, sizeof(struct key_so2spinode));
      CRITICAL_END;
      return(ENOBUFS);
    }

    /*
     * Add a hash entry into the so2spi table to
     * map socket to allocated secassoc.
     */
    DPRINTF(IDL_FINISHED,("key_alloc: adding entry to so2spi table..."));
    newnp->keynode = keynode;
    newnp->socket = socket;
    newnp->next = so2spitbl[((u_int32)socket) % SO2SPITBLSIZE].next; 
    so2spitbl[((u_int32)socket) % SO2SPITBLSIZE].next = newnp;
    DPRINTF(IDL_FINISHED,("done\n"));

    if (unique_key) {
      /*
       * Need to remove the allocation entry
       * since the secassoc is now unique and 
       * can't be allocated to any other socket
       */
      DPRINTF(IDL_FINISHED,("key_alloc: making keynode unique..."));
      keynode->secassoc->state |= K_UNIQUE;
      prevnp->next = np->next;
      KFREE_LEN(np, sizeof(struct key_so2spinode));
      DPRINTF(IDL_FINISHED,("done\n"));
    }
    keynode->secassoc->state |= K_USED;
    keynode->secassoc->state |= K_OUTBOUND;
    keynode->alloc_count++;

    /*
     * Add socket to list of socket using secassoc.
     */
    DPRINTF(IDL_FINISHED,("key_alloc: adding so to solist..."));
    newsp->socket = socket;
    newsp->next = keynode->solist->next;
    keynode->solist->next = newsp;
    DPRINTF(IDL_FINISHED,("done\n"));
    *keynodep = keynode;
    CRITICAL_END;
    return(0);
  } 
  *keynodep = NULL;
  return(0);
}


/*----------------------------------------------------------------------
 * key_free():
 *      Decrement the refcount for a key table entry.  If the entry is 
 *      marked dead, and the refcount is zero, we go ahead and delete it.
 ----------------------------------------------------------------------*/
void key_free DEFARGS((keynode),
     struct key_tblnode *keynode)
{
  DPRINTF(IDL_FINISHED,("Entering key_free w/keynode=0x%x\n",
			   (unsigned int)keynode));
  if (!keynode) {
    DPRINTF(IDL_ERROR,("Warning: key_free got null pointer\n"));
    return;
  }
  (keynode->ref_count)--;
  if (keynode->ref_count < 0) {
    DPRINTF(IDL_ERROR,("Warning: key_free decremented refcount to %d\n",keynode->ref_count));
  }
  if ((keynode->secassoc->state & K_DEAD) && (keynode->ref_count <= 0)) {
    DPRINTF(IDL_FINISHED,("key_free: calling key_delete\n"));
    key_delete(keynode->secassoc);
  }
}

/*----------------------------------------------------------------------
 * getassocbyspi():
 *      Get a security association for a given type, src, dst, and spi.
 *
 *      Returns: 0 if sucessfull
 *               -1 if error/not found
 *
 *      Caller must convert spi to host order.  Function assumes spi is  
 *      in host order!
 ----------------------------------------------------------------------*/
int getassocbyspi DEFARGS((type, src, dst, spi, keyentry),
     u_int type AND
     SOCKADDR *src AND
     SOCKADDR *dst AND
     u_int32 spi AND
     struct key_tblnode **keyentry)
{
  char buf[MAXHASHKEYLEN];
  int len, indx;
  struct key_tblnode *keynode, *prevkeynode = 0;

  DPRINTF(IDL_FINISHED,("Entering getassocbyspi w/type=%u spi=%u\n",(unsigned int)type,(unsigned int)spi));

  *keyentry = NULL;
  bzero((caddr_t)buf, sizeof(buf));
  len = key_createkey((caddr_t)buf, type, src, dst, spi, 0);
  indx = key_gethashval((caddr_t)buf, len, KEYTBLSIZE);
  DPRINTF(IDL_FINISHED,("getassocbyspi: indx=%d\n",indx));
  DDO(IDL_FINISHED,dump_sockaddr(src);dump_sockaddr(dst));
  keynode = key_search(type, src, dst, spi, indx, &prevkeynode);
  DPRINTF(IDL_FINISHED,("getassocbyspi: keysearch ret=0x%x\n",
			(unsigned int)keynode));
  if (keynode && !(keynode->secassoc->state & (K_DEAD | K_LARVAL))) {
    DPRINTF(IDL_FINISHED,("getassocbyspi: found secassoc!\n"));
    (keynode->ref_count)++;
    keynode->secassoc->state |= K_USED;
    *keyentry = keynode;
  } else {
    DPRINTF(IDL_FINISHED,("getassocbyspi: secassoc not found!\n"));
    return (-1);
  }
  return(0);
}


/*----------------------------------------------------------------------
 * getassocbysocket():
 *      Get a security association for a given type, src, dst, and socket.
 *      If not found, try to allocate one.
 *      Returns: 0 if successfull
 *              -1 if error condition/secassoc not found (*keyentry = NULL)
 *               1 if secassoc temporarily unavailable (*keynetry = NULL)
 *                 (e.g., key mgnt. daemon(s) called)
 ----------------------------------------------------------------------*/
int getassocbysocket DEFARGS((type, src, dst, socket, unique_key, keyentry),
     u_int type AND
     SOCKADDR *src AND
     SOCKADDR *dst AND
     SOCKET *socket AND
     u_int unique_key AND
     struct key_tblnode **keyentry)
{
  struct key_tblnode *keynode = 0;
  struct key_so2spinode *np;
  u_int realtype;
 
  DPRINTF(IDL_MAJOR_EVENT, (">getassocbysocket(type=%x, ..., so=%08x, unique_key=%x, ...), \n", type,(unsigned int)socket, unique_key));

  /*
   *  We treat esp-transport mode and esp-tunnel mode 
   *  as a single type in the keytable.  This has a side
   *  effect that socket using both esp-transport and
   *  esp-tunnel will use the same security association
   *  for both modes.  Is this a problem?
   */
  realtype = type;
  if ((np = key_sosearch(type, src, dst, socket))) {
    if (np->keynode && np->keynode->secassoc && 
	!(np->keynode->secassoc->state & (K_DEAD | K_LARVAL))) {
      DPRINTF(IDL_FINISHED,("getassocbysocket: found secassoc!\n"));
      (np->keynode->ref_count)++;
      *keyentry = np->keynode;
      return(0);
    }
  }

  /*
   * No secassoc has been allocated to socket, 
   * so allocate one, if available
   */
  DPRINTF(IDL_FINISHED,("getassocbyso: can't find it, trying to allocate!\n"));
  if (key_alloc(realtype, src, dst, socket, unique_key, &keynode) == 0) {
    if (keynode) {
      DPRINTF(IDL_FINISHED,("getassocbyso: key_alloc found secassoc!\n"));
      keynode->ref_count++;
      *keyentry = keynode;
      return(0);
    } else {
      /* 
       * Kick key mgnt. daemon(s) 
       * (this should be done in ipsec_output_policy() instead or
       * selectively called based on a flag value)
       */
      DPRINTF(IDL_FINISHED,("getassocbyso: calling key mgnt daemons!\n"));
      *keyentry = NULL;
      if (key_acquire(realtype, src, dst) == 0)
	return (1);
      else
	return(-1);
    }
  }
  *keyentry = NULL;
  return(-1);
}

/*----------------------------------------------------------------------
 * key_xdata():
 *      Parse message buffer for src/dst/from/iv/key if parseflag = 0
 *      else parse for src/dst only.
 ----------------------------------------------------------------------*/
int key_xdata DEFARGS((km, kip, parseflag),
     struct key_msghdr *km AND
     struct key_msgdata *kip AND
     int parseflag)
{
  char *cp, *cpmax;

  if (!km || (km->key_msglen <= 0))
    return (-1);

  cp = (caddr_t)km + sizeof(struct key_msghdr);
  cpmax = (caddr_t)km + km->key_msglen;

  /*
   * Assumes user process passes message with 
   * correct word alignment.
   */

  /* 
   * Need to clean up this code later.  
   */

  /* Grab src addr */
  kip->src = (SOCKADDR *)cp;
  if (!kip->src->sa_len) {
    DPRINTF(IDL_FINISHED,("key_xdata couldn't parse src addr\n"));
    return(-1);
  }

  ADVANCE(cp, kip->src->sa_len);

  /* Grab dest addr */
  kip->dst = (SOCKADDR *)cp;
  if (!kip->dst->sa_len) {
    DPRINTF(IDL_FINISHED,("key_xdata couldn't parse dest addr\n"));
    return(-1);
  }

  ADVANCE(cp, kip->dst->sa_len);
  if (parseflag == 1) {
    kip->from = 0;
    kip->key = kip->iv = 0;
    kip->keylen = kip->ivlen = 0;
    return(0);
  }
 
  /* Grab from addr */
  kip->from = (SOCKADDR *)cp;
  if (!kip->from->sa_len) {
    DPRINTF(IDL_FINISHED,("key_xdata couldn't parse from addr\n"));
    return(-1);
  }

  ADVANCE(cp, kip->from->sa_len);
 
  /* Grab key */
  if ((kip->keylen = km->keylen)) {
    kip->key = cp;
    ADVANCE(cp, km->keylen);
  } else 
    kip->key = 0;

  /* Grab iv */
  if ((kip->ivlen = km->ivlen))
    kip->iv = cp;
  else
    kip->iv = 0;

  return (0);
}


int key_parse DEFARGS((kmp, so, dstfamily),
struct key_msghdr **kmp AND
SOCKET *so AND
int *dstfamily)
{
  int error = 0, keyerror = 0;
  struct key_msgdata keyinfo;
  struct key_secassoc *secassoc = NULL;
  struct key_msghdr *km = *kmp;

  DPRINTF(IDL_FINISHED, (">key_parse(struct key_msghdr **kmp = %08x,\n", (unsigned int)kmp));
  DPRINTF(IDL_FINISHED, (" SOCKET *so = %08x, int *dstfamily = %08x\n", (unsigned int)so, (unsigned int)dstfamily));
  DP(FINISHED, (unsigned int)*kmp, 08x);
  DDO(IDL_FINISHED, if (*dstfamily) DP(FINISHED, *dstfamily, 08x));

#define senderr(e) \
  { error = (e); goto flush; }

  DPRINTF(IDL_FINISHED, ("0\n"));

  if (km->key_msgvers != KEY_VERSION) {
    DPRINTF(IDL_ERROR,("keyoutput: Unsupported key message version!\n"));
    senderr(EPROTONOSUPPORT);
  }

  DPRINTF(IDL_FINISHED, ("1\n"));

  km->key_pid = CURRENT_PID;

  DDO(IDL_FINISHED, printf("keymsghdr:\n"); dump_keymsghdr(km));

  /*
   * Parse buffer for src addr, dest addr, from addr, key, iv
   */
  bzero((caddr_t)&keyinfo, sizeof(keyinfo));

  switch (km->key_msgtype) {
  case KEY_ADD:
    DPRINTF(IDL_FINISHED,("key_output got KEY_ADD msg\n"));

    if (key_xdata(km, &keyinfo, 0) < 0)
      goto parsefail;

    if (!(secassoc = key_allocassoc(keyinfo.src, keyinfo.dst, keyinfo.from))) {
      DPRINTF(IDL_ERROR,("keyoutput: No more memory!\n"));
      senderr(ENOBUFS);
    }

    if (key_msghdr2secassoc(secassoc, km, &keyinfo) < 0) {
      DPRINTF(IDL_ERROR,("keyoutput: key_msghdr2secassoc failed!\n"));
      key_freeassoc(secassoc);
      senderr(EINVAL);
    }
    DPRINTF(IDL_FINISHED,("secassoc to add:\n"));
    DDO(IDL_FINISHED,dump_secassoc(secassoc));

    keyerror = key_add(secassoc);

    if (keyerror) {
      key_freeassoc(secassoc);
      DPRINTF(IDL_ERROR,("keyoutput: key_add failed\n"));
      if (keyerror == -2) {
	senderr(EEXIST);
      } else {
	senderr(ENOBUFS);
      }
    }
    break;
  case KEY_DELETE:
    DPRINTF(IDL_FINISHED,("key_output got KEY_DELETE msg\n"));

    if (key_xdata(km, &keyinfo, 1) < 0)
      goto parsefail;

    if (!(secassoc = key_allocassoc(keyinfo.src, keyinfo.dst, keyinfo.from))) {
      DPRINTF(IDL_ERROR,("keyoutput: No more memory!\n"));
      senderr(ENOBUFS);
    }

    if (key_msghdr2secassoc(secassoc, km, &keyinfo) < 0) {
      DPRINTF(IDL_ERROR,("keyoutput: key_msghdr2secassoc failed!\n"));
      key_freeassoc(secassoc);
      senderr(EINVAL);
    }

    keyerror = key_delete(secassoc);

    if (keyerror)
      senderr(ESRCH);
    break;
  case KEY_UPDATE:
    DPRINTF(IDL_FINISHED,("key_output got KEY_UPDATE msg\n"));

    if (key_xdata(km, &keyinfo, 0) < 0)
      goto parsefail;

    if (!(secassoc = key_allocassoc(keyinfo.src, keyinfo.dst, keyinfo.from))) {
      DPRINTF(IDL_ERROR,("keyoutput: No more memory!\n"));
      senderr(ENOBUFS);
    }

    if (key_msghdr2secassoc(secassoc, km, &keyinfo) < 0) {
      DPRINTF(IDL_ERROR,("keyoutput: key_msghdr2secassoc failed!\n"));
      key_freeassoc(secassoc);
      senderr(EINVAL);
    }
    keyerror = key_update(secassoc);
    if (keyerror) {
      key_freeassoc(secassoc);
      DPRINTF(IDL_ERROR,("Error updating key entry\n"));
      senderr(keyerror);
    }
    break;
  case KEY_GET:
    DPRINTF(IDL_FINISHED,("key_output got KEY_GET msg\n"));

    if (key_xdata(km, &keyinfo, 1) < 0)
      goto parsefail;

    if (key_get(km->type, (SOCKADDR *)keyinfo.src, 
		(SOCKADDR *)keyinfo.dst, 
		km->spi, &secassoc) != 0) {
      DPRINTF(IDL_FINISHED,("keyoutput: can't get key\n"));
      senderr(ESRCH);
    }

    if (secassoc) {
      int newlen;

      DPRINTF(IDL_FINISHED,("keyoutput: Found secassoc!\n"));
      newlen = sizeof(struct key_msghdr) + ROUNDUP(secassoc->src->sa_len) +
	ROUNDUP(secassoc->dst->sa_len) + ROUNDUP(secassoc->from->sa_len) +
	  ROUNDUP(secassoc->keylen) + ROUNDUP(secassoc->ivlen);
      DPRINTF(IDL_FINISHED,("keyoutput: newlen=%d\n", newlen));
      if (newlen > km->key_msglen) {
	struct key_msghdr *newkm;

	DPRINTF(IDL_FINISHED,("keyoutput: Allocating new buffer!\n"));
	KMALLOC(newkm, struct key_msghdr *, newlen); 
	if (newkm == 0) {
	  senderr(ENOBUFS);
	}
	bcopy((caddr_t)km, (caddr_t)newkm, km->key_msglen);
	DPRINTF(IDL_FINISHED,("keyoutput: 1\n"));
	KFREE_LEN(km, sizeof(struct key_msghdr));
	*kmp = km = newkm;
	DPRINTF(IDL_ERROR, ("km->key_msglen = %d, newlen = %d\n",
			       km->key_msglen, newlen));
	km->key_msglen = newlen;
      }
      DPRINTF(IDL_FINISHED,("keyoutput: 2\n"));
      if (key_secassoc2msghdr(secassoc, km, &keyinfo)) {
	DPRINTF(IDL_ERROR,("keyoutput: Can't create msghdr!\n"));
	senderr(EINVAL);
      }
      DPRINTF(IDL_FINISHED,("keyoutput: 3\n"));
    }
    break;
  case KEY_GETSPI:
    DPRINTF(IDL_FINISHED,("key_output got KEY_GETSPI msg\n"));

    if (key_xdata(km, &keyinfo, 1) < 0)
      goto parsefail;

    if ((keyerror = key_getspi(km->type, keyinfo.src, keyinfo.dst, 
			       km->lifetime1, km->lifetime2, 
			       &(km->spi))) != 0) {
      DPRINTF(IDL_ERROR,("keyoutput: getspi failed error=%d\n", keyerror));
      senderr(keyerror);
    }
    break;
  case KEY_REGISTER:
    DPRINTF(IDL_FINISHED,("key_output got KEY_REGISTER msg\n"));
    key_register(so, km->type);
    break;
  case KEY_DUMP:
    DPRINTF(IDL_FINISHED,("key_output got KEY_DUMP msg\n"));
    error = key_dump(so);
    return(error);
    break;
  case KEY_FLUSH:
    DPRINTF(IDL_FINISHED,("key_output got KEY_FLUSH msg\n"));
    key_flush();
    break;
  case KEY_DEBUGMSG:
    DPRINTF(IDL_FINISHED, ("key_parse: got KEY_DEBUG message\n"));
    error = key_debug(km);
    break;
  default:
    DPRINTF(IDL_ERROR,("key_output got unsupported msg type=%d\n", 
			     km->key_msgtype));
    senderr(EOPNOTSUPP);
  }

  goto flush;

parsefail:
  keyinfo.dst = NULL;
  error = EINVAL;

flush:
  if (km)
    km->key_errno = error;

  if (dstfamily)
    *dstfamily = keyinfo.dst ? keyinfo.dst->sa_family : 0;

  DPRINTF(IDL_FINISHED, ("key_parse exiting with error=%d\n", error));
  return error;
}
