#include <stdio.h>
#include <signal.h>
#include <sys/types.h>
#include <setjmp.h>
#include "define.h"
#define	SHELL_ESC	'!'
/************************************************************************/
/*									*/
/*  program:								*/
/*	mtip -- talk to a microprocessor from a UNIX process  		*/
/*									*/
/*  synopsis:								*/
/*      mtip  [-b board#] [-f file] [-x debuglevel] [-l loadaddr]        */
/*      Mtip  [-b board#] [-f file] [-x debuglevel] [-l loadaddr]        */
/*									*/
/*  description:							*/
/*      mtip attaches a user's terminal to a tty line that connects	*/
/*	to an MECB (mtip) or an ATT7300 standalone system (Mtip)        */
/*      so that keystrokes typed at the terminal are		        */
/*	sent to the MECB, and characters that the MECB emits		*/
/*	are sent to the terminal display (without any processing).	*/
/*	The standard input, standard output, and standard error		*/
/*	files remain attached to the terminal.  File descriptor		*/
/*	mecbfd is attached to the MECB tty line.			*/
/*	  Input from the keyboard is processed as follows: lines	*/
/*	beginning with ! are passed to a subshell.  A backslash /	*/
/*	causes mtip to change the keyboard mode to RAW for the next	*/
/*	character, making it possible to send special characters	*/
/*	like interrupt or kill characters.  Echo is turned off when	*/
/*	mtip begins.  Thus, any echoing of keystrokes is coming from	*/
/*	the MECB.							*/
/*									*/
/*	  Normally, mtip looks for any available MECB.  The argument	*/
/*	-b x restricts the search to the MECB with "number" x.		*/
/*									*/
/* Author/status:							*/
/*	Doug Comer, based on a program by Bob Brown			*/
/*	CS Dept., Purdue University					*/
/*  Brought up on UMB VAX by Rick Martin                                */
/*  Quite heavily modified by Bill Oliver and Betty O'Neil              */
/*      for use on Sun, for Computer Architecture students              */
/*									*/
/************************************************************************/

int	Debug=1;			/* Debugging level		*/
int     interactive;		/* real tty in charge */
extern	char	*mecbdevs[];
/*
 *========================
 *   M a i n   E n t r y
 *========================
 */
main(argc, argv)
int argc;
char *argv[];
{
  if (argc>1)	    
    {
      printf("For command help, type ~?\n");
      printf("For help on args, rerun without args\n");
    }

	procargs(argc,argv);	/* process args into struct A */
        arm_signals();   /* catch intr,hangup, suspend and continue signals*/

        /* find line if nec.,open it-- */
	if (setline(&mecbfd,A.xdev,&A.boardnum,A.Baud) < 0)
	     exit(1);		/* failed to get line */

	fprintf(stderr,"Using board # %d \r\n",A.boardnum);
        interactive = init_ctty(); /* commanding tty (stdin) in cbreak mode */

	/*
	 * start up the keyboard monitor and line monitor tasks
	 */
	if ((pid=fork())==-1)
	  {
	    perror("mtip: fork failed for reason: ");
	    exit(1);
	  }
        else if (pid !=0)
	  keymon();		/* parent */
	else
	  linemon();		/* child */
}

/*
 *------------------------------------------------------------------
 *
 * keymon() - terminal keyboard monitor process
 *
 *------------------------------------------------------------------
 */
keymon()
{
  int i;
  int	esc,line;
  char	c, buf[BUFSIZ];

  /*
   * loop forever over the following: process each keystroke
   * on the keyboard.
   */

  esc = 0;
  while((i=read(STDIN,&c,1)) != 0) {
    if (i == 1) {
      idle = 0;
      if ( c==A.boardesc && esc==0 ) 
	{
	  esc = 1;		/* entering escape-condition */
	  if (interactive)
	    printf("%c",c);	/* echo it */
	  fflush(stdout);
	}
      else if (c!=A.boardesc && esc==0) {
	if (!interactive)
	  if (c=='\n') c = '\r'; /* real CRs are expected from tty */
	write(mecbfd,&c,1);	/* normal output to board */
	  if (!interactive)	/* need to do this AFTER write, some reason */
	    usleep(10000);	/* 10 msecs <--->1000 bits/sec */
      }
      else {			/* case of esc-condition: handle next char c */
	switch( c & 0177) 
	  {
	  case '#' :		/* send a break to the board */

	    if (!interactive)
	      printf("%c",A.boardesc); /* catch up on output */
	    printf("%c",c);	/* echo it */
	    fflush(stdout);
	    fprintf(stderr," Break'ing...\r\n");
	    ioctl(mecbfd, TIOCSBRK, NULL);
	    sleep(1);
	    ioctl(mecbfd, TIOCCBRK, NULL);
	    fprintf(stderr," ...done\r\n");
	    break;

	  case '!' :		/* Shell escape */
	    if (!interactive)
	      printf("%c",A.boardesc); /* catch up on output */
	    printf("%c",c);	/* echo it */
	    set_orig_ctty();	/* normal */
	    gets(buf);		/* get a line from user */
	    ex(buf); 
	    set_cbreak_ctty();	/* back to cbreak */
	    break;  

	  case 'R':
	  case 'r':		/* reset the board */
	    if (!interactive)
	      printf("%c",A.boardesc); /* catch up on output */
	    printf("%cesetting board #%d",c,A.boardnum); /* echo it */
	    fflush(stdout);
	    reset_mecb(A.boardnum);
	    if (!interactive)
	      sleep(1);		/* time for mecb to react */
	    break;

	  case 'D':
	  case 'd' :		/* call the downloader */
	    if (!interactive)
	      printf("%c",A.boardesc); /* catch up on output */
	    printf("%c",c);	/* echo it */
	    set_orig_ctty();
	    kill(pid,SIGEMT);
	    if(download(mecbfd) < 0)
	      fprintf(stderr,"Sorry, download failed, try resetting board first\r\n");
	    kill(pid,SIGEMT);
	    set_cbreak_ctty();
	    if (!interactive)
	      sleep(2);		/* let settle down */
	    break;

	  case 'F':		/* set the filename for download */
	  case 'f':
	    if (!interactive)
	      printf("%c",A.boardesc); /* catch up on output */
	    printf("%c",c);	/* echo it */
	    set_orig_ctty();	/* restore line editting */
	    printf("ile for future downloads: ");
	    if (scanf("%s",buf)!=1) /* get 1 token */
	      printf("Error in scanning filename!\n");
	    else
	      setloadfile(buf);	/* prints error if any */
	    set_cbreak_ctty();	/* back to cbreak */
	    break; 

	    case '\\':		/* raw mode */
 	    printf("%c",c);	/* echo it */
	    fflush(stdout);
	    set_raw_ctty();
	    read(STDIN,&c,1);	/* get the char */
	    write(mecbfd,&c,1);	/* send it to mecb */
	    set_cbreak_ctty();
	    break;

	  case 'S': sleep(10);	/* unechoed command, for better batch output*/
	    break;
	  case 's': sleep(1);
	    break;
	    
	  case 'Q':
	  case 'q':
	    if (!interactive)
	      printf("%c",A.boardesc); /* catch up on output */
 	    printf("%c ",c);	/* echo it */
	    fflush(stdout);
	    quit();
	    break;
	  case '?':
	    if (!interactive)
	      printf("%c",A.boardesc); /* catch up on output */
  	    printf("%c",c);	/* echo it */
	    printf(" You are using board %d\r\n",A.boardnum);
	    printf(" Commands are: \r\n");
	    printf("  ~D  (or ~d) download file specified by -f or ~f\r\n");
	    printf("  ~F  (or ~f) set download filename\r\n");
	    printf("  ~R  (or ~r) reset the board\r\n");
	    printf("  ~!  UNIX command to lower shell\r\n");
	    printf("  ~#  send break to board\r\n");
	    printf("  ~\\c send char c literally (raw mode)\r\n");
	    printf("  ~?  this text\r\n");
	    printf("  ~Q  (or ~q) quit\r\n");
	    break;

	    default :		/* Not a cmd so complain */
	      fprintf(stderr,"\007\r\n");
	    break;
	  }			/* end switch */
	esc = 0;		/* cmd done */
      }				/* end case of escape-condition */
    }				/* end if (i==1) */
  }				/* end while not EOF */

  /* here on EOF on stdin */
  if (!interactive)
    sleep(2);			/* let finish up */
  quit();			/* but will return if mecb still outputting */
  printf("Resetting this board! For future, please add ~r to script\r\n");
  reset_mecb(A.boardnum);	/* no way to get user to do it  */
  exit(1);
}
/*
 *---------------------------------------------------------------------
 *
 * ex -- execute input following a ! and return to normal processing
 *
 *---------------------------------------------------------------------
 */
ex(bp)
char *bp;
{
  set_default_signals();	/* let shell take over sigs for now */
  system(bp);			/* do shell command */
  arm_signals();		/* back to mtip-controlled sigs */
  fprintf(stderr,"!\r\n");	/* mark that it's done */
}

/*
 *------------------------------------------------------------------
 *
 *  linemon  -- monitor MECB line and print characters that arrive
 *
 *------------------------------------------------------------------
 */
static int sig_hold(), sig_return();
static jmp_buf sig_frame;

linemon()
{
	char c;

	/*
	  Normally, keyboard mon kills us, but sometimes it
	  is killed -9, so this process should also respond
	  directly to signals (int,hup, etc.)
	 */
	set_default_signals();

	/* set up EMT to pause loop, come back here */
        /*--first EMT sig starts pause, second one aborts it,  */
        /*         -- third one starts another pause, ...               */

        /* do actions AFTER longjmp, to let it restore things first */

        switch (setjmp(sig_frame))
	  {
	  case 0:  signal(SIGEMT, sig_hold); /* just setting up */
	    break;
	  case 1:  signal(SIGEMT, sig_return); /* redirect EMT */
	    pause();		/* wait for next one */
	    sleep(1);		/* let output finist before signal */
	    signal(SIGEMT, sig_hold); /* back to initial setup */
	  };

	for (;;) {
		if (read(mecbfd, &c, 1) > 0)
			write(STDOUT, &c, 1);
	}
}

static sig_hold()
{
  longjmp(sig_frame,1);		/* make setjmp ret 1 */
}

static sig_return()
{				/* just terminates pause */
}
reset_mecb(boardnum)
int boardnum;
{
  char buf[100];
  int line;

  if (A.xdev)
    {
      switch (A.boardnum) {
      case 0:
	line = 15;
	break;
      case 1:
	line = 14;
	break;
      case 2:
	line = 10;
	break;
      default:
	sprintf(stderr,"reset for Mtip board %d NYI\r\n",A.boardnum);
      }
    }
  else /* orig series-- use board # for reset wire #-- */
    line = A.boardnum;
  sprintf(buf,"client %d\n\0",line);
  ex(buf);
}
