/*
 * Copyright © 2012 Intel Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include <gtest/gtest.h>
#include "main/glheader.h"

#include "glapi/glapi.h"
#include "glapitable.h"

struct name_offset {
   const char *name;
   unsigned int offset;
};

extern const struct name_offset linux_gl_abi[];
extern const struct name_offset known_dispatch[];

TEST(GetProcAddress, ABIOffsetByName)
{
   /* 408 functions have had their locations in the dispatch table set since
    * the dawn of time.  Verify that all of these functions are at the correct
    * locations.
    */
   for (unsigned i = 0; linux_gl_abi[i].name != NULL; i++) {
      EXPECT_EQ(linux_gl_abi[i].offset,
		_glapi_get_proc_offset(linux_gl_abi[i].name))
	 << "function name: " << linux_gl_abi[i].name;
   }
}

TEST(GetProcAddress, ABINameByOffset)
{
   /* 408 functions have had their locations in the dispatch table set since
    * the dawn of time.  Verify that all of these functions are at the correct
    * locations.
    */
   for (unsigned i = 0; linux_gl_abi[i].name != NULL; i++) {
      EXPECT_STREQ(linux_gl_abi[i].name,
		   _glapi_get_proc_name(linux_gl_abi[i].offset))
	 << "function offset: " << linux_gl_abi[i].offset;
   }
}

TEST(GetProcAddress, TableBigEnoughForABI)
{
   const unsigned table_entries = sizeof(struct _glapi_table) / sizeof(void *);

   EXPECT_GE(table_entries, 408u);
}

TEST(GetProcAddress, TableDidntShrink)
{
   const unsigned table_entries = sizeof(struct _glapi_table) / sizeof(void *);

   /* The dispatch table is not expected to shrink.  At GIT commit b45052b the
    * table had 978 entries.  Changes that intentionally reduce the size of
    * the table are very rare.  In those cases, the expected value should be
    * updated to reflect the change.
    *
    * Changes that accidentally reduce the size of the table are bugs, and
    * they should be fixed.
    *
    * 6 entries were removed when GL_SGIS_pixel_texture was removed from the
    * dispatch table.
    *
    * 1 entry was removed when GL_SGIX_pixel_texture was removed from the
    * dispatch table.
    *
    * 2 entries were removed when GL_APPLE_texture_range was removed from the
    * dispatch table.
    *
    * 13 entries were removed when GL_NV_register_combiners was removed from
    * the dispatch table.
    *
    * 7 entries were removed when GL_NV_fence was removed from the dispatch
    * table.
    *
    * 2 entries were removed when GL_NV_vertex_array_range was removed from
    * the dispatch table.
    */
   EXPECT_GE(table_entries, 978u - 6u - 1u - 2u - 13u - 7u - 2u);
}

TEST(GetProcAddress, QueriedDispatchSizeBigEnough)
{
   const unsigned table_entries = sizeof(struct _glapi_table) / sizeof(void *);

   /* _glapi_get_dispatch_table_size returns the size of the extended dispatch
    * table.  This is the size of the static table with some extra entries for
    * drivers to use for extensions that the core does not know about.
    */
   EXPECT_LT(table_entries, _glapi_get_dispatch_table_size());
}

TEST(GetProcAddress, KnownDispatchOffsetsAreConsistent)
{
   /* Verify that the queried dispatch offset for every known function is
    * consistent with its location in the static dispatch table.
    *
    * There is some redundancy between this test and ABIOffsetByName.  That's
    * okay.  The offsets in ABIOffsetByName comdirectly from the ABI
    * definition.  The offsets in this test come from locations in a structure
    * definition generated by scripts.
    */
   for (unsigned i = 0; known_dispatch[i].name != NULL; i++) {
      EXPECT_EQ(known_dispatch[i].offset,
		_glapi_get_proc_offset(known_dispatch[i].name))
	 << "function name: " << known_dispatch[i].name;
   }
}

TEST(GetProcAddress, KnownDispatchNamesAreConsistent)
{
   /* Verify that the queried dispatch name for every known function is
    * consistent with its location in the static dispatch table.
    *
    * There is some redundancy between this test and ABINameByOffset.  That's
    * okay.  The offsets in ABIOffsetByName comdirectly from the ABI
    * definition.  The offsets in this test come from locations in a structure
    * definition generated by scripts.
    */
   for (unsigned i = 0; known_dispatch[i].name != NULL; i++) {
      EXPECT_STREQ(known_dispatch[i].name,
		   _glapi_get_proc_name(known_dispatch[i].offset))
	 << "function offset: " << known_dispatch[i].offset;
   }
}

const struct name_offset linux_gl_abi[] = {
   { "glNewList", 0 },
   { "glEndList", 1 },
   { "glCallList", 2 },
   { "glCallLists", 3 },
   { "glDeleteLists", 4 },
   { "glGenLists", 5 },
   { "glListBase", 6 },
   { "glBegin", 7 },
   { "glBitmap", 8 },
   { "glColor3b", 9 },
   { "glColor3bv", 10 },
   { "glColor3d", 11 },
   { "glColor3dv", 12 },
   { "glColor3f", 13 },
   { "glColor3fv", 14 },
   { "glColor3i", 15 },
   { "glColor3iv", 16 },
   { "glColor3s", 17 },
   { "glColor3sv", 18 },
   { "glColor3ub", 19 },
   { "glColor3ubv", 20 },
   { "glColor3ui", 21 },
   { "glColor3uiv", 22 },
   { "glColor3us", 23 },
   { "glColor3usv", 24 },
   { "glColor4b", 25 },
   { "glColor4bv", 26 },
   { "glColor4d", 27 },
   { "glColor4dv", 28 },
   { "glColor4f", 29 },
   { "glColor4fv", 30 },
   { "glColor4i", 31 },
   { "glColor4iv", 32 },
   { "glColor4s", 33 },
   { "glColor4sv", 34 },
   { "glColor4ub", 35 },
   { "glColor4ubv", 36 },
   { "glColor4ui", 37 },
   { "glColor4uiv", 38 },
   { "glColor4us", 39 },
   { "glColor4usv", 40 },
   { "glEdgeFlag", 41 },
   { "glEdgeFlagv", 42 },
   { "glEnd", 43 },
   { "glIndexd", 44 },
   { "glIndexdv", 45 },
   { "glIndexf", 46 },
   { "glIndexfv", 47 },
   { "glIndexi", 48 },
   { "glIndexiv", 49 },
   { "glIndexs", 50 },
   { "glIndexsv", 51 },
   { "glNormal3b", 52 },
   { "glNormal3bv", 53 },
   { "glNormal3d", 54 },
   { "glNormal3dv", 55 },
   { "glNormal3f", 56 },
   { "glNormal3fv", 57 },
   { "glNormal3i", 58 },
   { "glNormal3iv", 59 },
   { "glNormal3s", 60 },
   { "glNormal3sv", 61 },
   { "glRasterPos2d", 62 },
   { "glRasterPos2dv", 63 },
   { "glRasterPos2f", 64 },
   { "glRasterPos2fv", 65 },
   { "glRasterPos2i", 66 },
   { "glRasterPos2iv", 67 },
   { "glRasterPos2s", 68 },
   { "glRasterPos2sv", 69 },
   { "glRasterPos3d", 70 },
   { "glRasterPos3dv", 71 },
   { "glRasterPos3f", 72 },
   { "glRasterPos3fv", 73 },
   { "glRasterPos3i", 74 },
   { "glRasterPos3iv", 75 },
   { "glRasterPos3s", 76 },
   { "glRasterPos3sv", 77 },
   { "glRasterPos4d", 78 },
   { "glRasterPos4dv", 79 },
   { "glRasterPos4f", 80 },
   { "glRasterPos4fv", 81 },
   { "glRasterPos4i", 82 },
   { "glRasterPos4iv", 83 },
   { "glRasterPos4s", 84 },
   { "glRasterPos4sv", 85 },
   { "glRectd", 86 },
   { "glRectdv", 87 },
   { "glRectf", 88 },
   { "glRectfv", 89 },
   { "glRecti", 90 },
   { "glRectiv", 91 },
   { "glRects", 92 },
   { "glRectsv", 93 },
   { "glTexCoord1d", 94 },
   { "glTexCoord1dv", 95 },
   { "glTexCoord1f", 96 },
   { "glTexCoord1fv", 97 },
   { "glTexCoord1i", 98 },
   { "glTexCoord1iv", 99 },
   { "glTexCoord1s", 100 },
   { "glTexCoord1sv", 101 },
   { "glTexCoord2d", 102 },
   { "glTexCoord2dv", 103 },
   { "glTexCoord2f", 104 },
   { "glTexCoord2fv", 105 },
   { "glTexCoord2i", 106 },
   { "glTexCoord2iv", 107 },
   { "glTexCoord2s", 108 },
   { "glTexCoord2sv", 109 },
   { "glTexCoord3d", 110 },
   { "glTexCoord3dv", 111 },
   { "glTexCoord3f", 112 },
   { "glTexCoord3fv", 113 },
   { "glTexCoord3i", 114 },
   { "glTexCoord3iv", 115 },
   { "glTexCoord3s", 116 },
   { "glTexCoord3sv", 117 },
   { "glTexCoord4d", 118 },
   { "glTexCoord4dv", 119 },
   { "glTexCoord4f", 120 },
   { "glTexCoord4fv", 121 },
   { "glTexCoord4i", 122 },
   { "glTexCoord4iv", 123 },
   { "glTexCoord4s", 124 },
   { "glTexCoord4sv", 125 },
   { "glVertex2d", 126 },
   { "glVertex2dv", 127 },
   { "glVertex2f", 128 },
   { "glVertex2fv", 129 },
   { "glVertex2i", 130 },
   { "glVertex2iv", 131 },
   { "glVertex2s", 132 },
   { "glVertex2sv", 133 },
   { "glVertex3d", 134 },
   { "glVertex3dv", 135 },
   { "glVertex3f", 136 },
   { "glVertex3fv", 137 },
   { "glVertex3i", 138 },
   { "glVertex3iv", 139 },
   { "glVertex3s", 140 },
   { "glVertex3sv", 141 },
   { "glVertex4d", 142 },
   { "glVertex4dv", 143 },
   { "glVertex4f", 144 },
   { "glVertex4fv", 145 },
   { "glVertex4i", 146 },
   { "glVertex4iv", 147 },
   { "glVertex4s", 148 },
   { "glVertex4sv", 149 },
   { "glClipPlane", 150 },
   { "glColorMaterial", 151 },
   { "glCullFace", 152 },
   { "glFogf", 153 },
   { "glFogfv", 154 },
   { "glFogi", 155 },
   { "glFogiv", 156 },
   { "glFrontFace", 157 },
   { "glHint", 158 },
   { "glLightf", 159 },
   { "glLightfv", 160 },
   { "glLighti", 161 },
   { "glLightiv", 162 },
   { "glLightModelf", 163 },
   { "glLightModelfv", 164 },
   { "glLightModeli", 165 },
   { "glLightModeliv", 166 },
   { "glLineStipple", 167 },
   { "glLineWidth", 168 },
   { "glMaterialf", 169 },
   { "glMaterialfv", 170 },
   { "glMateriali", 171 },
   { "glMaterialiv", 172 },
   { "glPointSize", 173 },
   { "glPolygonMode", 174 },
   { "glPolygonStipple", 175 },
   { "glScissor", 176 },
   { "glShadeModel", 177 },
   { "glTexParameterf", 178 },
   { "glTexParameterfv", 179 },
   { "glTexParameteri", 180 },
   { "glTexParameteriv", 181 },
   { "glTexImage1D", 182 },
   { "glTexImage2D", 183 },
   { "glTexEnvf", 184 },
   { "glTexEnvfv", 185 },
   { "glTexEnvi", 186 },
   { "glTexEnviv", 187 },
   { "glTexGend", 188 },
   { "glTexGendv", 189 },
   { "glTexGenf", 190 },
   { "glTexGenfv", 191 },
   { "glTexGeni", 192 },
   { "glTexGeniv", 193 },
   { "glFeedbackBuffer", 194 },
   { "glSelectBuffer", 195 },
   { "glRenderMode", 196 },
   { "glInitNames", 197 },
   { "glLoadName", 198 },
   { "glPassThrough", 199 },
   { "glPopName", 200 },
   { "glPushName", 201 },
   { "glDrawBuffer", 202 },
   { "glClear", 203 },
   { "glClearAccum", 204 },
   { "glClearIndex", 205 },
   { "glClearColor", 206 },
   { "glClearStencil", 207 },
   { "glClearDepth", 208 },
   { "glStencilMask", 209 },
   { "glColorMask", 210 },
   { "glDepthMask", 211 },
   { "glIndexMask", 212 },
   { "glAccum", 213 },
   { "glDisable", 214 },
   { "glEnable", 215 },
   { "glFinish", 216 },
   { "glFlush", 217 },
   { "glPopAttrib", 218 },
   { "glPushAttrib", 219 },
   { "glMap1d", 220 },
   { "glMap1f", 221 },
   { "glMap2d", 222 },
   { "glMap2f", 223 },
   { "glMapGrid1d", 224 },
   { "glMapGrid1f", 225 },
   { "glMapGrid2d", 226 },
   { "glMapGrid2f", 227 },
   { "glEvalCoord1d", 228 },
   { "glEvalCoord1dv", 229 },
   { "glEvalCoord1f", 230 },
   { "glEvalCoord1fv", 231 },
   { "glEvalCoord2d", 232 },
   { "glEvalCoord2dv", 233 },
   { "glEvalCoord2f", 234 },
   { "glEvalCoord2fv", 235 },
   { "glEvalMesh1", 236 },
   { "glEvalPoint1", 237 },
   { "glEvalMesh2", 238 },
   { "glEvalPoint2", 239 },
   { "glAlphaFunc", 240 },
   { "glBlendFunc", 241 },
   { "glLogicOp", 242 },
   { "glStencilFunc", 243 },
   { "glStencilOp", 244 },
   { "glDepthFunc", 245 },
   { "glPixelZoom", 246 },
   { "glPixelTransferf", 247 },
   { "glPixelTransferi", 248 },
   { "glPixelStoref", 249 },
   { "glPixelStorei", 250 },
   { "glPixelMapfv", 251 },
   { "glPixelMapuiv", 252 },
   { "glPixelMapusv", 253 },
   { "glReadBuffer", 254 },
   { "glCopyPixels", 255 },
   { "glReadPixels", 256 },
   { "glDrawPixels", 257 },
   { "glGetBooleanv", 258 },
   { "glGetClipPlane", 259 },
   { "glGetDoublev", 260 },
   { "glGetError", 261 },
   { "glGetFloatv", 262 },
   { "glGetIntegerv", 263 },
   { "glGetLightfv", 264 },
   { "glGetLightiv", 265 },
   { "glGetMapdv", 266 },
   { "glGetMapfv", 267 },
   { "glGetMapiv", 268 },
   { "glGetMaterialfv", 269 },
   { "glGetMaterialiv", 270 },
   { "glGetPixelMapfv", 271 },
   { "glGetPixelMapuiv", 272 },
   { "glGetPixelMapusv", 273 },
   { "glGetPolygonStipple", 274 },
   { "glGetString", 275 },
   { "glGetTexEnvfv", 276 },
   { "glGetTexEnviv", 277 },
   { "glGetTexGendv", 278 },
   { "glGetTexGenfv", 279 },
   { "glGetTexGeniv", 280 },
   { "glGetTexImage", 281 },
   { "glGetTexParameterfv", 282 },
   { "glGetTexParameteriv", 283 },
   { "glGetTexLevelParameterfv", 284 },
   { "glGetTexLevelParameteriv", 285 },
   { "glIsEnabled", 286 },
   { "glIsList", 287 },
   { "glDepthRange", 288 },
   { "glFrustum", 289 },
   { "glLoadIdentity", 290 },
   { "glLoadMatrixf", 291 },
   { "glLoadMatrixd", 292 },
   { "glMatrixMode", 293 },
   { "glMultMatrixf", 294 },
   { "glMultMatrixd", 295 },
   { "glOrtho", 296 },
   { "glPopMatrix", 297 },
   { "glPushMatrix", 298 },
   { "glRotated", 299 },
   { "glRotatef", 300 },
   { "glScaled", 301 },
   { "glScalef", 302 },
   { "glTranslated", 303 },
   { "glTranslatef", 304 },
   { "glViewport", 305 },
   { "glArrayElement", 306 },
   { "glColorPointer", 308 },
   { "glDisableClientState", 309 },
   { "glDrawArrays", 310 },
   { "glDrawElements", 311 },
   { "glEdgeFlagPointer", 312 },
   { "glEnableClientState", 313 },
   { "glGetPointerv", 329 },
   { "glIndexPointer", 314 },
   { "glInterleavedArrays", 317 },
   { "glNormalPointer", 318 },
   { "glTexCoordPointer", 320 },
   { "glVertexPointer", 321 },
   { "glPolygonOffset", 319 },
   { "glCopyTexImage1D", 323 },
   { "glCopyTexImage2D", 324 },
   { "glCopyTexSubImage1D", 325 },
   { "glCopyTexSubImage2D", 326 },
   { "glTexSubImage1D", 332 },
   { "glTexSubImage2D", 333 },
   { "glAreTexturesResident", 322 },
   { "glBindTexture", 307 },
   { "glDeleteTextures", 327 },
   { "glGenTextures", 328 },
   { "glIsTexture", 330 },
   { "glPrioritizeTextures", 331 },
   { "glIndexub", 315 },
   { "glIndexubv", 316 },
   { "glPopClientAttrib", 334 },
   { "glPushClientAttrib", 335 },
   { "glBlendColor", 336 },
   { "glBlendEquation", 337 },
   { "glDrawRangeElements", 338 },
   { "glColorTable", 339 },
   { "glColorTableParameterfv", 340 },
   { "glColorTableParameteriv", 341 },
   { "glCopyColorTable", 342 },
   { "glGetColorTable", 343 },
   { "glGetColorTableParameterfv", 344 },
   { "glGetColorTableParameteriv", 345 },
   { "glColorSubTable", 346 },
   { "glCopyColorSubTable", 347 },
   { "glConvolutionFilter1D", 348 },
   { "glConvolutionFilter2D", 349 },
   { "glConvolutionParameterf", 350 },
   { "glConvolutionParameterfv", 351 },
   { "glConvolutionParameteri", 352 },
   { "glConvolutionParameteriv", 353 },
   { "glCopyConvolutionFilter1D", 354 },
   { "glCopyConvolutionFilter2D", 355 },
   { "glGetConvolutionFilter", 356 },
   { "glGetConvolutionParameterfv", 357 },
   { "glGetConvolutionParameteriv", 358 },
   { "glGetSeparableFilter", 359 },
   { "glSeparableFilter2D", 360 },
   { "glGetHistogram", 361 },
   { "glGetHistogramParameterfv", 362 },
   { "glGetHistogramParameteriv", 363 },
   { "glGetMinmax", 364 },
   { "glGetMinmaxParameterfv", 365 },
   { "glGetMinmaxParameteriv", 366 },
   { "glHistogram", 367 },
   { "glMinmax", 368 },
   { "glResetHistogram", 369 },
   { "glResetMinmax", 370 },
   { "glTexImage3D", 371 },
   { "glTexSubImage3D", 372 },
   { "glCopyTexSubImage3D", 373 },
   { "glActiveTexture", 374 },
   { "glClientActiveTexture", 375 },
   { "glMultiTexCoord1d", 376 },
   { "glMultiTexCoord1dv", 377 },
   { "glMultiTexCoord1fARB", 378 },
   { "glMultiTexCoord1fvARB", 379 },
   { "glMultiTexCoord1i", 380 },
   { "glMultiTexCoord1iv", 381 },
   { "glMultiTexCoord1s", 382 },
   { "glMultiTexCoord1sv", 383 },
   { "glMultiTexCoord2d", 384 },
   { "glMultiTexCoord2dv", 385 },
   { "glMultiTexCoord2fARB", 386 },
   { "glMultiTexCoord2fvARB", 387 },
   { "glMultiTexCoord2i", 388 },
   { "glMultiTexCoord2iv", 389 },
   { "glMultiTexCoord2s", 390 },
   { "glMultiTexCoord2sv", 391 },
   { "glMultiTexCoord3d", 392 },
   { "glMultiTexCoord3dv", 393 },
   { "glMultiTexCoord3fARB", 394 },
   { "glMultiTexCoord3fvARB", 395 },
   { "glMultiTexCoord3i", 396 },
   { "glMultiTexCoord3iv", 397 },
   { "glMultiTexCoord3s", 398 },
   { "glMultiTexCoord3sv", 399 },
   { "glMultiTexCoord4d", 400 },
   { "glMultiTexCoord4dv", 401 },
   { "glMultiTexCoord4fARB", 402 },
   { "glMultiTexCoord4fvARB", 403 },
   { "glMultiTexCoord4i", 404 },
   { "glMultiTexCoord4iv", 405 },
   { "glMultiTexCoord4s", 406 },
   { "glMultiTexCoord4sv", 407 },
   { NULL, 0 }
};

#define _O(f) ((intptr_t) & (((struct _glapi_table *) 0)->f)) / sizeof(void *)

const struct name_offset known_dispatch[] = {
   { "glNewList", _O(NewList) },
   { "glEndList", _O(EndList) },
   { "glCallList", _O(CallList) },
   { "glCallLists", _O(CallLists) },
   { "glDeleteLists", _O(DeleteLists) },
   { "glGenLists", _O(GenLists) },
   { "glListBase", _O(ListBase) },
   { "glBegin", _O(Begin) },
   { "glBitmap", _O(Bitmap) },
   { "glColor3b", _O(Color3b) },
   { "glColor3bv", _O(Color3bv) },
   { "glColor3d", _O(Color3d) },
   { "glColor3dv", _O(Color3dv) },
   { "glColor3f", _O(Color3f) },
   { "glColor3fv", _O(Color3fv) },
   { "glColor3i", _O(Color3i) },
   { "glColor3iv", _O(Color3iv) },
   { "glColor3s", _O(Color3s) },
   { "glColor3sv", _O(Color3sv) },
   { "glColor3ub", _O(Color3ub) },
   { "glColor3ubv", _O(Color3ubv) },
   { "glColor3ui", _O(Color3ui) },
   { "glColor3uiv", _O(Color3uiv) },
   { "glColor3us", _O(Color3us) },
   { "glColor3usv", _O(Color3usv) },
   { "glColor4b", _O(Color4b) },
   { "glColor4bv", _O(Color4bv) },
   { "glColor4d", _O(Color4d) },
   { "glColor4dv", _O(Color4dv) },
   { "glColor4f", _O(Color4f) },
   { "glColor4fv", _O(Color4fv) },
   { "glColor4i", _O(Color4i) },
   { "glColor4iv", _O(Color4iv) },
   { "glColor4s", _O(Color4s) },
   { "glColor4sv", _O(Color4sv) },
   { "glColor4ub", _O(Color4ub) },
   { "glColor4ubv", _O(Color4ubv) },
   { "glColor4ui", _O(Color4ui) },
   { "glColor4uiv", _O(Color4uiv) },
   { "glColor4us", _O(Color4us) },
   { "glColor4usv", _O(Color4usv) },
   { "glEdgeFlag", _O(EdgeFlag) },
   { "glEdgeFlagv", _O(EdgeFlagv) },
   { "glEnd", _O(End) },
   { "glIndexd", _O(Indexd) },
   { "glIndexdv", _O(Indexdv) },
   { "glIndexf", _O(Indexf) },
   { "glIndexfv", _O(Indexfv) },
   { "glIndexi", _O(Indexi) },
   { "glIndexiv", _O(Indexiv) },
   { "glIndexs", _O(Indexs) },
   { "glIndexsv", _O(Indexsv) },
   { "glNormal3b", _O(Normal3b) },
   { "glNormal3bv", _O(Normal3bv) },
   { "glNormal3d", _O(Normal3d) },
   { "glNormal3dv", _O(Normal3dv) },
   { "glNormal3f", _O(Normal3f) },
   { "glNormal3fv", _O(Normal3fv) },
   { "glNormal3i", _O(Normal3i) },
   { "glNormal3iv", _O(Normal3iv) },
   { "glNormal3s", _O(Normal3s) },
   { "glNormal3sv", _O(Normal3sv) },
   { "glRasterPos2d", _O(RasterPos2d) },
   { "glRasterPos2dv", _O(RasterPos2dv) },
   { "glRasterPos2f", _O(RasterPos2f) },
   { "glRasterPos2fv", _O(RasterPos2fv) },
   { "glRasterPos2i", _O(RasterPos2i) },
   { "glRasterPos2iv", _O(RasterPos2iv) },
   { "glRasterPos2s", _O(RasterPos2s) },
   { "glRasterPos2sv", _O(RasterPos2sv) },
   { "glRasterPos3d", _O(RasterPos3d) },
   { "glRasterPos3dv", _O(RasterPos3dv) },
   { "glRasterPos3f", _O(RasterPos3f) },
   { "glRasterPos3fv", _O(RasterPos3fv) },
   { "glRasterPos3i", _O(RasterPos3i) },
   { "glRasterPos3iv", _O(RasterPos3iv) },
   { "glRasterPos3s", _O(RasterPos3s) },
   { "glRasterPos3sv", _O(RasterPos3sv) },
   { "glRasterPos4d", _O(RasterPos4d) },
   { "glRasterPos4dv", _O(RasterPos4dv) },
   { "glRasterPos4f", _O(RasterPos4f) },
   { "glRasterPos4fv", _O(RasterPos4fv) },
   { "glRasterPos4i", _O(RasterPos4i) },
   { "glRasterPos4iv", _O(RasterPos4iv) },
   { "glRasterPos4s", _O(RasterPos4s) },
   { "glRasterPos4sv", _O(RasterPos4sv) },
   { "glRectd", _O(Rectd) },
   { "glRectdv", _O(Rectdv) },
   { "glRectf", _O(Rectf) },
   { "glRectfv", _O(Rectfv) },
   { "glRecti", _O(Recti) },
   { "glRectiv", _O(Rectiv) },
   { "glRects", _O(Rects) },
   { "glRectsv", _O(Rectsv) },
   { "glTexCoord1d", _O(TexCoord1d) },
   { "glTexCoord1dv", _O(TexCoord1dv) },
   { "glTexCoord1f", _O(TexCoord1f) },
   { "glTexCoord1fv", _O(TexCoord1fv) },
   { "glTexCoord1i", _O(TexCoord1i) },
   { "glTexCoord1iv", _O(TexCoord1iv) },
   { "glTexCoord1s", _O(TexCoord1s) },
   { "glTexCoord1sv", _O(TexCoord1sv) },
   { "glTexCoord2d", _O(TexCoord2d) },
   { "glTexCoord2dv", _O(TexCoord2dv) },
   { "glTexCoord2f", _O(TexCoord2f) },
   { "glTexCoord2fv", _O(TexCoord2fv) },
   { "glTexCoord2i", _O(TexCoord2i) },
   { "glTexCoord2iv", _O(TexCoord2iv) },
   { "glTexCoord2s", _O(TexCoord2s) },
   { "glTexCoord2sv", _O(TexCoord2sv) },
   { "glTexCoord3d", _O(TexCoord3d) },
   { "glTexCoord3dv", _O(TexCoord3dv) },
   { "glTexCoord3f", _O(TexCoord3f) },
   { "glTexCoord3fv", _O(TexCoord3fv) },
   { "glTexCoord3i", _O(TexCoord3i) },
   { "glTexCoord3iv", _O(TexCoord3iv) },
   { "glTexCoord3s", _O(TexCoord3s) },
   { "glTexCoord3sv", _O(TexCoord3sv) },
   { "glTexCoord4d", _O(TexCoord4d) },
   { "glTexCoord4dv", _O(TexCoord4dv) },
   { "glTexCoord4f", _O(TexCoord4f) },
   { "glTexCoord4fv", _O(TexCoord4fv) },
   { "glTexCoord4i", _O(TexCoord4i) },
   { "glTexCoord4iv", _O(TexCoord4iv) },
   { "glTexCoord4s", _O(TexCoord4s) },
   { "glTexCoord4sv", _O(TexCoord4sv) },
   { "glVertex2d", _O(Vertex2d) },
   { "glVertex2dv", _O(Vertex2dv) },
   { "glVertex2f", _O(Vertex2f) },
   { "glVertex2fv", _O(Vertex2fv) },
   { "glVertex2i", _O(Vertex2i) },
   { "glVertex2iv", _O(Vertex2iv) },
   { "glVertex2s", _O(Vertex2s) },
   { "glVertex2sv", _O(Vertex2sv) },
   { "glVertex3d", _O(Vertex3d) },
   { "glVertex3dv", _O(Vertex3dv) },
   { "glVertex3f", _O(Vertex3f) },
   { "glVertex3fv", _O(Vertex3fv) },
   { "glVertex3i", _O(Vertex3i) },
   { "glVertex3iv", _O(Vertex3iv) },
   { "glVertex3s", _O(Vertex3s) },
   { "glVertex3sv", _O(Vertex3sv) },
   { "glVertex4d", _O(Vertex4d) },
   { "glVertex4dv", _O(Vertex4dv) },
   { "glVertex4f", _O(Vertex4f) },
   { "glVertex4fv", _O(Vertex4fv) },
   { "glVertex4i", _O(Vertex4i) },
   { "glVertex4iv", _O(Vertex4iv) },
   { "glVertex4s", _O(Vertex4s) },
   { "glVertex4sv", _O(Vertex4sv) },
   { "glClipPlane", _O(ClipPlane) },
   { "glColorMaterial", _O(ColorMaterial) },
   { "glCullFace", _O(CullFace) },
   { "glFogf", _O(Fogf) },
   { "glFogfv", _O(Fogfv) },
   { "glFogi", _O(Fogi) },
   { "glFogiv", _O(Fogiv) },
   { "glFrontFace", _O(FrontFace) },
   { "glHint", _O(Hint) },
   { "glLightf", _O(Lightf) },
   { "glLightfv", _O(Lightfv) },
   { "glLighti", _O(Lighti) },
   { "glLightiv", _O(Lightiv) },
   { "glLightModelf", _O(LightModelf) },
   { "glLightModelfv", _O(LightModelfv) },
   { "glLightModeli", _O(LightModeli) },
   { "glLightModeliv", _O(LightModeliv) },
   { "glLineStipple", _O(LineStipple) },
   { "glLineWidth", _O(LineWidth) },
   { "glMaterialf", _O(Materialf) },
   { "glMaterialfv", _O(Materialfv) },
   { "glMateriali", _O(Materiali) },
   { "glMaterialiv", _O(Materialiv) },
   { "glPointSize", _O(PointSize) },
   { "glPolygonMode", _O(PolygonMode) },
   { "glPolygonStipple", _O(PolygonStipple) },
   { "glScissor", _O(Scissor) },
   { "glShadeModel", _O(ShadeModel) },
   { "glTexParameterf", _O(TexParameterf) },
   { "glTexParameterfv", _O(TexParameterfv) },
   { "glTexParameteri", _O(TexParameteri) },
   { "glTexParameteriv", _O(TexParameteriv) },
   { "glTexImage1D", _O(TexImage1D) },
   { "glTexImage2D", _O(TexImage2D) },
   { "glTexEnvf", _O(TexEnvf) },
   { "glTexEnvfv", _O(TexEnvfv) },
   { "glTexEnvi", _O(TexEnvi) },
   { "glTexEnviv", _O(TexEnviv) },
   { "glTexGend", _O(TexGend) },
   { "glTexGendv", _O(TexGendv) },
   { "glTexGenf", _O(TexGenf) },
   { "glTexGenfv", _O(TexGenfv) },
   { "glTexGeni", _O(TexGeni) },
   { "glTexGeniv", _O(TexGeniv) },
   { "glFeedbackBuffer", _O(FeedbackBuffer) },
   { "glSelectBuffer", _O(SelectBuffer) },
   { "glRenderMode", _O(RenderMode) },
   { "glInitNames", _O(InitNames) },
   { "glLoadName", _O(LoadName) },
   { "glPassThrough", _O(PassThrough) },
   { "glPopName", _O(PopName) },
   { "glPushName", _O(PushName) },
   { "glDrawBuffer", _O(DrawBuffer) },
   { "glClear", _O(Clear) },
   { "glClearAccum", _O(ClearAccum) },
   { "glClearIndex", _O(ClearIndex) },
   { "glClearColor", _O(ClearColor) },
   { "glClearStencil", _O(ClearStencil) },
   { "glClearDepth", _O(ClearDepth) },
   { "glStencilMask", _O(StencilMask) },
   { "glColorMask", _O(ColorMask) },
   { "glDepthMask", _O(DepthMask) },
   { "glIndexMask", _O(IndexMask) },
   { "glAccum", _O(Accum) },
   { "glDisable", _O(Disable) },
   { "glEnable", _O(Enable) },
   { "glFinish", _O(Finish) },
   { "glFlush", _O(Flush) },
   { "glPopAttrib", _O(PopAttrib) },
   { "glPushAttrib", _O(PushAttrib) },
   { "glMap1d", _O(Map1d) },
   { "glMap1f", _O(Map1f) },
   { "glMap2d", _O(Map2d) },
   { "glMap2f", _O(Map2f) },
   { "glMapGrid1d", _O(MapGrid1d) },
   { "glMapGrid1f", _O(MapGrid1f) },
   { "glMapGrid2d", _O(MapGrid2d) },
   { "glMapGrid2f", _O(MapGrid2f) },
   { "glEvalCoord1d", _O(EvalCoord1d) },
   { "glEvalCoord1dv", _O(EvalCoord1dv) },
   { "glEvalCoord1f", _O(EvalCoord1f) },
   { "glEvalCoord1fv", _O(EvalCoord1fv) },
   { "glEvalCoord2d", _O(EvalCoord2d) },
   { "glEvalCoord2dv", _O(EvalCoord2dv) },
   { "glEvalCoord2f", _O(EvalCoord2f) },
   { "glEvalCoord2fv", _O(EvalCoord2fv) },
   { "glEvalMesh1", _O(EvalMesh1) },
   { "glEvalPoint1", _O(EvalPoint1) },
   { "glEvalMesh2", _O(EvalMesh2) },
   { "glEvalPoint2", _O(EvalPoint2) },
   { "glAlphaFunc", _O(AlphaFunc) },
   { "glBlendFunc", _O(BlendFunc) },
   { "glLogicOp", _O(LogicOp) },
   { "glStencilFunc", _O(StencilFunc) },
   { "glStencilOp", _O(StencilOp) },
   { "glDepthFunc", _O(DepthFunc) },
   { "glPixelZoom", _O(PixelZoom) },
   { "glPixelTransferf", _O(PixelTransferf) },
   { "glPixelTransferi", _O(PixelTransferi) },
   { "glPixelStoref", _O(PixelStoref) },
   { "glPixelStorei", _O(PixelStorei) },
   { "glPixelMapfv", _O(PixelMapfv) },
   { "glPixelMapuiv", _O(PixelMapuiv) },
   { "glPixelMapusv", _O(PixelMapusv) },
   { "glReadBuffer", _O(ReadBuffer) },
   { "glCopyPixels", _O(CopyPixels) },
   { "glReadPixels", _O(ReadPixels) },
   { "glDrawPixels", _O(DrawPixels) },
   { "glGetBooleanv", _O(GetBooleanv) },
   { "glGetClipPlane", _O(GetClipPlane) },
   { "glGetDoublev", _O(GetDoublev) },
   { "glGetError", _O(GetError) },
   { "glGetFloatv", _O(GetFloatv) },
   { "glGetIntegerv", _O(GetIntegerv) },
   { "glGetLightfv", _O(GetLightfv) },
   { "glGetLightiv", _O(GetLightiv) },
   { "glGetMapdv", _O(GetMapdv) },
   { "glGetMapfv", _O(GetMapfv) },
   { "glGetMapiv", _O(GetMapiv) },
   { "glGetMaterialfv", _O(GetMaterialfv) },
   { "glGetMaterialiv", _O(GetMaterialiv) },
   { "glGetPixelMapfv", _O(GetPixelMapfv) },
   { "glGetPixelMapuiv", _O(GetPixelMapuiv) },
   { "glGetPixelMapusv", _O(GetPixelMapusv) },
   { "glGetPolygonStipple", _O(GetPolygonStipple) },
   { "glGetString", _O(GetString) },
   { "glGetTexEnvfv", _O(GetTexEnvfv) },
   { "glGetTexEnviv", _O(GetTexEnviv) },
   { "glGetTexGendv", _O(GetTexGendv) },
   { "glGetTexGenfv", _O(GetTexGenfv) },
   { "glGetTexGeniv", _O(GetTexGeniv) },
   { "glGetTexImage", _O(GetTexImage) },
   { "glGetTexParameterfv", _O(GetTexParameterfv) },
   { "glGetTexParameteriv", _O(GetTexParameteriv) },
   { "glGetTexLevelParameterfv", _O(GetTexLevelParameterfv) },
   { "glGetTexLevelParameteriv", _O(GetTexLevelParameteriv) },
   { "glIsEnabled", _O(IsEnabled) },
   { "glIsList", _O(IsList) },
   { "glDepthRange", _O(DepthRange) },
   { "glFrustum", _O(Frustum) },
   { "glLoadIdentity", _O(LoadIdentity) },
   { "glLoadMatrixf", _O(LoadMatrixf) },
   { "glLoadMatrixd", _O(LoadMatrixd) },
   { "glMatrixMode", _O(MatrixMode) },
   { "glMultMatrixf", _O(MultMatrixf) },
   { "glMultMatrixd", _O(MultMatrixd) },
   { "glOrtho", _O(Ortho) },
   { "glPopMatrix", _O(PopMatrix) },
   { "glPushMatrix", _O(PushMatrix) },
   { "glRotated", _O(Rotated) },
   { "glRotatef", _O(Rotatef) },
   { "glScaled", _O(Scaled) },
   { "glScalef", _O(Scalef) },
   { "glTranslated", _O(Translated) },
   { "glTranslatef", _O(Translatef) },
   { "glViewport", _O(Viewport) },
   { "glArrayElement", _O(ArrayElement) },
   { "glBindTexture", _O(BindTexture) },
   { "glColorPointer", _O(ColorPointer) },
   { "glDisableClientState", _O(DisableClientState) },
   { "glDrawArrays", _O(DrawArrays) },
   { "glDrawElements", _O(DrawElements) },
   { "glEdgeFlagPointer", _O(EdgeFlagPointer) },
   { "glEnableClientState", _O(EnableClientState) },
   { "glIndexPointer", _O(IndexPointer) },
   { "glIndexub", _O(Indexub) },
   { "glIndexubv", _O(Indexubv) },
   { "glInterleavedArrays", _O(InterleavedArrays) },
   { "glNormalPointer", _O(NormalPointer) },
   { "glPolygonOffset", _O(PolygonOffset) },
   { "glTexCoordPointer", _O(TexCoordPointer) },
   { "glVertexPointer", _O(VertexPointer) },
   { "glAreTexturesResident", _O(AreTexturesResident) },
   { "glCopyTexImage1D", _O(CopyTexImage1D) },
   { "glCopyTexImage2D", _O(CopyTexImage2D) },
   { "glCopyTexSubImage1D", _O(CopyTexSubImage1D) },
   { "glCopyTexSubImage2D", _O(CopyTexSubImage2D) },
   { "glDeleteTextures", _O(DeleteTextures) },
   { "glGenTextures", _O(GenTextures) },
   { "glGetPointerv", _O(GetPointerv) },
   { "glIsTexture", _O(IsTexture) },
   { "glPrioritizeTextures", _O(PrioritizeTextures) },
   { "glTexSubImage1D", _O(TexSubImage1D) },
   { "glTexSubImage2D", _O(TexSubImage2D) },
   { "glPopClientAttrib", _O(PopClientAttrib) },
   { "glPushClientAttrib", _O(PushClientAttrib) },
   { "glBlendColor", _O(BlendColor) },
   { "glBlendEquation", _O(BlendEquation) },
   { "glDrawRangeElements", _O(DrawRangeElements) },
   { "glColorTable", _O(ColorTable) },
   { "glColorTableParameterfv", _O(ColorTableParameterfv) },
   { "glColorTableParameteriv", _O(ColorTableParameteriv) },
   { "glCopyColorTable", _O(CopyColorTable) },
   { "glGetColorTable", _O(GetColorTable) },
   { "glGetColorTableParameterfv", _O(GetColorTableParameterfv) },
   { "glGetColorTableParameteriv", _O(GetColorTableParameteriv) },
   { "glColorSubTable", _O(ColorSubTable) },
   { "glCopyColorSubTable", _O(CopyColorSubTable) },
   { "glConvolutionFilter1D", _O(ConvolutionFilter1D) },
   { "glConvolutionFilter2D", _O(ConvolutionFilter2D) },
   { "glConvolutionParameterf", _O(ConvolutionParameterf) },
   { "glConvolutionParameterfv", _O(ConvolutionParameterfv) },
   { "glConvolutionParameteri", _O(ConvolutionParameteri) },
   { "glConvolutionParameteriv", _O(ConvolutionParameteriv) },
   { "glCopyConvolutionFilter1D", _O(CopyConvolutionFilter1D) },
   { "glCopyConvolutionFilter2D", _O(CopyConvolutionFilter2D) },
   { "glGetConvolutionFilter", _O(GetConvolutionFilter) },
   { "glGetConvolutionParameterfv", _O(GetConvolutionParameterfv) },
   { "glGetConvolutionParameteriv", _O(GetConvolutionParameteriv) },
   { "glGetSeparableFilter", _O(GetSeparableFilter) },
   { "glSeparableFilter2D", _O(SeparableFilter2D) },
   { "glGetHistogram", _O(GetHistogram) },
   { "glGetHistogramParameterfv", _O(GetHistogramParameterfv) },
   { "glGetHistogramParameteriv", _O(GetHistogramParameteriv) },
   { "glGetMinmax", _O(GetMinmax) },
   { "glGetMinmaxParameterfv", _O(GetMinmaxParameterfv) },
   { "glGetMinmaxParameteriv", _O(GetMinmaxParameteriv) },
   { "glHistogram", _O(Histogram) },
   { "glMinmax", _O(Minmax) },
   { "glResetHistogram", _O(ResetHistogram) },
   { "glResetMinmax", _O(ResetMinmax) },
   { "glTexImage3D", _O(TexImage3D) },
   { "glTexSubImage3D", _O(TexSubImage3D) },
   { "glCopyTexSubImage3D", _O(CopyTexSubImage3D) },
   { "glActiveTexture", _O(ActiveTexture) },
   { "glClientActiveTexture", _O(ClientActiveTexture) },
   { "glMultiTexCoord1d", _O(MultiTexCoord1d) },
   { "glMultiTexCoord1dv", _O(MultiTexCoord1dv) },
   { "glMultiTexCoord1fARB", _O(MultiTexCoord1fARB) },
   { "glMultiTexCoord1fvARB", _O(MultiTexCoord1fvARB) },
   { "glMultiTexCoord1i", _O(MultiTexCoord1i) },
   { "glMultiTexCoord1iv", _O(MultiTexCoord1iv) },
   { "glMultiTexCoord1s", _O(MultiTexCoord1s) },
   { "glMultiTexCoord1sv", _O(MultiTexCoord1sv) },
   { "glMultiTexCoord2d", _O(MultiTexCoord2d) },
   { "glMultiTexCoord2dv", _O(MultiTexCoord2dv) },
   { "glMultiTexCoord2fARB", _O(MultiTexCoord2fARB) },
   { "glMultiTexCoord2fvARB", _O(MultiTexCoord2fvARB) },
   { "glMultiTexCoord2i", _O(MultiTexCoord2i) },
   { "glMultiTexCoord2iv", _O(MultiTexCoord2iv) },
   { "glMultiTexCoord2s", _O(MultiTexCoord2s) },
   { "glMultiTexCoord2sv", _O(MultiTexCoord2sv) },
   { "glMultiTexCoord3d", _O(MultiTexCoord3d) },
   { "glMultiTexCoord3dv", _O(MultiTexCoord3dv) },
   { "glMultiTexCoord3fARB", _O(MultiTexCoord3fARB) },
   { "glMultiTexCoord3fvARB", _O(MultiTexCoord3fvARB) },
   { "glMultiTexCoord3i", _O(MultiTexCoord3i) },
   { "glMultiTexCoord3iv", _O(MultiTexCoord3iv) },
   { "glMultiTexCoord3s", _O(MultiTexCoord3s) },
   { "glMultiTexCoord3sv", _O(MultiTexCoord3sv) },
   { "glMultiTexCoord4d", _O(MultiTexCoord4d) },
   { "glMultiTexCoord4dv", _O(MultiTexCoord4dv) },
   { "glMultiTexCoord4fARB", _O(MultiTexCoord4fARB) },
   { "glMultiTexCoord4fvARB", _O(MultiTexCoord4fvARB) },
   { "glMultiTexCoord4i", _O(MultiTexCoord4i) },
   { "glMultiTexCoord4iv", _O(MultiTexCoord4iv) },
   { "glMultiTexCoord4s", _O(MultiTexCoord4s) },
   { "glMultiTexCoord4sv", _O(MultiTexCoord4sv) },
   { "glAttachShader", _O(AttachShader) },
   { "glCreateProgram", _O(CreateProgram) },
   { "glCreateShader", _O(CreateShader) },
   { "glDeleteProgram", _O(DeleteProgram) },
   { "glDeleteShader", _O(DeleteShader) },
   { "glDetachShader", _O(DetachShader) },
   { "glGetAttachedShaders", _O(GetAttachedShaders) },
   { "glGetProgramInfoLog", _O(GetProgramInfoLog) },
   { "glGetProgramiv", _O(GetProgramiv) },
   { "glGetShaderInfoLog", _O(GetShaderInfoLog) },
   { "glGetShaderiv", _O(GetShaderiv) },
   { "glIsProgram", _O(IsProgram) },
   { "glIsShader", _O(IsShader) },
   { "glStencilFuncSeparate", _O(StencilFuncSeparate) },
   { "glStencilMaskSeparate", _O(StencilMaskSeparate) },
   { "glStencilOpSeparate", _O(StencilOpSeparate) },
   { "glUniformMatrix2x3fv", _O(UniformMatrix2x3fv) },
   { "glUniformMatrix2x4fv", _O(UniformMatrix2x4fv) },
   { "glUniformMatrix3x2fv", _O(UniformMatrix3x2fv) },
   { "glUniformMatrix3x4fv", _O(UniformMatrix3x4fv) },
   { "glUniformMatrix4x2fv", _O(UniformMatrix4x2fv) },
   { "glUniformMatrix4x3fv", _O(UniformMatrix4x3fv) },
   { "glClearBufferfi", _O(ClearBufferfi) },
   { "glClearBufferfv", _O(ClearBufferfv) },
   { "glClearBufferiv", _O(ClearBufferiv) },
   { "glClearBufferuiv", _O(ClearBufferuiv) },
   { "glGetStringi", _O(GetStringi) },
   { "glFramebufferTexture", _O(FramebufferTexture) },
   { "glGetBufferParameteri64v", _O(GetBufferParameteri64v) },
   { "glGetInteger64i_v", _O(GetInteger64i_v) },
   { "glLoadTransposeMatrixd", _O(LoadTransposeMatrixd) },
   { "glLoadTransposeMatrixf", _O(LoadTransposeMatrixf) },
   { "glMultTransposeMatrixd", _O(MultTransposeMatrixd) },
   { "glMultTransposeMatrixf", _O(MultTransposeMatrixf) },
   { "glSampleCoverage", _O(SampleCoverage) },
   { "glCompressedTexImage1D", _O(CompressedTexImage1D) },
   { "glCompressedTexImage2D", _O(CompressedTexImage2D) },
   { "glCompressedTexImage3D", _O(CompressedTexImage3D) },
   { "glCompressedTexSubImage1D", _O(CompressedTexSubImage1D) },
   { "glCompressedTexSubImage2D", _O(CompressedTexSubImage2D) },
   { "glCompressedTexSubImage3D", _O(CompressedTexSubImage3D) },
   { "glGetCompressedTexImage", _O(GetCompressedTexImage) },
   { "glDisableVertexAttribArray", _O(DisableVertexAttribArray) },
   { "glEnableVertexAttribArray", _O(EnableVertexAttribArray) },
   { "glGetProgramEnvParameterdvARB", _O(GetProgramEnvParameterdvARB) },
   { "glGetProgramEnvParameterfvARB", _O(GetProgramEnvParameterfvARB) },
   { "glGetProgramLocalParameterdvARB", _O(GetProgramLocalParameterdvARB) },
   { "glGetProgramLocalParameterfvARB", _O(GetProgramLocalParameterfvARB) },
   { "glGetProgramStringARB", _O(GetProgramStringARB) },
   { "glGetProgramivARB", _O(GetProgramivARB) },
   { "glGetVertexAttribdv", _O(GetVertexAttribdv) },
   { "glGetVertexAttribfv", _O(GetVertexAttribfv) },
   { "glGetVertexAttribiv", _O(GetVertexAttribiv) },
   { "glProgramEnvParameter4dARB", _O(ProgramEnvParameter4dARB) },
   { "glProgramEnvParameter4dvARB", _O(ProgramEnvParameter4dvARB) },
   { "glProgramEnvParameter4fARB", _O(ProgramEnvParameter4fARB) },
   { "glProgramEnvParameter4fvARB", _O(ProgramEnvParameter4fvARB) },
   { "glProgramLocalParameter4dARB", _O(ProgramLocalParameter4dARB) },
   { "glProgramLocalParameter4dvARB", _O(ProgramLocalParameter4dvARB) },
   { "glProgramLocalParameter4fARB", _O(ProgramLocalParameter4fARB) },
   { "glProgramLocalParameter4fvARB", _O(ProgramLocalParameter4fvARB) },
   { "glProgramStringARB", _O(ProgramStringARB) },
   { "glVertexAttrib1d", _O(VertexAttrib1d) },
   { "glVertexAttrib1dv", _O(VertexAttrib1dv) },
   { "glVertexAttrib1fARB", _O(VertexAttrib1fARB) },
   { "glVertexAttrib1fvARB", _O(VertexAttrib1fvARB) },
   { "glVertexAttrib1s", _O(VertexAttrib1s) },
   { "glVertexAttrib1sv", _O(VertexAttrib1sv) },
   { "glVertexAttrib2d", _O(VertexAttrib2d) },
   { "glVertexAttrib2dv", _O(VertexAttrib2dv) },
   { "glVertexAttrib2fARB", _O(VertexAttrib2fARB) },
   { "glVertexAttrib2fvARB", _O(VertexAttrib2fvARB) },
   { "glVertexAttrib2s", _O(VertexAttrib2s) },
   { "glVertexAttrib2sv", _O(VertexAttrib2sv) },
   { "glVertexAttrib3d", _O(VertexAttrib3d) },
   { "glVertexAttrib3dv", _O(VertexAttrib3dv) },
   { "glVertexAttrib3fARB", _O(VertexAttrib3fARB) },
   { "glVertexAttrib3fvARB", _O(VertexAttrib3fvARB) },
   { "glVertexAttrib3s", _O(VertexAttrib3s) },
   { "glVertexAttrib3sv", _O(VertexAttrib3sv) },
   { "glVertexAttrib4Nbv", _O(VertexAttrib4Nbv) },
   { "glVertexAttrib4Niv", _O(VertexAttrib4Niv) },
   { "glVertexAttrib4Nsv", _O(VertexAttrib4Nsv) },
   { "glVertexAttrib4Nub", _O(VertexAttrib4Nub) },
   { "glVertexAttrib4Nubv", _O(VertexAttrib4Nubv) },
   { "glVertexAttrib4Nuiv", _O(VertexAttrib4Nuiv) },
   { "glVertexAttrib4Nusv", _O(VertexAttrib4Nusv) },
   { "glVertexAttrib4bv", _O(VertexAttrib4bv) },
   { "glVertexAttrib4d", _O(VertexAttrib4d) },
   { "glVertexAttrib4dv", _O(VertexAttrib4dv) },
   { "glVertexAttrib4fARB", _O(VertexAttrib4fARB) },
   { "glVertexAttrib4fvARB", _O(VertexAttrib4fvARB) },
   { "glVertexAttrib4iv", _O(VertexAttrib4iv) },
   { "glVertexAttrib4s", _O(VertexAttrib4s) },
   { "glVertexAttrib4sv", _O(VertexAttrib4sv) },
   { "glVertexAttrib4ubv", _O(VertexAttrib4ubv) },
   { "glVertexAttrib4uiv", _O(VertexAttrib4uiv) },
   { "glVertexAttrib4usv", _O(VertexAttrib4usv) },
   { "glVertexAttribPointer", _O(VertexAttribPointer) },
   { "glBindBuffer", _O(BindBuffer) },
   { "glBufferData", _O(BufferData) },
   { "glBufferSubData", _O(BufferSubData) },
   { "glDeleteBuffers", _O(DeleteBuffers) },
   { "glGenBuffers", _O(GenBuffers) },
   { "glGetBufferParameteriv", _O(GetBufferParameteriv) },
   { "glGetBufferPointerv", _O(GetBufferPointerv) },
   { "glGetBufferSubData", _O(GetBufferSubData) },
   { "glIsBuffer", _O(IsBuffer) },
   { "glMapBuffer", _O(MapBuffer) },
   { "glUnmapBuffer", _O(UnmapBuffer) },
   { "glBeginQuery", _O(BeginQuery) },
   { "glDeleteQueries", _O(DeleteQueries) },
   { "glEndQuery", _O(EndQuery) },
   { "glGenQueries", _O(GenQueries) },
   { "glGetQueryObjectiv", _O(GetQueryObjectiv) },
   { "glGetQueryObjectuiv", _O(GetQueryObjectuiv) },
   { "glGetQueryiv", _O(GetQueryiv) },
   { "glIsQuery", _O(IsQuery) },
   { "glAttachObjectARB", _O(AttachObjectARB) },
   { "glCompileShader", _O(CompileShader) },
   { "glCreateProgramObjectARB", _O(CreateProgramObjectARB) },
   { "glCreateShaderObjectARB", _O(CreateShaderObjectARB) },
   { "glDeleteObjectARB", _O(DeleteObjectARB) },
   { "glDetachObjectARB", _O(DetachObjectARB) },
   { "glGetActiveUniform", _O(GetActiveUniform) },
   { "glGetAttachedObjectsARB", _O(GetAttachedObjectsARB) },
   { "glGetHandleARB", _O(GetHandleARB) },
   { "glGetInfoLogARB", _O(GetInfoLogARB) },
   { "glGetObjectParameterfvARB", _O(GetObjectParameterfvARB) },
   { "glGetObjectParameterivARB", _O(GetObjectParameterivARB) },
   { "glGetShaderSource", _O(GetShaderSource) },
   { "glGetUniformLocation", _O(GetUniformLocation) },
   { "glGetUniformfv", _O(GetUniformfv) },
   { "glGetUniformiv", _O(GetUniformiv) },
   { "glLinkProgram", _O(LinkProgram) },
   { "glShaderSource", _O(ShaderSource) },
   { "glUniform1f", _O(Uniform1f) },
   { "glUniform1fv", _O(Uniform1fv) },
   { "glUniform1i", _O(Uniform1i) },
   { "glUniform1iv", _O(Uniform1iv) },
   { "glUniform2f", _O(Uniform2f) },
   { "glUniform2fv", _O(Uniform2fv) },
   { "glUniform2i", _O(Uniform2i) },
   { "glUniform2iv", _O(Uniform2iv) },
   { "glUniform3f", _O(Uniform3f) },
   { "glUniform3fv", _O(Uniform3fv) },
   { "glUniform3i", _O(Uniform3i) },
   { "glUniform3iv", _O(Uniform3iv) },
   { "glUniform4f", _O(Uniform4f) },
   { "glUniform4fv", _O(Uniform4fv) },
   { "glUniform4i", _O(Uniform4i) },
   { "glUniform4iv", _O(Uniform4iv) },
   { "glUniformMatrix2fv", _O(UniformMatrix2fv) },
   { "glUniformMatrix3fv", _O(UniformMatrix3fv) },
   { "glUniformMatrix4fv", _O(UniformMatrix4fv) },
   { "glUseProgram", _O(UseProgram) },
   { "glValidateProgram", _O(ValidateProgram) },
   { "glBindAttribLocation", _O(BindAttribLocation) },
   { "glGetActiveAttrib", _O(GetActiveAttrib) },
   { "glGetAttribLocation", _O(GetAttribLocation) },
   { "glDrawBuffers", _O(DrawBuffers) },
   { "glClampColor", _O(ClampColor) },
   { "glDrawArraysInstancedARB", _O(DrawArraysInstancedARB) },
   { "glDrawElementsInstancedARB", _O(DrawElementsInstancedARB) },
   { "glRenderbufferStorageMultisample", _O(RenderbufferStorageMultisample) },
   { "glFramebufferTexture", _O(FramebufferTexture) },
   { "glProgramParameteri", _O(ProgramParameteri) },
   { "glVertexAttribDivisor", _O(VertexAttribDivisor) },
   { "glFlushMappedBufferRange", _O(FlushMappedBufferRange) },
   { "glMapBufferRange", _O(MapBufferRange) },
   { "glTexBuffer", _O(TexBuffer) },
   { "glBindVertexArray", _O(BindVertexArray) },
   { "glGenVertexArrays", _O(GenVertexArrays) },
   { "glCopyBufferSubData", _O(CopyBufferSubData) },
   { "glClientWaitSync", _O(ClientWaitSync) },
   { "glDeleteSync", _O(DeleteSync) },
   { "glFenceSync", _O(FenceSync) },
   { "glGetInteger64v", _O(GetInteger64v) },
   { "glGetSynciv", _O(GetSynciv) },
   { "glIsSync", _O(IsSync) },
   { "glWaitSync", _O(WaitSync) },
   { "glDrawElementsBaseVertex", _O(DrawElementsBaseVertex) },
   { "glDrawElementsInstancedBaseVertex", _O(DrawElementsInstancedBaseVertex) },
   { "glDrawRangeElementsBaseVertex", _O(DrawRangeElementsBaseVertex) },
   { "glMultiDrawElementsBaseVertex", _O(MultiDrawElementsBaseVertex) },
   { "glBlendEquationSeparateiARB", _O(BlendEquationSeparateiARB) },
   { "glBlendEquationiARB", _O(BlendEquationiARB) },
   { "glBlendFuncSeparateiARB", _O(BlendFuncSeparateiARB) },
   { "glBlendFunciARB", _O(BlendFunciARB) },
   { "glBindFragDataLocationIndexed", _O(BindFragDataLocationIndexed) },
   { "glGetFragDataIndex", _O(GetFragDataIndex) },
   { "glBindSampler", _O(BindSampler) },
   { "glDeleteSamplers", _O(DeleteSamplers) },
   { "glGenSamplers", _O(GenSamplers) },
   { "glGetSamplerParameterIiv", _O(GetSamplerParameterIiv) },
   { "glGetSamplerParameterIuiv", _O(GetSamplerParameterIuiv) },
   { "glGetSamplerParameterfv", _O(GetSamplerParameterfv) },
   { "glGetSamplerParameteriv", _O(GetSamplerParameteriv) },
   { "glIsSampler", _O(IsSampler) },
   { "glSamplerParameterIiv", _O(SamplerParameterIiv) },
   { "glSamplerParameterIuiv", _O(SamplerParameterIuiv) },
   { "glSamplerParameterf", _O(SamplerParameterf) },
   { "glSamplerParameterfv", _O(SamplerParameterfv) },
   { "glSamplerParameteri", _O(SamplerParameteri) },
   { "glSamplerParameteriv", _O(SamplerParameteriv) },
   { "glColorP3ui", _O(ColorP3ui) },
   { "glColorP3uiv", _O(ColorP3uiv) },
   { "glColorP4ui", _O(ColorP4ui) },
   { "glColorP4uiv", _O(ColorP4uiv) },
   { "glMultiTexCoordP1ui", _O(MultiTexCoordP1ui) },
   { "glMultiTexCoordP1uiv", _O(MultiTexCoordP1uiv) },
   { "glMultiTexCoordP2ui", _O(MultiTexCoordP2ui) },
   { "glMultiTexCoordP2uiv", _O(MultiTexCoordP2uiv) },
   { "glMultiTexCoordP3ui", _O(MultiTexCoordP3ui) },
   { "glMultiTexCoordP3uiv", _O(MultiTexCoordP3uiv) },
   { "glMultiTexCoordP4ui", _O(MultiTexCoordP4ui) },
   { "glMultiTexCoordP4uiv", _O(MultiTexCoordP4uiv) },
   { "glNormalP3ui", _O(NormalP3ui) },
   { "glNormalP3uiv", _O(NormalP3uiv) },
   { "glSecondaryColorP3ui", _O(SecondaryColorP3ui) },
   { "glSecondaryColorP3uiv", _O(SecondaryColorP3uiv) },
   { "glTexCoordP1ui", _O(TexCoordP1ui) },
   { "glTexCoordP1uiv", _O(TexCoordP1uiv) },
   { "glTexCoordP2ui", _O(TexCoordP2ui) },
   { "glTexCoordP2uiv", _O(TexCoordP2uiv) },
   { "glTexCoordP3ui", _O(TexCoordP3ui) },
   { "glTexCoordP3uiv", _O(TexCoordP3uiv) },
   { "glTexCoordP4ui", _O(TexCoordP4ui) },
   { "glTexCoordP4uiv", _O(TexCoordP4uiv) },
   { "glVertexAttribP1ui", _O(VertexAttribP1ui) },
   { "glVertexAttribP1uiv", _O(VertexAttribP1uiv) },
   { "glVertexAttribP2ui", _O(VertexAttribP2ui) },
   { "glVertexAttribP2uiv", _O(VertexAttribP2uiv) },
   { "glVertexAttribP3ui", _O(VertexAttribP3ui) },
   { "glVertexAttribP3uiv", _O(VertexAttribP3uiv) },
   { "glVertexAttribP4ui", _O(VertexAttribP4ui) },
   { "glVertexAttribP4uiv", _O(VertexAttribP4uiv) },
   { "glVertexP2ui", _O(VertexP2ui) },
   { "glVertexP2uiv", _O(VertexP2uiv) },
   { "glVertexP3ui", _O(VertexP3ui) },
   { "glVertexP3uiv", _O(VertexP3uiv) },
   { "glVertexP4ui", _O(VertexP4ui) },
   { "glVertexP4uiv", _O(VertexP4uiv) },
   { "glBindTransformFeedback", _O(BindTransformFeedback) },
   { "glDeleteTransformFeedbacks", _O(DeleteTransformFeedbacks) },
   { "glDrawTransformFeedback", _O(DrawTransformFeedback) },
   { "glGenTransformFeedbacks", _O(GenTransformFeedbacks) },
   { "glIsTransformFeedback", _O(IsTransformFeedback) },
   { "glPauseTransformFeedback", _O(PauseTransformFeedback) },
   { "glResumeTransformFeedback", _O(ResumeTransformFeedback) },
   { "glClearDepthf", _O(ClearDepthf) },
   { "glDepthRangef", _O(DepthRangef) },
   { "glGetShaderPrecisionFormat", _O(GetShaderPrecisionFormat) },
   { "glReleaseShaderCompiler", _O(ReleaseShaderCompiler) },
   { "glShaderBinary", _O(ShaderBinary) },
   { "glDebugMessageCallback", _O(DebugMessageCallback) },
   { "glDebugMessageControl", _O(DebugMessageControl) },
   { "glDebugMessageInsert", _O(DebugMessageInsert) },
   { "glGetDebugMessageLog", _O(GetDebugMessageLog) },
   { "glPushDebugGroup", _O(PushDebugGroup) },
   { "glPopDebugGroup", _O(PopDebugGroup) },
   { "glGetObjectLabel", _O(GetObjectLabel) },
   { "glGetObjectPtrLabel", _O(GetObjectPtrLabel) },
   { "glObjectLabel", _O(ObjectLabel) },
   { "glObjectPtrLabel", _O(ObjectPtrLabel) },
   { "glGetGraphicsResetStatusARB", _O(GetGraphicsResetStatusARB) },
   { "glGetnColorTableARB", _O(GetnColorTableARB) },
   { "glGetnCompressedTexImageARB", _O(GetnCompressedTexImageARB) },
   { "glGetnConvolutionFilterARB", _O(GetnConvolutionFilterARB) },
   { "glGetnHistogramARB", _O(GetnHistogramARB) },
   { "glGetnMapdvARB", _O(GetnMapdvARB) },
   { "glGetnMapfvARB", _O(GetnMapfvARB) },
   { "glGetnMapivARB", _O(GetnMapivARB) },
   { "glGetnMinmaxARB", _O(GetnMinmaxARB) },
   { "glGetnPixelMapfvARB", _O(GetnPixelMapfvARB) },
   { "glGetnPixelMapuivARB", _O(GetnPixelMapuivARB) },
   { "glGetnPixelMapusvARB", _O(GetnPixelMapusvARB) },
   { "glGetnPolygonStippleARB", _O(GetnPolygonStippleARB) },
   { "glGetnSeparableFilterARB", _O(GetnSeparableFilterARB) },
   { "glGetnTexImageARB", _O(GetnTexImageARB) },
   { "glGetnUniformdvARB", _O(GetnUniformdvARB) },
   { "glGetnUniformfvARB", _O(GetnUniformfvARB) },
   { "glGetnUniformivARB", _O(GetnUniformivARB) },
   { "glGetnUniformuivARB", _O(GetnUniformuivARB) },
   { "glReadnPixelsARB", _O(ReadnPixelsARB) },
   { "glTexStorage1D", _O(TexStorage1D) },
   { "glTexStorage2D", _O(TexStorage2D) },
   { "glTexStorage3D", _O(TexStorage3D) },
   { "glTextureStorage1DEXT", _O(TextureStorage1DEXT) },
   { "glTextureStorage2DEXT", _O(TextureStorage2DEXT) },
   { "glTextureStorage3DEXT", _O(TextureStorage3DEXT) },
   { "glSampleMaskSGIS", _O(SampleMaskSGIS) },
   { "glSamplePatternSGIS", _O(SamplePatternSGIS) },
   { "glColorPointerEXT", _O(ColorPointerEXT) },
   { "glEdgeFlagPointerEXT", _O(EdgeFlagPointerEXT) },
   { "glIndexPointerEXT", _O(IndexPointerEXT) },
   { "glNormalPointerEXT", _O(NormalPointerEXT) },
   { "glTexCoordPointerEXT", _O(TexCoordPointerEXT) },
   { "glVertexPointerEXT", _O(VertexPointerEXT) },
   { "glPointParameterf", _O(PointParameterf) },
   { "glPointParameterfv", _O(PointParameterfv) },
   { "glLockArraysEXT", _O(LockArraysEXT) },
   { "glUnlockArraysEXT", _O(UnlockArraysEXT) },
   { "glSecondaryColor3b", _O(SecondaryColor3b) },
   { "glSecondaryColor3bv", _O(SecondaryColor3bv) },
   { "glSecondaryColor3d", _O(SecondaryColor3d) },
   { "glSecondaryColor3dv", _O(SecondaryColor3dv) },
   { "glSecondaryColor3fEXT", _O(SecondaryColor3fEXT) },
   { "glSecondaryColor3fvEXT", _O(SecondaryColor3fvEXT) },
   { "glSecondaryColor3i", _O(SecondaryColor3i) },
   { "glSecondaryColor3iv", _O(SecondaryColor3iv) },
   { "glSecondaryColor3s", _O(SecondaryColor3s) },
   { "glSecondaryColor3sv", _O(SecondaryColor3sv) },
   { "glSecondaryColor3ub", _O(SecondaryColor3ub) },
   { "glSecondaryColor3ubv", _O(SecondaryColor3ubv) },
   { "glSecondaryColor3ui", _O(SecondaryColor3ui) },
   { "glSecondaryColor3uiv", _O(SecondaryColor3uiv) },
   { "glSecondaryColor3us", _O(SecondaryColor3us) },
   { "glSecondaryColor3usv", _O(SecondaryColor3usv) },
   { "glSecondaryColorPointer", _O(SecondaryColorPointer) },
   { "glMultiDrawArrays", _O(MultiDrawArrays) },
   { "glMultiDrawElementsEXT", _O(MultiDrawElementsEXT) },
   { "glFogCoordPointer", _O(FogCoordPointer) },
   { "glFogCoordd", _O(FogCoordd) },
   { "glFogCoorddv", _O(FogCoorddv) },
   { "glFogCoordfEXT", _O(FogCoordfEXT) },
   { "glFogCoordfvEXT", _O(FogCoordfvEXT) },
   { "glBlendFuncSeparate", _O(BlendFuncSeparate) },
   { "glResizeBuffersMESA", _O(ResizeBuffersMESA) },
   { "glWindowPos2d", _O(WindowPos2d) },
   { "glWindowPos2dv", _O(WindowPos2dv) },
   { "glWindowPos2f", _O(WindowPos2f) },
   { "glWindowPos2fv", _O(WindowPos2fv) },
   { "glWindowPos2i", _O(WindowPos2i) },
   { "glWindowPos2iv", _O(WindowPos2iv) },
   { "glWindowPos2s", _O(WindowPos2s) },
   { "glWindowPos2sv", _O(WindowPos2sv) },
   { "glWindowPos3d", _O(WindowPos3d) },
   { "glWindowPos3dv", _O(WindowPos3dv) },
   { "glWindowPos3f", _O(WindowPos3f) },
   { "glWindowPos3fv", _O(WindowPos3fv) },
   { "glWindowPos3i", _O(WindowPos3i) },
   { "glWindowPos3iv", _O(WindowPos3iv) },
   { "glWindowPos3s", _O(WindowPos3s) },
   { "glWindowPos3sv", _O(WindowPos3sv) },
   { "glWindowPos4dMESA", _O(WindowPos4dMESA) },
   { "glWindowPos4dvMESA", _O(WindowPos4dvMESA) },
   { "glWindowPos4fMESA", _O(WindowPos4fMESA) },
   { "glWindowPos4fvMESA", _O(WindowPos4fvMESA) },
   { "glWindowPos4iMESA", _O(WindowPos4iMESA) },
   { "glWindowPos4ivMESA", _O(WindowPos4ivMESA) },
   { "glWindowPos4sMESA", _O(WindowPos4sMESA) },
   { "glWindowPos4svMESA", _O(WindowPos4svMESA) },
   { "glMultiModeDrawArraysIBM", _O(MultiModeDrawArraysIBM) },
   { "glMultiModeDrawElementsIBM", _O(MultiModeDrawElementsIBM) },
   { "glAreProgramsResidentNV", _O(AreProgramsResidentNV) },
   { "glBindProgramARB", _O(BindProgramARB) },
   { "glDeleteProgramsARB", _O(DeleteProgramsARB) },
   { "glExecuteProgramNV", _O(ExecuteProgramNV) },
   { "glGenProgramsARB", _O(GenProgramsARB) },
   { "glGetProgramParameterdvNV", _O(GetProgramParameterdvNV) },
   { "glGetProgramParameterfvNV", _O(GetProgramParameterfvNV) },
   { "glGetProgramStringNV", _O(GetProgramStringNV) },
   { "glGetProgramivNV", _O(GetProgramivNV) },
   { "glGetTrackMatrixivNV", _O(GetTrackMatrixivNV) },
   { "glGetVertexAttribPointerv", _O(GetVertexAttribPointerv) },
   { "glGetVertexAttribdvNV", _O(GetVertexAttribdvNV) },
   { "glGetVertexAttribfvNV", _O(GetVertexAttribfvNV) },
   { "glGetVertexAttribivNV", _O(GetVertexAttribivNV) },
   { "glIsProgram", _O(IsProgram) },
   { "glLoadProgramNV", _O(LoadProgramNV) },
   { "glProgramParameters4dvNV", _O(ProgramParameters4dvNV) },
   { "glProgramParameters4fvNV", _O(ProgramParameters4fvNV) },
   { "glRequestResidentProgramsNV", _O(RequestResidentProgramsNV) },
   { "glTrackMatrixNV", _O(TrackMatrixNV) },
   { "glVertexAttrib1dNV", _O(VertexAttrib1dNV) },
   { "glVertexAttrib1dvNV", _O(VertexAttrib1dvNV) },
   { "glVertexAttrib1fNV", _O(VertexAttrib1fNV) },
   { "glVertexAttrib1fvNV", _O(VertexAttrib1fvNV) },
   { "glVertexAttrib1sNV", _O(VertexAttrib1sNV) },
   { "glVertexAttrib1svNV", _O(VertexAttrib1svNV) },
   { "glVertexAttrib2dNV", _O(VertexAttrib2dNV) },
   { "glVertexAttrib2dvNV", _O(VertexAttrib2dvNV) },
   { "glVertexAttrib2fNV", _O(VertexAttrib2fNV) },
   { "glVertexAttrib2fvNV", _O(VertexAttrib2fvNV) },
   { "glVertexAttrib2sNV", _O(VertexAttrib2sNV) },
   { "glVertexAttrib2svNV", _O(VertexAttrib2svNV) },
   { "glVertexAttrib3dNV", _O(VertexAttrib3dNV) },
   { "glVertexAttrib3dvNV", _O(VertexAttrib3dvNV) },
   { "glVertexAttrib3fNV", _O(VertexAttrib3fNV) },
   { "glVertexAttrib3fvNV", _O(VertexAttrib3fvNV) },
   { "glVertexAttrib3sNV", _O(VertexAttrib3sNV) },
   { "glVertexAttrib3svNV", _O(VertexAttrib3svNV) },
   { "glVertexAttrib4dNV", _O(VertexAttrib4dNV) },
   { "glVertexAttrib4dvNV", _O(VertexAttrib4dvNV) },
   { "glVertexAttrib4fNV", _O(VertexAttrib4fNV) },
   { "glVertexAttrib4fvNV", _O(VertexAttrib4fvNV) },
   { "glVertexAttrib4sNV", _O(VertexAttrib4sNV) },
   { "glVertexAttrib4svNV", _O(VertexAttrib4svNV) },
   { "glVertexAttrib4ubNV", _O(VertexAttrib4ubNV) },
   { "glVertexAttrib4ubvNV", _O(VertexAttrib4ubvNV) },
   { "glVertexAttribPointerNV", _O(VertexAttribPointerNV) },
   { "glVertexAttribs1dvNV", _O(VertexAttribs1dvNV) },
   { "glVertexAttribs1fvNV", _O(VertexAttribs1fvNV) },
   { "glVertexAttribs1svNV", _O(VertexAttribs1svNV) },
   { "glVertexAttribs2dvNV", _O(VertexAttribs2dvNV) },
   { "glVertexAttribs2fvNV", _O(VertexAttribs2fvNV) },
   { "glVertexAttribs2svNV", _O(VertexAttribs2svNV) },
   { "glVertexAttribs3dvNV", _O(VertexAttribs3dvNV) },
   { "glVertexAttribs3fvNV", _O(VertexAttribs3fvNV) },
   { "glVertexAttribs3svNV", _O(VertexAttribs3svNV) },
   { "glVertexAttribs4dvNV", _O(VertexAttribs4dvNV) },
   { "glVertexAttribs4fvNV", _O(VertexAttribs4fvNV) },
   { "glVertexAttribs4svNV", _O(VertexAttribs4svNV) },
   { "glVertexAttribs4ubvNV", _O(VertexAttribs4ubvNV) },
   { "glAlphaFragmentOp1ATI", _O(AlphaFragmentOp1ATI) },
   { "glAlphaFragmentOp2ATI", _O(AlphaFragmentOp2ATI) },
   { "glAlphaFragmentOp3ATI", _O(AlphaFragmentOp3ATI) },
   { "glBeginFragmentShaderATI", _O(BeginFragmentShaderATI) },
   { "glBindFragmentShaderATI", _O(BindFragmentShaderATI) },
   { "glColorFragmentOp1ATI", _O(ColorFragmentOp1ATI) },
   { "glColorFragmentOp2ATI", _O(ColorFragmentOp2ATI) },
   { "glColorFragmentOp3ATI", _O(ColorFragmentOp3ATI) },
   { "glDeleteFragmentShaderATI", _O(DeleteFragmentShaderATI) },
   { "glEndFragmentShaderATI", _O(EndFragmentShaderATI) },
   { "glGenFragmentShadersATI", _O(GenFragmentShadersATI) },
   { "glPassTexCoordATI", _O(PassTexCoordATI) },
   { "glSampleMapATI", _O(SampleMapATI) },
   { "glSetFragmentShaderConstantATI", _O(SetFragmentShaderConstantATI) },
   { "glPointParameteri", _O(PointParameteri) },
   { "glPointParameteriv", _O(PointParameteriv) },
   { "glActiveStencilFaceEXT", _O(ActiveStencilFaceEXT) },
   { "glDeleteVertexArrays", _O(DeleteVertexArrays) },
   { "glIsVertexArray", _O(IsVertexArray) },
   { "glGetProgramNamedParameterdvNV", _O(GetProgramNamedParameterdvNV) },
   { "glGetProgramNamedParameterfvNV", _O(GetProgramNamedParameterfvNV) },
   { "glProgramNamedParameter4dNV", _O(ProgramNamedParameter4dNV) },
   { "glProgramNamedParameter4dvNV", _O(ProgramNamedParameter4dvNV) },
   { "glProgramNamedParameter4fNV", _O(ProgramNamedParameter4fNV) },
   { "glProgramNamedParameter4fvNV", _O(ProgramNamedParameter4fvNV) },
   { "glPrimitiveRestartIndex", _O(PrimitiveRestartIndex) },
   { "glPrimitiveRestartNV", _O(PrimitiveRestartNV) },
   { "glDepthBoundsEXT", _O(DepthBoundsEXT) },
   { "glBlendEquationSeparate", _O(BlendEquationSeparate) },
   { "glBindFramebuffer", _O(BindFramebuffer) },
   { "glBindRenderbuffer", _O(BindRenderbuffer) },
   { "glCheckFramebufferStatus", _O(CheckFramebufferStatus) },
   { "glDeleteFramebuffers", _O(DeleteFramebuffers) },
   { "glDeleteRenderbuffers", _O(DeleteRenderbuffers) },
   { "glFramebufferRenderbuffer", _O(FramebufferRenderbuffer) },
   { "glFramebufferTexture1D", _O(FramebufferTexture1D) },
   { "glFramebufferTexture2D", _O(FramebufferTexture2D) },
   { "glFramebufferTexture3D", _O(FramebufferTexture3D) },
   { "glGenFramebuffers", _O(GenFramebuffers) },
   { "glGenRenderbuffers", _O(GenRenderbuffers) },
   { "glGenerateMipmap", _O(GenerateMipmap) },
   { "glGetFramebufferAttachmentParameteriv", _O(GetFramebufferAttachmentParameteriv) },
   { "glGetRenderbufferParameteriv", _O(GetRenderbufferParameteriv) },
   { "glIsFramebuffer", _O(IsFramebuffer) },
   { "glIsRenderbuffer", _O(IsRenderbuffer) },
   { "glRenderbufferStorage", _O(RenderbufferStorage) },
   { "glBlitFramebuffer", _O(BlitFramebuffer) },
   { "glBufferParameteriAPPLE", _O(BufferParameteriAPPLE) },
   { "glFlushMappedBufferRangeAPPLE", _O(FlushMappedBufferRangeAPPLE) },
   { "glBindFragDataLocation", _O(BindFragDataLocation) },
   { "glGetFragDataLocation", _O(GetFragDataLocation) },
   { "glGetUniformuiv", _O(GetUniformuiv) },
   { "glGetVertexAttribIiv", _O(GetVertexAttribIiv) },
   { "glGetVertexAttribIuiv", _O(GetVertexAttribIuiv) },
   { "glUniform1ui", _O(Uniform1ui) },
   { "glUniform1uiv", _O(Uniform1uiv) },
   { "glUniform2ui", _O(Uniform2ui) },
   { "glUniform2uiv", _O(Uniform2uiv) },
   { "glUniform3ui", _O(Uniform3ui) },
   { "glUniform3uiv", _O(Uniform3uiv) },
   { "glUniform4ui", _O(Uniform4ui) },
   { "glUniform4uiv", _O(Uniform4uiv) },
   { "glVertexAttribI1iEXT", _O(VertexAttribI1iEXT) },
   { "glVertexAttribI1iv", _O(VertexAttribI1iv) },
   { "glVertexAttribI1uiEXT", _O(VertexAttribI1uiEXT) },
   { "glVertexAttribI1uiv", _O(VertexAttribI1uiv) },
   { "glVertexAttribI2iEXT", _O(VertexAttribI2iEXT) },
   { "glVertexAttribI2ivEXT", _O(VertexAttribI2ivEXT) },
   { "glVertexAttribI2uiEXT", _O(VertexAttribI2uiEXT) },
   { "glVertexAttribI2uivEXT", _O(VertexAttribI2uivEXT) },
   { "glVertexAttribI3iEXT", _O(VertexAttribI3iEXT) },
   { "glVertexAttribI3ivEXT", _O(VertexAttribI3ivEXT) },
   { "glVertexAttribI3uiEXT", _O(VertexAttribI3uiEXT) },
   { "glVertexAttribI3uivEXT", _O(VertexAttribI3uivEXT) },
   { "glVertexAttribI4bv", _O(VertexAttribI4bv) },
   { "glVertexAttribI4iEXT", _O(VertexAttribI4iEXT) },
   { "glVertexAttribI4ivEXT", _O(VertexAttribI4ivEXT) },
   { "glVertexAttribI4sv", _O(VertexAttribI4sv) },
   { "glVertexAttribI4ubv", _O(VertexAttribI4ubv) },
   { "glVertexAttribI4uiEXT", _O(VertexAttribI4uiEXT) },
   { "glVertexAttribI4uivEXT", _O(VertexAttribI4uivEXT) },
   { "glVertexAttribI4usv", _O(VertexAttribI4usv) },
   { "glVertexAttribIPointer", _O(VertexAttribIPointer) },
   { "glFramebufferTextureLayer", _O(FramebufferTextureLayer) },
   { "glColorMaski", _O(ColorMaski) },
   { "glDisablei", _O(Disablei) },
   { "glEnablei", _O(Enablei) },
   { "glGetBooleani_v", _O(GetBooleani_v) },
   { "glGetIntegeri_v", _O(GetIntegeri_v) },
   { "glIsEnabledi", _O(IsEnabledi) },
   { "glClearColorIiEXT", _O(ClearColorIiEXT) },
   { "glClearColorIuiEXT", _O(ClearColorIuiEXT) },
   { "glGetTexParameterIiv", _O(GetTexParameterIiv) },
   { "glGetTexParameterIuiv", _O(GetTexParameterIuiv) },
   { "glTexParameterIiv", _O(TexParameterIiv) },
   { "glTexParameterIuiv", _O(TexParameterIuiv) },
   { "glBeginConditionalRender", _O(BeginConditionalRender) },
   { "glEndConditionalRender", _O(EndConditionalRender) },
   { "glBeginTransformFeedback", _O(BeginTransformFeedback) },
   { "glBindBufferBase", _O(BindBufferBase) },
   { "glBindBufferOffsetEXT", _O(BindBufferOffsetEXT) },
   { "glBindBufferRange", _O(BindBufferRange) },
   { "glEndTransformFeedback", _O(EndTransformFeedback) },
   { "glGetTransformFeedbackVarying", _O(GetTransformFeedbackVarying) },
   { "glTransformFeedbackVaryings", _O(TransformFeedbackVaryings) },
   { "glProvokingVertex", _O(ProvokingVertex) },
   { "glGetObjectParameterivAPPLE", _O(GetObjectParameterivAPPLE) },
   { "glObjectPurgeableAPPLE", _O(ObjectPurgeableAPPLE) },
   { "glObjectUnpurgeableAPPLE", _O(ObjectUnpurgeableAPPLE) },
   { "glActiveProgramEXT", _O(ActiveProgramEXT) },
   { "glCreateShaderProgramEXT", _O(CreateShaderProgramEXT) },
   { "glUseShaderProgramEXT", _O(UseShaderProgramEXT) },
   { "glTextureBarrierNV", _O(TextureBarrierNV) },
   { "glStencilFuncSeparateATI", _O(StencilFuncSeparateATI) },
   { "glProgramEnvParameters4fvEXT", _O(ProgramEnvParameters4fvEXT) },
   { "glProgramLocalParameters4fvEXT", _O(ProgramLocalParameters4fvEXT) },
   { "glGetQueryObjecti64v", _O(GetQueryObjecti64v) },
   { "glGetQueryObjectui64v", _O(GetQueryObjectui64v) },
   { "glEGLImageTargetRenderbufferStorageOES", _O(EGLImageTargetRenderbufferStorageOES) },
   { "glEGLImageTargetTexture2DOES", _O(EGLImageTargetTexture2DOES) },
   { NULL, 0 }
};

#ifdef GLX_INDIRECT_RENDERING
extern "C" {
GLAPI GLboolean GLAPIENTRY
glAreTexturesResidentEXT(GLsizei n, const GLuint *textures,
			 GLboolean *residences)
{
   (void) n;
   (void) textures;
   (void) residences;
   return GL_FALSE;
}

GLAPI void GLAPIENTRY
glDeleteTexturesEXT(GLsizei n, const GLuint *textures)
{
   (void) n;
   (void) textures;
}

GLAPI void GLAPIENTRY
glGenTexturesEXT(GLsizei n, GLuint *textures)
{
   (void) n;
   (void) textures;
}

GLAPI GLboolean GLAPIENTRY
glIsTextureEXT(GLuint texture)
{
   (void) texture;
   return GL_FALSE;
}

GLAPI void GLAPIENTRY
glGetColorTableEXT(GLenum target, GLenum format, GLenum type, GLvoid *table)
{
   (void) target;
   (void) format;
   (void) type;
   (void) table;
}

GLAPI void GLAPIENTRY
glGetColorTableParameterfvEXT(GLenum target, GLenum pname, GLfloat *params)
{
   (void) target;
   (void) pname;
   (void) params;
}

GLAPI void GLAPIENTRY
glGetColorTableParameterivEXT(GLenum target, GLenum pname, GLint *params)
{
   (void) target;
   (void) pname;
   (void) params;
}

void GLAPIENTRY
gl_dispatch_stub_356(GLenum target, GLenum format, GLenum type, GLvoid * image)
{
   (void) target;
   (void) format;
   (void) type;
   (void) image;
}

void GLAPIENTRY
gl_dispatch_stub_357(GLenum target, GLenum pname, GLfloat * params)
{
   (void) target;
   (void) pname;
   (void) params;
}

void GLAPIENTRY
gl_dispatch_stub_358(GLenum target, GLenum pname, GLint * params)
{
   (void) target;
   (void) pname;
   (void) params;
}

void GLAPIENTRY
gl_dispatch_stub_359(GLenum target, GLenum format, GLenum type, GLvoid * row, GLvoid * column, GLvoid * span)
{
   (void) target;
   (void) format;
   (void) type;
   (void) row;
   (void) column;
   (void) span;
}

void GLAPIENTRY
gl_dispatch_stub_361(GLenum target, GLboolean reset, GLenum format, GLenum type, GLvoid * values)
{
   (void) target;
   (void) reset;
   (void) format;
   (void) type;
   (void) values;
}

void GLAPIENTRY
gl_dispatch_stub_362(GLenum target, GLenum pname, GLfloat * params)
{
   (void) target;
   (void) pname;
   (void) params;
}

void GLAPIENTRY
gl_dispatch_stub_363(GLenum target, GLenum pname, GLint * params)
{
   (void) target;
   (void) pname;
   (void) params;
}

void GLAPIENTRY
gl_dispatch_stub_364(GLenum target, GLboolean reset, GLenum format, GLenum type, GLvoid * values)
{
   (void) target;
   (void) reset;
   (void) format;
   (void) type;
   (void) values;
}

void GLAPIENTRY
gl_dispatch_stub_365(GLenum target, GLenum pname, GLfloat * params)
{
   (void) target;
   (void) pname;
   (void) params;
}

void GLAPIENTRY
gl_dispatch_stub_366(GLenum target, GLenum pname, GLint * params)
{
   (void) target;
   (void) pname;
   (void) params;
}

}
#endif
