/* PageScrollView.m
*  Purpose: A subclass of ScrollView which adds controls to the scroller area.
*
*  You may freely copy, distribute, and reuse the code in this example.
*  NeXT disclaims any warranty of any kind, expressed or  implied, as to its fitness
*  for any particular use.
*
*/

#import <appkit/Application.h>
#import <appkit/Button.h>
#import <appkit/ButtonCell.h>
#import <dpsclient/psops.h>
#import "PageScrollView.h"

#define GADGET_HEIGHT		16.0

@implementation PageScrollView : ScrollView

/* We need the popup list to be smaller than IB will allow to fit comfortably */
/* into the scroller. */
- setZoomPopUpList: anObject
{
	NXRect	aRect;
	
	zoomPopUpList = anObject;
	[zoomPopUpList getFrame: &aRect];
	aRect.size.height = GADGET_HEIGHT;
	[zoomPopUpList setFrame: &aRect];
	return self;
}

/* We need the page form cell to be smaller than IB will allow to fit comfortably */
/* into the scroller */
- setPageForm: anObject
{
	NXRect	aRect;
	
	pageForm = anObject;
	[pageForm getFrame: &aRect];
	aRect.size.height = GADGET_HEIGHT + 4.0;
	[pageForm setFrame: &aRect];
	return self;
}

/* We need the page label to be smaller than IB will allow to fit comfortably */
/* into the scroller. */
- setPageLabel: anObject
{
	NXRect	aRect;
	
	pageLabel = anObject;
	[pageForm getFrame: &aRect];
	aRect.size.height = GADGET_HEIGHT + 2.0;
	[pageForm setFrame: &aRect];
	return self;
}


- initFrame: (const NXRect *)frameRect
{
	[super initFrame: frameRect];
	
	/* load our gadgets in */
	[NXApp loadNibSection: "ScrollerGadgets.nib"  owner:self];

	/* Add the page up button at the bottom of the vertical scroller */
	[self addSubview:pageUpButton];

	/* Add the page down button at the bottom of the vertical scroller, */
	/* below the page up button */
	[self addSubview:pageDownButton];
	
	/* Add the page number cell and it's label */
	[self addSubview:pageLabel];
	[self addSubview:pageForm];
	
	/* Add the zoom list */
	[self addSubview: zoomPopUpList];
	
	/* Now that I've added all my subviews to my ScrollView, free the window */
	[[pageUpButton window] free];
	
	return self;
}

/* - tile 
 *     Override the tile method to draw the subviews in the scroll bars correctly.
 *
 * From the ScrollView spec sheet:
 *	Tiles the subviews of the ScrollView.  You never send a tile message directly, 
 * but you may override it if you need to have the ScrollView manage additional views.  
 * When tile is invoked, it's responsible for sizing each of the subviews of the ScrollView, 
 * including the content view.  This is accomplished by sending each of its subviews a 
 * setFrame: message.  The width of the vertical scroller and the height of the horizontal 
 * scroller (if present) are set to NX_SCROLLERWIDTH.  A tile message is sent whenever 
 * the ScrollView is resized, or a vertical or horizontal scroller is added or removed.  
 * The method invoking tile should then send a  display message to the ScrollView.  
 * Returns self.
 */
- tile
{
	NXRect	aRect, ctlRect;
	float zoom_width, page_label_width, page_cell_width;
	
	[super tile];
	
	/* take the zoom popup list & page display into account on the horizontal scroller */
	[hScroller getFrame: &aRect];
	[zoomPopUpList getFrame: &ctlRect];
	zoom_width = ctlRect.size.width;
	[pageLabel getFrame: &ctlRect];
	page_label_width = ctlRect.size.width;
	[pageForm getFrame: &ctlRect];
	page_cell_width = ctlRect.size.width;
	aRect.size.width -= zoom_width + page_label_width + page_cell_width;
	[hScroller setFrame: &aRect];
	
	/* position the zoom popup list in the correct place */
	aRect.origin.x += aRect.size.width;
	aRect.size.width = zoom_width;
	horzScrollerArea = aRect;
	horzScrollerArea.size.width += page_label_width + page_cell_width;
	[zoomPopUpList moveTo: aRect.origin.x : aRect.origin.y +1.0];
	
	/* position the page display after the popuplist in the horizontal scroller */
	aRect.origin.x += zoom_width;
	aRect.size.width = page_label_width;
	[pageLabel moveTo:aRect.origin.x :aRect.origin.y];
	aRect.origin.x += page_label_width;
	aRect.size.width = page_cell_width;
	[pageForm moveTo: aRect.origin.x :aRect.origin.y];
	
	/* take the page up/down buttons into account on the vertical scroller */
	[vScroller getFrame: &aRect];
	aRect.size.height -= (2.0 * GADGET_HEIGHT) + 2.0;
	[vScroller setFrame: &aRect];
	
	/* position the buttons in the correct place */
	aRect.origin.y += aRect.size.height;
	vertScrollerArea = aRect;
	aRect.size.height = (2.0 * GADGET_HEIGHT) + 2.0;
	[pageUpButton moveTo:1.0 :aRect.origin.y];
	[pageDownButton moveTo:1.0 :aRect.origin.y + GADGET_HEIGHT + 1.0];
	return self;
}

/* We need to override drawSelf to make the background behind the new gadgets */
/* grey instead of the default white */
- drawSelf:(const NXRect *)rects :(int)rectCount
{
    PSsetgray(NX_LTGRAY);
    NXRectFill(&vertScrollerArea);
    NXRectFill(&horzScrollerArea);
    [super drawSelf:rects:rectCount];
    return self;
}

/* This action is connected to the page up/down buttons in the vertical scroller */
- pageButton:sender
{
	fprintf(stderr,"page up/down!\n");
	return self;
}

/* This action is connected to the page formCell in the horizontal scroller */
- pageTo:sender
{
	fprintf(stderr,"move to page!\n");
	return self;
}

/* This action is connected to the zoom popup list in the horizontal scroller */
- zoomTo:sender
{
	fprintf(stderr,"zoom in/out!\n");
	return self;
}

@end
