/*
    objc-load - Dynamically load in Obj-C modules (Classes, Categories)

    Copyright (C) 1993, Adam Fedor.

    objc-load.c,v 1.4 1994/06/28 20:41:08 kamerer Exp
    
    BUGS: Kind-of hacked for dld especially finding class and
    category symbols (objc_find_symbols()).
	I still don't know how to find the Category structure that was 
    loaded (this is apparently a local symbol that is not stored by dld), 
    so this information is not returned in the callback.
	Possible bug in objc_find_symbols if a loaded module name is a 
    substring of an already loaded module name.
	Since we have to sort classes to find the principal class, there is
    an explicit limit of MAXCLASSES classes in a bundle.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/param.h>
#include <objc/objc-api.h>

/* From objc/runtime.h -- needed when loading categories */
extern void __objc_update_dispatch_table_for_class (Class*);

extern char *objc_find_executable(const char *name);

/* We need dld, otherwise we can't load in modules */
#if HAVE_LIBDLD
#include <dld/defs.h>
#endif

#ifdef DEBUG
#define DEBUG_PRINTF printf
#else
#define DEBUG_PRINTF
#endif

#define MAXCLASSES	1024

/* The compiler generates a constructor function for each class.  The function
   has the prefix given below, with the class name appended to it, and a 
   "_m" suffix.
*/
#define GLOBAL_PREFIX	"_GLOBAL_$I$"
#define GLOBAL_SUFFIX	".m"
#define GLOBAL_SUFFIX2	"_m"

/* Class and category symbol prefixes */
#define CLASS_PREFIX	"_objc_class_name_"
#define CATEGORY_PREFIX	"_objc_category_name_"

/* Class and category structure prefixes */
#define CLASS_STRUCT_PREFIX     "_OBJC_CLASS_"
#define CATEGORY_STRUCT_PREFIX  "_OBJC_CATEGORY_"

/* External variables defined in GNU objcX.  We needs these because dld needs
   to know the path to the executable program.
*/
#define NXArgv	argv		/* latest definition? */
extern char     **NXArgv;

/* dynamic_loaded is YES if dld was sucessfully initialized. */
static BOOL	dynamic_loaded;

/* Check to see if there are any undefined symbols. Print them out.
*/
static int
objc_check_undefineds(FILE *errorStream)
{

#ifdef HAVE_LIBDLD
    if (dld_undefined_sym_count) {
        int  i;
        char **undefs;
        undefs = dld_list_undefined_sym();
        fprintf(errorStream, "Undefined symbols:\n");
        for (i=0; i < dld_undefined_sym_count; i++)
            fprintf(errorStream, "  %s\n", undefs[i]);
        free(undefs);
	return 1;
    }
#endif

    return 0;
}

char *
objc_executable_location()
{
    return objc_find_executable(NXArgv[0]);
}

/* Initialize dld for dynamic loading */
static int 
objc_initialize_loading()
{
    char *path;

    dynamic_loaded = NO;
    path   = objc_executable_location();
#ifdef HAVE_LIBDLD
    DEBUG_PRINTF("Debug (objc-load): init'ing dld for %s\n", path);
    if (dld_init(path)) {
	dld_perror("Error (objc-load): cannot initalize dld\n");
	return 1;
    } else
	dynamic_loaded = YES;
#endif /* HAVE_LIBDLD */
    free(path);

#ifdef HAVE_LIBDLD
    return 0;
#else
    /* Can't initialize if we don't have dld */
    return 1;
#endif
}

/* Load the Objective-C symbol (either a Class or a Category) into the
   runtime by calling the global constructor function.  Guaranteed to be
   called only if dynamic_loaded.
*/
static int
objc_load_symbol(const char *name)
{
    char    sym[MAXPATHLEN+1];
    void    (*constructor)();

    /* Find the symbol that corresponds to the class constructor */
    strcpy(sym, GLOBAL_PREFIX);
    strcat(sym, name);
    strcat(sym, GLOBAL_SUFFIX);

#ifdef HAVE_LIBDLD
    DEBUG_PRINTF("Debug (objc-load): Looking for constructor %s\n", sym);
    if (!(constructor = (void (*)()) dld_get_func(sym))) {
    	strcpy(sym, GLOBAL_PREFIX);
    	strcat(sym, name);
    	strcat(sym, GLOBAL_SUFFIX2);
    	if (!(constructor = (void (*)()) dld_get_func(sym))) {
            DEBUG_PRINTF("Error (objc-load): Can't find symbol %s\n", name);
            return 1;
	}
    }

    (*constructor)(); 

#endif /* HAVE_LIBDLD */
    return 0;
}

/* Un-Load the Objective-C symbol (either a Class or a Category) from the
   runtime.  Guaranteed to be called only if dynamic_loaded.
   Not implemented since I don't know how to do it and it may not be possible.
*/
static int
objc_unload_symbol(const char *name)
{
    return 1;
}

#ifdef HAVE_LIBDLD
static int
sym_addr_compare(struct glosym **sym1, struct glosym **sym2)
{
    return (*sym1)->value - (*sym2)->value;
}
#endif

#ifdef HAVE_LIBDLD
static BOOL
is_objc_symbol(struct glosym *sym_entry, char *class_name, char *cat_name)
{
    char *s;
    
    s = strstr(sym_entry->name, CLASS_PREFIX );
    if (!s) {
    	char *e;
    	s = strstr(sym_entry->name, CATEGORY_PREFIX);
    	if (!s)
	    return NO;

	if (class_name) {
	    /* No really, you tell me why Sun computers don't store the
	       name of the category in the value like there supposed too.
	    */
	    if (sym_entry->value && ((char *)sym_entry->value)[0] != 0) 
	        strcpy(cat_name, (char *)sym_entry->value);
	    else {
		/* Ack! No value!!! */
		e = rindex(sym_entry->name, '_');
	        strcpy(cat_name, e+1);
	    }

	    e = strstr(sym_entry->name, cat_name);
	    s += strlen(CATEGORY_PREFIX);
	    strcpy(class_name, s);
	    *(class_name + (size_t)e-(size_t)s-1) = '\0';
	}
    } else if (class_name) {
    	strcpy(class_name, (char *)sym_entry->value);
        cat_name[0] = '\0';
    }
	
    if (class_name)
        DEBUG_PRINTF("Debug (objc-load): Found class %s, category %s\n", 
    	    class_name, cat_name);
    return YES;
}
#endif /* HAVE_LIBDLD */

/* Look through the dld symbol table for Objective-C symbols, call the
   appropriate loading or unloading functions and the callback routine
   for each symbol found.  Guaranteed only to be called if dynamic_loaded.
*/
static int
objc_find_symbols(
	char *module,
	FILE *errorStream,
	int  (*loadFunction)(const char *),
	void (*loadCallback)(Class*, Category*)
)
{
    int  i;
    int  classes = 0;
    int  errors = 0;
    
#ifdef HAVE_LIBDLD
    struct glosym *sym_table[MAXCLASSES];

    for (i=0; i < TABSIZE; i++) {
        struct glosym *sym_entry = _dld_symtab[i];
   	for (; sym_entry; sym_entry = sym_entry->link) {
	    char *s;

	    if (!(s = strstr(sym_entry->defined_by->filename, module)))
	    	continue;

	    if (is_objc_symbol(sym_entry, NULL, NULL)) {
		sym_table[classes++] = sym_entry;

		if (classes > MAXCLASSES) 
		    return 1;
	    }
	}
    }

    /* Now sort the symbols by address */
    qsort(sym_table, classes, sizeof(struct glosym *), sym_addr_compare);

    /* Load the classes in */
    for (i=0; i < classes; i++) {
        char class_name[MAXPATHLEN];
        char cat_name[MAXPATHLEN];
	Class *class;
	is_objc_symbol(sym_table[i], class_name, cat_name);
	if (cat_name[0] != '\0')
	    errors += loadFunction(cat_name);
	else
	    errors += loadFunction(class_name);
		    
	if (errors)
	    continue;
	class = objc_lookUpClass(class_name);
	if (!class) {
	    errors++;
	    continue;
	}

	/* Need to update the dispatch tables for classes that were 
	   already loaded.
	*/
    	if (cat_name[0] != '\0')
	    __objc_update_dispatch_table_for_class(class);

	/* only callback on classes for now */
	if (cat_name[0] == '\0')
	    loadCallback(class, NULL);
    }
#endif

    return errors;
}

long 
objc_load_module(
	char *module,
	FILE *errorStream,
	void (*loadCallback)(Class*, Category*)
)
{
    if (!dynamic_loaded)
        if (objc_initialize_loading())
            return 1;

#ifdef HAVE_LIBDLD
    /* Link in the object file via dld*/
    DEBUG_PRINTF("Debug (objc-load): Linking file %s\n", module);
    if (dld_link(module)) {
	dld_perror("Error (objc-load)");	/* Should go to errorStream */
	return 1;
    }
#endif

    /* If there are any undefined symbols, we can't load the bundle */
    if (objc_check_undefineds(errorStream)) {
#ifdef HAVE_LIBDLD
	dld_unlink_by_file(module, 0);
#endif
	return 1;
    }

    return objc_find_symbols(module, 
    		errorStream, 
		objc_load_symbol, 
		loadCallback);
}

long 
objc_unload_module(
	FILE *errorStream,
	void (*unloadCallback)(Class*, Category*)
)
{
    if (!dynamic_loaded)
        return 1;

#ifdef HAVE_LIBDLD
    /* Make sure not to "unload" the main bundle */
    if (!_dld_latest_entry || !_dld_latest_entry->chain \
	|| !_dld_latest_entry->chain->chain)
	return 1;

    /* Unload the classes */
    objc_find_symbols(_dld_latest_entry->filename, 
    		errorStream, 
		objc_unload_symbol, 
		unloadCallback);
    
    /* Unload the code */
    return dld_unlink_by_file(_dld_latest_entry->filename, 0);
#else
    return 0;
#endif
}
