/* Protocol for Objective-C objects holding (keyElement,contentElement) pairs.
   Copyright (C) 1993 Free Software Foundation, Inc.

   Written by:  R. Andrew McCallum <mccallum@cs.rochester.edu>
   Dept. of Computer Science, U. of Rochester, Rochester, NY  14627

   This file is part of the GNU Objective-C Collection library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/ 

/* The <KeyedCollecting> protocol inherits from the <Collecting> protocol.

   The <KeyedCollecting> protocol defines the interface to a
   collection of elements that are accessible by a key, where the key is
   some unique element.  Pairs of (key element, content element) may be
   added, removed and replaced.  The keys and contents may be tested,
   enumerated and copied.  
*/

#ifndef __KeyedCollecting_h_GNU_INCLUDE
#define __KeyedCollecting_h_GNU_INCLUDE

#include <coll/objc-gnu2next.h>
#include <coll/Collecting.h>

@protocol KeyedCollecting <Collecting>

// ADDING;
- putObject: newContentObject atKey: (elt)aKey;

// REPLACING AND SWAPPING;
- replaceObjectAtKey: (elt)aKey with: newContentObject;
- swapAtKeys: (elt)key1 : (elt)key2;

// REMOVING;
- removeObjectAtKey: (elt)aKey;

// GETTING ELEMENTS AND KEYS;
- objectAtKey: (elt)aKey;
- keyObjectOfObject: aContentObject;

// TESTING;
- (BOOL) includesKey: (elt)aKey;

// ENUMERATIONS;
- withKeyObjectsCall: (void(*)(id))aFunc;
- withKeyObjectsAndContentObjectsCall: (void(*)(id,id))aFunc;
- withKeyObjectsAndContentObjectsCall: (void(*)(id,id))aFunc 
    whileTrue: (BOOL *)flag;

// ENUMERATING WHILE CHANGING CONTENTS;
- safeWithKeyObjectsCall: (void(*)(id))aFunc;
- safeWithKeyObjectsAndContentObjectsCall: (void(*)(id,id))aFunc;
- safeWithKeyObjectsAndContentObjectsCall: (void(*)(id,id))aFunc 
    whileTrue: (BOOL *)flag;


// NON-OBJECT ELEMENT METHOD NAMES;

// INITIALIZING;
- initEncoding: (const char *)contentsEncoding
    keyEncoding: (const char *)keyEncoding;
- initKeyEncoding: (const char *)keyEncoding;

// ADDING;
- putElement: (elt)newContentElement atKey: (elt)aKey;

// REPLACING;
- (elt) replaceElementAtKey: (elt)aKey with: (elt)newContentElement;
- (elt) replaceElementAtKey: (elt)aKey with: (elt)newContentElement
    ifAbsentCall: (elt(*)(arglist_t))excFunc;

// REMOVING;
- (elt) removeElementAtKey: (elt)aKey;
- (elt) removeElementAtKey: (elt)aKey ifAbsentCall: (elt(*)(arglist_t))excFunc;

// GETTING ELEMENTS AND KEYS;
- (elt) elementAtKey: (elt)aKey;
- (elt) elementAtKey: (elt)aKey ifAbsentCall: (elt(*)(arglist_t))excFunc;
- (elt) keyElementOfElement: (elt)aContentObject;
- (elt) keyElementOfElement: (elt)aContentObject
    ifAbsentCall: (elt(*)(arglist_t))excFunc;

// TESTING;
- (const char *) keyEncoding;

// ENUMERATING;
- (BOOL) getNextKey: (elt*)aKeyPtr content: (elt*)anElementPtr 
  withEnumState: (void**)enumState;
- withKeyElementsCall: (void(*)(elt))aFunc;
- withKeyElementsAndContentElementsCall: (void(*)(elt,elt))aFunc;
- withKeyElementsAndContentElementsCall: (void(*)(elt,elt))aFunc 
    whileTrue: (BOOL *)flag;

// ENUMERATING WHILE CHANGING CONTENTS;
- safeWithKeyElementsCall: (void(*)(elt))aFunc;
- safeWithKeyElementsAndContentElementsCall: (void(*)(elt,elt))aFunc;
- safeWithKeyElementsAndContentElementsCall: (void(*)(elt,elt))aFunc 
    whileTrue: (BOOL *)flag;

@end

#endif /* __KeyedCollecting_h_GNU_INCLUDE */
