/* Implementation of View class
 *
 * Copyright (C)  1993  The Board of Trustees of  
 * The Leland Stanford Junior University.  All Rights Reserved.
 *
 * Authors: Scott Francis, Paul Kunz, Tom Pavel, 
 *	    Imran Qureshi, and Libing Wang
 *	    Adam Fedor (U Colorado)
 *          Mike Kienenberger (Alaska)
 *
 * This file is part of an Objective-C class library for a window system
 *
 * View.m,v 1.117 1994/08/17 05:11:19 fxmlk Exp
 */

// In this file, comments that start with '//' are normal, permanent comments.
// Those that start with '/*' mark bugs or inconsistencies with the way
// things should be, or comment out code that doesn't yet work or exist, or
// is likely to change in the near future.
 
#include "View.h"

/* Required for implementation: */
#include  <coll/List.h>
#include "Application.h"
#include "CustomView.h"
#include <stdlib.h>
#include <stdio.h>
#include <objc2/typedstream2.h>
#include <dpsclient/dpsNeXT.h>

extern char *ViewInstanceName(void);

/* Are we drawing, printing, or copying PostScript to the scrap? */

short NXDrawingStatus = NX_DRAWING;

@implementation View:Responder

// Private methods not declared in interface file

- _setSuperview:(View *)aView
{
  superview = aView;
  [self setNextResponder:superview];
  [self windowChanged:window];
  window = [superview window];
//  [self update];  /* don't need to display */
  return self;
}

- _removeSubview:(View *)subview
{
  [subviews removeObject:subview];
  /* For each object below removed object, we need to calculate rectangle
     overlap with this subview, and send -update messages. -km */
  return self;
}

// A modified -findViewWithTag: that returns how deep the tag was found.
- _searchForTag:(int)aTag deep:(unsigned int *)deepret
{
  unsigned int i;
  unsigned int numchildren = [subviews count];
  unsigned int deepness[numchildren];
  unsigned int shallowest;
  id  deepobj[numchildren];
  id  shallowobj;
  
  if ([self tag] == aTag) {
    *deepret = 0;
    return self;
  }
  
  for (i = 0; i < numchildren; i++) {
    deepobj[i] = [[subviews objectAt:i] _searchForTag:aTag deep:&(deepness[i])];
    if (deepness[i] == 0)
      return deepobj[i];
  }
  shallowest = ~0;
  shallowobj = nil;
  for (i = 0; i < numchildren; i++)
    if (deepness[i] < shallowest) {
      shallowest = deepness[i];
      shallowobj = deepobj[i];
    }
  *deepret = shallowest;
  return shallowobj;
}

// Public methods

//
// Initializing and freeing View objects
//
- initFrame:(const NXRect *)frameRect
{
    [super init];
    instancename = ViewInstanceName();
    frame.origin    = frameRect->origin;
    frame.size      = frameRect->size;
    bounds.origin.x = 0;
    bounds.origin.y = 0;
    bounds.size     = frame.size;
    vFlags2.autosizing = NX_NOTSIZABLE;
    vFlags2.autoresizeSubviews = NO;
#ifdef	RESIZE_DEBUG
    fprintf(stderr, "View: autosizing = NX_NOTSIZABLE in initFrame:\n");
#endif	/* RESIZE_DEBUG */
    [self _init];
    return self;
}

- init
{
  NXRect rect = { {0,0}, {0,0} };
  
  [self initFrame:&rect];
  return self;
}

- free
{
  if (subviews) {
    [subviews freeObjects];
    subviews = [subviews free];
    subviews = nil;
  }
  [self removeFromSuperview];
  [self _destroy];
  return [super free];
}

//
// Managing the View hierarchy
//
- addSubview:aView
{
  return [self addSubview:aView :NX_ABOVE relativeTo:nil];
}

- addSubview:aView :(int)place relativeTo:otherView;
{
  unsigned int index;
  
  if (aView == nil)
    return nil;
  
  if ([aView isKindOf:[View class]] == NO)
    fprintf(stderr, "Adding subview that's not a view\n");
  
  if (!subviews)
    subviews = [[List alloc] init];
  [aView removeFromSuperview];
  index = [subviews indexOf:otherView];
  if (index == GNU_NOT_IN_LIST)
    if (place == NX_ABOVE)
      index = 0;
    else
      index = [subviews count];
  [subviews insertObject:aView at:index];
  [aView _setSuperview:self];
  
  if (widgetid)
    [aView _managedBy:self];
  
  return aView;
}

- findAncestorSharedWith:aView
{
  // This method isn't very efficient; it's too brute-force.
  if (aView == self)
    return self;
  if (aView == nil || [aView isKindOf:[View class]] == NO)
    return nil;
  
  if ([aView isDescendantOf:self] == YES)
    return self;
  if (superview == nil)
    return nil;
  return [superview findAncestorSharedWith:aView];
}

- (BOOL)isDescendantOf:aView
{
  unsigned int index, count;
  id object;
  
  if (aView == self)
    return YES;
  if (aView == nil || [aView isKindOf:[View class]] == NO)
    return NO;
  if (subviews == nil)
    return NO;
  count = [subviews count];
  for (index = 0; index < count; index++) {
    object = [subviews objectAt:index];
    if ([object isDescendantOf:aView] == YES)
      return YES;
  }
  return NO;
}

- opaqueAncestor
{
  if (vFlags.opaque || superview == nil)
    return self;
  return [superview opaqueAncestor];
}

- removeFromSuperview
{
  if (superview == nil)
    return self;
  [self setNextResponder:nil];
  [superview _removeSubview:self];
  [self windowChanged:nil];
  superview = nil;
  window = nil;
  [self _unManage];
  return self;
}

- replaceSubview:oldView with:newView
{
  int index;
  
  if (newView == nil || [newView isKindOf:[View class]] == NO ||
      subviews == nil)
    return nil;
  index = [subviews indexOf:oldView];
  if (index == GNU_NOT_IN_LIST)
    return nil;
  [oldView removeFromSuperview];
  [subviews insertObject:newView at:index];
  [newView _setSuperview:self];
  [newView _managedBy:self];
  return oldView;
}

- subviews
{
  return subviews;
}

- superview
{
  return superview;
}

- window
{
  return window;
}

- windowChanged:newWindow
{
  /* Is this method supposed to do anything in this class?  The NS GenRef
     doesn't really say. -km */
  return self;
}

//
// Modifying the frame rectangle
//
- (float)frameAngle
{
  if (!vFlags.rotatedFromBase)
    return 0;
  /* I think that this value is supposed to come from the matrix in the
     DPS server, but for now it's an instance variable. -km */
  return frameAngle;
}

- getFrame:(NXRect *)theRect
{
  theRect->origin = frame.origin;
  theRect->size   = frame.size;
  return self;
}

- moveBy:(NXCoord)deltaX :(NXCoord)deltaY
{
  NXPoint point;
  
  point.x = frame.origin.x + deltaX;
  point.y = frame.origin.y + deltaY;
  return [self moveTo:point.x :point.y];
}    

- moveTo:(NXCoord)x :(NXCoord)y
{
  frame.origin.x = x;
  frame.origin.y = y;
  if (vFlags2.needsAncestorNotify && vFlags2.ancestorNotifyWasEnabled)
    [superview descendantFrameChanged:self];
  [self _moveTo:x :y];
  [self update];
  return self;
}

- rotateBy:(NXCoord)deltaAngle
{
  NXCoord angle;
  
  angle = frameAngle + deltaAngle;
  return [self rotateTo:angle];
}    

- rotateTo:(NXCoord)angle
{
  frameAngle = angle;
  /* Does this method send descendantFrameChanged?  It doesn't, according to
     the NS GenRef. -km */
/*
  if (vFlags2.needsAncestorNotify && vFlags2.ancestorNotifyWasEnabled)
    [superview descendantFrameChanged:self];
*/
  [self update];
  return self;
}

- setFrame:(const NXRect *)frameRect
{
  frame.origin = frameRect->origin;
  frame.size   = frameRect->size;
  /* Need to calculate changes to bounds.  bounds may be translated, scaled,
     and rotated.  This assignment assumes no translating, scaling, or
     rotating, and therefore is wrong. -km */
  [self _setFrame:&frame];
  bounds.size  = frame.size;
  /* Does this method send descendantFrameChanged?  It doesn't, according to
     the NS GenRef, but logically it should. -km */
/*
  if (vFlags2.needsAncestorNotify && vFlags2.ancestorNotifyWasEnabled)
    [superview descendantFrameChanged:self];
*/
//   [self update];  /* don't need to update */
  return self;
}

- sizeBy:(NXCoord)deltaWidth :(NXCoord)deltaHeight
{
  NXSize size;
  
  size.width = frame.size.width + deltaWidth;
  size.height = frame.size.height + deltaHeight;
  return [self sizeTo:size.width :size.height];
}    

- sizeTo:(NXCoord)width :(NXCoord)height;
{
    NXSize oldSize;
    
    oldSize = frame.size;
    frame.size.width   = width;
    frame.size.height  = height;
    /* This assignment is wrong.  The bounds changes in relation to it's CTM. 
	See note in -setFrame:, above. -km */

    bounds.size.width  = width;
    bounds.size.height = height;

    [self _sizeTo:width :height];
    
    if (vFlags2.autoresizeSubviews)
	[self resizeSubviews:&oldSize];
    
    return self;
}

//
// Modifying the coordinate system
//
- (float)boundsAngle
{
  /* This value needs to come from the CTM. -km */
  return 0;
}

- drawInSuperview
{
  /* More playing with the coordinate system.  I'll save this for later. -km */
  return self;
}

- getBounds:(NXRect *)theRect
{
  theRect->origin = bounds.origin;
  theRect->size   = bounds.size;
  return self;
}

- (BOOL)isFlipped
{
  return vFlags.needsFlipped;
}

- (BOOL)isRotatedFromBase
{
  return vFlags.rotatedFromBase;
}

- (BOOL)isRotatedOrScaledFromBase
{
  return vFlags.rotatedOrScaledFromBase;
}

- rotate:(NXCoord)angle
{
  /* Save CTM transformations for later.  Perhaps all of these should go into
     categories? -km */
  return self;
}

- scale:(NXCoord)x :(NXCoord)y
{
  /* Save CTM transformations for later. -km */
  return self;
}

- setDrawOrigin:(NXCoord)x :(NXCoord)y
{
  /* Save CTM transformations for later. -km */
  return self;
}

- setDrawRotation:(NXCoord)angle
{
  /* Save CTM transformations for later. -km */
  return self;
}

- setDrawSize:(NXCoord)width :(NXCoord)height
{
  /* Save CTM transformations for later. -km */
  return self;
}

- setFlipped:(BOOL)flag
{
  BOOL origflag;
  
  origflag = vFlags.needsFlipped;
  vFlags.needsFlipped = !(!flag); // Insure that the BOOL is 1, not just "not 0"
  if (origflag != flag && vFlags2.notifyWhenFlipped)
    [superview descendantFlipped:self];
  return self;
}

- translate:(NXCoord)x :(NXCoord)y
{
  /* Save CTM transformations for later. -km */
  return self;
}

//
// Converting coordinates
//
- centerScanRect:(NXRect *)aRect
{
  /* Save coordinate conversion for after CTMs. -km */
  return self;
}

- convertPoint:(NXPoint *)aPoint fromView:aView
{
  /* Save coordinate conversion for after CTMs. -km */
  return self;
}

- convertPoint:(NXPoint *)aPoint toView:aView
{
  /* Save coordinate conversion for after CTMs. -km */
  return self;
}

- convertPointFromSuperview:(NXPoint *)aPoint
{
  /* Save coordinate conversion for after CTMs. -km */
  return self;
}

- convertPointToSuperview:(NXPoint *)aPoint
{
  /* Save coordinate conversion for after CTMs. -km */
  return self;
}

- convertRect:(NXRect *)aRect fromView:aView
{
  /* Save coordinate conversion for after CTMs. -km */
  return self;
}

- convertRect:(NXRect *)aRect toView:aView
{
  /* Save coordinate conversion for after CTMs. -km */
  return self;
}

- convertRectFromSuperview:(NXRect *)aRect
{
  /* Save coordinate conversion for after CTMs. -km */
  return self;
}

- convertRectToSuperview:(NXRect *)aRect
{
  /* Save coordinate conversion for after CTMs. -km */
  return self;
}

- convertSize:(NXSize *)aSize fromView:aView
{
  /* Save coordinate conversion for after CTMs. -km */
  return self;
}

- convertSize:(NXSize *)aSize toView:aView
{
  /* Save coordinate conversion for after CTMs. -km */
  return self;
}

//
// Notifying ancestor Views
//
- descendantFlipped:sender
{
  if (superview == nil)
    return self;
  return [superview descendantFlipped:sender];
}

- descendantFrameChanged:sender
{
  if (superview == nil)
    return self;
  return [superview descendantFrameChanged:sender];
}

- notifyAncestorWhenFrameChanged:(BOOL)flag
{
  vFlags2.needsAncestorNotify = flag;
  return self;
}

- notifyWhenFlipped:(BOOL)flag
{
  vFlags2.notifyWhenFlipped = flag;
  return self;
}

- suspendNotifyAncestorWhenFrameChanged:(BOOL)flag
{
    if ( flag ) {
	vFlags2.ancestorNotifyWasEnabled = NO;
    } else {
	vFlags2.ancestorNotifyWasEnabled = YES;
    }
    return self;
}

//
// Resizing subviews
//
- resizeSubviews:(const NXSize *)oldSize
{
    if (subviews)
    {
	int count = [subviews count] - 1;
	while (count >= 0)
	{
	    [[subviews objectAt:count] superviewSizeChanged:oldSize];
	    --count;
	}
    }
    return self;
}

- setAutoresizeSubviews:(BOOL)flag
{
  vFlags2.autoresizeSubviews = !(!flag); // Insure that the BOOL is 1, not
                                         // just "not 0"
  return self;
}

- setAutosizing:(unsigned int)mask
{
    vFlags2.autosizing = mask;
#ifdef	RESIZE_DEBUG
    fprintf(stderr, "View: autosizing = mask in setAutosizing:\n");
#endif	/* RESIZE_DEBUG */
    return self;
}

- (unsigned int)autosizing
{
  return vFlags2.autosizing;
}

- superviewSizeChanged:(const NXSize *)oldSize
{
    NXRect newSuperviewFrame, newFrame = frame;
    NXSize myOldSize = frame.size;
    int widthDif, heightDif;
    
#ifdef	RESIZE_DEBUG
    fprintf(stderr, "Autosizing flags: ");
    if (NX_NOTSIZABLE == vFlags2.autosizing)
      fprintf(stderr, "NX_NOTSIZABLE ");
    if (vFlags2.autosizing & NX_WIDTHSIZABLE)
      fprintf(stderr, "NX_WIDTHSIZABLE ");
    if (vFlags2.autosizing & NX_MINXMARGINSIZABLE)
      fprintf(stderr, "NX_MINXMARGINSIZABLE ");
    if (vFlags2.autosizing & NX_MAXXMARGINSIZABLE)
      fprintf(stderr, "NX_MAXXMARGINSIZABLE ");
    if (vFlags2.autosizing & NX_HEIGHTSIZABLE)
      fprintf(stderr, "NX_HEIGHTSIZABLE ");
    if (vFlags2.autosizing & NX_MINYMARGINSIZABLE)
      fprintf(stderr, "NX_MINYMARGINSIZABLE ");
    if (vFlags2.autosizing & NX_MAXYMARGINSIZABLE)
      fprintf(stderr, "NX_MAXYMARGINSIZABLE ");
    fprintf(stderr, "\n");
#endif	/* RESIZE_DEBUG */

    if (vFlags2.autosizing & NX_NOTSIZABLE)  return self;
    
    [superview getFrame:&newSuperviewFrame];
    widthDif = newSuperviewFrame.size.width - oldSize->width;
    if ((vFlags2.autosizing & NX_WIDTHSIZABLE)
     && (vFlags2.autosizing & NX_MINXMARGINSIZABLE)
     && (vFlags2.autosizing & NX_MAXXMARGINSIZABLE))
    {
      newFrame.size.width += (widthDif / 3.0);
      newFrame.origin.x += (widthDif / 3.0);
    }
    else if ((vFlags2.autosizing & NX_WIDTHSIZABLE)
     && (vFlags2.autosizing & NX_MINXMARGINSIZABLE))
    {
      newFrame.size.width += (widthDif / 2.0);
      newFrame.origin.x += (widthDif / 2.0);
    }
    else if ((vFlags2.autosizing & NX_WIDTHSIZABLE)
     && (vFlags2.autosizing & NX_MAXXMARGINSIZABLE))
    {
      newFrame.size.width += (widthDif / 2.0);
    }
    else if ((vFlags2.autosizing & NX_MINXMARGINSIZABLE)
     && (vFlags2.autosizing & NX_MAXXMARGINSIZABLE))
    {
      newFrame.origin.x += (widthDif / 2.0);
    }
    else if (vFlags2.autosizing & NX_WIDTHSIZABLE)
    {
      newFrame.size.width += widthDif;
#ifdef	RESIZE_DEBUG
      fprintf(stderr, "View:  stretching width\n");
#endif	/* RESIZE_DEBUG */
    }
    else if (vFlags2.autosizing & NX_MINXMARGINSIZABLE)
    {
      newFrame.origin.x += widthDif;
    }
    else if (vFlags2.autosizing & NX_MAXXMARGINSIZABLE)
    {
      /* do nothing */
    }

    heightDif = newSuperviewFrame.size.height - oldSize->height;
    if ((vFlags2.autosizing & NX_HEIGHTSIZABLE)
     && (vFlags2.autosizing & NX_MINYMARGINSIZABLE)
     && (vFlags2.autosizing & NX_MAXYMARGINSIZABLE))
    {
      newFrame.size.height += (heightDif / 3.0);
      newFrame.origin.y += (heightDif / 3.0);
    }
    else if ((vFlags2.autosizing & NX_HEIGHTSIZABLE)
     && (vFlags2.autosizing & NX_MINYMARGINSIZABLE))
    {
      newFrame.size.height += (heightDif / 2.0);
      newFrame.origin.y += (heightDif / 2.0);
    }
    else if ((vFlags2.autosizing & NX_HEIGHTSIZABLE)
     && (vFlags2.autosizing & NX_MAXYMARGINSIZABLE))
    {
      newFrame.size.height += (heightDif / 2.0);
    }
    else if ((vFlags2.autosizing & NX_MINYMARGINSIZABLE)
     && (vFlags2.autosizing & NX_MAXYMARGINSIZABLE))
    {
      newFrame.origin.y += (heightDif / 2.0);
    }
    else if (vFlags2.autosizing & NX_HEIGHTSIZABLE)
    {
      newFrame.size.height += heightDif;
#ifdef	RESIZE_DEBUG
      fprintf(stderr, "View:  stretching height\n");
#endif	/* RESIZE_DEBUG */
    }
    else if (vFlags2.autosizing & NX_MINYMARGINSIZABLE)
    {
      newFrame.origin.y += heightDif;
    }
    else if (vFlags2.autosizing & NX_MAXYMARGINSIZABLE)
    {
      /* do nothing */
    }

#ifdef	RESIZE_DEBUG
    fprintf(stderr,
          "View:  changing from (%d, %d, %d, %d) to (%d, %d, %d, %d)\n",
          frame.origin.x, frame.origin.y, frame.size.width,
          frame.size.height, newFrame.origin.x, newFrame.origin.y,
          newFrame.size.width, newFrame.size.height);
#endif	/* RESIZE_DEBUG */

    frame = newFrame;

    if (YES == vFlags2.autoresizeSubviews )
      if ((vFlags2.autosizing & NX_HEIGHTSIZABLE)
       || (vFlags2.autosizing & NX_WIDTHSIZABLE))
          [self resizeSubviews:&myOldSize];

    frame = newFrame;
    [self _setFrame:&frame];

    return self;
}

- _windowRectChangedFrom:(const NXRect *)oldFrame to:(const NXRect *)newFrame
{
    NXRect myNewFrame = frame;
    NXSize myOldSize = frame.size;
    
    myNewFrame.size.width += newFrame->size.width - oldFrame->size.width;
    myNewFrame.size.height += newFrame->size.height - oldFrame->size.height;
    
#ifdef	RESIZE_DEBUG
    fprintf(stderr, "ContentView:  changing from (%d, %d) to (%d, %d)\n",
          frame.size.width, frame.size.height,
          myNewFrame.size.width, myNewFrame.size.height);
#endif	/* RESIZE_DEBUG */

    frame = myNewFrame;

    [self resizeSubviews:&myOldSize];
    frame = myNewFrame;
    [self _setFrame:&frame];

    return self;
}
// Graphics state objects
- allocateGState
{
  // Override this method in the DPS category.
  return self;
}

- freeGState
{
  // Override this method in the DPS category.
  return self;
}

- (int)gState
{
  // Override this method in the DPS category.
  return 0;
}

- initGState
{
  return self;
}

- renewGState
{
  // Override this method in the DPS category.
  return self;
}

- notifyToInitGState:(BOOL)flag
{
  vFlags2.notifyToInitGState = flag;
  return self;
}

//
// Focusing
//
- clipToFrame:(const NXRect *)frameRect
{
  /* Does the default version of this method do anything? -km */
  return self;
}

- (BOOL)doesClip
{
  return !(vFlags.noClip);
}

- setClipping:(BOOL)flag
{
  vFlags.noClip = !flag;
  return self;
}

- (BOOL)isFocusView
{
  /* I need to add some global mechanism (stack?) that keeps track of focusing
     the right way so that I can answer this properly (and maybe even do
     drawing correctly). -km */
  return focusLocked;
}

- (BOOL)lockFocus
{
  /* Focusing is somewhat stack based... This isn't. -km */
  /* Yes it is! - adam */
  BOOL retval;
  retval = focusLocked;
    if (!focusLocked) {
        if (superview)
            [superview lockFocus];
        [self _displayLockFocus];
    }
  focusLocked = YES;
  return retval;
}

- unlockFocus
{
  /* Focusing is somewhat stack based... This isn't. -km */
    /* Yes it is! - adam */
    if (focusLocked) {
        [self _displayUnlockFocus];
        if ( superview )
            [superview unlockFocus];
    }
  focusLocked = NO;
  return self;
}

//
// Displaying
//
- (BOOL)canDraw
{
  if (window != nil) {	/* should check if window is enabled */
    return YES;
  }
  return NO;
}

- display
{
  [self display:NULL :0 :NO];
  return self;
}

- display:(const NXRect *)rects :(int)rectCount
{
  [self display:rects :rectCount];
  return self;
}

- display:(const NXRect *)rects :(int)rectCount :(BOOL)clipFlag
{
  NXRect i_rects[3];
  int rcount;
  
  /* NXBrowser class (and maybe other classes) is incompatible with
     -display.  I think it's a problem with nested focuses.  Until
     it's fixed, I'll leave lockFocus and unlockFocus out of display.
     Nothing calls it yet anyway. -km */
  /* examples/NXimage/EasyViewer does - adam */
  [self lockFocus];
  
  /* clipFlag needs to be checked for. -km */
  if ((rectCount == 0) || (rects == NULL)) {
    i_rects[0] = frame; /* fake it */
    rcount = 1;
    [self drawSelf:i_rects :rcount];
  } else {
    [self drawSelf:rects :rectCount];
  }
  /* Need to calculate whether each subview is inside rects, and if so,
     compute a new array of rects for the subview.  Currently, I cheat. -km */
  [subviews makeObjectsPerform:@selector(display)];
  
  [self unlockFocus];
  return self;
}

- displayFromOpaqueAncestor:(const NXRect *)rects
    :(int)rectCount
    :(BOOL)clipFlag
{
  [self notImplemented:_cmd];
  return self;
}

- displayIfNeeded
{
  if (vFlags.disableAutodisplay && vFlags.needsDisplay)
    [self display];
  vFlags.needsDisplay = 0;
  return self;
}
    
- drawSelf:(const NXRect *)rects :(int)rectCount
{
  return self;
}

- (BOOL)getVisibleRect:(NXRect *)theRect
{
  /* Do calculations with parent Views; in other words, clip theRect to
     the parent and see if there's a rect remaining.  Right now, cheat and
     return the entire frameRect. -km */
  if (!window)
    return NO;
  theRect->origin = frame.origin;
  theRect->size = frame.size;
  return YES;
}

- (BOOL)isAutodisplay
{
  return !(vFlags.disableAutodisplay);
}

- setAutodisplay:(BOOL)flag
{
  vFlags.disableAutodisplay = !flag;
  if (vFlags.needsDisplay) {
    [self display];
    vFlags.needsDisplay = NO;
  }
  return self;
}

- (BOOL)isOpaque
{
  return vFlags.opaque;
}

- setOpaque:(BOOL)flag
{
  vFlags.opaque = !(!flag);  // Make sure that the first bit is on.  A BOOL in
                             // C can mean ANY bit is on, and we need the first.
  return self;
}

- (BOOL)needsDisplay
{
  return vFlags.needsDisplay;
}

- setNeedsDisplay:(BOOL)flag
{
  // Make sure that the first bit is on.  A BOOL in
  // C can mean ANY bit is on, and we need the first.
  vFlags.needsDisplay = !(!flag);
  return self;
}

- (BOOL)shouldDrawColor
{
  /* This should check the X server that the window is on to see if it is
     color.  Should be done in a category for both Xlib AND Xm/Xt.  The
     default of this will return YES for the time being. -km */
  return YES;
}

- update
{
  if (vFlags.disableAutodisplay)
    vFlags.needsDisplay = YES;
  else
    [self display];
  return self;
}

//
// Scrolling
//
/* I don't understand these; it must be too late at night.
   I'll look at them again later. -km */
- adjustScroll:(NXRect *)newVisible
{
  return self;
}

- autoscroll:(NXEvent *)theEvent
{
  return nil;
}

- (BOOL)calcUpdateRects:(NXRect *)rects
    :(int *)rectCount
    :(NXRect *)enclRect
    :(NXRect *)goodRect
{
  if (*rectCount > 0)
    return YES;
  else
    return NO;
}

- invalidate:(const NXRect *)rects
    :(int)rectCount
{
  return self;
}

- scrollPoint:(const NXPoint *)aPoint
{
  return self;
}

- scrollRect:(const NXRect *)aRect
    by:(const NXPoint *)delta
{
  return self;
}

- scrollRectToVisible:(const NXRect *)aRect
{
  return nil;
}

//
// Managing the cursor
//

// This can't be done with any Widget set (well, it can, but it's very messy).
// It requires a low level graphics library interface.  Override these methods
// in an appropriate category to implement.

- addCursorRect:(const NXRect *)aRect
    cursor:anNXCursor
{
  /* For rotated cursor rects, there's two ways to do it:
     1) Create a shaped InputOnly window (X only).
     2) Create many InputOnly (or equivalent non-X systems) windows so that
        it appears that there's a consecutive non-rectangular region.
        Using X, the efficiency difference between (1) and (2) is almost nil.
        (1) will save resources (fewer windows), but will only work if the
        shape extension exists.
     According to the NS GenRef, rotated cursor rects don't work right anyway,
     so maybe we don't have to worry about this... -km */
  return self;
}

- discardCursorRects
{
  return self;
}

- removeCursorRect:(const NXRect *)aRect
    cursor:anNXCursor
{
  return self;
}

- resetCursorRects
{
  return self;
}

//
// Assigning a tag
//
- findViewWithTag:(int)aTag
{
  unsigned int i;
  unsigned int numchildren = [subviews count];
  unsigned int deepness[numchildren];
  unsigned int shallowest;
  id  deepobj[numchildren];
  id  shallowobj;
  
  if ([self tag] == aTag)
    return self;
  
  for (i = 0; i < numchildren; i++) {
    deepobj[i] = [[subviews objectAt:i] _searchForTag:aTag deep:&(deepness[i])];
    if (deepness[i] == 0)
      return deepobj[i];
  }
  shallowest = ~0;
  shallowobj = nil;
  for (i = 0; i < numchildren; i++)
    if (deepness[i] < shallowest) {
      shallowest = deepness[i];
      shallowobj = deepobj[i];
    }
  return shallowobj;
}

- (int)tag
{
  return -1;
}

//
// Aiding event handling
//
- (BOOL)acceptsFirstMouse
{
  // Do the default
  return NO;
}

- hitTest:(NXPoint *)aPoint
{
  NXPoint newPoint;
  unsigned int i;
  unsigned int numchildren = [subviews count];
  id obj;
  
  if (aPoint->x >= frame.origin.x &&
      aPoint->x <= frame.origin.x+frame.size.width &&
      aPoint->y >= frame.origin.y &&
      aPoint->y <= frame.origin.y+frame.size.height) {
    newPoint = *aPoint;
    [self convertPointFromSuperview:&newPoint];
    for (i = 0; i < numchildren; i++)
      if ((obj = [[subviews objectAt:i] hitTest:&newPoint]) != nil)
        return obj;
  }
  return nil;
}

- (BOOL)mouse:(NXPoint *)aPoint
    inRect:(NXRect *)aRect
{
  /* Uncomment the following line, and delete the one after, when this
     function is implemented. -km */
  /* return NXMouseInRect(aPoint, aRect, vFlags.needsFlipped); */
  return NO;
}

- (BOOL)performKeyEquivalent:(NXEvent *)theEvent
{
  unsigned int i;
  unsigned int numchildren = [subviews count];
  
  for (i = 0; i < numchildren; i++)
    if ([[subviews objectAt:i] performKeyEquivalent:theEvent] == YES)
      return YES;
  return NO;
}

- (BOOL)shouldDelayWindowOrderingForEvent:(NXEvent *)anEvent
{
  return NO;
}

/* I am here, for implementation. -km */

//
// Dragging
//
- dragFile:(const char *)filename
    fromRect:(NXRect *)rect
    slideBack:(BOOL)aFlag
    event:(NXEvent *)event
{
  [self notImplemented:_cmd];
  return self;
}

- dragImage:anImage
    at:(NXPoint *)location
    offset:(NXPoint *)mouseOffset
    event:(NXEvent *)theMouseDown
    pasteboard:(id /* really (Pasteboard *) but it doesn't exist yet */)pboard
    source:sourceObject
    slideBack:(BOOL)slideFlag
{
  [self notImplemented:_cmd];
  return self;
}

- registerForDraggedTypes:(const char *const *)pbTypes
    count:(int)count
{
  [self notImplemented:_cmd];
  return self;
}

- unregisterDraggedTypes
{
  [self notImplemented:_cmd];
  return self;
}

//
// Printing
//
- printPSCode:sender
{
  [self notImplemented:_cmd];
  return self;
}

- faxPSCode:sender
{
  [self notImplemented:_cmd];
  return self;
}

- faxPSCode:sender
    toList:(const char *const *)names
    numberList:(const char *const *)numbers
    sendAt:(time_t)time
    wantsCover:(BOOL)coverFlag
    wantsNotify:(BOOL)notifyFlag
    wantsHires:(BOOL)hiresFlag
    faxName:(const char *)string
{
  [self notImplemented:_cmd];
  return self;
}

- copyPSCodeInside:(const NXRect *)rect
    to:(NXStream *)stream
{
  [self notImplemented:_cmd];
  return self;
}

- writePSCodeInside:(const NXRect *)aRect
    to:pasteboard
{
  [self notImplemented:_cmd];
  return self;
}

- openSpoolFile:(char *)filename
{
  [self notImplemented:_cmd];
  return self;
}

- spoolFile:(const char *)filename
{
  [self notImplemented:_cmd];
  return self;
}

- (BOOL)canPrintRIB
{
  return NO;
}

//
// Setting up pages
//
- (BOOL)knowsPagesFirst:(int *)firstPageNum
    last:(int *)lastPageNum
{
  [self notImplemented:_cmd];
  return NO;
}

- (BOOL)getRect:(NXRect *)theRect
    forPage:(int)page
{
  [self notImplemented:_cmd];
  return NO;
}

- placePrintRect:(const NXRect *)aRect
    offset:(NXPoint *)location
{
  [self notImplemented:_cmd];
  return self;
}

- (float)heightAdjustLimit
{
  [self notImplemented:_cmd];
  return 0;
}

- (float)widthAdjustLimit
{
  [self notImplemented:_cmd];
  return 0;
}

//
// Writing conforming PostScript
//

// This section does nothing but return self for all of the methods.
// Actually outputing PostScript code is something that should be done
// in the DPS category.  So it'll be done there.  Some methods below
// are actually considered to be implemented because they just return
// self, according to the NS GenRef.

- beginPSOutput
{
  return self;
}

- beginPrologueBBox:(const NXRect *)boundingbox
    creationDate:(const char *)dateCreated
    createdBy:(const char *)anApplication
    fonts:(const char *)fontNames
    forWhom:(const char *)user
    pages:(int)numPages
    title:(const char *)aTitle
{
  return self;
}

- endHeaderComments
{
  return self;
}

- endPrologue
{
  return self;
}

- beginSetup
{
  return self;
}

- endSetup
{
  return self;
}

- adjustPageWidthNew:(float *)newRight
    left:(float)oldLeft
    right:(float)oldRight
    limit:(float)rightLimit
{
  return self;
}

- adjustPageHeightNew:(float *)newBottom
    top:(float)oldTop
    bottom:(float)oldBottom
    limit:(float)bottomLimit
{
  return self;
}

- beginPageSetupRect:(const NXRect *)aRect
    placement:(const NXPoint *)location
{
  return self;
}

// This method is complete.  It doesn't do anything except return self.
- drawSheetBorder:(float)width
    :(float)height
{
  return self;
}

// This method is complete.  It doesn't do anything except return self.
- drawPageBorder:(float)width
    :(float)height
{
  return self;
}

// This method is complete.  It doesn't do anything except return self.
- addToPageSetup
{
  return self;
}

- endPageSetup
{
  return self;
}

- endPage
{
  return self;
}

- beginTrailer
{
  return self;
}

- endTrailer
{
  return self;
}

- endPSOutput
{
  return self;
}

//
// Archiving
//
- awake
{
  /* This is just copied from the previous version of View.  The
     bounds = frame assignment isn't right.  I just haven't gotten
     to this part of the file yet. -km */
  [super awake];
  
  bounds = frame;
  instancename = ViewInstanceName();
  [self _init];
  return self;
}

- read:(NXTypedStream *)stream
{
    unsigned int autosizing, autoresizeSubviews;
    
    /* This is just copied from the previous version of View.  The
       bounds = frame assignment isn't right.  I just haven't gotten
       to this part of the file yet. -km */
       
    [super read:stream];
    
    NXReadRect(stream, &frame);
    bounds = frame;
    objc_read_type(stream, "i", &autosizing);
    vFlags2.autosizing = autosizing & 0x3F;
    
#ifdef	RESIZE_DEBUG
    fprintf(stderr, "reading Autosizing flags: ");
    if (NX_NOTSIZABLE == vFlags2.autosizing)
    fprintf(stderr, "NX_NOTSIZABLE ");
    if (vFlags2.autosizing & NX_WIDTHSIZABLE)
    fprintf(stderr, "NX_WIDTHSIZABLE ");
    if (vFlags2.autosizing & NX_MINXMARGINSIZABLE)
    fprintf(stderr, "NX_MINXMARGINSIZABLE ");
    if (vFlags2.autosizing & NX_MAXXMARGINSIZABLE)
    fprintf(stderr, "NX_MAXXMARGINSIZABLE ");
    if (vFlags2.autosizing & NX_HEIGHTSIZABLE)
    fprintf(stderr, "NX_HEIGHTSIZABLE ");
    if (vFlags2.autosizing & NX_MINYMARGINSIZABLE)
    fprintf(stderr, "NX_MINYMARGINSIZABLE ");
    if (vFlags2.autosizing & NX_MAXYMARGINSIZABLE)
    fprintf(stderr, "NX_MAXYMARGINSIZABLE ");
    fprintf(stderr, "\n");
#endif	/* RESIZE_DEBUG */
    
    objc_read_type(stream, "i", &autoresizeSubviews);
    vFlags2.autoresizeSubviews = autoresizeSubviews & 0x01;
    objc_read_object(stream, &subviews);
    return self;
}

- write:(NXTypedStream *)stream
{
  [self notImplemented:_cmd];
  return self;
}



/*
 * From here on, stuff added to make it work right.  These shouldn't be public
 * methods.  Some of them don't really need to exist anymore.  But we'll deal
 * with all of that later.  -km
 */

- translateCoords:(NXEvent *)nx
{
    /* simply translate the y coordinates! */

    nx->location.y = bounds.size.height - nx->location.y;
    return self;
}


/* Archiving methods */
- awakeFromNib
{
    CustomView	*subview, *custom;
    int		i;
    
    i = [subviews count];
    while ( i-- ) {
	subview = [subviews objectAt:i];
	if ( [subview isKindOf:[CustomView class]] ) {
	    subview = [subview nibInstantiate];
	    custom = [subviews replaceObjectAt:i with:subview];
	    [custom free];
	} else {
	    [subview awakeFromNib];
	}
    }
    return self;
}

- (NXRect *)frame
{
    return &frame;
}

- (void *)_widget
{
    return widgetid;
}

- _addCallback
{
    return self;
}

- _managedBy:parent
{
    if (![parent _widget]) {
    	fprintf(stderr, "%s: parent widget not found\n", [self name]);
	return nil;
    }
    if ( [parent isKindOf:[View class]] ) {
	superview = parent;
	window = [parent window];
    }
    return [self _managedBy:parent wid:[parent _widget]];
}

- _setWindow:(Window *)aWindow
{
    return [self _setWindow:aWindow andSuperview:nil];
}

- _setWindow:(Window *)aWindow andSuperview:(View *)aView
{
    int		i;

    /* build window and superview chain */
    window = aWindow;
    superview = aView;
    i = [subviews count];
    while (i--)  [[subviews objectAt:i] _setWindow:aWindow andSuperview:self];

    return self;
}

- _manageChildren
{
    View	*view;
    int		i;

    i = [subviews count];
    while ( i-- ) {
	view = [subviews objectAt:i];
	[view _managedBy:self];
    }
    return self;
}

/* really, only boxes should call this method:  
 	Right now, only does a setFrame, but could require special 
	additions later... at first it did, maybe it will again...?
 */
- _setBorderRect:(NXRect *)brect
{
    [self setFrame:brect];
    return self;
}

@end
