static char rcsid[] = "$Id: gdb_parser.c,v 1.2 1992/03/02 16:25:26 jtsillas Exp $";

/*****************************************************************************
 *
 *  Copyright 1989 The University of Texas at Austin
 *  Copyright 1990 Microelectronics and Computer Technology Corporation
 *  Copyright 1990 Thomson Consumer Electronics, Inc.
 *  Copyright 1991 Bull HN Worldwide Info Systems, Inc.
 *
 *****************************************************************************/

/*  gdb_parser.c:
 *
 *    Parse output messages from gdb using regular expression pattern matching,
 *    and take appropriate action.
 *
 *    parse():		Parse the gdb output and invoke the appropriate action
 *			handler.
 *    filter():		Modify the gdb output before it gets displayed on the
 *			dialog window.
 *    gdb_source_command(): Test for source command.
 *    gdb_define_command(): Test for define command.
 *    filter_reading_symbols(): filters 'Reading in symbols' from string 
 *                      (static).
 *    filter_display_info(): filters all display info from string (static).
 */

#include "global.h"
#include "regex.h"
#include "gdb_regex.h"

static void filter_reading_symbols(char*);
static void filter_display_info(char*);

/*
 * Function to remove all 'Reading in symbols' message from a string. This 
 * function is used in parser() before matching the output because this 
 * message can happen any time.
 */
static void filter_reading_symbols(output)
     char *output;
{
  struct re_registers regs;
  int	r;
  char *p1;
  char *p2;
  
  /* test for reading symbols message */
	
  while (re_match(output_pattern[O_READING_SYMBOLS].buf,output,strlen(output),
		  0,&regs) > 0)
    {
      /* we found a "Reading in symbols for 	...done." pattern */
      
      r = output_pattern[O_READING_SYMBOLS].reg_token[TK_MESG];
      p1=  output+regs.start[r];
      p2 = output+regs.end[r];
      
      /* remove "Reading in symbols for 	...done." */
      
      while(*(p1++) = *(p2++));
    }
}

/*
 * This routine first parses the command string. If the command is 
 * one of run, cont, next, step, stop at, stop in, where, up, or down, 
 * it parses the gdb output to decide what action to take and dispatch
 * it to one of the handlers. For other commands, the appropriate 
 * handler is called. !!! This routine has to be re-entrant.
 */
void parse(output, command)
     char *output;
     char *command;
{
  int  command_type;
  char *output_string;

#ifdef DEBUG
  if (debug) {
    char *temp;
    if(!command)temp="";else temp=command;
    fprintf(stderr, "parse(output = %s, command = %s)\n", output, temp);
  }
#endif
  /* Make a local copy of `output' and use that instead */
  output_string = XtNewString(output);
  if (output) strcpy(output, "");
  
  if (!command) {
    if (match(output_pattern, output_string, O_DEBUG) != -1)
      debug_handler(); 
    debug_init();
    XtFree(output_string);
    return;
  }
  
  if (!Parse)
    {
      XtFree(output_string);
      return;
    }
  
  if (match(output_pattern, output_string, O_BELL) != -1)
    {
      bell(0);
      XtFree(output_string);
      return;
    }
  
  command_type = match(command_pattern, command, C_ANY);
  
  
  /* remove all "Reading in symbols for pw.c...done."  */
  
  filter_reading_symbols(output_string);
  
  switch (command_type) {
  case C_EXEC: 
  case C_FINISH:
    {
      char * message;
      int signal;
      message = 0;
      if (match(output_pattern, output_string, O_RECEIVED_SIGNAL) != -1)
	{
	  message = XtNewString(Token.mesg);
	  signal = Token.stop;	/* signal number received */
	}
      
      /* warning : the order of the matching tests is important */
      
      if ((match(output_pattern, output_string, O_EXEC_MESS_AFTER) != -1)
	  || (match(output_pattern, output_string, O_EXEC_MESS_BEFORE) != -1)
	  || (match(output_pattern, output_string, O_EXEC) != -1))
	{
	  exec_handler(message,signal);
	}
      else 
	{
	  if (match(output_pattern, output_string, O_DONE) != -1)
	    done_handler(message,signal);
	  else
	    bell(0);
	}
      
      if (message)
	{
	  bell(0);
	  XtFree(message);
	}
    }
    break;
    
  case C_UPDOWN:
    if (match(output_pattern, output_string, O_UPDOWN) != -1)
      updown_handler();
    else
      bell(0);
    break;
  case C_SEARCH:
    if (match(output_pattern, output_string, O_SEARCH) != -1)
      search_handler();
    else
      bell(0);
    break;
  case C_CLEAR:
    clear_handler(); 
    break;
  case C_LIST:
    if (match(output_pattern, output_string, O_LIST) != -1)
      list_handler();
    else
      bell(0);
    break;
    
  case C_BREAK: 
    if (match(output_pattern, output_string, O_BREAK) != -1)
      break_handler();
    else
      bell(0);
    break;
	    
  case C_INFO_DIR:
    if (match(output_pattern, output_string, O_INFO_DIR) != -1)
      info_dir_handler();
    else
      bell(0);	
    break;
    
  case C_DIRECTORY:
    directory_handler(); 
    break;
    
  case C_INFO_LINE:
    if (match(output_pattern, output_string, O_INFO_LINE) != -1)
      info_line_handler();		/* command was 'info line' */
    else
      bell(0);
    break;
    
  case C_INFO_BREAK:
    info_break_handler(output_string);
    break;
    
  case C_DISPLAY:
    {
      if ((!strcmp(output_string, "")) ||
	  (match(output_pattern, output_string, O_DISPLAY) != -1))
	display_handler();
      else
	bell(0);
    }
    break;
    
  case C_UNDISPLAY:
    if (!strcmp(output_string, ""))
      display_handler();
    else
      bell(0);
    break;
    
  case C_DISPLAY_INFO:
    {
      if ((!strcmp(output_string, "")) ||
	  (match(output_pattern, output_string, O_DISPLAY_INFO) != -1))
	display_info_handler(TRUE);
      else
	bell(0);
    }
    break;
    
  case C_CD:
    if (match(output_pattern, output_string, O_CD) != -1)
      cd_handler(Token.mesg); 
    else
      bell(0);
    break;
    
  case C_PWD:
    if (match(output_pattern, output_string, O_PWD) != -1)
      pwd_handler(Token.mesg);
    else
      bell(0);
    break;
    
  case C_FRAME_CURR:
    if (match(output_pattern, output_string, O_FRAME_CURR) != -1)
      frame_curr_handler();
    else
      bell(0);
    break;
    
  case C_PRINT:
    match(output_pattern, output_string, O_PRINT);
    break;

  case C_HELP:
    help_handler(command, output_string);
    break;

  case C_SYMBOL_FILE:
    debug_handler(); 
    break;
    
  case C_SOURCE:	/* WE SHOULD NEVER ARRIVE HERE */
    break;
    
  case C_EXEC_FILE:
    break;
    
  default:
    break;
  }
  XtFree(output_string);
}

/*
 * Function to filter all the display information in a string
 * input : string pointer.
 * output : none.
 * See O_EXEC_DISPLAY in gdb_regex.h for the display pattern. Take 
 * care when GDB send some message after '\032\032...\n' which is 
 * not a display line.
 */
static void filter_display_info(output)
     char *output;
{
  struct re_registers regs;
  int	r;
  char *p;
  char *p1;
  char *p2;
  char *cp_output;
  int begin_struct;
  
  p = cp_output = XtNewString(output);

  /* find beginning of special gdb line */
  p1 = strstr(p,"\032\032");		
  
  if (!p1 || !(p2 = strchr(p1+1,'\n')))
    {
      AppendDialogText(p);		/* sowrong here */
      XtFree(cp_output);
      return;
    }
  
  *p1 = 0;
  AppendDialogText(p);		/* print everything before that line */
  p = p2 + 1;			/* end of that line + skip \n */
  
  /* test for beginning of a display */
  
  while (re_match(output_pattern[O_EXEC_DISPLAY].buf,p,strlen(p),0,&regs) > 0)
    {
      /* we found a "X:....\n" pattern */
      
      r = output_pattern[O_EXEC_DISPLAY].reg_token[TK_DISP];
      p1 = p+regs.start[r];
      p2 = p+regs.end[r];
      
      /* count number of { and } : if not equal, the next 
	 lines are part of this display */
      begin_struct = 0;
      while (p1 < p2)
	{
	  switch(*(p1++))
	    {
	    case '{':
	      begin_struct++;
	      break;
	    case '}':
	      begin_struct--;
	      break;
	    }
	}
      
      p1=p+regs.start[r];
      *p1 = 0;
      if (p != p1)
	AppendDialogText(p);	/* print what is before display */
      p = p2;			/* skip display text */
      
      if (begin_struct)
	{
	  /* try to find end of display structure */
	  
	  if (p1 = strstr(p,"\n}\n"))
	    p = p1 + strlen("\n}\n");
	}
    }
  
  AppendDialogText(p);		/* print what is after display */
  XtFree(cp_output);
}

/*
 * This function edits the gdb output so that unnecessary information is 
 * not displayed on the dialog window. It filters away the some output 
 * returned by the execution commands; output from the search commands, 
 * and the display command.
 */
void filter(string, output, command)
     char *string, *output, *command;
{
  struct re_registers regs;
  char 		*p;
  char 		*p2;
  char 		*p3;
  int			r;
  static Boolean	deleteRest = False;
  int			command_type = -1;
  static unsigned int already_taken_care;	
  
  if (output == NULL || !strcmp(output, "")) 
    return;
  
  if (!string)
    string = "";
  
  if (command)
    	command_type = match(command_pattern, command, C_ANY);
  
  if ((command_type == C_EXEC)||(command_type == C_FINISH))
    {
      if ((re_match(output_pattern[O_EXEC_MESS_AFTER].buf,
		    string,strlen(string),0,&regs) > 0)
	  || (re_match(output_pattern[O_EXEC_MESS_BEFORE].buf,
		       string,strlen(string),0,&regs) > 0)
	  || (re_match(output_pattern[O_EXEC].buf,
		       string,strlen(string),0,&regs) > 0))
	{
	  /* remember what was the output length was output was matched */
	  already_taken_care = strlen(output) - strlen(string);
	  
	  /* Remove display messages from output and print 
	     what is no a display */
	  
	  if (Prompt)
	    filter_display_info(output + already_taken_care);	
	  else
	    deleteRest = True;
	  
	  return;
	}
      else
	{
	  if (deleteRest)		
	    {
	      /* Remove display messages from output and print 
		 what is no a display */
	      if (Prompt)
		{
		  filter_display_info(output + already_taken_care);	
		  deleteRest = False;
		}
	      return;
	    }
	}
    }
		
  /* filter any line starting with \032\032 */
  
  p = strchr(string,'\032');
  if (p && (*(p+1) == '\032') && (p == string || 
				  *(p-1) == '\n') && (p2 = strchr(p,'\n')))
		{
		  while (*(p++) = *(++p2));
		}
  
  if ((command_type == C_EXEC)||(command_type == C_FINISH))
    {
      AppendDialogText(string);
      return;
    }
  
  if (Prompt)
    {
      char *s;
      
      s = XtNewString(output);
      switch (command_type)
	{
	case C_DISPLAY:
	  if (match(output_pattern, s, O_DISPLAY) != -1)
	    strcpy(s, "");
	  break;
	case C_SEARCH:
	  if (match(output_pattern, s, O_SEARCH) != -1)
	    strcpy(s, "");
	  break;
	case C_LIST:
	  if (match(output_pattern, s, O_LIST) != -1)
	    strcpy(s, "");
	  break;
	case C_PRINT:
	  if(match(output_pattern, s, O_PRINT) == -1)
	    strcpy(s, "");
	  break;
	  
	default:
	  XtFree(s);
	  s = XtNewString(string);		/* append 'string' only */
	  break;
	}
      AppendDialogText(s);
      XtFree(s);
    }
  else
    {
      switch (command_type)
	{
	case C_DISPLAY:
	case C_SEARCH:
	case C_LIST:
	case C_PRINT:
	  break;
	default:
	  AppendDialogText(string);
	  break;
	}
    }
}


/*
 * Function to filter 'source' command
 * input : command (from .gdbinit or source command or keyboard), echo 
 * is TRUE if source command must be echoed.
 * output : TRUE if source command was recognized
 * In case source command is recognized, it is executed here.
 */
int gdb_source_command(command,echo)
     char *command;
     int echo;
{
  if (command && (match(command_pattern, command, C_SOURCE) != -1))
    {
      if (echo)
	AppendDialogText(command);
      source_handler();	
      send_command(" \n");	/* ask GDB to redisplay its prompt */
      return TRUE;
    }
  
  return FALSE;
}


/*
 * Function to filter 'define', 'document' & 'command' commands
 * input : command (from .gdbinit or source command), fp = file pointer.
 * output : TRUE if define or document command was recognized.
 * In case the command is recognized, it is executed here.
 */
int gdb_define_command(command,fp)
     char *command;
     FILE *fp;
{
  char s[LINESIZ];
  int error_cmd;
  int end_found;
  
  if ((match(command_pattern, command, C_DEFINE) != -1)
      || (match(command_pattern, command, C_DOCUMENT) != -1)
      )
    {
      AppendDialogText(command);
      send_command(command);
      
      /*
	gdb could ask :
	
	"Redefine command \"%s\"? "
	"Really redefine built-in command \"%s\"? "
	
	gdb could display 
	
	"Type commands for definition of \"%s\".\n"
	"End with a line saying just \"end\"
	
	or
	"Type documentation for \"%s\".\n"
	"End with a line saying just \"end\".\n"
	
	or
	"Command \"%s\" is built-in."
	*/
      
      error_cmd = FALSE;
      
      /* read message from gdb */
      
      while (1)
#ifdef STR_PTY
	if (fgets(s, LINESIZ, gdbfpin))
#else
	if (fgets(s, LINESIZ, gdbfp))
#endif
	  {
#ifdef DEBUG
	    if (debug)
	      fprintf(stderr, "=>%s", s);
#endif
	    /* error for document of built-in command 
	       or no break point number */
	    if (strstr(s," is built-in.") 
		|| strstr(s, "No breakpoint number"))	
	      {
		AppendDialogText(s);
		error_cmd = TRUE;
		bell(0);
		break;
	      }
	    
	    while (strstr(s,"Redefine command ")
		   || strstr(s,"Really redefine built-in command "))
	      {
		write_gdb ("y\n"); 	/* answer to question if any */
#ifdef STR_PTY
		fgets(s, LINESIZ, gdbfpin);
#else
		fgets(s, LINESIZ, gdbfp);
#endif
	      }
	    if (!strcmp(s,"End with a line saying just \"end\".\n"))
	      break;
	  }
      
      /* write command definition */
      
      end_found = FALSE;
      
      while (fgets (s, LINESIZ, fp))
	{
	  if (!error_cmd)
	    {
	      AppendDialogText(s);
	      write_gdb (s);
	    }
	  
	  if (match(command_pattern, s, C_END) != -1)
	    {
	      end_found = TRUE;
	      break;
	    }
	}
      
      if ((!error_cmd) && (!end_found))
	{
	  AppendDialogText("Error : missing \"end\" in file.\n");
	  bell(0);
	  write_gdb ("end\n");
	}
      
      return TRUE;
    }
  return FALSE;
}




