static char rcsid[] = "$Id: gdb.c,v 1.2 1992/03/02 16:25:24 jtsillas Exp $";

/*****************************************************************************
 *
 *  Copyright 1989 The University of Texas at Austin
 *  Copyright 1990 Microelectronics and Computer Technology Corporation
 *  Copyright 1990 Thomson Consumer Electronics, Inc.
 *  Copyright 1991 Bull HN Worldwide Info Systems, Inc.
 *
 *****************************************************************************/

/*
 *  gdb.c
 *
 *    Handle gdb command initialization file (.gdbinit) and communication 
 *    between gdb and mxgdb.
 *
 *    gdb_init():        Handle .gdbinit (static).
 *    debug_init():	
 *    read_from_gdb():	 Read gdb output, parse and filter it before 
 *                       displaying onto the dialog window.
 *    write_gdb():	 Send a command to gdb.
 *    query_gdb():	 Send a command to gdb and process it.
 *    ReadGdbCallback(): An Xt input callback to handle input from gdb.
 */

#include <errno.h>
#include "global.h"

static void gdb_init(char*);
static char *freadline(char *s, int maxlen, FILE *stream);

Boolean	Prompt;			/* True when gdb prompt arrives */
char 	*concat();
char	*gdbprompt; 
char	*mxgdbprompt;

/*  Given a gdb command initialization file, this routine executes each gdb 
 *  command in the file.  It sends the command to gdb, and calls 
 *  read_from_gdb() to process output returned from gdb.
 */
 
static void gdb_init(mxgdbinit)
char *mxgdbinit;
{
  FILE *fp;
  char s[LINESIZ];
  
  if (!strcmp(mxgdbinit, ""))
    return;
  if (fp = fopen(mxgdbinit, "r")) 
    {
      while (fgets(s, LINESIZ, fp)) 
	{
	  if ((*s != '#') && strcmp(s,"\n"))
	    {
	      if ((!gdb_source_command(s,TRUE)) && 
		  (!gdb_define_command(s,fp))) 
		{
		  write_gdb(s);
		  insert_command(s);
		  AppendDialogText(s);
		}
	      Prompt = False;
	      while (!Prompt)
		read_from_gdb();
	    }
	}
      close((int)fp);
    }
}

/*
 *  This routine is called after getting the first gdb prompt.  
 *  > check the use list to create a list of directories for searching
 *    source files.
 *  > ask gdb for the source file and display it if it exists.
 *  > open the command initialization file and executed the commands;
 *    if Tstartup is true, remove the initialization file.
 */
void debug_init()
{
    static visited = False;

    if (!visited) {
	visited = True;
	gdb_init(mxgdbinit);
	if (Tstartup)
	    unlink(mxgdbinit);
	strcpy(mxgdbinit, "");
    }
}

/*
 *  This is a callback procedure invoked everytime when input is pending
 *  on the file descriptor to gdb.
 *  o reads all the data available on the file descriptor line by line
 *    into local variable 'string' and global variable 'output'.
 *    'output' records the entire gdb output whereas 'string' records
 *    only the data read in this invocation of read_gdb().
 *  o in Echo mode, the contents in 'string' is edited by filter()
 *    before it gets displayed on the dialog window.
 *  o once the gdb prompt is read, calls parse() to analyse the gdb output
 *    and take appropriate action.
 */
void ReadGdbCallback(master, source, id)
     XtPointer master;
     int *source;
     XtInputId *id;
{
  read_from_gdb();
}

void read_from_gdb()
{
  static char *output = NULL; 	/* buffer for gdb output */
  static char *next_string = NULL;
  static char *command;
  char 	*string = NULL;
  char 	s[LINESIZ];
  Boolean 	more;
  
  more = True;
  while (more) 
    {
      Prompt = False;
      /* keep reading until no more or until prompt arrives */
      while (more =
#ifdef STR_PTY
	     fgets(s, LINESIZ, gdbfpin)
#else 
#ifdef GETS_PROBLEM
	     freadline(s, LINESIZ, gdbfp)
#else
	     fgets(s, LINESIZ, gdbfp)
#endif	  
#endif 
	     && !Prompt) 
	{
#ifdef DEBUG
	  if (debug)
	    fprintf(stderr, "=>%s", s);
#endif
	  /* receive prompt? */
	  if (!strncmp(s, gdbprompt, strlen(gdbprompt))) 
	    {
	      Prompt = True;
	      /* more stuff behind prompt? */
	      if (s[strlen(gdbprompt)])
		/* remember it */
		next_string = XtNewString(s+strlen(mxgdbprompt));
	      /* destroy contents */
	      strcpy(s, "");
	    }
	  string = concat(string, s);
	  strcpy(s, "");
	}
      output = concat(output, string);
      command = get_command();
      
      if (Echo) 
	{
	  filter(string, output, command);
	  if (Prompt) AppendDialogText(mxgdbprompt);
	}
      if (string) 
	{
	  XtFree(string);
	  string = NULL;
	}
      if (next_string) 
	{
	  string = concat(string, next_string);
	  XtFree(next_string);
	  next_string = NULL;
	}
      if (Prompt)
	{
	  parse(output, command);
	  delete_command();
	  XtFree(output);
	  output = NULL;
	}
    }
}

/*  Write string s to gdb, and flush the output.  */
void write_gdb(s)
char *s;
{
#ifdef DEBUG
  if (debug)
    fprintf(stderr, ">>%s", s);
#endif

#ifdef STR_PTY
  fputs(s, gdbfpout);
  fflush(gdbfpout);
#else		
  fputs(s, gdbfp);
  fflush(gdbfp);
#endif
}

/*  Sends a command to gdb and read the corresponding output through
 *  read_from_gdb().
 */
void query_gdb(command)
char *command;
{
    write_gdb(command);
    insert_command(command);

    Echo = False;
    Prompt = False;
    while (!Prompt)
        read_from_gdb();

    Parse = True;	/* Always reset Parse and Echo to True */
    Echo = True;
}


static char *freadline(char *s, int maxlen, FILE *stream)
{
  int fd = fileno(stream);
  int n, rc;
  char c;
  char *ptr = s;
  
  for (n = 1; n < maxlen; n++) {
    if ( (rc = read(fd, &c, 1)) == 1) {
      *ptr++ = c;
      if (c == '\n')
	break;
    }
    else if ( (rc == 0) || (rc == -1 && errno == EAGAIN)) {
      if (n == 1)
	return(NULL);
      else break;
    }
    else return(NULL);
  }
  
  *ptr = 0;
  return(s);
}

