/****************************************************************************** 
 *
 *  mixview - X Window System based soundfile editor and processor
 *
 *  Copyright 1989 by Douglas Scott
 *
 *  Author:     Douglas Scott 
 *  Date:       May 1, 1989
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The author makes no representations about
 *  the suitability of this software for any purpose.  
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/
#include "main.h"
#include "dialog.h"
#include "mesg.h"
#include "debug.h"
#include "play.h"

extern int peak_rescan, playtag;
extern int errno;
STUFF stuff;
int devfd = 0;			/* fd for DAC device */

#ifdef ADC_DACS
#if defined(NeXT) || defined SPARC_DACS
int use_adc = 0;
#else
int use_adc = 1;
#endif
short AdcNumControllers;
struct ADCCONFIG AdcConfig[MAX_CONTROLLERS];
#else
#define use_adc (0)
#endif /* ADC_DACS */

#ifdef SPARC_DACS
double volume = 0.5;
double savedVolume = 0.5;
int use_jack = False;
extern int use_native;
#endif /* SPARC_DACS */

#ifdef NeXT
extern int use_native, peak_rescan;
#endif

int
startPlaying(es)
	ebuf_struct *es;
{
#ifdef NeXT
	static SNDSoundStruct Snd;
	int err, play_mesg(), done_mesg();
#endif
	ebuf_struct *b;
	char *buff;
	int bufsize = -1, code = -1;

	playtag = 0;	/* reset */
	if((bufsize =checkSound(es, &b, &buff)) < 0)
		return(playtag = code);
#if defined(SPARC_DACS) || defined(ADC_DACS) || defined(SOUNDBLASTER)
#ifdef NeXT
	if(!use_native) {
#endif
	if((devfd = openDevice()) < 0) /* Open up the converter */
		goto error;
	if(initDevice(devfd, b ? b : es, &stuff) < 0) /* Set up info for cnverter */
		goto error;
	mesg->On(mesg, "Playing...");

	if((code = doConversion(devfd, &stuff, buff, bufsize)) < 0)
		goto error;
#ifdef NeXT
	}
#endif
#endif	/* SPARC_DACS || ADC_DACS || SOUNDBLASTER */
#ifdef NeXT
	/* create new header for NeXT SNDStruct */
	/* es can be used as source for this data even if it is floats */
	if(use_native) {
		if((code = makeTemp(es, buff, bufsize, &Snd)) < 0)
			goto error;
		/* go for it */
		if((err = SNDStartPlaying(&Snd, 1, 5, 1, play_mesg, done_mesg))
				== SND_ERR_NONE)
			code = 1;
		else {
			playtag = code = -1;
			mv_alert(SNDSoundError(err));
			setStartButton();
		}
	}
#endif
error:	;
	if(devfd) close(devfd);
	resetSigs();
	mesg->Off(mesg);
	if(b) destroyBuffer(b);
	return(code);	
#if !defined(NeXT) && !defined(SPARC_DACS) && !defined(ADC_DACS) && !defined(SOUNDBLASTER)
	return -1;	/* no converters available */
#endif
}

#ifdef NeXT
int
play_mesg(sp, tag, err)
	SNDSoundStruct *sp;
	int tag, err;
{
	/* set this here and nowhere else on NeXT */
	playtag = (err == SND_ERR_NONE) ? 1 : -1;
	/* if(err != SND_ERR_NONE)
		mv_alert(SNDSoundError(err));
	else mesg->On(mesg, "Playing..."); */
}

int
done_mesg(sp, tag, err)
	SNDSoundStruct *sp;
	int tag, err;
{
	/* no need to call SNDStop again */
	playtag = (err == SND_ERR_NONE) ? 0 : -1;
	/* mesg->Off(mesg);
	if(err != SND_ERR_NONE) mv_alert(SNDSoundError(err)); */
	setStartButton();
}

int
makeTemp(e, buff, bufsize, Snd)
	ebuf_struct *e;
	char *buff;
	int bufsize;
	SNDSoundStruct *Snd;
{
	SNDSoundStruct *snd;
	int err;
	if((err = SNDAlloc(&snd, 1, 
		(e->size == 1)?SND_FORMAT_MULAW_8:SND_FORMAT_LINEAR_16, 
		(e->srate<30000)?((e->srate<10000)?
			SND_RATE_CODEC:SND_RATE_LOW):SND_RATE_HIGH,
		e->nchans, 4)) != SND_ERR_NONE) {
			mv_alert(SNDSoundError(err));
			return -1;
	}
	/* copy data from new header to static header */
	bcopy((char *) snd, (char *) Snd, sizeof(*Snd));

	/* set data offset to beginning of sample data */
	Snd->dataLocation = buff - (char *) Snd;
	Snd->dataSize = bufsize;
	SNDFree(snd);	/* and free the temporary sound struct */
	return 1;
}	
#endif /* NeXT */

#ifdef SPARC_DACS
void
get_audio_defaults()
{
	unsigned port;
	int err;

	/* Open up the converter */
	if ((devfd = open(DEVICE,MODE)) < 0) {
		mv_error(errno, "Can't open audio device.");
		return;
	}
	err = audio_get_play_gain(devfd, &volume);
	err = audio_get_play_port(devfd, &port);
	if(err != AUDIO_SUCCESS) {
		mv_error(errno, "Can't retrieve audio defaults.");
		close(devfd);
		return;
	}
	savedVolume = volume;
	use_jack = (port == AUDIO_HEADPHONE) ? True : False;
	close(devfd);
}
		
/* Convert local gain into device parameters */
int
scale_gain(g)
    int g;
{
	return (AUDIO_MIN_GAIN + 
		irint(((double) (AUDIO_MAX_GAIN-AUDIO_MIN_GAIN)) * (g/100.0)));
} 

/* vice versa */

int
unscale_gain(g)
    int g;
{
	double val = 100.0 * ((g - AUDIO_MIN_GAIN)/
		(double) (AUDIO_MAX_GAIN - AUDIO_MIN_GAIN));
	return irint(val);
} 

void
set_volume(p, pi)
Panel *p;
Panel_item *pi;
{
	int panelvalue = *((int *) panelitem_get(p, pi, LXPSLIDER_VALUE));
	/* volume is scaled for use by audio macros */
	volume = scale_gain(panelvalue) / (double) AUDIO_MAX_GAIN;
}

void
set_audio_output(p, pi)
	Panel *p;
	Panel_item *pi;
{
	use_jack = *((int *) panelitem_get(p, pi, LXPENUM_VALUE));
}
#endif /* SPARC_DACS */

void
stopPlaying()
{
#ifdef NeXT
	if(!use_adc && playtag == 1) {
		SNDStop(playtag);
		playtag = 0;	/* to avoid duplicate stops */
	}
#endif
#if defined(ADC_DACS) || defined(SPARC_DACS) || defined(SOUNDBLASTER)
	if(playtag > 0) playtag = 0;
#ifdef SPARC_DACS
	if(devfd > 0) {
		audio_flush_play(devfd);
		audio_set_play_gain(devfd, &savedVolume);	/* reset to saved */
	}
#endif
#endif
}

trapSigs()
{
	signal(SIGHUP,stopPlaying);
	signal(SIGBUS,stopPlaying);
	signal(SIGSEGV,stopPlaying);
	signal(SIGTERM,stopPlaying);
#ifdef SIGTSTP
	signal(SIGTSTP,stopPlaying);
#endif
}

resetSigs()
{
	signal(SIGHUP,SIG_DFL);
	signal(SIGBUS,SIG_DFL);
	signal(SIGSEGV,SIG_DFL);
	signal(SIGTERM,SIG_DFL);
#ifdef SIGTSTP
	signal(SIGTSTP,SIG_DFL);
#endif
}

int
checkSound(es, bptr, sdata)
	ebuf_struct *es, **bptr;	/* ptr to ptr passed in to allow later freeing */
	char **sdata;
{
	int bufsize = -1;
	ebuf_struct *b;
	if(checkSRate(es) < 0 || checkChans(es) < 0)
		return -1;
	if(use_adc)
		if(checkSize(es) < 0) return -1;
#if defined(NeXT) || (defined(SPARC_DACS) && defined(SPARC_16_BIT))
	if((!use_adc && es->size != 1 && es->size != 2) || (use_adc && es->size != 2)) {
#else
	if(es->size != CONVERT_FORMAT) {
#endif
		if(peak_rescan != 1) {
			mesg->On(mesg, "Checking peak amplitude...");
			es->peakamp = get_peakamp(1); 
		}
		/* create temp buffer to rescale into */
#if defined(SOUNDBLASTER)
		mesg->On(mesg, "Converting to SB format...");
		*bptr = b = (ebuf_struct *) copyUB(es);
#else
#if defined(SPARC_DACS) && !defined(SPARC_16_BIT)
		mesg->On(mesg, "Converting to MuLaw...");
		*bptr = b = (ebuf_struct *) copyMuLaw(es);
#else
		mesg->On(mesg, "Rescaling...");
		*bptr = b = (ebuf_struct *) copyRescaled(es);
#endif /* SPARC_DACS */
#endif /* SOUNDBLASTER */
		bufsize = b->bufsize;
		*sdata = b->sfbuff;
	}
	else {
		*bptr = b = 0;
		bufsize = es->bufsize;
		*sdata = es->sfbuff;
	}
	return bufsize;
}

int
checkChans(e)
	ebuf_struct *e;
{
	if(!use_adc) {
		if(e->size == 1 && e->nchans != 1) {
			mv_alert("Only monaural mulaw files may be played.");
			return(-1);
		}
#if defined(NeXT) || (defined(SPARC_DACS) && defined(SPARC_16_BIT))
		else if(e->nchans != 1 && e->nchans != 2) {
			mv_alert("Only monaural and stereo files may be played.");
			return(-1);
		};
#endif
	}
	else
		if(e->nchans > 4 || e->nchans == 3) {
			mv_alert("Only mono, stereo, or quad files may be played.");
			return -1;
		};
	return 1;
}

int
checkSRate(e)
	ebuf_struct *e;
{
	char msg[80];
	if(use_adc) {
		if(e->srate != 44100 && e->srate != 48000) {
		sprintf(msg, "Sample rate is %d but must be 44.1K or 48K.", e->srate);
		mv_alert(msg);
		return -1;
		}
	}
#if defined(NeXT) || defined(SPARC_DACS)
	else if(e->size == 1 && e->srate != MULAW_SRATE) {
		sprintf(msg, "Sample rate for MuLaw files must be %d", MULAW_SRATE);
		mv_alert(msg);
		return(-1);
	}
#endif
#ifdef SOUNDBLASTER
	else if(e->srate < 4000 || e->srate > 24000) {
		char msg[80];
		sprintf(msg, "Sample rate must be 4000-24000 to be played.");
		mv_alert(msg);
		return(-1);
	}
}
#endif
	return 1;
}

int
checkSize(e)
	ebuf_struct *e;
{
#ifdef ADC_DACS
	if(e->bufsize < (ADC_BUFLEN * 3)) {
		mv_alert("Too small a sound segment for ADC conversion.");
		return -1;
	}
#endif
	return 1;
}

int
openDevice()
{
	struct stat st;
	int fd = -1;
#ifdef ADC_DACS
	if(use_adc) {
		if(AdcInit() == FALSE) {
			mv_error(errno, "Cannot initialize Adc toolbox.");
			return -1;
		}
		fd = 1;
	}
#endif /* ADC_DACS */
#ifdef SPARC_DACS
	if(!use_adc) {
		if(stat(DEVICE, &st) < 0) {
			mv_error(errno, "Cannot stat converter device.");
			return -1;
		}
		if (!S_ISCHR(st.st_mode)) {
			char msg[128];
			sprintf(msg, "%s %s", DEVICE, "is not an audio device.");
			mv_alert(msg);
			return -1;
		}
	}
#endif
#if defined(SPARC_DACS) || defined(SOUNDBLASTER)
	if ((fd = open(DEVICE,MODE)) < 0) {
		mv_error(errno, "Cannot open converter device.");
		return -1;
	}
#endif /* SPARC_DACS || SOUNDBLASTER */
	return fd;
}

int
initDevice(fd, es, st)
	int fd;
	ebuf_struct *es;
	STUFF *st;
{
#ifdef ADC_DACS
	if(use_adc)
		if(setAdcFIFO(es) < 0) return -1;
#endif
#if defined(SPARC_DACS) || defined(SOUNDBLASTER)
	/* Initialize converter */
	if (!use_adc) {
#if defined(SOUNDBLASTER)
		/* Initialize speed.  SB driver needs more complex set-up. */
		if (ioctl(devfd, DEVICE_SET, (int) es->srate) < 0) {
			mv_error(errno, "Can't initialize device!");
			return -1;
		}
#else
		if(configureSparcDevice(fd, es, st) < 0) return -1;
#endif /* SOUNDBLASTER */
	}
#endif /* SPARC_DACS || SOUNDBLASTER */
	return 1;
}

int
doConversion(fd, st, sdata, bufsize)
	int fd;
	STUFF *st;
	char *sdata;
	int bufsize;
{
	int ret = 1;
	trapSigs();
#ifdef ADC_DACS
	if(use_adc) 
		return (doAdcFIFO(sdata, bufsize));
#endif
#if defined(SPARC_DACS) || defined(SOUNDBLASTER)
	if(!use_adc) ret = writeToDev(fd, sdata, bufsize);
#endif
	if(ret < 0) {
		mv_error(errno, "Error writing to device.");
		return -1;
	}
	return 1;
}

#if defined(SPARC_DACS) || defined(SOUNDBLASTER)
int
writeToDev(fd, data, bufsize)
	int fd;
	char *data;
	int bufsize;
{
	char *ptr;
	int ret = -1, size = BUFLEN;	/* size of buffers to write */
	/* convert to sound */
	size = bufsize < size ? bufsize : size;
	ptr = data;	/* beginning of data to convert */
	setStopButton();
	playtag = 1;
	while((ret=write(fd, ptr, size)) == size) {
		clear_events();	/* check for stop button press */
		if(playtag == 0) {	/* stopped */
			return 0;
		}
		bufsize -= size;	/* decrement total to play */
		ptr += size;	/* increment to next buffer */
		size = bufsize < size ? bufsize : size;
		if(size <= 0) break;
	}
	return ret;
}
#endif

#ifdef SPARC_DACS

#define	SAMPLE_RATE_THRESHOLD	(.01)

int
configureSparcDevice(fd, es, st)
int fd;
ebuf_struct *es;
STUFF *st;
{
	Audio_hdr	Dev_hdr;		/* audio header for device */
	Audio_hdr	File_hdr;		/* audio header for file */
	char msg[256];
	char errmsg[128];
	unsigned int audio_port;
	
	/* fill in values from buffer to be played */
	File_hdr.sample_rate = es->srate;
	File_hdr.samples_per_unit = 1;	/* is this correct? */
	File_hdr.bytes_per_unit = es->size;
	File_hdr.channels = es->nchans;
	File_hdr.encoding = es->size == 1 ? MULAW_FORMAT : SHORT_FORMAT;
	File_hdr.sample_rate = es->srate;

	/* Get the device output encoding configuration */
	if (audio_get_play_config(fd, &Dev_hdr) != AUDIO_SUCCESS) {
		mv_error("The device is not an audio device.");
		return -1;
	}
	if (audio_cmp_hdr(&Dev_hdr, &File_hdr) != 0) {
		Dev_hdr = File_hdr;	
		switch (audio_set_play_config(fd, &Dev_hdr)) {
		case AUDIO_SUCCESS:
			break;
		case AUDIO_ERR_NOEFFECT:
			/*
			* Couldn't change the device.
			* Check to see if we're nearly compatible.
			* audio_cmp_hdr() returns >0 if only sample rate difference.
			*/
			if (audio_cmp_hdr(&Dev_hdr, &File_hdr) > 0) {
				double	ratio;
	
				ratio = (double) abs((int)
					(Dev_hdr.sample_rate - File_hdr.sample_rate)) /
					(double) File_hdr.sample_rate;
				if (ratio <= SAMPLE_RATE_THRESHOLD) {
					sprintf(msg, "Srate was %d, playing at %d",
							File_hdr.sample_rate, Dev_hdr.sample_rate);
					mesg->On(mesg, msg);
				}
				else {
					sprintf(msg, "Sample rate %d not available",
						File_hdr.sample_rate);
					mv_alert(msg);
					return -1;
				}
			}
			(void) audio_enc_to_str(&File_hdr, errmsg);
			sprintf(msg, "play error: Encoding not available: %s", errmsg);
			mv_alert(msg);
			return -1;
		default:
			mv_alert("play: i/o error (set config)");
			return -1;
		}
	}

	if (audio_set_play_gain(fd, &volume) != AUDIO_SUCCESS) {
		mv_error(errno, "Can't set audio gain!");
		return -1;
	}
	audio_port = use_jack ? AUDIO_HEADPHONE : AUDIO_SPEAKER;
	if (audio_set_play_port(fd, &audio_port) != AUDIO_SUCCESS) {
		mv_error(errno, "Can't set audio port!");
		return -1;
	}
}

#endif /* SPARC_DACS */

#ifdef ADC_DACS
int
setAdcFIFO(e)
	ebuf_struct *e;
{
	int rate = e->srate == 44100 ? ADCSampRate44 : ADCSampRate48;
	static int updated = 0;
	AdcSetupSampRate(1, rate);
	if(AdcSetupSampRate(3, rate) == FALSE) {
		mv_error(errno, "Error in AdcSetupSampRate.");
		return -1;
	}
	if(!updated) {
		/* make sure byte order is reset */
		AdcSetupByteOrder(1, ADCByteHighLow);
		if(AdcSetupByteOrder(3, ADCByteHighLow) == FALSE) {
			mv_error(errno, "Error in AdcSetupByteOrder.");
			return -1;
		}
		AdcUpdateSetups(1);        /* update device configuration */
		AdcUpdateSetups(3);        /* update device configuration */
		updated = TRUE;
        }
	/* flush buffers */
	if(AdcFIFOStop(1, ADCFIFOFlush) == FALSE) {
		mv_error(errno, "Error flushing buffers with AdcFIFOStop.");
		return -1;
	}
	/* set FIFO for play */
	if(AdcFIFOPlay(1, e->nchans) == FALSE) {
		mv_error(errno, "Error in AdcFIFOPlay.");
		return -1;
	}
	return 1;
}

int
doAdcFIFO(data, bufsize)
	char *data;
	int bufsize;
{
	struct  ADCERRORINFO ErrorInfo;      /* ADC Error Info structure */
	struct  ADCINFO Info; 
	static char lastbuff[ADC_BUFLEN];
	int prev_type = 0;
	int nbuffs = ADC_MINBUFFS, nsamps = bufsize/(sizeof(short));
	int writesize = ADC_BUFLEN; /* size of write in bytes */
	int writesamps = ADC_BUFSAMPS;
	int lastsamps = nsamps % writesamps;  /* nsamps in last buffer */
	/* load final buffer if last buff shorter than ADC_BUFSAMPS samps */
	if(lastsamps) {
		int lastsize = lastsamps*sizeof(short);
		bzero(lastbuff, writesize);
		bcopy(data+(bufsize-lastsize), lastbuff, lastsize);
	}
/*	fprintf(stderr, "ADC General Information\n");
	AdcGetInfo (1, &Info);
	fprintf(stderr, "    ADC Buffer size   = %u\n",Info.BufferSize);
	fprintf(stderr, "    ADC Buffer count  = %u\n",Info.BufferCount);
	fprintf(stderr, "    ADC List Elements = %u\n",Info.ListElements);
*/
	AdcGetErrorInfo (1, &ErrorInfo);
	while(ErrorInfo.Type != 0) {
	/*	fprintf(stderr, "ADC Error Type    = %u\n",ErrorInfo.Type);
		fprintf(stderr, "ADC Error Number  = %u\n",ErrorInfo.Number);
		fprintf(stderr, "ADC Error Channel = %u\n",ErrorInfo.Channel);
		fprintf(stderr, "ADC Error Text = %s\n",ErrorInfo.Text);
	*/
		if(ErrorInfo.Type != prev_type) {
			mv_alert(ErrorInfo.Text);
			prev_type = ErrorInfo.Type;
		}	
		else fprintf(stderr, "ADC Error Dump: = %s\n",ErrorInfo.Text);
		if(AdcGetErrorInfo (0, &ErrorInfo) == FALSE)
			break;
	}
	/* write buffers to prepare for conversion */
	while(nbuffs--) {
	/*	fprintf(stderr, "   loading %d samps, nsamps= %d\n", writesamps, nsamps);
	*/
		if(AdcWriteFIFO(1, data, writesamps) == FALSE) {
			mv_error(errno, "Error in AdcWriteFIFO.");
			return -1;
		}
		data += writesize;
		nsamps -= writesamps;
		if(nsamps < writesamps) break;
	}
	setStopButton();	/* set button for stopping play */
	playtag = 1;		/* set flag */
	/* start the FIFO playing */
	if(AdcCommand(1, ADCStartFIFO, 1) == FALSE) {
		mv_error(errno, "Unable to start Adc FIFO.");
		return -1;
	}
	/* setup for auto stop if FIFO runs out of buffers */
	AdcFIFOStop(1, ADCFIFODelayed);

	while(nsamps > writesamps) {
		struct ADCCHANNELINFO ChannelInfo;
		clear_events();	/* check for stop button press */
		if(playtag == 0) {
			/* stop and flush anything left */
			if(AdcFIFOStop(1, ADCFIFOFlush) == FALSE) {
				mv_error(errno, 
				"Error flushing buffers with AdcFIFOStop.");
			}
			return 0;
		}
		if(AdcGetChannelInfo(1, &ChannelInfo) == FALSE)
			return -1;
		if(ChannelInfo.FIFOActive == FALSE) {
			if(nsamps > 0 && playtag > 0)
				mv_alert("FIFO play error:  buffer under-run.");
			return 0;
		}
		if(ChannelInfo.NumFIFOBlocks < ADC_MINBUFFS) {
			nbuffs = ADC_MINBUFFS;
			while(nbuffs--) {
				if(AdcWriteFIFO(1, data, writesamps) == FALSE) {
					mv_error(errno, "Error in AdcWriteFIFO.");
					return -1;
				}
				data += writesize;
				nsamps -= writesamps;
				if(nsamps < writesamps) break;
			}
		}
	}
	if(nsamps > 0) {
		AdcWriteFIFO(1, lastbuff, writesamps);
	}
	playtag = 0;	/* no need to call AdcStopPlaying now */
	return 1;
}

int
waitForStop()
{
	struct ADCCHANNELINFO ChannelInfo;
	struct  ADCERRORINFO ErrorInfo;      /* ADC Error Info structure */
	int prev_type = 0;
	if(playtag == 1) {	/* finishing normally */
		do {
			sleep(1);
			clear_events();
			if(AdcGetChannelInfo(1, &ChannelInfo) == FALSE)
				break;
		} while(ChannelInfo.FIFOActive == TRUE);
	}
	if(playtag != -1) {
		if(AdcGetErrorInfo (1, &ErrorInfo) == FALSE)
			ErrorInfo.Type = 0;
		while(ErrorInfo.Type != 0) {
		/*
		   fprintf(stderr, "ADC Error Type    = %u\n",ErrorInfo.Type);
		   fprintf(stderr, "ADC Error Number  = %u\n",ErrorInfo.Number);
		   fprintf(stderr, "ADC Error Channel = %u\n",ErrorInfo.Channel);
		   fprintf(stderr, "ADC Error Text = %s\n",ErrorInfo.Text);
		*/
			if(ErrorInfo.Type != prev_type) {
				mv_alert(ErrorInfo.Text);
				prev_type = ErrorInfo.Type;
			}	
			else fprintf(stderr,
				"ADC Error Dump: = %s\n",ErrorInfo.Text);
		  	if(AdcGetErrorInfo (1, &ErrorInfo) == FALSE)
				break;
		}
	}
	AdcKill();		/* else we were done and we go here */
	playtag = 0;
	setStartButton();
	clear_events();
	return 1;
}
#endif /* ADC_DACS */
