/* Standalone analysis program, reading fixed-point monaural sound files.
 * Currently set for maximum of 34 poles, & max anal segment of 500 samples,
 * meaning that the frame slices cannot exceed 250 samples.
 * Program size expands linearly with slice size, and as square of npoles.
 */

#include "main.h"
#include <pwd.h>
#include <math.h>
#include "dialog.h"
#include "mesg.h"
#include "lpsf.h"
#include "lp.h"

#define INCR 4

static int NPOLE;
static int FRAME;
static int alpol(), gauss();

#define NDATA 	4	/* number of data values stored with frame */

int anallpc (sfbuff, lpdata)
cbuf_struct *sfbuff;
LPDATA *lpdata;
{
        int	j, jj, slice, counter;
        float	coef[POLEMAX+NDATA], inskip, dur;
	char    *oname;
        double	errn, rms1, rms2, cc[40];
        double	sigi[FRAMAX], out[4];
        long	i, nsamps, skipbytes, outskip;
        int	anal, sfd, nbytes, nblpc, firstframe, nread;

	int	n, hsize, esize;		/* new declarations */
	char	framerpt[64];
	int Hflag = 0;
	LPHEADER *lph;
	char	*lpbuf, *tp;
	char    c;
	int nframes;
	
	lpbuf = (char *) mv_alloc(LPBUFSIZ);
	lph = (LPHEADER *) lpbuf;
	tp = lph->text;

	oname = lpdata->lpfname;
	NPOLE = lpdata->npoles;	
	slice = lpdata->offset;
	strncat(tp, lpdata->comment, (LPBUFSIZ - sizeof(LPHEADER) + 4));
	tp += strlen(tp);

	/* error checking... */
	if(!slice) {
		mv_alert("Offset must be nonzero value.");
		return 0;
	}
	if (NPOLE > POLEMAX) {
		mv_alert("Max number of poles allowed is 34.");
		return 0;
	}
	FRAME = slice * 2;
	if (FRAME > FRAMAX) {
		mv_alert("Framesize (inter-frame-offset*2) too large.");
		return 0;
	}
	nsamps = sfbuff->nsamps/sfbuff->nchans;
	if (nsamps < FRAME) {
		mv_alert("Selected region too small for given framesize.");
		return 0;
	}
	if ((anal = creat (oname,0644)) < 0) {
		mv_error(errno, "Cannot open output data file.");
		return 0;
	}
	lph->lpmagic = LP_MAGIC;
	lph->srate = sfbuff->srate;
	lph->npoles = NPOLE;
	lph->nvals = NPOLE + 4;
	lph->framrate = lph->srate / (float) slice;
	lph->duration = sfbuff->nsamps/(double) sfbuff->srate;

	if(lpdata->use_header) {
		/* compute header size */
		hsize = tp - (char *) lph;	/* sizeof text */
		esize = INCR - (hsize % INCR); /* round up to 4 byte boundary */
		if (esize == 4) esize = 0;	/* if we need it */
		lph->headersize = hsize + esize;
		if (lph->headersize >= (LPBUFSIZ - sizeof(LPHEADER) + 4))
			lph->headersize = LPBUFSIZ;
		if (write(anal, (char *) lph, lph->headersize) < lph->headersize) {
			mv_error(errno, "anallpc: can't write header");
			close (anal);
			return 0;
		}
	}
	
/*  for now we'll just say to start at 0 ('C' mind set)  	*/

	firstframe = 0;

	nblpc = (NPOLE + NDATA)*FLOAT;
	outskip = firstframe * nblpc + lph->headersize;	/* headersize is new */

	if ((lseek (anal,outskip,0)) < 0) {
	  	mv_error(errno, "Bad lseek on analysis file");
		close (anal);
	  	return 0;
	}

	sfbuff->inptr = 0;	/* reset pointers in buffer */

	for (i = 0; i < FRAME; i++) {
		GETSAMP(sfbuff, out);
		sigi[i] = 0.0;
		for (j=0; j < sfbuff->nchans; j++) sigi[i] += out[j]; 
	}
	counter = 1;
	nframes = nsamps / slice;
	for (i = 0; i < nsamps; ) {
		if (!alpol(sigi,&errn,&rms1,&rms2,cc)) {
			mesg->Off(mesg);
			close (anal);
			return 0;
		}
		coef[0] = (float)rms2; 
		coef[1] = (float)rms1; 
		coef[2] = (float)errn; 
		coef[3] = 0.;			  /* for pitch of this frame */
			/* reverse the coefs & change sign */
		for(jj=NDATA; jj<NPOLE+NDATA; jj++)
			coef[jj] = (float)-cc[NPOLE-jj+(NDATA - 1)];  
		/* then write the anal frame */
		if ((n = write(anal, (char *)coef, nblpc)) != nblpc) {
			mesg->Off(mesg);
			mv_error(errno, "analpc: write error");
			close (anal);
			return 0;
		}
		sprintf (framerpt, "Running LPC:  frame %5d of %5d", 
			counter, nframes);
		if(!(counter++ % 5)) mesg->On(mesg, framerpt);
		for(jj=0; jj<slice; jj++,i++)	   /* now move slice forward */
			sigi[jj] = sigi[jj+slice];	/*   & refill:	     */
		if(sfbuff->inptr+FRAME > sfbuff->nsamps) break; /* EOF */
		for (jj = slice; jj < FRAME; jj++) {
			GETSAMP(sfbuff, out);
			sigi[jj] = 0.0;
			for (j=0; j < sfbuff->nchans; j++) sigi[jj] += out[j]; 
		}
	}
	close (anal);
	cfree(lpbuf);
	mesg->Off(mesg);
	return 1;
}

static int alpol(sig, errn, rms1, rms2, c)
double *sig, *errn, *rms1, *rms2, *c;
{
	double a[POLEMAX][POLEMAX], v[POLEMAX], b[POLEMAX];
	double *vp=v, *bp=b;
	double sum, sumx, sumy, sqrt();
	int k1, i, l, k, limit, j;

	k1 = NPOLE + 1;
	for (i = 0; i < NPOLE; ++i)  {
		sum = (double) 0.0;
		for (k = NPOLE; k < FRAME; ++k) sum += sig[k-(i+1)] * sig[k];
		v[i] = -sum;
		if (i != NPOLE - 1)  {
			limit = NPOLE - (i+1);
			for (l=0; l < limit ;++l)  {
				sum +=	sig[NPOLE-(i+1)-(l+1)] *
					sig[NPOLE-(l+1)] -
					sig[FRAME-(i+1)-(l+1)] *
					sig[FRAME-(l+1)];
				a[(i+1)+l][l] = a[l][(i+1)+l] = sum;
				}
			}
		}
	sum = (double) 0.0;
	for (k = NPOLE; k < FRAME; ++k) sum += sig[k] * sig[k];
	sumy = sumx = sum;
	for (l = 0; l < NPOLE; ++l)  {
		sum +=	sig[NPOLE-(l+1)] * sig[NPOLE-(l+1)] -
			sig[FRAME-(l+1)] * sig[FRAME-(l+1)];
		a[l][l] = sum;
		}
	if (!gauss (a, v, b)) return 0;
	for (i = 0; i < NPOLE; ++i) sumy = sumy - b[i] * v[i];
	*rms1 = sqrt(sumx/(double) (FRAME - k1 + 1) );
	*rms2 = sqrt(sumy/(double) (FRAME - k1 + 1) );
	*errn = (*rms2 * *rms2) / (*rms1 * *rms1);
	for (bp = b; bp - b < NPOLE; ++bp, ++c) *c = *bp;
	return 1;
}

static int gauss(aold, bold, b)
double aold[POLEMAX][POLEMAX], *bold, *b;
{
	double amax, dum, pivot;
	double c[POLEMAX], a[POLEMAX][POLEMAX];
	int i, j, k, l, istar, ii, lp;

	/* aold and bold untouched by this subroutine */
	for (i=0; i < NPOLE ;++i)  {
		c[i] = bold[i];
		for (j = 0; j < NPOLE; ++j) a[i][j] = aold[i][j];
		}
	/* eliminate i-th unknown */
	for (i = 0; i < NPOLE - 1; ++i)  {        /* find largest pivot */
		amax = 0.0;
		for (ii = i; ii < NPOLE; ++ii)  {
			if (fabs (a[ii][i]) >= amax)  {
				istar = ii;
				amax = fabs (a[ii][i]);
				}
			}
		if (amax < 1e-20) {
			mv_alert("error: frames with zero amplitude not allowed.");
			return 0;
			}

		for (j = 0; j < NPOLE; ++j)  {    /* switch rows */
			dum = a[istar][j];
			a[istar][j] = a[i][j];
			a[i][j] = dum;
			}
		dum = c[istar];
		c[istar] = c[i];
		c[i] = dum;
		/* pivot */
		for (j = i + 1; j < NPOLE; ++j)  {
			pivot = a[j][i] / a[i][i];
			c[j] = c[j] - pivot * c[i];
			for (k = 0; k < NPOLE; ++k)
				a[j][k] = a[j][k] - pivot * a[i][k];
			}
		}       /* return if last pivot is too small */
	if (fabs (a[NPOLE-1][NPOLE-1]) < 1e-20) {
		mv_alert("error: frames with zero amplitude not allowed.");
		return 0;
		}

	*(b+NPOLE-1) = c[NPOLE-1] / a[NPOLE-1][NPOLE-1];
	/* back substitute */
	for (k = 0; k < NPOLE - 1; ++k)  {
		l = NPOLE-1 -(k+1);
		*(b+l) = c[l];
		lp = l + 1;
		for (j = lp; j < NPOLE; ++j) *(b+l) += -a[l][j] * *(b+j);
		*(b+l) /= a[l][l];
		}
	return(1);
}

#define PFRAMSIZ  2                    /* words per frame (pitch&rms */
#define PCHSIZE   16                   /* record size in float words */
#define BPPFRAME  FLOAT*PFRAMSIZ       /* bytes per frame */
#define BPPREC    BPPFRAME*PCHSIZE     /* bytes per record */
#define FPPREC    PCHSIZE/PFRAMSIZ     /* frames per record */

static float PFPS;                     /* pitch frames per second */
static float sampspf;
static int pitches;
static int lpclast;
static float getfreq();
static float ptframerate;

int lpmerge (input, output)
	char  *input, *output;
{
        int anal,j;
	int npoles,lpcframe,pchframe,pchlast,nbpch;
	long nskiplpc,nskippch,nblpc;
	float pch[2],val,sr,time;
	LPHEADER *lph;
	char lpbuf[LPBUFSIZ];
	char c;

	PFPS = ptframerate;	/* this is default SR/JSLIDE */
	lpcframe = 0;
	lpclast = -1;		/* to EOF */

	if((anal = open (output,2)) < 0) {
		mv_error(errno, "Can't open lpc analysis file.");
		return 0;
	}
	if((pitches = open (input,0)) < 0) {
		mv_error(errno, "Can't open pitch analysis file.");
		return 0;
	}
	if ((j = read (anal, lpbuf, LPBUFSIZ)) <= 0)	{
		mv_error(errno, "Can't read header on lpc analysis file.");
		return 0;
	}
	lph = (LPHEADER *) lpbuf;
	if(lph->lpmagic != LP_MAGIC) {
		mv_alert("LPC datafile must have header to be concatted.");
		close(anal); close(pitches);
		return 0;
	}
	npoles = lph->npoles;
	sr = lph->srate;
	sampspf = sr / lph->framrate;

	nblpc = (npoles+4)*FLOAT - FLOAT;/*to beginning of next pchloc*/
	nskiplpc = (long)(lpcframe)*(long)(nblpc+FLOAT) + (3*FLOAT);  
				 /* pch is 4th data value in fr*/
	if((lseek (anal,nskiplpc + lph->headersize,0)) < 0)  { /* add */
		mv_error(errno, "Bad lseek on analysis file."); /* headersize */
		return 0;
	}
	mesg->On(mesg, "Merging pitch data into LPC file...");
	sleep(1);
	for(j = lpcframe; ;j++) {
		time = (float)j*sampspf/sr;
		if ((j > lpclast && lpclast != -1) || (time > lph->duration))
			break;
		if ((val = getfreq (time)) < 0.0) return 0;
		if((write(anal,(char *)&val,FLOAT)) != FLOAT) {
			mv_error(errno, "Bad write on lpc analysis file.");
			mesg->Off(mesg);
			return 0;
		}
		lseek (anal,nblpc,1);
	}
	mesg->Off(mesg);
	return 1;
}

static float getfreq(time) 
float time;
{
	int i,j,frame;
	float fframe,fraction,cps,error;
	static int oldframe = 0;
	static int endframe = 0;
	static float parray[PCHSIZE*2];

	fframe = time * PFPS;
	frame = (int)fframe;
	fraction = fframe - (float)frame;
	if(!((frame >= oldframe) && (frame < endframe))) {
		if(lseek(pitches,((long)frame*(long)BPPFRAME),0) < 0) {
			mv_error(errno, "Bad lseek on pitch file.");
			return -1.0;
		} 
		read (pitches,(char *)parray,BPPREC);
		oldframe = frame;
		endframe = oldframe + FPPREC - 1;
	}
	cps = (1.-fraction) * *(parray+(frame-oldframe)*PFRAMSIZ) +
	    fraction  * *((parray+(frame-oldframe)*PFRAMSIZ)+2);
	error = (1.-fraction) * *(parray+(frame-oldframe)*PFRAMSIZ+1) +
	    fraction  * *((parray+(frame-oldframe)*PFRAMSIZ)+3);
	return(cps);
}

/*  PTRACK - Standalone pitch analysis program.  Revised dll@ems.
 *     Reads a fixed-point monaural sound file (with or without header).
 *     
 *     Program accepts command-line data (see man entry),
 *     or with -q will querie the user for the following:
 *
 *    -Name of soundfile.
 *    -Sampling rate (if no header).
 *    -Name of file to contain analysis data--can be an old or null file.
 *    -Framesize.  The size of the frame to be analyzed, in samples.
 *     Maximum is 350, which has been good working number.  Frame should
 *     be able to contain at least one expected pitch period of signal.
 *    -JSLIDE.  The number of new samples to add per frame.  This number 
 *     should normally be less than half of LSLICE.  125 is good for 15k
 *     and 250 is good for 30k sampling rate.
 *     This number reflects the real time period of the frame, i.e. how
 *     far you creep up for each frame in the signal.
 *    -pchlow and pchigh-- expected lower and upper bounds of analysis.
 *     the better you guess at this, the better the quantization will
 *     be in the analysis.
 *     At the present time the highest fundamental analyzable is about SR/20
 *     cps, due to the settings of the lowpass filter.  This can be fixed
 *     by using interpolation, as in the fortran version of this program, 
 *     or, more sensibly, different filters.
 *    -inskip, amount of time to skip on input file before beginning analysis.
 *     The program is now written to write data sequentially beginning at the
 *     start of the data file, but this can be changed by simple adding an
 *     lseek.  
 *     (inskip should match the -s value in soundout).
 *    -dur,   amount of time to analyze. This should equal, but not
 *     exceed the -d value from soundout. 
 *  The following is good for a 15k signal, for example:
 *
 *	inname 15000 outname
 *	350 125
 *	200 300
 *	0 .5
 *    
 */
 
extern double lowpass();
extern void init_lowpass();

int LSLICE, JSLIDE, JMAX, MM;
float SR, NYQ; 

static float gtphi[50][5][MAXMM],gtpsi[50][6][MAXMM];
static float gtgamph[50][5],gtgamps[50][6];

int ptrack (sfb, ptdata)
cbuf_struct *sfb;
PTDATA *ptdata;
{
	double sig[PFRAMAX], out[4];
	int MIDPOINT;
	float freq[50];
	float getpch(),getrms(), ptable();
	float pchlow,pchigh,input[4];
	int i,n=0,jj,anal,framesize;
	long nsamps;
	float data[2];
	char *name, ptrpt[64];
	char *sigtemp;
	double rmsm,freqm,rmsx,freqx;
	int counter;
	int Hflag = 0;
	char c;
	int rflag = 0;

	name = ptdata->ptfname;
	LSLICE = ptdata->framesize;		/* 350 is max! */
	JSLIDE = ptdata->offset;
	pchlow = ptdata->lowest;
	pchigh = ptdata->highest;
	SR = sfb->srate;

	if(LSLICE > 350) {
		mv_alert("Maximum framesize is 350 samples.");
		return 0;
	}	
	if(LSLICE > sfb->nsamps) {
		mv_alert("Framesize is too large for selected region.");
		return 0;
	}	
	if(JSLIDE >= LSLICE) {
		mv_alert("Offset must be smaller than framesize.");
		return 0;
	}	
	if(JSLIDE == 0) {
		mv_alert("Offset must be a nonzero value.");
		return 0;
	}	
	if ((anal = creat (name,0644)) < 0) { /* create if not exists */
		mv_error(errno, "Unable to open output data file.");
		return 0;
	}
	/* set these globals for pitch routines */
	NYQ = SR/2.;
	JMAX = LSLICE/10;
	MM = ((LSLICE/10+1)/2);
	MIDPOINT = LSLICE - JSLIDE;
	ptframerate = SR/JSLIDE;	/* global for lpmerge */
	ptable(pchlow,pchigh,gtphi,gtpsi,gtgamph,gtgamps,freq,n); 
	init_lowpass ();
	for(i=0; i<LSLICE; i++) {
		GETSAMP(sfb, out);
		sig[i] = lowpass(*out);	/* just scan first channel */
	}
	nsamps = sfb->nsamps;
	counter = 1;
	while (nsamps) {
		data[0] = getpch(sig,gtphi,gtpsi,gtgamph,gtgamps,freq,n);
		data[1] = getrms(sig);
		write (anal,(char *)data,8);
		sprintf (ptrpt, "Running pitchtrack:  frame %5d, pitch = %8.2f",
			counter, data[0]);
		if(!(counter++ % 5)) mesg->On(mesg, ptrpt);
		for (i = 0; i < MIDPOINT; i++) sig[i] = sig[i+JSLIDE];

		if (sfb->inptr+JSLIDE > sfb->nsamps) break;		
		for(jj=MIDPOINT; jj<LSLICE; jj++) {
			GETSAMP(sfb, out);
			sig[jj] = lowpass(*out);
		}
		nsamps -= JSLIDE; 
	}
	close (anal);
	mesg->Off(mesg);
	return 1;
}

int
lpdump (ifile, ofile)
	char *ifile, *ofile;
{
	int fd, nbytes, l, framsize, i, frmno = 0;
	char lpbuf[LPBUFSIZ], dumprpt[32], msg[128];
	float frmbuf[MAXFRAME];
	LPHEADER *lph;
	FILE *outfile;
	
	if((outfile = fopen(ofile, "w+")) == NULL) {
		mv_error(errno, "lpdump: can't open temp output file.");
		return 0;
	}
	if ((fd = open (ifile, O_RDONLY)) < 0)	{
		mv_error(errno, "lpdump: can't open datafile");
		return 0;
	}
	if ((nbytes = read (fd, lpbuf, LPBUFSIZ)) <= 0)	{
		mv_error(errno, "lpdump: read error on datafile");
		close (fd);
		return 0;
	}
	lph = (LPHEADER *) lpbuf;
	if (lph->lpmagic != LP_MAGIC)	{
		sprintf(msg, "lpdump: %s is not an lpfile", ifile);
		mv_alert(msg);
		close (fd);
		return 0;
		}
	if ((l = lseek (fd, lph->headersize, 0)) < 0)	{
		sprintf(msg, "lpdump: bad lseek on %s", ifile);
		mv_error(errno, msg);
		close (fd);
		return 0;
		}
	framsize = lph->nvals * FLOAT;
	while (++frmno)	{
		if ((read (fd, (char *) frmbuf, framsize)) < framsize) {
			mesg->On(mesg, "End of lp file.");
			fflush(outfile);
			sleep(1);
			mesg->Off(mesg);
			fclose(outfile);
			close (fd);
			return 1;
			}
		
		fprintf(outfile, "\n\nFrame #%d\n",frmno);
		sprintf (dumprpt, "dumping frame %d", frmno);
		if(!(frmno % 5)) mesg->On(mesg, dumprpt);
		fprintf(outfile, "rms1: %f	rms2: %f	error: %f	cps: %f\n",frmbuf[0], frmbuf[1], frmbuf[2], frmbuf[3]);
		fprintf(outfile, "Coeffs:");
		for (i = 0; i < lph->npoles; i++) {
			if ((i % 6) == 0) fprintf (outfile, "\n  ");
			fprintf (outfile, "%10.6f ", frmbuf[4 + i]);
			}
		fprintf (outfile, "\n");
	}
	fflush(outfile);
	fclose(outfile);
	return 1;
}
