/****************************************************************************** 
 *
 *  mixview - X Window System based soundfile editor and processor
 *
 *  Copyright 1989 by Douglas Scott
 *
 *  Author:     Douglas Scott 
 *  Date:       May 1, 1989
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The author makes no representations about
 *  the suitability of this software for any purpose.  
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/
/******************************************************************************

FFT / Inverse FFT routine

Based on William D.Stanley, "Digital Signal Processing", Second Edition,
         publisher and copyright not given
This is an implementation of Figure 9-14 on page 277, except that the
factor of 1/N appears in the FFT rather than the inverse FFT (see page 268).

The parameter N is the number of time samples and the number of frequency
samples and is rounded down to the nearest power of two.
If N is negative, the inverse FFT is applied with abs(N).

The arrays xreal[] and ximag[] serve for both input and output, and
are indexed from 0 to N-1

******************************************************************************/
#include "main.h"
#include "mesg.h"
#include <math.h>
#include "fft.h"

int
fftran(b, data)
cbuf_struct *b;
FFTDATA *data;
{
	FFTHEADER fh;
	double xreal[FMAX], ximag[FMAX], input[SF_MAXCHAN], xrealsum;
	double mag, phase;
	int anal, nsamps, decibel=False, status=0;
	register int i, chan;
	nsamps = data->npoints * 2;
	decibel = data->mode;	/* if true, store as decibel values */

	if(b->nsamps < nsamps) {
		mv_alert("Selected region too small for given frame size.");
		return 0;
	}
        if ((anal = open(data->fftfname, O_CREAT|O_TRUNC|O_RDWR, 0644)) < 0) {
		mv_error(errno, "Unable to open output data file.");
		return 0;
	}
	if(data->use_header) {
		fh.fftmagic = FFT_MAGIC;
		fh.headersize = sizeof(fh) - sizeof(fh.text);
		fh.npoints = data->npoints;
		fh.nvals = 2;
		fh.srate = data->srate;
		fh.mode = data->mode;
		if(write(anal, (char *) &fh, fh.headersize) != fh.headersize) {
			mv_error(errno, "Unable to write data file header.");
			return 0;
		}
	}
	mesg->On(mesg, "Running fft...");
	for(i=0; i<nsamps && i < FMAX; i++) {	/* load current frame */
		xrealsum = 0.0;
		GETSAMP(b, input);
		for(chan=0; chan < b->nchans; chan++) xrealsum += input[chan];
		xreal[i] = xrealsum;
		ximag[i] = 0.0;
	}
	fft(nsamps, xreal, ximag);	/* do the fft */

	/* calculate magnitude and convert to dB */

	nsamps /= 2;	/* only convert and write first half */
	for(i=0; i<nsamps; i++) {
		mag = sqrt(xreal[i]*xreal[i] + ximag[i]*ximag[i]);
		phase = atan2(ximag[i], xreal[i]) * 180.0 / PI;
		if(decibel) mag = 20. * log10(mag / b->peakamp);
		if((status=write(anal, (char *) &mag, sizeof(double))) < 0)
			break;
		if((status=write(anal, (char *) &phase, sizeof(double))) < 0)
			break;
	}
	if(status < 0)
		mv_error(status, "fft: Bad write on output file.");
	mesg->Off(mesg);
	return 1;
}

fft(N,xreal,ximag)
int N;
double *xreal,*ximag;
{
	int l,n;

	l = log( (double)(abs(N)) ) / log(2.0) + 0.001;
	n = pow( 2.0,(double)(l) ) + 0.5;

	fftalgorithm( (int)(N/abs(N)),l,n,xreal,ximag);

	descramble( ((N>=0)?n:1),l,n,xreal,ximag);
}


fftalgorithm(wexpsign,l,n,xr,xi)
int wexpsign,l,n;
double *xr,*xi;
{
	int ia,ib,ic,id,ie;
	int i,k,m;
	double wexp,zr,zi,ar,ai,br,bi;

/* printf("fft: wes,l,n = %d,%d,%d\n",wexpsign,l,n);
*/	ia = n/2;
	ib = 1;

	for (i = 0; i < l; ++i)
	{
/* printf("level %d\n",i);
*/		ic = 0;
		id = ia;

		for (k = 0; k < ib; ++k)
		{
			ie = scramble( l,n,(int)(ic/ia) );
/* printf("ic = %d, ia = %d\n",ic,ia);
printf("W power = %d\n",ie);
*/			wexp = -wexpsign*2.0*PI*(double)(ie)/(double)(n);
			zr = cos(wexp);
			zi = sin(wexp);

			for (m = ic; m < id; ++m)
			{
				ar = xr[m];
				ai = xi[m];

				br = xr[m+ia] * zr - xi[m+ia] * zi;
				bi = xr[m+ia] * zi + xi[m+ia] * zr;

/* printf("x(%d) = x(%d) + W^(%d) * x(%d)\n",m,m,ie,m+ia);
*/				xr[m] = ar + br;
				xi[m] = ai + bi;

/* printf("x(%d) = x(%d) - W^(%d) * x(%d)\n",m+ia,m,ie,m+ia);
*/				xr[m+ia] = ar - br;
				xi[m+ia] = ai - bi;
			}

			ic += 2*ia;
			id += 2*ia;
		}

		ia /= 2;
		ib *= 2;
	}

}


descramble(scalediv,l,n,xreal,ximag)

	int scalediv,l,n;
	double *xreal,*ximag;

{
	int m1,m2;
	double swap;

	for (m1 = 0; m1 < n; ++m1)
	{
		m2 = scramble(l,n,m1);
		if (m1 == m2)
		{
			xreal[m1] /= scalediv;
			ximag[m1] /= scalediv;
		}
		else if (m1 > m2)
		{

			swap = xreal[m1]/scalediv;
			xreal[m1] = xreal[m2]/scalediv;
			xreal[m2] = swap;

			swap = ximag[m1]/scalediv;
			ximag[m1] = ximag[m2]/scalediv;
			ximag[m2] = swap;
		}
	}
}


int scramble(l,n,min)

	int l,n,min;

{
	int mout,mplus,mminus,i;

	mout = 0;
	mplus = 1;
	mminus = n/2;

	for (i = 0; i < l; ++i)
	{
		if (min >= mminus)
		{	mout += mplus;
			min -= mminus;
		}

		mplus *= 2;
		mminus /= 2;
	}

	return(mout);
}
