/****************************************************************************** 
 *
 *  mixview - X Window System based soundfile editor and processor
 *
 *  Copyright 1989 by Douglas Scott
 *
 *  Author:     Douglas Scott 
 *  Date:       May 1, 1989
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The author makes no representations about
 *  the suitability of this software for any purpose.  
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/
/* mvcurves.c -- the routines for the line & curve entry panels used in mixview
	Douglas Scott  3/89
*/

#include "main.h"
#include "windows.h"
#include "dialog.h"
#include "menus.h"
#include "nobug.h"

#define NPTS	512
#define MAXPT	511
#define XMIN	30
#define XMAX	541
#define YMIN	30
#define YMAX	429
#define S_HEIGHT  460
#define S_WIDTH   572

extern Display *dpy;
extern Window mixwin, lwin, lcwin;
extern Cursor arrow_curs, wait_curs, draw_curs, cross_curs;
extern sf_struct *v;
extern char *progname;
extern unsigned long myfg, mybg;
extern double array[SIZE];
extern int lineset;

Window sc_win;
Pixmap scrn_pm;
Mv_panel *c_panel;
Canvas *screen;
Menu *curve_menu;
GC s_gc, s_dgc;
Time evt_time;

int ret_code;
static FLAG locflag = 0;
static int ybase;
static double vscale, maxval, minval;
static double list[NPTS*2+2];	/* array of {loc0,val0, loc1,val1, ...} */
static XPoint esegs[30], dsegs[3], points[S_WIDTH], old, new;

static
char *menu_cmds[]= { "set range", "blank", "blank", 0
};

setup_screen()
{
	XGCValues gcv;

	if ((screen= canvas_create(progname, dpy, lcwin,
			LXC_WIDTH, S_WIDTH,
			LXC_HEIGHT, S_HEIGHT,
			LXC_NULL)) == (Canvas *) NULL)
		exit(-1);

	sc_win= *((Window *) canvas_get(screen, LXC_WINDOW));
	scrn_pm= *((Pixmap *) canvas_get(screen, LXC_PIXMAP));

	gcv.function= GXxor; 
	gcv.foreground= myfg^mybg;	/* this is to permit compatibility */
	gcv.background= mybg;		/* with color systems w/ reversed  */
	gcv.plane_mask = myfg^mybg;	/* pixel values                    */
	s_gc= XCreateGC(dpy, sc_win,
			GCFunction|GCForeground|GCBackground|GCPlaneMask,
			&gcv);
	XSetLineAttributes(dpy, s_gc, 0, LineSolid, CapNotLast, JoinMiter);
	s_dgc= XCreateGC(dpy, sc_win,
			GCFunction|GCForeground|GCBackground|GCPlaneMask,
			&gcv);
	XSetLineAttributes(dpy, s_dgc, 0, LineOnOffDash, CapNotLast, JoinMiter);
	XSelectInput(dpy, sc_win,
	   ExposureMask | ButtonPressMask | ButtonMotionMask|ButtonReleaseMask);
}

extern Mv_panel *new_panel();
setup_curve_window()
{
        void  reset_curve(), set_range(), cancel_curve(), confirm_curve();

	/* Create the panel and panelitems for this window */
        c_panel = new_panel(lwin, LFRAME_WIDTH, 100, 0, 0, 0);
        c_panel->addLabel(c_panel, 20, 10, 
		"Draw continuous and/or segmented curve for values:",
		"label 1");
        c_panel->addLabel(c_panel, 20, 32,
		"Left button: segmented; Center button: continuous.",
		"label 2");
        c_panel->addButton(c_panel, 20, 60, "confirm", confirm_curve);
        c_panel->addButton(c_panel, 125, 60, "cancel", cancel_curve);
        c_panel->addButton(c_panel, 220, 60, " reset ", reset_curve);

	/* create and initialize canvas for screen */

	setup_screen();

	/* create menu and menuitems for screen */

        curve_menu = new_menu(sc_win, CNULL);
        add_menuitem(curve_menu, menu_cmds[0], set_range, 0);
        add_menuitem(curve_menu, menu_cmds[1], PROC_NULL, 0);
        add_menuitem(curve_menu, menu_cmds[2], PROC_NULL, 0);
}

FLAG again;
static unsigned e_mask = ButtonPressMask | ButtonMotionMask |
	ButtonReleaseMask /* | KeyPressMask */ ;

get_curve()	/* displays window/panel/canvas for entering data curve */
{
	XSetWindowAttributes xswa;
	XEvent evt;
	unsigned wait_mask = 0;
	void reset_curve();
	int x, y;
	static int first_time = 1;
	get_win_coords(mixwin, &x, &y); /* get position relative to root */
	move_window(lwin, x+150, y+50); /* locate dialog box on root */
	xswa.override_redirect= TRUE;
	XChangeWindowAttributes(dpy, lwin, CWOverrideRedirect, &xswa);
	XMapRaised(dpy, lwin);
	XMapSubwindows(dpy, lwin);
	XGrabPointer(dpy, lwin, True, e_mask,
			GrabModeAsync, GrabModeAsync,
			lwin, arrow_curs, CurrentTime);


	/* reset, draw scale and base line, etc. */

	if(first_time) {
		int n_args;
		ybase = YMAX;	/* the 0 value vertical coordinate */
		maxval = 1.0;
		minval = 0.0;
		DBPRT(ybase);
		vscale = (maxval-minval)/(double) (YMAX-YMIN);
		DBPRT(vscale);
		reset_curve((Panel *) NULL, (Panel_item *) NULL); 
		curve_to_vals(vscale, ybase, points, list, &n_args);
		first_time = 0;
	}
	else {
		draw_scrn();
	}

	again = 1;

	while(again) {
		XNextEvent(dpy, &evt);
		if (lxt_event(&evt)) 
			continue;
		switch (evt.type) {    /* process pointer & kbd. events */
		case MotionNotify:
			cpm_select((XPointerMovedEvent *) &evt);
			break;
		case ButtonPress:
			cbp_select((XButtonPressedEvent *) &evt);
			break;
		case ButtonRelease:
			cbr_select((XButtonReleasedEvent *) &evt);
			break;
		}
	}
									
	if(ret_code == YES) {
		int n_args;
		XChangeActivePointerGrab(dpy, wait_mask, wait_curs, evt_time);

		/* create interpolated table */
		curve_to_vals(vscale, ybase, points, list, &n_args);
		vals_to_array(list, array, n_args);
	}
	XUngrabPointer(dpy, evt_time);
	XUnmapSubwindows(dpy, lwin);
	XUnmapWindow(dpy, lwin);
	return(ret_code);
}

move_window(win, xpos, ypos)
Window win;
int xpos, ypos;
{
	int max_x = DisplayWidth(dpy, DefaultScreen(dpy));
	int max_y = DisplayHeight(dpy, DefaultScreen(dpy));
	if(xpos+LFRAME_WIDTH > max_x) xpos = max_x - LFRAME_WIDTH - 5;
	if(ypos+LFRAME_HEIGHT > max_y) ypos = max_y - LFRAME_HEIGHT - 5;
	XMoveWindow(dpy, win, xpos, ypos);
	XSync(dpy, 0);	/* try syncing to avoid window flitting */
}

void
confirm_curve(p, pi)
Panel *p;
Panel_item *pi;
{
	again = 0;
	evt_time = *((Time *) panelitem_get(p, pi, LXPI_TIMESTAMP));
	ret_code = YES;
	XFlush(dpy);
}

void
cancel_curve(p, pi)
Panel *p;
Panel_item *pi;
{
	again = 0;
	evt_time = *((Time *) panelitem_get(p, pi, LXPI_TIMESTAMP));
	ret_code = CANCEL;
	lineset = 0;
	XFlush(dpy);
}

void
reset_curve(p, pi)	/* restores flat curve at 0.0 */
Panel *p;
Panel_item *pi;
{
	register int i;
	DBCOM(entering reset_curve);

	/* reset data points */

	reset_list(list);
	for(i=0; i < S_WIDTH; i++) points[i].x = -1;

	dsegs[0].x = points[XMIN].x = XMIN;	/* left endpoint */
	dsegs[0].y = points[XMIN].y = ybase;
	dsegs[2].x = points[XMAX].x = XMAX;	/* right endpoint */
	dsegs[2].y = points[XMAX].y = ybase;

	draw_scrn();	/* redisplay the curve */
	DBCOM(leaving reset_curve);
}

void
set_range(m, mi)
Menu *m;
Menu_item *mi;
{
	int n_args;
	DBCOM(entering set_range);
	if(dialog->call(dialog, D_QRANGE, "") < 0) return;
	curve_to_vals(vscale, ybase, points, list, &n_args); /* store vals */
	minval = dialog->getValue(dialog, 0);
	maxval = dialog->getValue(dialog, 1);
	DBPRT(maxval);
	DBPRT(minval);
	vscale = (maxval-minval)/(double) (YMAX-YMIN);
	ybase = (YMAX - YMIN) * (maxval/(maxval-minval)) + YMIN;
	vals_to_curve(vscale, ybase, points, list); /* rescale coords of vals */
	draw_scrn();
	DBCOM(leaving set_range);
}

cbp_select(evt)
XButtonPressedEvent *evt;
{
	switch (evt->button){
	case Button1:
		XChangeActivePointerGrab(dpy, e_mask, cross_curs, evt->time);
		new_point(evt->x,evt->y);
		break;
	case Button2:
		XChangeActivePointerGrab(dpy, e_mask, draw_curs, evt->time);
		new_line(evt->x,evt->y);
		break;
	case Button3:
		menu_show(curve_menu, evt);
		break;
	}
}

cpm_select(evt)
XPointerMovedEvent *evt;
{
	if(evt->state & Button1Mask) 
		move_point(evt->x,evt->y);
	else if(evt->state & Button2Mask) 
		draw_line(evt->x,evt->y);
}

cbr_select(evt)
XButtonReleasedEvent *evt;
{
	switch (evt->button){
	case Button1:
		XChangeActivePointerGrab(dpy, e_mask, arrow_curs, evt->time);
		set_point(evt->x,evt->y);
		break;
	case Button2:
		XChangeActivePointerGrab(dpy, e_mask, arrow_curs, evt->time);
		break;
	}
}

/**********************************************************************/

static int npoints;
FLAG lbflag, ubflag;

draw_scrn()
{
	register int i, j = 0, npts;
	XPoint graph_pts[S_WIDTH];
	DBCOM(entering draw_scrn);

	canvas_clear(screen);
	locflag = 0;

	/* find all set points for current screen and plot the curve */

	for(i = 0; i < S_WIDTH; i++) {
		if(points[i].x >= 0.0)
			graph_pts[j++] = points[i];
	}
	npts = j;
	DBPRT(npts);
	XDrawLines(dpy, sc_win, s_gc, graph_pts, npts, CoordModeOrigin);
	XDrawLines(dpy, scrn_pm, s_gc, graph_pts, npts, CoordModeOrigin);

	/* draw scale on screen */

	print_scale(ybase);
	DBCOM(leaving draw_scrn);
}

print_scale(base)
int base;
{
	XDrawLine(dpy, sc_win,s_gc, XMIN-1,YMIN-1,XMIN-1,YMAX);
	XDrawLine(dpy, scrn_pm,s_gc, XMIN-1,YMIN-1,XMIN-1,YMAX);

	XDrawLine(dpy, sc_win, s_dgc, XMIN, base, XMAX-1, base);
	XDrawLine(dpy, scrn_pm, s_dgc, XMIN, base, XMAX-1, base);
}

new_point(x,y)	/* button 1 clicked */
int x,y;
{
	register int resetpt = 0;

	DBCOM(entering new_point);

	x = (x < XMIN) ? XMIN : (x > XMAX) ? XMAX : x;
	y = (y <= YMIN) ? YMIN : (y >= YMAX) ? YMAX : y;
	npoints = 3;
	new.x =  x;
	new.y = y;

	/* if resetting (erasing) an existing point */

	if(get_endpoints(x, &esegs[0], &esegs[2])) { 
		if(esegs[0].x == x) {		/* reset lower edge */
			dsegs[0] = new;
			dsegs[2] = dsegs[1] = esegs[2];
			ubflag = 0;
			lbflag = 1;
			npoints = 2;
		}
		else if(esegs[2].x == x) {	/* reset upper edge */
			dsegs[0] = esegs[0];
			dsegs[2] = dsegs[1] = new;
			ubflag = 1;
			lbflag = 0;
			npoints = 2;
		}
		else {
			esegs[1].x = points[x].x; /* middle point to erase */
			esegs[1].y = points[x].y;
			resetpt = 1;
			dsegs[0] = esegs[0];	  /* lower point to draw  */
			dsegs[1] = new;		  /* middle point to draw */
			dsegs[2] = esegs[2];	  /* upper point to draw  */
			lbflag = ubflag = 0;	  /* reset boundry flags  */
		}
		/* remove old inflection pt. from list */

		points[x].x = -1; 
		DBCOM(resetting existing point);
	}
	else {			/* if setting new point */
		dsegs[0] = esegs[0];
		dsegs[1] = new;
		dsegs[2] = esegs[2];
		lbflag = ubflag = 0;
	}

	/* erase old segment(s) */

	if(resetpt) {			/* if shifting (erasing) old point */
		DBCOM(erase two old segments AND infl. pt.);
		DBPRT(npoints);
		XDrawLines(dpy,sc_win,s_gc,esegs,npoints,CoordModeOrigin);
		XDrawLines(dpy,scrn_pm,s_gc,esegs,npoints,CoordModeOrigin);
	}
	else {		/* if setting new inflection pt. */
		DBCOM(no resetpt - just erase the old segment);
		XDrawLine(dpy,sc_win,s_gc,
			esegs[0].x,esegs[0].y,esegs[2].x,esegs[2].y);
		XDrawLine(dpy,scrn_pm,s_gc,
			esegs[0].x,esegs[0].y,esegs[2].x,esegs[2].y);
	}

	/* draw new segments */

	DBCOM(draw new segments);
	DBPRT(npoints);
	XDrawLines(dpy,sc_win,s_gc,dsegs,npoints,CoordModeOrigin);
	XDrawLines(dpy,scrn_pm,s_gc,dsegs,npoints,CoordModeOrigin);
	print_loc(x, y);
	old = new;
	DBCOM(leaving new_point);
}

move_point(x,y)	/* button 1 dragged */
int x, y;
{
	int xloc, lpoint = esegs[0].x, rpoint = esegs[2].x;
	DBCOM(entering move_point);
	npoints = 3;

	y = (y <= YMIN) ? YMIN : (y >= YMAX) ? YMAX : y;

	/* always bounded by previous hi and lo inflection pts. */

	if(lbflag) {			/* moving lower boundry */
		xloc = XMIN;
		npoints = 2;
		esegs[0] = dsegs[0];
		esegs[1] = dsegs[1];
		dsegs[0].x = xloc;
		dsegs[0].y = y;
		old = dsegs[0];
	}
	else if(ubflag) {		/* moving upper boundry */
		xloc = XMAX;
		npoints = 2;
		esegs[0] = dsegs[0];
		esegs[1] = dsegs[1];
		dsegs[1].x = xloc;
		dsegs[1].y = y;
		old = dsegs[1];
	}
	else {				/* moving intermediate points */
		xloc = (x <= lpoint) ? lpoint+1 : (x >= rpoint) ? rpoint-1 : x;
		esegs[0] = dsegs[0];
		esegs[1] = dsegs[1];
		esegs[2] = dsegs[2];
		dsegs[1].x = xloc;
		dsegs[1].y = y;
		old = dsegs[1];
	}
	/* erase old segments */

	XDrawLines(dpy, sc_win, s_gc, esegs, npoints, CoordModeOrigin);
	XDrawLines(dpy, scrn_pm, s_gc, esegs, npoints, CoordModeOrigin);

	/* draw new segment(s) */

	XDrawLines(dpy, sc_win, s_gc, dsegs, npoints, CoordModeOrigin);
	XDrawLines(dpy, scrn_pm, s_gc, dsegs, npoints, CoordModeOrigin);
	print_loc(xloc, y);
	DBCOM(leaving move_point);
}

set_point(x,y)	/* button 1 released */
int x, y;
{
	int xloc, lpoint = esegs[0].x, rpoint = esegs[2].x;
	DBCOM(entering set_point);
	if(x >= XMAX) xloc = XMAX;
	else if(x <= XMIN) xloc = XMIN;
	else xloc = (x < lpoint) ? lpoint : (x > rpoint) ? rpoint : x;
	y = (y <= YMIN) ? YMIN : (y >= YMAX) ? YMAX : y;
	points[xloc].x = xloc;
	points[xloc].y = y;
	DBCOM(leaving set_point);
}

new_line(x,y)	/* button 2 depressed */
int x,y;
{
	XPoint lower, upper;
	int nepts = 2, ndpts = 3;

	DBCOM(entering new_line);

	x = (x <= XMIN) ? XMIN+1 : (x >= XMAX) ? XMAX-1 : x;
	y = (y <= YMIN) ? YMIN : (y >= YMAX) ? YMAX : y;
	new.x = x;
	new.y = y;

	/* if resetting an existing point */

	if(get_endpoints(x, &lower, &upper)) {
		esegs[1] = points[x];
		esegs[2] = lower;
		nepts = 3;
		points[x].x = -1; /* remove point from list -- to be reset */
		DBCOM(resetting existing point);
	}
	else esegs[1] = lower;
	esegs[0] = upper; 	/* segment to erase */

	DBPRT(new.x);
	DBPRT(old.x);
	DBPRT(lower.x);
	DBPRT(upper.x);

	dsegs[0] = upper;	/* segment to draw: */
	dsegs[1] = new;
	dsegs[2] = lower;

	/* erase old segment(s) */

	XDrawLines(dpy, sc_win, s_gc, esegs, nepts, CoordModeOrigin);
	XDrawLines(dpy, scrn_pm, s_gc, esegs, nepts, CoordModeOrigin);

	/* draw new segments */

	XDrawLines(dpy, sc_win, s_gc, dsegs, ndpts, CoordModeOrigin);
	XDrawLines(dpy, scrn_pm, s_gc, dsegs, ndpts, CoordModeOrigin);
	print_loc(x, y);
	points[x] = new;
	old = new;
	DBCOM(leaving new_line);
}

draw_line(x,y)	/* button 2 dragged */
int x, y;
{
	XPoint lower, upper;
	register int i, nepts, ndpts = 3;
	DBCOM(entering draw_line);

	x = (x <= XMIN) ? XMIN+1 : (x >= XMAX) ? XMAX-1 : x;
	y = (y <= YMIN) ? YMIN : (y >= YMAX) ? YMAX : y;
	new.x = x;
	new.y = y;

	/* set the old segment endpoints for erase */

	get_endpoints(x, &lower, &upper); 
	nepts = 0;

	if(x > old.x) { /* if moving to right */
		esegs[nepts++] = upper; /* upper endpt. */
		i = upper.x;
		DBCOM(intermed pts);
		while(--i > old.x) { /* get all intermediate pts. to erase */
			if(points[i].x >= 0) {
				esegs[nepts++] = points[i];
				points[i].x = -1;
			}
		}
		esegs[nepts++] = old; /* lower endpt */

		/* set the new segment endpts */

		dsegs[0] = upper;
		dsegs[1] = new;
		dsegs[2] = old; /* = lower or old */
	}
	else if(x < old.x ) {
		nepts = 0;
		esegs[nepts++] = lower; /* lower endpt. */
		i = lower.x;
		DBCOM(intermed. pts:);
		while(++i < old.x) { /* get all intermediate pts. to erase */
			if(points[i].x >= 0) {
				esegs[nepts++] = points[i];
				points[i].x = -1;
			}
		}
		esegs[nepts++] = old; /* upper endpt */
		DBPRT(nepts);

		/* set the new segment endpts */

		dsegs[0] = lower;
		dsegs[1] = new;
		dsegs[2] = old; /* = upper or old */
	}	
	else { /* if x = x.old */
		esegs[0] = upper; 
		esegs[1] = old; 
		esegs[2] = lower;

		/* set the new segment endpts */

		dsegs[0] = upper;
		dsegs[1] = new;
		dsegs[2] = lower;
		points[old.x].x = -1;
		nepts = ndpts = 3;
	}
	points[x].x = x; /* add current point to point list */
	points[x].y = y;

	XDrawLines(dpy, sc_win, s_gc, esegs, nepts, CoordModeOrigin);
	XDrawLines(dpy, scrn_pm, s_gc, esegs, nepts, CoordModeOrigin);
	XDrawLines(dpy, sc_win, s_gc, dsegs, ndpts, CoordModeOrigin);
	XDrawLines(dpy, scrn_pm, s_gc, dsegs, ndpts, CoordModeOrigin);
	DBPRT(ndpts);
	print_loc(x, y);
	old = new;
	DBCOM(leaving draw_line);
}

/* find current upper and lower graph point bounds for new point */

get_endpoints(xstart, lpoint, upoint)
int xstart;
XPoint *lpoint, *upoint;
{
	register int i = xstart, val = 0;
	DBCOM(entering get_endpoints);
	DBPRT(points[xstart].x);
	if(points[xstart].x >= 0) val = 1;	/* if new pt. already set */
	DBCOM(upper);
	if(xstart == XMAX) {		/* resetting right edge value */
		upoint->x = points[XMAX].x;
		upoint->y = points[XMAX].y;
	}
	else {				/* find regular upper bound */
		while(i++ <= XMAX) {
			if(points[i].x >= 0) {
				upoint->x = points[i].x;
				upoint->y = points[i].y;
				break;
			}
		}
	}
	DBCOM(lower);
	i = xstart;
	if(xstart == XMIN) {		/* resetting left edge value */
		lpoint->x = points[XMIN].x;
		lpoint->y = points[XMIN].y;
	}
	else {				/* find regular lower bound */
		while(i-- >= XMIN) {
			if(points[i].x >= 0) {
				lpoint->x = points[i].x;
				lpoint->y = points[i].y;
				break;
			}
		}
	}
	DBCOM(leaving get_endpoints);
	return(val);
}

print_loc(x, y) /* prints current loc value at the bottom of screen */
int x, y;
{
        static int len, len2, tx, ty, ax;
	static int ly = YMAX+15, vx = 10, vy = -20, axmax = XMAX-30;
        static char string[6], string2[16];
	double val;

        /* draw over (erase) previous location and value labels if present */

        if (locflag) {
                XDrawString(dpy, sc_win, s_gc, tx, ly, string, len);
                XDrawString(dpy, scrn_pm, s_gc, tx, ly, string, len);
                XDrawString(dpy, sc_win, s_gc, ax+vx, ty+vy, string2, len2);
                XDrawString(dpy, scrn_pm, s_gc, ax+vx, ty+vy, string2, len2);
        }
        /* print new location and value  */

	val = (ybase - y) * vscale; 
        sprintf(string, "%d", x-XMIN);
	sprintf(string2, "%3.4f", val);
        len = strlen(string);
        len2 = strlen(string2);
	ax = (x < axmax) ? x : axmax;
        tx = x;
	ty = y;
        XDrawString(dpy, sc_win, s_gc, tx, ly, string, len);
        XDrawString(dpy, scrn_pm, s_gc, tx, ly, string, len);
	XDrawString(dpy, sc_win, s_gc, ax+vx, ty+vy, string2, len2);
	XDrawString(dpy, scrn_pm, s_gc, ax+vx, ty+vy, string2, len2);
	locflag = 1;
}

/* converts array of point structs into interpolated data values */

curve_to_vals(scl, base, pts, vlist, n_args)
double scl, *vlist;
int base, *n_args;
XPoint *pts;
{
	register int loc, i, j = 0;
	DBCOM(entering curve_to_vals);

	for(i = XMIN; i <= XMAX; i++) {
		loc = i - XMIN;			/* 0 to 511 */
		if(pts[i].x > 0) {
			vlist[j] = loc;		/* loc of point set */
			DBPRT(loc);
			vlist[j+1] = (base - pts[i].y) * scl; /* value */
			j += 2;
		}
	}
	*n_args = j;
	DBCOM(leaving curve_to_vals);
}

vals_to_array(vlist, arr, n_args)
double *vlist, *arr;
int n_args;
{
	DBCOM(entering vals_to_array);
	setline(vlist, n_args, SIZE, arr);	/* do interpolation */
	lineset = 1;
	DBCOM(leaving vals_to_array);
}

/* converts list of values into points for screen */

vals_to_curve(scl, base, pts, vlist)
double scl, *vlist;
int base;
XPoint *pts;
{
	register int i, loc;
	static int end = SIZE * 2;
	DBCOM(entering vals_to_curve);
	DBPRT(scl);
	DBPRT(base);

	for(i = 0; i < end; i += 2) {	/* for all loc, val pairs */
		if(vlist[i] >= 0.0) {
			loc = (vlist[i]/(SIZE-1)) * (XMAX-XMIN) + XMIN;
			DBPRT(loc);
			pts[loc].x = loc;
			pts[loc].y = base - (vlist[i+1] / scl);
			DBPRT(pts[loc].y);
		}
	}
	DBCOM(leaving vals_to_curve);
}

reset_list(vlist)
double *vlist;
{
	register int i;
	static int end = SIZE * 2;
	DBCOM(entering reset_list);
	for(i = 0; i < end; i+=2) {
		vlist[i] = -1.0;
		vlist[i+1] = 0.0;
	}

	vlist[0] = 0.0;			/* first pair in list */
	vlist[1] = 0.0;
	vlist[2] = SIZE - 1;		/* last pair in list */
	vlist[3] = 0.0;
	DBCOM(leaving reset_list);
}
