/*  A variation on Hume & Sunday's Tuned Boyer-Moore algorithm
 *  for searching for a literal string pattern in a text.
 *
 *  References:
 *    Hume, A., D.M. Sunday. "Fast String Searching."
 *        _Software--Practice and Experience_. Vol. 21.
 *        No. 11. p. 1221-48. November 1991.
 *
 *  Copyright (c) 1994 Christopher J. Kane.
 *
 *  Version 1.2 (March 22, 1994)
 */

#ifndef _MISC_TBMK_H
#define _MISC_TBMK_H

/* The type Misc_TBMKpattern is a pointer to a structure used to hold
 * a compiled pattern. Create a compiled pattern by using the function
 * Misc_TBMKpattern_alloc(), destroy it with Misc_TBMKpattern_free().
 * The members of the structure should never be accessed directly. */

typedef struct Misc_tbmk_pattern * Misc_TBMKpattern;


/* Creates and compiles a pattern structure from the parameters, and
 * returns a pointer to it. The search is for a fixed-length sequence of
 * characters; the pattern can contain embedded \0 characters. To change
 * the case sensitivity of a search or the direction, the pattern must
 * be recompiled. Returns NULL if 'pattern_length' is zero, or memory
 * couldn't be allocated. Note that if the parameter 'nocases' is true,
 * searches with the returned pattern will be case insensitive. */

Misc_TBMKpattern
Misc_TBMKpattern_alloc(const unsigned char *pattern,
                       unsigned int pattern_length,
                       unsigned char reverse,
                       unsigned char nocases);


/* Destroys a pattern structure created with Misc_TBMKpattern_alloc().
 * Note that a *pointer* to the value returned by that function is
 * passed as the paramter. On return, the value pointed to by 'pattern'
 * has been set to NULL. This function does nothing if 'pattern' or the
 * address 'pattern' points to is NULL, or does not point to a valid
 * Misc_TBMKpattern. */

void
Misc_TBMKpattern_free(Misc_TBMKpattern *pattern);


/* Both of these search function search a source of characters for a
 * literal text as compiled into a Misc_TBMKpattern structure. The first,
 * Misc_TBMKsearch_memory(), uses a contiguous block of memory as the
 * source of characters; the parameter 'text' points to the first
 * character that might possibly be examined by the search. The second
 * function, Misc_TBMKsearch_stream() gets its characters from the
 * NXStream 'stream'; searching begins at the current position of the
 * stream. The search encompasses at most 'text_extent' characters from
 * (and including) the character pointed to by 'text'. 'text_extent' is
 * positive for forward searches, negative for reverse searches (as
 * specified when the pattern was compiled). If 'all_matches' is true,
 * the entire extent is searched and the total number of matches is
 * returned; otherwise, the search stops at the first match. The number
 * of non-overlapping matches is returned (zero if no matches found), and
 * the offset from 'text' of the last match is returned by reference in
 * 'match_position' (for reverse searches, the 'match_position' will be
 * negative). Both functions return -1 if 'pattern' is clearly not a
 * valid pattern, or the sign of 'text_extent' is not valid for the
 * direction of the search specified when the pattern was compiled.
 *
 * Additionally, for Misc_TBMKsearch_stream(): When searching a stream,
 * if the end of stream arrives before 'text_extent' is reached, the
 * function returns as it does when 'text_extent' is reached. Note that
 * since seeks on a stream that is writable extend the size of the
 * stream, the write flag of the stream is turned off while this function
 * is executing; this may impact, for instance, signal handlers which are
 * called during the search. The flag is restored before the function
 * returns or upon the generation of an exception (before an non-handled
 * exeption is propogated outside the function). In addition to the
 * errors named above, -1 is returned if the stream is not readable or
 * not seekable. Exceptions other than NX_illegalSeek are propogated up
 * the function call stack as usual; NX_illegalSeek exceptions cause a
 * normal return. Finally, the function guarentees nothing about the
 * current position of the stream when the function returns. */

int
Misc_TBMKsearch_memory(const Misc_TBMKpattern pattern,
                       const unsigned char *text,
                       int text_extent,
                       unsigned char all_matches,
                       int *match_position);

#if defined(NeXT)
#include <streams/streams.h>

int
Misc_TBMKsearch_stream(const Misc_TBMKpattern pattern,
                       NXStream *stream,
                       int text_extent,
                       unsigned char all_matches,
                       int *match_position);

#endif /* defined(NeXT) */

/* For compatibility with version 1.0 of this module. */
#define Misc_TBMKsearch Misc_TBMKsearch_memory

#endif /* _MISC_TBMK_H */
