/* Grab a copy of the keyboard translation table so we can find the charCode
   for a given keyCode, ignoring the Alternate key.

   For legal stuff see the file COPYRIGHT.  */

#include <dpsclient/dpsclient.h>
#include <drivers/event_status_driver.h>
#include <libc.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/file.h>
#include <sys/types.h>

static char keycode[256][4];
static havekeymap = 0;

/* Given the char mask MASK for a key return the number of chars generated
   by that key.  */
static int
mask_skip (char mask)
{
  int i, skip;

  if (mask == -1)
    return 0;

  for (i = CHAR_BIT, skip = 1; i > 0; i--, mask >>=1)
    if (mask & 01)
      skip <<= 1;
  return skip;
} /* mask_skip */

/* Get the current system keymap and put it into an understandable array
   KEYCODE.  */
void
kc_init (void)
{
  int kc, kcs, mask, skip;
  NXEventHandle handle;
  NXKeyMapping map;
  char evfs, nks;
  char *mapping;

  /* Get the current keymap from the Event Status Driver.  */
  handle = NXOpenEventStatus ();
  if (!handle)
    {
      perror ("Emacs: NXOpenEventStatus");
      return;
    }
  map.size = NXKeyMappingLength (handle);
  map.mapping = malloc (map.size);
  if (!NXGetKeyMapping (handle, &map))
    {
      perror ("Emacs: NXGetKeyMapping");
      return;
    }
  NXCloseEventStatus (handle);

  /* Check mapping format. */
  if (map.mapping[0] || map.mapping[1])
    {
      fprintf(stderr, "Emacs: unknown keyboard map format.\n");
      return;
    }
  mapping = map.mapping + 2;

  /* Skip event flags. */
  for (evfs = *mapping++; evfs > 0; evfs--)
    {
      /* Skip bit number. */
      mapping++;
      /* Number of keys assigned to bit. */
      nks = *mapping++;
      /* Skip key codes. */
      mapping += nks;
    }

  /* kcs = number of key defs.  */
  kcs = (unsigned char) *mapping++;

  for (kc = 0; kc < kcs; kc++)
    {
      /* Get mask */
      mask = *mapping++;
      skip = mask_skip (mask);

      keycode[kc][0] = mapping[1];
      /* Shift key */
      if (mask & 3)
	{
	  /* Assumes only one bit will be set */
	  keycode[kc][1] = mapping[3];
	  /* Shift key & Control key */
	  if (mask & 4)
	    {
	      keycode[kc][2] = mapping[5];
	      keycode[kc][3] = mapping[7];
	    }
	  else
	    {
	      keycode[kc][2] = keycode[kc][0];
	      keycode[kc][3] = keycode[kc][1];
	    }
	}
      else
	{
	  keycode[kc][1] = keycode[kc][0];
	  /* Control key */
	  if (mask & 4)
	    {
	      keycode[kc][2] = keycode[kc][3] = mapping[3];
	    }
	  else
	    {
	      keycode[kc][2] = keycode[kc][3] = keycode[kc][0];
	    }
	}
      /* Skip defs. */
      mapping += 2 * skip;
    }
  havekeymap = 1;
} /* kc_init */

int
kc_keyforcode (int code, int flags)
{
  int modifier = 0;

  if (!havekeymap)
    return -1;

  if (flags & NX_CONTROLMASK)
    modifier += 2;
  if (flags & NX_SHIFTMASK)
    modifier++;

  return keycode[code][modifier];
} /* kc_keyforcode */
