/*
 * Netbufs don't come with nifty queuing functions
 * like mbufs.  So make some simple ones that we will
 * use.  We keep stats on what was dropped from
 * which queue.  This is independent from the stats
 * kept for pppstats.  We may not use them, but
 * then again... we may modify later.
 *
 *
 * Note:  many of these inlines use information private to the data
 * structures and we really shouldn't be peeking at this since the
 * interface might change in the next release (and very likely will),
 * but we at least try to abstract this behind a function...
 */

/*
 * N.B. Not all of these functions "protect" themselves, so look for
 * splimp()/splx() sequence...
 */

#if defined(m68k)
#import "spl.h"
#else
#import <kernserv/machine/spl.h>
#endif
#include <kernserv/kern_server_types.h>
#include <kernserv/kalloc.h>
#include "nbq.h"

/*
 * There is no driver kit for the Moto release.
 */
#ifndef IOLog
#define IOLog printf
#define	IOLogDbg		if (sc->sc_flags & SC_DEBUG) printf
#else
#define	IOLogDbg		if (sc->sc_flags & SC_DEBUG) IOLog
#endif

extern kern_server_t instance;


/*
 * Careful about using this function.  Some places
 * in the code drop packets based on this count
 * but they never free them.
 */

static inline int
nbq_full(struct nb_queue* nbq)
{
    return (nbq->len >= nbq->max);
}

static inline int
nbq_empty(struct nb_queue* nbq)
{
    return nbq->len == 0;
}

static inline int
nbq_low(struct nb_queue* nbq)
{
    return (nbq->len <= nbq->low);
}

static inline int
nbq_high(struct nb_queue* nbq)
{
    return (nbq->len >= nbq->high);
}

static inline netbuf_t
nbq_peek(struct nb_queue* nbq)
{
    int s;
    netbuf_t nb;

    if (nbq->head == NULL)
      return NULL;
    
    s = splimp();
    nb = (nbq->head)->data;
    splx(s);
    return nb;
}

/*
 * N.B. We aren't passed the pointer to the packet we want to delete but
 * rather the address of a pointer to the packet we want to delete...  also
 * note that the packet isn't freed.  The caller must do that.
 */
#if 0
  void
nbq_delete(struct nb_queue *nbq, netbuf_t *nb)
{
    int s = splimp();
    netbuf_t nb2 = *nb;

    if ((*nb = nb2->m_nextpkt) == NULL)
	nbq->tail = (netbuf_t) nb;	/* we count on the pointer being at
					 * at offset 0 in the structure!  hack!!
					 */
    nb2->m_nextpkt = NULL;
    --nbq->len;
    splx(s);
}
#endif
		
static inline netbuf_t
nbq_dequeue(struct nb_queue* nbq)
{
  int s;
  struct netbuf_item *item;
  netbuf_t nb;
  
  
  if (nbq->head == NULL)
      return NULL;
  else
    {
      s = splimp();
      item = nbq->head;
      nbq->head = nbq->head->next;
      if (nbq->head == NULL)
	nbq->tail = NULL;
      nb = item->data;

      if (num_empty_netbuf_items > ENB_MAX_MARK)
	kfree(item, sizeof (struct netbuf_item));
      else
	{
	  item->next = empty_netbuf_items;
	  empty_netbuf_items = item;
	  ++num_empty_netbuf_items;
	}

      --nbq->len;
      splx(s);
    }
  return nb;
}

/*
 * One simple note about nbq_enqueue: it will enqueue packets even if
 * it is full, so the caller is responsible for checking this first...
 *
 * We return 1 if we added, else we return 0
 * if there was a problem. We leave it up to the caller
 * to detect an error return value (0) and print
 * an appropriate message/update stats.  However, in the spirit of
 * keeping the code as close to the netbsd version as is possible,
 * WE WILL FREE a packet that can't be enqueued.  This should be the
 * responsibility of the caller but that is currently not the case.
 *
 * Note that since we may be at interrupt priority
 * we have to be careful when calling
 * topoff_empty_netbuf_item_list()
 *
 * P.S. I refuse to use gotos.  Wouldn't unwind-protect
 *      be nice.
 */

static inline int
nbq_enqueue(struct nb_queue* nbq, netbuf_t nb)
{
  struct netbuf_item *newitem;
  int s, success;
  
  /*
   * Make sure we have plenty of netbuf_items.
   * While there is the potential this will be
   * called many times before the callback
   * ever gets activated, emperical tests
   * show this not to be the case.
   */
  

  if (num_empty_netbuf_items < ENB_LOW_MARK)
    {
      success = kern_serv_callout(&instance,
				  topoff_empty_netbuf_item_list,
				  (void *) NULL);
      if (success != KERN_SUCCESS)
	IOLog("topoff+empty_netubf_item_list kernel server callout failed at %s %d\n",
	       __FILE__, __LINE__);
    }

  s = splimp();
  newitem = (struct netbuf_item *) empty_netbuf_items;
  if (newitem == NULL)
    {
      IOLog("Warning: empty_netbuf_items list at %s %d\n",
	       __FILE__, __LINE__);
      ++nbq->dropped;
      nb_free(nb);
      splx(s);
      return 0;
    }
  
  empty_netbuf_items = empty_netbuf_items->next;
  --num_empty_netbuf_items;
  newitem->data = nb;
  newitem->next = NULL;

  if (nbq->tail)
    nbq->tail->next = newitem;
  else
    nbq->head = newitem;
  nbq->tail = newitem;
  ++nbq->len;
  splx(s);
  return 1;
}

static inline void
nbq_flush(struct nb_queue *nbq)
{
    struct netbuf_item *temp;
    int s = splimp();

  while (nbq->head != NULL)
    {
      nb_free(nbq->head->data);
      temp = nbq->head;
      nbq->head = nbq->head->next;

      if (num_empty_netbuf_items > ENB_MAX_MARK)
	kfree(temp, sizeof(struct netbuf_item));
      else
	{
	  temp->next = empty_netbuf_items;
	  empty_netbuf_items = temp;
	  ++num_empty_netbuf_items;
	}
    }

    nbq->head = nbq->tail = NULL;
    nbq->len = 0;
    nbq->dropped = 0;
    splx(s);
}

/*
 * Must not be called at interrupt priority
 */

static inline void
nbq_init(struct nb_queue *nbq, struct qparms *qp)
{

  if (num_empty_netbuf_items < ENB_LOW_MARK)
    topoff_empty_netbuf_item_list(NULL);
  
  
  nbq->head = nbq->tail = NULL;
  nbq->low = qp->q_low;
  nbq->high = qp->q_high;
  nbq->max = qp->q_max;
  nbq->len = 0;
  nbq->dropped = 0;
}

static inline void
nbq_drop(struct nb_queue *nbq)
{
    ++nbq->dropped;
}

/*
 * Don't use these functions.  They assume the old
 * netbuf filling model.
 */

#if 0

static inline void
nb_write_byte(netbuf_t nb, char b)
{
    int ret;
    if (nb == NULL)
	panic("nb_write_byte");
    nb_grow_bot(nb, 1);
    if ((ret = nb_write(nb, nb_size(nb) - 1, 1, &b)) < 0)
	IOLog("nb_write failed: ret = %x\n", ret);
}

static inline void
nb_write_bytes(netbuf_t nb, char* bs, int n)
{
    int ret;
    if (nb == NULL)
	panic("nb_write_byte");
    nb_grow_bot(nb, n);
    if ((ret = nb_write(nb, nb_size(nb) - n, n, bs)) < 0)
	IOLog("nb_write failed: ret = %x\n", ret);
}

#endif

/*
 * Not very pretty, but it makes for less "diffs"...
 */
#define mtod(m,type)	((type) nb_map(m))

typedef void (*pfv)(void *);

/* used by both ppp_tty.c and if_ppp.c */
static inline kern_return_t
pppsched(pfv func, struct ppp_softc *sc)
{
    extern kern_server_t instance;
    kern_return_t result;

    if ((result = kern_serv_callout(&instance, func, (void *)sc)) != KERN_SUCCESS)
      IOLog("kern_serv_callout failed: ret = %x\n", result);

    return result;
}

#undef	u

static inline thread_t
current_thread(void)
{
	extern thread_t active_threads[];

	return active_threads[0];
}

extern struct proc *proc_from_thread(thread_t);
extern struct uthread *uthread_from_thread(thread_t);

#define	curproc		(proc_from_thread(current_thread()))

#ifdef NBPFILTER
#include <bpf/bpf.h>

extern struct bpf_fns fnarg;

static inline void
bpfattach(caddr_t *driverp, netif_t ifp, u_int dlt, u_int hdrlen)
{
  struct bpf_attachargs atarg = {driverp, (caddr_t) ifp, dlt, hdrlen};

  if (cdevsw[BPF_MAJOR_CHAR].d_ioctl != 0)
    {
      (*cdevsw[BPF_MAJOR_CHAR].d_ioctl)(0, BIOCATTACH, &atarg, 0);
      (*cdevsw[BPF_MAJOR_CHAR].d_ioctl)(0, BIOCGFNS, &fnarg, 0);
    }
}


static inline void
bpf_tap(caddr_t arg, u_char *pkt, u_int pktlen)
{
  (*fnarg.tapfn)(arg, pkt, pktlen);
}
#endif
